/*
MobileRobots Advanced Robotics Navigation and Localization (ARNL)
Version 1.7.0

Copyright (C) 2004, 2005 ActivMedia Robotics LLC
Copyright (C) 2006, 2007, 2008, 2009 MobileRobots Inc.

All Rights Reserved.

MobileRobots Inc does not make any representations about the
suitability of this software for any purpose.  It is provided "as is"
without express or implied warranty.

The license for this software is distributed as LICENSE.txt in the top
level directory.

robots@mobilerobots.com
MobileRobots
10 Columbia Drive
Amherst, NH 03031
800-639-9481

*/
#include "Aria.h"
#include "ArNetworking.h"
#include "ArExport.h"
#include "ArDocking.h"
#include "ArPathPlanningTask.h"
#include "ArLocalizationTask.h"

/* -------- Dock: --------- */

AREXPORT ArServerModeDock *ArServerModeDock::createDock(
	ArServerBase *serverBase, ArRobot *robot, ArLocalizationTask *locTask, 
	ArPathPlanningTask *pathTask, ArFunctor *shutdownFunctor)
{
  if (!robot->isConnected())
  {
    ArLog::log(ArLog::Normal, "ArServerModeDock::createDock: Robot is not connected, cannot create dock");
    return NULL;
  }

  ArServerModeDock *modeDock = NULL;
  const ArRobotConfigPacketReader *origConfig;
  if ((origConfig = robot->getOrigRobotConfig()) != NULL)
  {
    if (strcmp(robot->getRobotName(), "simler") == 0 || 
        strcmp(robot->getRobotName(), "MobileSim") == 0)
    {
      ArLog::log(ArLog::Normal, "Creating simulator dock mode");
      modeDock = new ArServerModeDockSimulator(serverBase, robot, locTask, 
					       pathTask, shutdownFunctor);
    }
    else if (origConfig->getHasCharger() == 3)
    {
      ArLog::log(ArLog::Normal, "Creating patrolbot dock mode");
      modeDock = new ArServerModeDockPatrolBot(serverBase, robot, locTask, 
					       pathTask, shutdownFunctor);
    }
    else if (origConfig->getHasCharger() == 2)
    {
      ArLog::log(ArLog::Normal, "Creating old powerbot dock mode");
      modeDock = new ArServerModeDockPowerBot(serverBase, robot, locTask, 
					      pathTask, true, 
					      shutdownFunctor, false, 0);
    }
    else if (origConfig->getHasCharger() == 4)
    {
      ArLog::log(ArLog::Normal, "Creating new ARCOS powerbot dock mode");
      modeDock = new ArServerModeDockPowerBot(serverBase, robot, locTask, 
					      pathTask, false, 
					      shutdownFunctor, false, 0);
    }
    else if (origConfig->getHasCharger() == 5)
    {
      ArLog::log(ArLog::Normal, "Creating new uARCS powerbot dock mode");
      modeDock = new ArServerModeDockPowerBot(serverBase, robot, locTask, 
					      pathTask, false, 
					      shutdownFunctor, true, 0);
    }
    else if (origConfig->getHasCharger() == 1)
    {
      ArLog::log(ArLog::Normal, "Creating pioneer dock mode");
      modeDock = new ArServerModeDockPioneer(serverBase, robot, locTask, 
					     pathTask, shutdownFunctor);
    }
  }
  return modeDock;
}

AREXPORT ArServerModeDock::ArServerModeDock(ArServerBase *server, 
					    ArRobot *robot, 
					    ArLocalizationTask *locTask, 
					    ArPathPlanningTask *pathTask, 
					    bool useChargeState, 
					    ArFunctor *shutdownFunctor) :
  ArServerMode(robot, server, "dock"),
  myDockUserTaskCB(this, &ArServerModeDock::dockUserTask),
  myServerDockCB(this, &ArServerModeDock::serverDock),
  myServerUndockCB(this, &ArServerModeDock::serverUndock),
  myServerDockInfoCB(this, &ArServerModeDock::serverDockInfo),
  myServerGetAutoDockCB(this, &ArServerModeDock::serverGetAutoDock),
  myServerSetAutoDockCB(this, &ArServerModeDock::serverSetAutoDock),
  myServerAutoDockingDisableCB(this, &ArServerModeDock::setAutoDock, false),
  myServerAutoDockingEnableCB(this, &ArServerModeDock::setAutoDock, true),
  myProcessFileCB(this, &ArServerModeDock::processFile)
{
  myMode = "Dock";
  myLocTask = locTask;
  myPathTask = pathTask;
  myUseChargeState = useChargeState;
  myShutdownFunctor = shutdownFunctor;

  myForcedDockCBList.setName("ArServerModeDock::forcedDockCBList");
  myIdleDockCBList.setName("ArServerModeDock::idleDockCBList");
  myRequestedDockCBList.setName("ArServerModeDock::requestedDockCBList");
  myDockedCBList.setName("ArServerModeDock::dockedCBList");
  myUndockingCBList.setName("ArServerModeDock::undockingCBList");
  myUndockedCBList.setName("ArServerModeDock::undockedCBList");
  myDockNowUnforcedCBList.setName("ArServerModeDock::dockNowUnforcedCBList");
  myDockNowForcedCBList.setName("ArServerModeDock::dockNowForcedCBList");
  mySingleShotDockedCBList.setName("ArServerModeDock::singleShotDockedCBList");
  mySingleShotDockedCBList.setSingleShot(true);

  myRobot->addUserTask("dock", 50, &myDockUserTaskCB);
  if (myServer != NULL)
  {
    addModeData("dock", "sends the robot to the dock", 
			  &myServerDockCB,
			  "none", "none", "Navigation", "RETURN_NONE");
    addModeData("undock", "undocks the robot", &myServerUndockCB,
		"none", "none", "Navigation", "RETURN_NONE");
    myServer->addData("dockInfo", "get info about the docking state", 
			  &myServerDockInfoCB, "none", 
			  "ubyte: 0 undocked, 1 docking, 2 docked, 3 undocking; ubyte: 0 docking not forced, 1 docking forced (can't undock now); ubyte2: shutdown_countdown, 0 means no shutdown imminent, any other value means a shutdown will occur in that many seconds",
		      "NavigationInfo", "RETURN_SINGLE");
    myServer->addData("dockInfoChanged", 
											"Broadcast when the docking state changes", 
			 								NULL, 
											"none", 
			  						  "ubyte: 0 docked, 1 docking, 2 docked, 3 undocking; ubyte: 0 docking not forced, 1 docking forced (can't undock now); ubyte2: shutdown_countdown, 0 means no shutdown imminent, any other value means a shutdown will occur in that many seconds",
		      						"NavigationInfo", 
											"RETURN_SINGLE"); 
    myServer->addData("getAutoDock", "gets whether we auto dock or not",
		      &myServerGetAutoDockCB, "none", 
		      "ubyte: 0 no auto docking, 1 autodocking",
		      "NavigationInfo", "RETURN_SINGLE");
    myServer->addData(
	    "setAutoDock", "Sets whether we auto dock or not",
	    &myServerSetAutoDockCB, 
	    "ubyte: 0 = no autdocking, anything else = auto docking",
	    "none",
	    "Navigation", "RETURN_NONE");
  }
  myDockingVoltage = 0;
  myDockingIdleTime = 5;

  myDoneChargingVoltage = 0;
  if (myUseChargeState)
  {
    myDoneChargingAtFloat = true;
    myDoneChargingMinutes = 0;
  }
  else
  {
    myDoneChargingAtFloat = false;
    myDoneChargingMinutes = 60;
  }
  myMinimumMinutesBetweenAutoDock = 0;
  clearInterrupted();
  myForcedDock = false;
  myLastForcedDock = false;
  myForcedDockRequested = false;
  myIdleDock = false;
  myFailedGotoNum = 0;
  myHandlerCommands = NULL;
  myHaveDocked = false;
  myAutoDock = false;
  myLastAutoDock = false;
  myUsingAutoDock = false;
  myState = UNDOCKED;
  myLastState = UNDOCKED;
  myPreferredDock[0] = '\0';
  checkDock();
  myStartedState.setToNow();
  myProcessFileCB.setName("ArServerModeDock");
  myWasCharging = (myRobot->getFlags() & ArUtil::BIT10);
}

AREXPORT ArServerModeDock::~ArServerModeDock()
{
  myRobot->remUserTask(&myDockUserTaskCB);
}

AREXPORT void ArServerModeDock::addControlCommands(
	ArServerHandlerCommands *handlerCommands)
{
  myHandlerCommands = handlerCommands;
  myHandlerCommands->addCommand(
          "autoDockDisable",
          "This disables the auto docking and undocking",
          &myServerAutoDockingDisableCB);
  myHandlerCommands->addCommand(
          "autoDockEnable",
          "This enables the auto docking and undocking",
          &myServerAutoDockingEnableCB);
}


AREXPORT void ArServerModeDock::dockUserTask(void)
{
  ArMapInterface *arMap;
  if (myIsActive && myState == DOCKED)
  {
    myLastDocked.setToNow();
    myHaveDocked = true;
  } 
  
  if (myIsActive && (myState == DOCKING || myState == UNDOCKING))
  {
    setActivityTimeToNow();
  }
  
  double dockingVoltage;
  // if our docking voltage is set to auto figure it out then do that
  if (!myRobot->haveStateOfCharge())
  {
    if (myDockingVoltage < .1)
    {
      if (myRobot->getOrigRobotConfig() != NULL && 
	  myRobot->getOrigRobotConfig()->hasPacketArrived() && 
	  myRobot->getOrigRobotConfig()->getLowBattery() > .1)
	dockingVoltage = myRobot->getOrigRobotConfig()->getLowBattery();
      else
	dockingVoltage = 12;
    }
    // otherwise just use the one we have
    else
      dockingVoltage = myDockingVoltage;
  }

  // dock if auto docking is enabled and we're below docking voltage
  // and its been at least the minimum time between docks (or we
  // haven't docked)
  if (!myIsActive && myUsingAutoDock && //!myServer->idleProcessingPending() &&
      !(myRobot->getFlags() & ArUtil::BIT10) &&
      ((!myRobot->haveStateOfCharge() && 
	myRobot->getRealBatteryVoltage() < dockingVoltage) || 
       (myRobot->haveStateOfCharge() && 
	myRobot->getStateOfCharge() < myDockingStateOfCharge) || 
       myForcedDockRequested) && 
      (myLastDocked.secSince() > myMinimumMinutesBetweenAutoDock * 60 ||
       !myHaveDocked) && 
      (arMap = myPathTask->getAriaMap()) != NULL)
  {
    if (findDock(arMap) != NULL)
    {
      myForcedDockRequested = false;
      myForcedDock = true;
      activate();
      if (myIsActive)
      {
	lockMode(false);
	if (myForcedDockRequested)
	  ArLog::log(ArLog::Normal,
		     "Docking because forced dock requested");
	else if (!myRobot->haveStateOfCharge())
	  ArLog::log(ArLog::Normal,
		     "Docking because voltage is %.1f (docks at %.1f)",
		     myRobot->getRealBatteryVoltage(), dockingVoltage);
	else 
	  ArLog::log(ArLog::Normal,
	     "Docking because state of charge is %.1f%% (docks at %.1f%%)",
		     myRobot->getStateOfCharge(), myDockingStateOfCharge);
      }

    }
  }

  // if we're docking already but are unforced, see if it becomes
  // forced on the way (otherwise if there's no idle shutdown we may
  // wind up discharging the batteries)
  if (myIsActive && myUsingAutoDock && myState == DOCKING && !myForcedDock  &&
      ((!myRobot->haveStateOfCharge() && 
	myRobot->getRealBatteryVoltage() < dockingVoltage) || 
       (myRobot->haveStateOfCharge() && 
	myRobot->getStateOfCharge() < myDockingStateOfCharge) || 
       myForcedDockRequested))
  {
    myForcedDockRequested = false;
    myForcedDock = true;
    if (myIsActive)
    {
      lockMode(false);
      if (myForcedDockRequested)
	ArLog::log(ArLog::Normal,
		   "Making docking forced because forced dock requested");
      else if (!myRobot->haveStateOfCharge())
	ArLog::log(ArLog::Normal,
		   "Making docking forced because voltage is %.1f (docks at %.1f)",
		   myRobot->getRealBatteryVoltage(), dockingVoltage);
      else 
	ArLog::log(ArLog::Normal,
		   "Making docking forced because state of charge is %.1f%% (docks at %.1f%%)",
		   myRobot->getStateOfCharge(), myDockingStateOfCharge);
    }
  }

  // dock if auto docking is enabled and not docked and we're below 
  // docking voltage and its been at least the minimum time between 
  // docks (or we haven't docked)
  if (!myIsActive &&  myUsingAutoDock && //!myServer->idleProcessingPending() &&
      !(myRobot->getFlags() & ArUtil::BIT10) && myDockingIdleTime != 0 && 
      getActiveMode() != NULL && getActiveMode()->hasSetActivityTime() &&
      getActiveMode()->getActivityTime().secSince() >= myDockingIdleTime * 60 &&
      (myLastDocked.secSince() > myMinimumMinutesBetweenAutoDock * 60 ||
       !myHaveDocked) && 
      (arMap = myPathTask->getAriaMap()) != NULL)
  {
    if (findDock(arMap) != NULL)
    {
      myIdleDock = true;
      ArTime activityTime = getActiveMode()->getActivityTime();
      activate();
      if (isActive())
	
	ArLog::log(ArLog::Normal,
		   "Docking because idle for %d minutes %d seconds (docks at %d minutes idle)",
		   activityTime.secSince() / 60, 
		   activityTime.secSince() % 60, 
		   myDockingIdleTime);
    }
  }
  if (!myUsingAutoDock && myForcedDock)
  {
    myForcedDock = false;
    lockMode(true);
    stateChanged();
  }
  // if we're to a high enough voltage undock or for long enough time
  // or at float
  if (myIsActive && myForcedDock && myState == DOCKED)
  {
    if (myDoneChargingMinutes == -1)
    {
      ArLog::log(ArLog::Normal, 
	 "Done docking because MinutesToChargeFor set to debug value (-1)");
      myForcedDock = false;
      lockMode(true);
    }
    else if (myUseChargeState && myDoneChargingAtFloat && 
	myRobot->getChargeState() == ArRobot::CHARGING_FLOAT)
    {
      ArLog::log(ArLog::Normal, 
	 "Done docking because charging finished (got to float)");
      myForcedDock = false;
      lockMode(true);
    }
    else if (!myRobot->haveStateOfCharge() && myDoneChargingVoltage > 0.1 && 
	     myRobot->getRealBatteryVoltage() > myDoneChargingVoltage)
    {
      ArLog::log(ArLog::Normal, 
		 "Done docking because voltage is %.1f (finishes at %.1f)", 
		 myRobot->getRealBatteryVoltage(), myDoneChargingVoltage);
      myForcedDock = false;
      lockMode(true);
    }
    else if (myRobot->haveStateOfCharge() && 
	     myDoneChargingStateOfCharge > 0.1 && 
	     myRobot->getStateOfCharge() > myDoneChargingStateOfCharge)
    {
      ArLog::log(ArLog::Normal, 
		 "Done docking because state of charge is %.1f%% (finishes at %.1f%%)", 
		 myRobot->getStateOfCharge(), myDoneChargingStateOfCharge);
      myForcedDock = false;
      lockMode(true);
    }
    else if (myDoneChargingMinutes > 0 && 
	     myStartedState.secSince() > myDoneChargingMinutes * 60)
    {
      ArLog::log(ArLog::Normal, 
		 "Done docking because docked for %d minutes", 
		 myDoneChargingMinutes);
      myForcedDock = false;
      lockMode(true);
    }
    // now see if we should undock
    if (!myForcedDock)
    {
      // first just change the string so its clear we're not forced anymore
      myStatus = "Docked";
      stateChanged();
      resumeInterrupted(false);
    }
  }

  if (myShutdownFunctor != NULL && myIsActive && myState == DOCKING)
  {
    int secSince;

    if (myShutdownLastPose.squaredFindDistanceTo(myRobot->getPose()) > 20 * 20)
    {
      myShutdownLastPose = myRobot->getPose();
      myShutdownLastMove.setToNow();
      //printf("1\n");
      if (myShuttingDownSeconds > 0)
      {
	ArLog::log(ArLog::Normal, "ServerModeDock: Robot moved, shutdown cancelled");
	myShuttingDownSeconds = 0;
      }
    }
    else if ((myForcedDock && myShutdownMinutesForced > 0 && 
	      myShutdownLastMove.secSince() >= 
	       myShutdownMinutesForced * 60 + 30) ||
	     (!myForcedDock && myShutdownMinutesIdle > 0 && 
	      myShutdownLastMove.secSince() >= 
	       myShutdownMinutesIdle * 60 + 30))
    {
      //printf("2\n");
      if (myForcedDock)
	ArLog::log(ArLog::Normal, "ServerModeDock: Shutting down since have not made progress towards the dock (forced) in %d minutes", myShutdownMinutesForced);
      else
	ArLog::log(ArLog::Normal, "ServerModeDock: Shutting down since have not made progress towards the dock (idle) in %d minutes", myShutdownMinutesIdle);
      myShutdownFunctor->invoke();
    }
    else if ((secSince = myShutdownLastMove.secSince()) > 30)
    {
      //printf("3 %d %d %d\n", myForcedDock, secSince, myShuttingDownSeconds);
      bool wasShuttingDown = (myShuttingDownSeconds != 0);
      if (myForcedDock && myShutdownMinutesForced > 0)
	myShuttingDownSeconds = myShutdownMinutesForced * 60 + 30 - secSince;
      else if (!myForcedDock && myShutdownMinutesIdle > 0)
	myShuttingDownSeconds = myShutdownMinutesIdle * 60 + 30 - secSince;
      if (!wasShuttingDown && myShuttingDownSeconds > 0)
	ArLog::log(ArLog::Normal, "ServerModeDock: Robot has not moved in 30 seconds while trying to dock (%s), shutting down in %d seconds", 
		   myForcedDock ? "forced" : "idle",
		   myShuttingDownSeconds);
    }
    //printf("4 %d\n", myShuttingDownSeconds);
    if (myShuttingDownSeconds != myLastShuttingDownSeconds)
      broadcastDockInfoChanged();
    myLastShuttingDownSeconds = myShuttingDownSeconds;
  }


  // now we see if we're undocked but our robot started charging suddently, if
  // so we call checkDock
  if (myState == UNDOCKED && !myWasCharging && 
      (myRobot->getFlags() & ArUtil::BIT10))
  {
    checkDock();
  }
  
  myWasCharging = (myRobot->getFlags() & ArUtil::BIT10);

}

AREXPORT void ArServerModeDock::serverDock(ArServerClient *client, ArNetPacket * /*packet*/)
{
  ArLog::log(ArLog::Normal, "Docking for %s", client->getIPString());
  myRobot->lock();
  dock();
  myRobot->unlock();
}

AREXPORT void ArServerModeDock::serverUndock(ArServerClient *client, ArNetPacket * /*packet*/)
{
  ArLog::log(ArLog::Normal, "Undocking for %s", client->getIPString());
  myRobot->lock();
  undock();
  myRobot->unlock();
}

AREXPORT void ArServerModeDock::serverDockInfo(ArServerClient *client, ArNetPacket * /*packet*/)
{
  ArNetPacket send;
  myRobot->lock();
  makeDockInfoPacket(&send);
  myRobot->unlock();
  client->sendPacketTcp(&send);
}


AREXPORT void ArServerModeDock::serverGetAutoDock(ArServerClient *client, ArNetPacket * /*packet*/ )
{
  ArNetPacket send;
  myRobot->lock();
  if (myUsingAutoDock)
    send.uByteToBuf(1);
  else
    send.uByteToBuf(0);
  myRobot->unlock();
  client->sendPacketTcp(&send);
}

AREXPORT void ArServerModeDock::serverSetAutoDock(ArServerClient *client, ArNetPacket *packet)
{
  ArNetPacket send;
  bool autoDock;
  if (packet->bufToUByte() == 0)
  {
    ArLog::log(ArLog::Normal, "Clearing autodock for %s", 
	       client->getIPString());
    autoDock = false;
  }
  else
  {
    ArLog::log(ArLog::Normal, "Setting autodock for %s", 
	       client->getIPString());
    autoDock = true;
  }
  myRobot->lock();
  myUsingAutoDock = autoDock;
  myRobot->unlock();
}

AREXPORT void ArServerModeDock::activate()
{
  bool idleDock = myIdleDock;
  myIdleDock = false;
  if (myIsActive)
    return;

  ArServerMode *activeMode = getActiveMode();
  ArServerMode *lastActiveMode = getLastActiveMode();

  if (activeMode == NULL && lastActiveMode == ArServerMode::getIdleMode())
  {
    ArLog::log(ArLog::Normal, "DOCK: activeMode was NULL, but ourLastActiveMode was idle, so using that");
    activeMode = lastActiveMode;
  }

  if (activeMode != NULL)
    ArLog::log(ArLog::Normal, "DOCK: %s was activeMode", 
	       activeMode->getName());
  if (activeMode != NULL && activeMode == ArServerMode::getIdleMode())
  {
    activeMode = ArServerMode::getIdleMode()->getModeInterrupted();
    if (activeMode != NULL)
      ArLog::log(ArLog::Normal, "DOCK: Made %s new activeMode", 	       
		 activeMode->getName());
    else
      ArLog::log(ArLog::Normal, "DOCK: NULL new activeMode...");
  }

  if (myForcedDock && activeMode != NULL)
  {
    //printf("Interrupted %s\n", getActiveMode()->getName());
    if (activeMode->isAutoResumeAfterInterrupt()) 
    {
      myModeInterrupted = activeMode;
      ArLog::log(ArLog::Normal, "DOCK: Made %s new modeInterrupted",
		 myModeInterrupted->getName());
    }
  }
  else
  {
    clearInterrupted();
  }
  if (!baseActivate())
  {
    ArLog::log(ArLog::Normal, "DOCK: Couldn't activate, clearing interrupted");
    clearInterrupted();
    return;
  }

  lockMode(true);
  if (myState != DOCKED)
  {
    if (myForcedDock)
      myForcedDockCBList.invoke();
    else if (idleDock)
      myIdleDockCBList.invoke();
    else 
      myRequestedDockCBList.invoke();
    dock();
  }
}

AREXPORT void ArServerModeDock::deactivate()
{
  if (myState == UNDOCKED)
  {
    myForcedDock = false;
    myShuttingDownSeconds = 0;
    broadcastDockInfoChanged();
    baseDeactivate();
  }
  else
    undock();
}

AREXPORT void ArServerModeDock::requestUnlock()
{
  if (!myForcedDock || myState == UNDOCKED)
    deactivate();
}

AREXPORT void ArServerModeDock::forceUnlock(void)
{
  myState = UNDOCKED;
  ArServerMode::forceUnlock();
}

void ArServerModeDock::switchState(State state)
{
  State oldState = myState;
  myState = state;
  myStartedState.setToNow();
  if (oldState != myState)
  {
    if (myState == DOCKED)
    {
      myDockedCBList.invoke();
      mySingleShotDockedCBList.invoke();
    }
    else if (myState == UNDOCKING)
      myUndockingCBList.invoke();
    else if (myState == UNDOCKED)
      myUndockedCBList.invoke();
  }

  if (myShutdownFunctor != NULL && myState == DOCKING && 
      myShutdownLastPose.squaredFindDistanceTo(myRobot->getPose()) > 20 * 20)
  {
    myShutdownLastPose = myRobot->getPose();
    myShutdownLastMove.setToNow();
  }
  
  stateChanged();
}


AREXPORT void ArServerModeDock::setDockingVoltage(double dockingVoltage)
{ 
  myDockingVoltage = dockingVoltage;
}

AREXPORT double ArServerModeDock::getDockingVoltage(void) const
{ 
  return myDockingVoltage; 
}

AREXPORT void ArServerModeDock::setDoneChargingVoltage(double doneChargingVoltage)
{ 
  myDoneChargingVoltage = doneChargingVoltage; 
}

AREXPORT double ArServerModeDock::getDoneChargingVoltage(void) const
{ 
  return myDoneChargingVoltage; 
}

AREXPORT void ArServerModeDock::setDoneChargingMinutes(int doneChargingMinutes)
{ 
  myDoneChargingMinutes = doneChargingMinutes;
}

AREXPORT int ArServerModeDock::getDoneChargingMinutes(void)
{ 
  return myDoneChargingMinutes; 
}

AREXPORT void ArServerModeDock::setMinimumMinutesBetweenAutoDock(
	int minutesBetween)
{ 
  myMinimumMinutesBetweenAutoDock = minutesBetween;
}

AREXPORT int ArServerModeDock::getMinimumMinutesBetweenAutoDock(void)
{ 
  return myMinimumMinutesBetweenAutoDock;
}

AREXPORT bool ArServerModeDock::getUseChargeState(void)
{
  return myUseChargeState;
}

AREXPORT void ArServerModeDock::setDoneChargingAtFloat(
	bool doneChargingAtFloat)
{
  myDoneChargingAtFloat = doneChargingAtFloat;
}

AREXPORT bool ArServerModeDock::getDoneChargingAtFloat(void)
{
  return myDoneChargingAtFloat;
}



AREXPORT void ArServerModeDock::setAutoDock(bool autoDocking) 
{ 
  if (myUsingAutoDock != autoDocking)
  {
    if (autoDocking)
      ArLog::log(ArLog::Normal, "Enabling autodocking");
    else
      ArLog::log(ArLog::Normal, "Disabling autodocking");
  }
  myUsingAutoDock = autoDocking; 
}

AREXPORT bool ArServerModeDock::getAutoDock(void)
{ 
  return myUsingAutoDock; 
}

AREXPORT bool ArServerModeDock::processFile(void)
{
  if (myAutoDock != myLastAutoDock)
    myUsingAutoDock = myAutoDock;
  myLastAutoDock = myAutoDock;
  if (myDockName.empty())
  {
    ArMapInterface *arMap;
    arMap = myPathTask->getAriaMap();
    ArMapObject *dockObject = NULL;
    dockObject = findDock(arMap);
    if (dockObject != NULL)
      myDockName = dockObject->getName();
    else 
      myDockName = myPreferredDock;
  }
  return true;
}

/**
   This call will have the server mode do a forced dock
**/ 
AREXPORT void ArServerModeDock::requestForcedDock(void)
{
  if (!myIsActive)
  {
    myForcedDockRequested = true;
    myUsingAutoDock = true;
  }
  else
  {
    myForcedDock = true;
    lockMode(false);
    myUsingAutoDock = true;
  }
}

AREXPORT void ArServerModeDock::addToConfig(ArConfig *config)
{
  std::string section;
  section = "Docking";

  if (config == NULL)
    return;
  
  char displayHint[2048];
  if (myDockType.empty())
    sprintf(displayHint, "MapItem:Dock|Parent|Optional");
  else
    sprintf(displayHint, "MapItem:%s|SubTypeAndParent|Optional", 
	   myDockType.c_str());

  config->addParam(
	  ArConfigArg("AutoDock", &myAutoDock, 
		      "True if we should automatically dock"), 
	  section.c_str(), ArPriority::NORMAL);

  if (myRobot->haveStateOfCharge())
    config->addParam(
	  ArConfigArg("AutoDockStateOfCharge", &myDockingStateOfCharge, 
		      "If auto docking the robot state of charge to automatically dock at", 0, 100),
	  section.c_str(), ArPriority::NORMAL);
  else
    config->addParam(
	  ArConfigArg("AutoDockVoltage", &myDockingVoltage, 
		      "If auto docking the robot voltage to automatically dock at (0 will use the lowBatteryVoltage from the firmware if it exists, if it does not exist it will use 12 volts)", 0),
	  section.c_str(), ArPriority::NORMAL);
  config->addParam(
	  ArConfigArg("AutoDockIdleTime", &myDockingIdleTime, 
		      "If auto docking the idle time in minutes until auto dock (0 means never auto dock because of idle)", 0),
	  section.c_str(), ArPriority::NORMAL);
  config->addParam(
	  ArConfigArg("PreferredDock", myPreferredDock,
		      "If there is a charger with this name this robot will use that dock, if not the default behavior of using the first dock in the map applies",
		      sizeof(myPreferredDock)),
	  section.c_str(), ArPriority::NORMAL, displayHint);
  if (myUseChargeState)
    config->addParam(
	    ArConfigArg("DockUntilDoneCharging", &myDoneChargingAtFloat,
			"If true then the robot will dock until it is done charging (in float mode)"),
	    section.c_str(), ArPriority::NORMAL);
  config->addParam(
	  ArConfigArg("MinutesToChargeFor", &myDoneChargingMinutes,
		      "If 0 don't finish charging based on time, if -1 then just undock as soon as docked (mostly for testing) if greater than 0 then be done charging after this many minutes", 
		      -1),
	  section.c_str(), ArPriority::NORMAL);

  if (myRobot->haveStateOfCharge())
    config->addParam(
	    ArConfigArg("StateOfChargeToChargeTo", 
			&myDoneChargingStateOfCharge,
			"If 0 don't finish charging based on sate of charge, if greater than 0 then be done charging after robot's state of charge reachs this point", 
			0, 100),
	    section.c_str(), ArPriority::NORMAL);
  else
    config->addParam(
	    ArConfigArg("VoltageToChargeTo", &myDoneChargingVoltage,
			"If 0 don't finish charging based on voltage, if greater than 0 then be done charging after robot's voltage reachs this point", 
			0),
	    section.c_str(), ArPriority::NORMAL);

  config->addParam(
	  ArConfigArg("MinutesBetweenAutoDock",
		      &myMinimumMinutesBetweenAutoDock,
		      "If 0 don't limit how often the robot can charge, if greater than 0 then this many minutes will elapse between when the robot last docked and when it will autodock",
		      0),
	  section.c_str(), ArPriority::TRIVIAL);

  if (myShutdownFunctor != NULL)
  {
    config->addParam(
	    ArConfigArg("AutoDockShutdownMinutes",
			&myShutdownMinutesForced,
			"This controls how long the robot should wait before shutting down if it cannot move on its way to the dock while in a low battery situation.  0 means do not automatically shutdown for this reason, any other value is how many minutes to wait before shutting down",
			0, 60),
	    section.c_str(), ArPriority::TRIVIAL);
    config->addParam(
	    ArConfigArg("UnforcedDockShutdownMinutes",
			&myShutdownMinutesIdle,
			  "This controls how long the robot should wait before shutting down if it cannot move on its way to the dock if docking because the robot was idle or an explicit dock request was made.  0 means do not automatically shutdown for this reason, any other value is how many minutes to wait before shutting down",
			0, 60),
	      section.c_str(), ArPriority::TRIVIAL);
  }
  config->addProcessFileCB(&myProcessFileCB, 50);
}

AREXPORT ArMapObject *ArServerModeDock::findDock(ArMapInterface *arMap)
{
  ArMapObject *dockObject = NULL;
  arMap->lock();
  if (myPreferredDock[0] != '\0' && !myDockType.empty())
    dockObject = arMap->findMapObject(myPreferredDock, myDockType.c_str());
  if (dockObject == NULL && myPreferredDock[0] != '\0')
    dockObject = arMap->findMapObject(myPreferredDock, "Dock");
  if (dockObject == NULL && !myDockType.empty())
    dockObject = arMap->findMapObject(NULL, myDockType.c_str());
  if (dockObject == NULL)
    dockObject = arMap->findMapObject(NULL, "Dock");
  arMap->unlock();
  return dockObject;
}

AREXPORT void ArServerModeDock::clearInterrupted(void)
{
  myModeInterrupted = NULL;
  ArLog::log(ArLog::Normal, "DOCK: Clearing modeInterrupted");
}

AREXPORT void ArServerModeDock::resumeInterrupted(bool assureDeactivation)
{
  ArLog::log(ArLog::Normal, "DOCK: Resume interrupted start %d", assureDeactivation);
  std::list<ArServerMode *> *requestedActivateModes = getRequestedActivateModes();

  // if something else wanted to activate we can just deactivate and 
  // that'll get activated
  if (isActive() && 
      (requestedActivateModes != NULL) &&
      requestedActivateModes->begin() != requestedActivateModes->end())
  {
    deactivate();
    if (getActiveMode() != NULL)
      ArLog::log(ArLog::Normal, 
		 "DOCK: Resume interrupted deactivating and returning... %s got activated", 
		 getActiveMode()->getName());
    else
      ArLog::log(ArLog::Normal, 
		 "DOCK: Resume interrupted deactivating and returning... nothing active...");
    return;
  }
  // if we're still active and we interrupted something when we
  // activated then start that up again, if it was stop we interrupted
  // just stay at the dock

  if (myModeInterrupted != NULL)
    ArLog::log(ArLog::Normal, "DOCK: Resume interrupted another... isActive %d modeInterrupted %p %s", 
	       isActive(), myModeInterrupted, myModeInterrupted->getName());
  else
    ArLog::log(ArLog::Normal, "DOCK: Resume interrupted another... isActive %d modeInterrupted NULL", isActive());


  if (isActive() && myModeInterrupted != NULL && 
      strcmp(myModeInterrupted->getName(), "stop") != 0)
  {
    ArLog::log(ArLog::Normal, "DOCK: Trying to activate interrupted mode %s", 
	       myModeInterrupted->getName());
    myModeInterrupted->activate();
    myModeInterrupted = NULL;
    if (getActiveMode() != NULL)
      ArLog::log(ArLog::Normal, 
		 "DOCK: Did activate mode %s", 
		 getActiveMode()->getName());
    return;
  }

  ArLog::log(ArLog::Normal, "DOCK: Resume interrupted later");
  
  // if we're supposed to assure deactivation and we're still here
  // then deactivate
  if (isActive() && assureDeactivation)
  {
    ArLog::log(ArLog::Normal, "DOCK: Deactivating");
    deactivate();
    return;
  }
  ArLog::log(ArLog::Normal, "DOCK: Resume interrupted end");
}

AREXPORT void ArServerModeDock::addStateChangedCB(ArFunctor *functor, 
						 ArListPos::Pos position)
{
  if (position == ArListPos::FIRST)
    myStateChangedCBList.push_front(functor);
  else if (position == ArListPos::LAST)
    myStateChangedCBList.push_back(functor);
  else
    ArLog::log(ArLog::Terse, 
	       "ArServerModeDock::addStateChangeCB: Invalid position.");
}

AREXPORT void ArServerModeDock::remStateChangedCB(ArFunctor *functor)
{
  myStateChangedCBList.remove(functor);
}

void ArServerModeDock::stateChanged(void)
{
  // if we were and are docked see if forced toggled
  if (myLastState == DOCKED && myState == DOCKED && 
      myLastForcedDock != myForcedDock)
  {
    if (myForcedDock)
      myDockNowForcedCBList.invoke();
    else
      myDockNowUnforcedCBList.invoke();
  }
  myLastForcedDock = myForcedDock;
  myLastState = myState;

  std::list<ArFunctor *>::iterator it;
  for (it = myStateChangedCBList.begin(); 
       it != myStateChangedCBList.end(); 
       it++)
    (*it)->invoke();

  broadcastDockInfoChanged();
}

void ArServerModeDock::broadcastDockInfoChanged(void)
{
  // Think that the robot is already locked at this point.... (so it's 
  // not necessary here)
  if (myServer != NULL) 
  {
    ArNetPacket send;
    makeDockInfoPacket(&send);
    myServer->broadcastPacketTcp(&send, "dockInfoChanged");
  }
}

void ArServerModeDock::makeDockInfoPacket(ArNetPacket *packet)
{
  packet->empty();
  packet->uByteToBuf(myState);
  packet->uByteToBuf(myForcedDock);
  if (myShutdownFunctor == NULL)
    packet->uByte2ToBuf(0);
  else 
    packet->uByte2ToBuf(myShuttingDownSeconds);
}

AREXPORT void ArServerModeDock::activateAsDocked(void)
{
  myStatus = "Docked";
  switchState(DOCKED);
  activate();
}

AREXPORT ArServerModeDockPioneer::ArServerModeDockPioneer(
	ArServerBase *server, ArRobot *robot, 
	ArLocalizationTask *locTask, ArPathPlanningTask *pathTask, 
	ArFunctor *shutdownFunctor) :
  ArServerModeDock(server, robot, locTask, pathTask, true, shutdownFunctor),
  myGroup(robot),
  myGoalDoneCB(this, &ArServerModeDockPioneer::goalDone),
  myGoalFailedCB(this, &ArServerModeDockPioneer::goalFailed)
{
  myPathTask->addGoalDoneCB(&myGoalDoneCB);
  myPathTask->addGoalFailedCB(&myGoalFailedCB);

  myGroup.addAction(new ArActionLimiterForwards("limiter", 150, 0, 0, 1.3),
		    70);
  myGroup.addAction(new ArActionLimiterBackwards,
		   69);

  myDriveTo.setMaxLateralDist(500);
  myDriveTo.setMaxDistBetweenLinePoints(50);
  myGroup.addAction(&myDriveTo, 60);

  myGroup.addAction(new ArActionStop, 50);
  myDockType = "DockPioneer";
  myNeedToPathPlanToDock = false;
}

AREXPORT ArServerModeDockPioneer::~ArServerModeDockPioneer()
{

}

AREXPORT void ArServerModeDockPioneer::userTask(void)
{
  bool printing = false;
  if (printing)
    printf("State %d\n", myState);
  if (myState == DOCKED)
  {
    if (!(myRobot->getFlags() & ArUtil::BIT10))
    {
      //myStatus = "Dock failed";
      //switchState(UNDOCKED);
      if (mySentDockCommandTime.secSince() > 10)
      {
	myFindingDockTry++;
	if (myFindingDockTry == 2)
	{
	  ArLog::log(ArLog::Normal, "Moving forwards to get dock back");
	  myNeedToResendDockCommand = true;
	  myRobot->move(25);
	  myRobot->enableMotors();
	  mySentDockCommandTime.setToNow();
	}
	if (myFindingDockTry == 3)
	{
	  ArLog::log(ArLog::Normal, "Moving backwards to get dock back");
	  myRobot->enableMotors();
	  myRobot->move(-50);
	  myNeedToResendDockCommand = true;
	  mySentDockCommandTime.setToNow();
	}
	if (myFindingDockTry > 3)
	{
	  ArLog::log(ArLog::Normal, 
		     "Failed dock: Failed 3 tries to dock");
	  myStatus = "Failed Dock";
	  undock();
	}
      }
      else if (myNeedToResendDockCommand && 
	       mySentDockCommandTime.secSince() > 2)
      {
	ArLog::log(ArLog::Normal, "Resending dock command");
	myNeedToResendDockCommand = false;
	myRobot->comInt(68, 1);
      }
    }

  }
  if (myState == DOCKING)
  {
    if (printing)
      printf("Docking... %ld %d\n", myStartedDriveTo.secSince(),
	     myRobot->getFlags() & ArUtil::BIT10);
    // if we're docked
    if (mySentDockCommand && (myRobot->getFlags() & ArUtil::BIT10))
    {
      ArLog::log(ArLog::Normal, "Docked");
      if (myForcedDock)
	myStatus = "Docked because of low voltage";
      else
	myStatus = "Docked";
      switchState(DOCKED);
    }
    // if we sent the command but it didn't work
    else if (mySentDockCommand && mySentDockCommandTime.secSince() > 10)
    {
      myFindingDockTry++;
      if (myFindingDockTry == 2)
      {
	ArLog::log(ArLog::Normal, "Moving forwards to find dock");
	myNeedToResendDockCommand = true;
	myRobot->move(25);
	myRobot->enableMotors();
	mySentDockCommandTime.setToNow();
      }
      if (myFindingDockTry == 3)
      {
	ArLog::log(ArLog::Normal, "Moving backwards to find dock");
	myRobot->enableMotors();
	myRobot->move(-50);
	myNeedToResendDockCommand = true;
	mySentDockCommandTime.setToNow();
      }
      if (myFindingDockTry > 3)
      {
	ArLog::log(ArLog::Normal, 
		   "Dock Failed: Failed 3 tries to dock");
	undock();
      }
    }
    else if (mySentDockCommand && myNeedToResendDockCommand && 
	     mySentDockCommandTime.secSince() > 2)
    {
      ArLog::log(ArLog::Normal, "Resending dock command");
      myNeedToResendDockCommand = false;
      myRobot->comInt(68, 1);
    }
    // if we sent the dock command just chill
    else if (mySentDockCommand)
    {

    }
    // if we're there and haven't sent the command
    else if (myDriveTo.isActive() && 
	     myDriveTo.getState() == ArActionTriangleDriveTo::STATE_SUCCEEDED)
    {
      if (printing)
	printf("Sent dock\n");
      myStatus = "Trying to dock";
      mySentDockCommand = true;
      myRobot->comInt(68, 1);
      mySentDockCommandTime.setToNow();
      ArLog::log(ArLog::Normal, "Sending dock command");
    }
    // if we failed getting there
    else if (myDriveTo.isActive() && 
	     myDriveTo.getState() == ArActionTriangleDriveTo::STATE_FAILED)
    {
      ArLog::log(ArLog::Normal, 
		 "Dock Failed: Could not find dock target");
      undock();
    }
    // if it took longer than 30 seconds to drive in
    else if (myDriveTo.isActive() && myStartedDriveTo.secSince() > 30)
    {
      ArLog::log(ArLog::Normal, 
		 "Dock Failed: Took too long to find target");
      myStatus = "Failed dock";
      undock();
    }
    // if we've gotten to our goal and need to drive into the dock
    else if (myGoalDone && !myDriveTo.isActive())
    {
      ArLog::log(ArLog::Normal, "Driving to docking triangle");
      myGroup.activateExclusive();
      myGoalDone = false;
      myStartedDriveTo.setToNow();
      myDrivingIntoDockCBList.invoke();
    }
  }
  else if (myState == UNDOCKING)
  {
    if (printing)
      printf("Undocking %ld\n", myStartedState.secSince());
    if (myStartedState.secSince() >= 4)
    {
      myRobot->stop();
      ArLog::log(ArLog::Normal, "Undocked");
      switchState(UNDOCKED);
      resumeInterrupted(true);
    }
    else if (myStartedState.secSince() >= 2 && !myUndockingMoveSent)
    {
      myUndockingMoveSent = true;
      myRobot->enableMotors();      
      myRobot->move(-25);
    }
  }
  else if (myState == UNDOCKED)
  {
    if (myNeedToPathPlanToDock && !myServer->idleProcessingPending())
    {
      myGoalDone = false;
      myDriveTo.deactivate();
      mySentDockCommand = false;
      myNeedToResendDockCommand = false;
      myNeedToPathPlanToDock = false;

      ArMapInterface *arMap;
      ArMapObject *dockObject = NULL;
      arMap = myPathTask->getAriaMap();
      
      dockObject = findDock(arMap);
      
      if (dockObject == NULL || 
	  !myPathTask->pathPlanToPose(dockObject->getPose(), true))
      {
	myStatus = "Failed dock";
	resumeInterrupted(false);
	// no callback here since this'll only fail if the path planning
	// is completely hosed (if there's no path the goal failed will be
	// called, not this)
	return;
      }
      //myStatus = "Driving to dock";
      myDockName = dockObject->getName();
      myStatus = "Going to dock at ";
      myStatus += myDockName;
      ArLog::log(ArLog::Normal, "Docking at %s", myDockName.c_str());
      myFindingDockTry = 1;
      switchState(DOCKING);
      myDrivingToDockCBList.invoke();
    }
    else
    {
      myStatus = "Waiting to dock";
    }

  }
}

AREXPORT void ArServerModeDockPioneer::dock(void)
{
  if (myIsActive && myState == DOCKED)
    return;

  activate();
  if (!myIsActive || myState != UNDOCKED)
    return;
  myNeedToPathPlanToDock = true;
  switchState(UNDOCKED);
  /*
  myGoalDone = false;
  myDriveTo.deactivate();
  mySentDockCommand = false;
  myNeedToResendDockCommand = false;

  ArMapInterface *arMap;
  ArMapObject *dockObject = NULL;
  arMap = myPathTask->getAriaMap();

  dockObject = findDock(arMap);

  if (dockObject == NULL || 
      !myPathTask->pathPlanToPose(dockObject->getPose(), true))
  {
    myStatus = "Failed dock";
    resumeInterrupted(false);
    // no callback here since this'll only fail if the path planning
    // is completely hosed (if there's no path the goal failed will be
    // called, not this)
    return;
  }
  myStatus = "Driving to dock";
  ArLog::log(ArLog::Normal, "Docking");
  myDockName = dockObject->getName();
  myFindingDockTry = 1;
  switchState(DOCKING);
  myDrivingToDockCBList.invoke();
  */
}

AREXPORT void ArServerModeDockPioneer::undock(void)
{
  if (myState == DOCKED && myForcedDock)
    return;
  if (myState == UNDOCKED)
    deactivate();
  else if (myState == DOCKED)
  {
    myStatus = "Undocking";
    myUndockingMoveSent = false;
    switchState(UNDOCKING);
    ArLog::log(ArLog::Normal, "Undocking");
    myRobot->comInt(68, 0);
  }
  else if (myState == DOCKING)
  {
    ArLog::log(ArLog::Normal, "Undocking");
    if (mySentDockCommand)
    {
      myStatus = "Undocking";
      myUndockingMoveSent = false;
      switchState(UNDOCKING);
      myRobot->comInt(68, 0);
    }
    else
    {
      switchState(UNDOCKED);
      ArLog::log(ArLog::Normal, "Undocked");
      resumeInterrupted(true);
    }
  }
  // if we're already undocking ignore this
  else if (myState == UNDOCKING)
  {
  }
    

}

AREXPORT void ArServerModeDockPioneer::deactivate()
{
  myNeedToPathPlanToDock = false;
  ArServerModeDock::deactivate();
}

AREXPORT void ArServerModeDockPioneer::checkDock(void)
{
  if (myRobot->isConnected() && (myRobot->getFlags() & ArUtil::BIT10))
  {
    ArLog::log(ArLog::Normal, "Robot is already docked, activating dock mode");
    myStatus = "Docked";
    switchState(DOCKED);
    activate();
    clearInterrupted();
  }
}

AREXPORT void ArServerModeDockPioneer::forceUnlock(void)
{
  myRobot->comInt(68, 0);
  ArUtil::sleep(10);
  myState = UNDOCKED;
  ArServerMode::forceUnlock();
}

AREXPORT void ArServerModeDockPioneer::goalDone(ArPose /*pose*/)
{
  if (!myIsActive)
    return;
  myStatus = "Driving into dock";
  myGoalDone = true;
}

AREXPORT void ArServerModeDockPioneer::goalFailed(ArPose /*pose*/)
{
  if (!myIsActive || myState != DOCKING)
    return;
  gotoFailed();
  resumeInterrupted(false);
  switchState(UNDOCKED);
}

/**
   @param server the server where we put the docking commands

   @param robot the robot we're using

   @param locTask the task for localization (so we can say its okay to
   be in a wall)

   @param pathTask the task for driving around

   @param useChargeState whether to use the charge state to know when
   docking or not

   @param approachDist distance in front of triangle vertex to use as
   the approach point (before the final drive in)

   @param backOutDist distance to back out, 0 means use old method
   which goes out as far as it had to drive in
 **/
AREXPORT ArServerModeDockTriangleBump::ArServerModeDockTriangleBump(
	ArServerBase *server, ArRobot *robot, 
	ArLocalizationTask *locTask, ArPathPlanningTask *pathTask,
	bool useChargeState, double approachDist, double backOutDist, 
	ArFunctor *shutdownFunctor) :
  ArServerModeDock(server, robot, locTask, pathTask, useChargeState,
		   shutdownFunctor),
  myGroup(robot),
  myDriveTo("triangleDriveTo", 0, approachDist, 100, 100, 30),
  myBackGroup(robot),
  myGoalDoneCB(this, &ArServerModeDockTriangleBump::goalDone),
  myGoalFailedCB(this, &ArServerModeDockTriangleBump::goalFailed)
{
  myPathTask->addGoalDoneCB(&myGoalDoneCB);
  myPathTask->addGoalFailedCB(&myGoalFailedCB);
  myStallsAsBumps = false;

  myDisableDockCalled = true;
  myHitDock = false;
  myDriveFromValid = false;
  myHitDock = false;

  myDriveTo.setGotoVertex(true);
  myDriveTo.setMaxLateralDist(500);
  myDriveTo.setMaxDistBetweenLinePoints(50);
  myGroup.addAction(&myDriveTo, 60);
  myGroup.addAction(new ArActionStop, 50);
  
  myDesiredBackOutDist = backOutDist;
  myBackGroup.addAction(new ArActionLimiterBackwards("backwards limiter",
						     -200, 0, -200, 1.2), 51);
  myBackGroup.addAction(&myGoto, 50);
  myIsSim = false;
  myDockType = "";
  myNeedToPathPlanToDock = false;
}

AREXPORT ArServerModeDockTriangleBump::~ArServerModeDockTriangleBump()
{

}

AREXPORT void ArServerModeDockTriangleBump::userTask(void)
{
  bool printing = false;
  int frontBump;
  int frontBumpMask = (ArUtil::BIT1 | ArUtil::BIT2 | ArUtil::BIT3 | 
		       ArUtil::BIT4 | ArUtil::BIT5 | ArUtil::BIT6);
  if (printing)
    printf("State %d\n", myState);
  if (myState == DOCKED)
  {
    // if we've been docked for more than a second and our motors are
    // enabled disable them
    if (!myIsSim && myRobot->areMotorsEnabled() && 
	myStartedState.mSecSince() > 1000)
    {
      ArLog::log(ArLog::Normal, "Dock disabling motors");
      myRobot->disableMotors();
    }

    if (isDocked())
    {
      myLastPowerGood.setToNow();
    }
    if (myLastPowerGood.secSince() >= 30)
    {
      //switchState(UNDOCKED);
      ArLog::log(ArLog::Normal, 
	  "Failed dock lost power after there, last power was %d seconds ago", 
		 myLastPowerGood.secSince());
      
      myStatus = "Failed Dock";
      myRetryDock = true;
      undock();
    }
  }
  if (myState == DOCKING)
  {
    frontBump = ((myRobot->getStallValue() & 0xff00) >> 8) & frontBumpMask;
    if (myStallsAsBumps && (myRobot->isLeftMotorStalled() ||
			    myRobot->isRightMotorStalled()))
      frontBump |= ArUtil::BIT0;
      
    if (printing)
      printf("Docking... %ld %d\n", myStartedDriveTo.secSince(),
	     isDocked());
    // if we're there and haven't sent the command
    if (myHitDock && isDocked())
    {
      ArLog::log(ArLog::Normal, "Docked");
      if (myForcedDock)
	myStatus = "Docked because of low voltage";
      else
	myStatus = "Docked";
      myLastPowerGood.setToNow();
      switchState(DOCKED);
    }
    // if we sent the command but it didn't work
    else if (myHitDock && myHitDockTime.secSince() > 10)
    {
      ArLog::log(ArLog::Normal, 
		 "Dock Failed");
      myRetryDock = true;
      undock();
    }
    // if we sent the dock command just chill
    else if (myHitDock)
    {

    }
    else if (myDriveTo.isActive() && 
	     frontBump)
    {
      ArLog::log(ArLog::Normal, "Hit dock");
      enableDock();
      myHitDock = true;
      myHitDockTime.setToNow();
      myDriveTo.deactivate();
    }
    // if we failed getting there
    else if (myDriveTo.isActive() && 
	     myDriveTo.getState() == ArActionTriangleDriveTo::STATE_FAILED)
    {
      ArLog::log(ArLog::Normal, 
		 "Dock Failed: Could not find dock target");
      myRetryDock = true;
      undock();
    }
    // if it took longer than 30 seconds to drive in
    else if (myDriveTo.isActive() && myStartedDriveTo.secSince() > 30)
    {
      ArLog::log(ArLog::Normal, 
		 "Dock Failed: Took too long to find target");
      myStatus = "Failed dock";
      myRetryDock = true;
      undock();
    }
    // if we've gotten to our goal and need to drive into the dock
    else if (myGoalDone && !myDriveTo.isActive())
    {
      ArLog::log(ArLog::Normal, "Driving into dock");
      beforeDriveInCallback();
      myDriveFromValid = true;
      myDriveFrom = myRobot->getEncoderPose();
      myGroup.activateExclusive();
      myGoalDone = false;
      myStartedDriveTo.setToNow();
      myDrivingIntoDockCBList.invoke();
    }
  }
  else if (myState == UNDOCKING)
  {
    myHitDock = false;
    if (printing)
      printf("Undocking %ld\n", myStartedState.secSince());

    // make it so that we're always trying to move while undocking, so
    // the sonar kick in before we start backing up
    myRobot->forceTryingToMove();

    if (!myDisableDockCalled)
    {
      disableDock();
      myDisableDockCalled = true;
    }

    if (myStartedState.secSince() < 2 )
    {
      //ArLog::log(ArLog::Normal, "Sending motor enable");
      myRobot->enableMotors();
    }

    if (myStartedState.secSince() >= 3 && !myRobot->areMotorsEnabled())
      myRobot->enableMotors();

    if (myStartedState.secSince() >= 3 && !myStartedBacking)
    {
      myStartedBacking = true;
      //myGoto.setEncoderGoalRel(myDistanceToBack, -180, true);
      myGoto.setDistance(-myDistanceToBack, true);
      myRobot->clearDirectMotion();
      myBackGroup.activateExclusive();
    }
    if (myStartedBacking && myGoto.haveAchievedDistance())
    {
      backoutCallback();
    }

    if (myStartedBacking && myGoto.haveAchievedDistance() && !isDocked())
    {
      afterDriveOutCallback();
      if (myRetryDock)
      {
	myRetryDock = false;
	switchState(UNDOCKED);
	ArLog::log(ArLog::Normal, "Retrying dock");
	dock();
	return;
      }
      myStatus = "Stopped";
      switchState(UNDOCKED);
      ArLog::log(ArLog::Normal, "Undocked");
      resumeInterrupted(true);
    }
    if (myStartedState.secSince() >= 60)
    {
      ArLog::log(ArLog::Normal, "Error undocking, taken too long");
      myStartedState.setToNow();
      myStatus = "Undocking failing";
    }
  }
  else if (myState == UNDOCKED)
  {
    if (myNeedToPathPlanToDock && !myServer->idleProcessingPending())
    {
      myNeedToPathPlanToDock = false;
      myGoalDone = false;
      myDriveTo.deactivate();
      myRobot->clearDirectMotion();
      
      
      ArMapInterface *arMap;
      ArMapObject *dockObject = NULL;
      arMap = myPathTask->getAriaMap();
      dockObject = findDock(arMap);
      // go to either the first dock object or to the goal dock if thats not around
      if (dockObject == NULL || 
	  !myPathTask->pathPlanToPose(dockObject->getPose(), true))
      {
	if (dockObject != NULL)
	{
	  myStatus = "Cannot drive to dock at ";
	  myStatus += dockObject->getName();
	}
	else
	{
	  myStatus = "Cannot drive to dock";
	}
	if (myForcedDock)
	{
	  myRetryDock = true;
	  undock();
	  // no callback here since this'll only fail if the path planning
	  // is completely hosed (if there's no path the goal failed will
	  // be called, not this)... in fact this whole my retry dock and
	  // all that is probably not needed
	}
	else
	  resumeInterrupted(false);
	return;
      }
      myDockName = dockObject->getName();
      myStatus = "Going to dock at ";
      myStatus += dockObject->getName();
      ArLog::log(ArLog::Normal, "Docking at %s", dockObject->getName());
      switchState(DOCKING);
      myDrivingToDockCBList.invoke();
    }
    else
    {
      myStatus = "Waiting to dock";
    }

  }
 
}

AREXPORT void ArServerModeDockTriangleBump::dock(void)
{
  if (myIsActive && myState == DOCKED)
    return;

  activate();
  /// MPL changed
  //if (!myIsActive || myState != UNDOCKED)
  // to this  for the shutdown change
  if (!myIsActive || (myState != UNDOCKED && !myRetryDock))
    return;
  myNeedToPathPlanToDock = true;
  switchState(UNDOCKED);
  /*
  myGoalDone = false;
  myDriveTo.deactivate();
  myRobot->clearDirectMotion();


  ArMapInterface *arMap;
  ArMapObject *dockObject = NULL;
  arMap = myPathTask->getAriaMap();
  dockObject = findDock(arMap);
  // go to either the first dock object or to the goal dock if thats not around
  if (dockObject == NULL || 
      !myPathTask->pathPlanToPose(dockObject->getPose(), true))
  {
    if (dockObject != NULL)
    {
      myStatus = "Cannot drive to dock at ";
      myStatus += dockObject->getName();
    }
    else
    {
      myStatus = "Cannot drive to dock";
    }
    if (myForcedDock)
    {
      myRetryDock = true;
      undock();
      // no callback here since this'll only fail if the path planning
      // is completely hosed (if there's no path the goal failed will
      // be called, not this)... in fact this whole my retry dock and
      // all that is probably not needed
    }
    else
      resumeInterrupted(false);
    return;
  }
  myDockName = dockObject->getName();
  myStatus = "Going to dock at ";
  myStatus += dockObject->getName();
  ArLog::log(ArLog::Normal, "Docking at %s", dockObject->getName());
  switchState(DOCKING);
  myDrivingToDockCBList.invoke();
  */
}

AREXPORT void ArServerModeDockTriangleBump::undock(void)
{
  if (myState == DOCKED && myForcedDock && !myRetryDock)
    return;

  myRobot->enableMotors();
  if (myState == UNDOCKED)
  {
    resumeInterrupted(true);
  }
  else if (myState == DOCKED)
  {
    if (myDesiredBackOutDist > .1)
      myDistanceToBack = myDesiredBackOutDist;
    else if (myDriveFromValid)
      myDistanceToBack = myRobot->getEncoderPose().findDistanceTo(myDriveFrom);
    else
      myDistanceToBack = myRobot->getRobotRadius() + 1000;
    myStatus = "Undocking";
    myStartedBacking = false;
    myDisableDockCalled = false;
    switchState(UNDOCKING);
    ArLog::log(ArLog::Normal, "Undocking");
  }
  else if (myState == DOCKING)
  {
    myRobot->stop();
    ArLog::log(ArLog::Normal, "Undocking");
    if (myDesiredBackOutDist > .1)
      myDistanceToBack = myDesiredBackOutDist;
    else if (myDriveFromValid)
      myDistanceToBack = myRobot->getEncoderPose().findDistanceTo(myDriveFrom);
    else
      myDistanceToBack = myRobot->getRobotRadius() + 1000;
    myStartedBacking = false;
    if (myHitDock || myDriveTo.isActive())
    {
      myStatus = "Undocking";
      myDisableDockCalled = false;
      switchState(UNDOCKING);
    }
    else
    {
      myDisableDockCalled = true;
      switchState(UNDOCKED);
      resumeInterrupted(true);
    }
  }
  // if we're already undocking ignore this
  else if (myState == UNDOCKING)
  {
  }
}

AREXPORT void ArServerModeDockTriangleBump::deactivate()
{
  // if we're not forced docking and retry dock is set then cancel retry dock
  if (!myForcedDock && myRetryDock)
  {
    ArLog::log(ArLog::Normal, "Cancelling dock retry");
    myRetryDock = false;
  }
  myNeedToPathPlanToDock = false;
  ArServerModeDock::deactivate();
}

AREXPORT void ArServerModeDockTriangleBump::checkDock(void)
{
  if (myRobot->isConnected() && isDocked())
  {
    ArLog::log(ArLog::Normal, "Robot is already docked, activating dock mode");
    myStatus = "Docked";
    switchState(DOCKED);
    myHitDock = true;
    activate();
    clearInterrupted();
  }
}

AREXPORT void ArServerModeDockTriangleBump::forceUnlock(void)
{
  myRobot->comInt(68, 0);
  ArUtil::sleep(10);
  myState = UNDOCKED;
  ArServerMode::forceUnlock();
}


AREXPORT void ArServerModeDockTriangleBump::goalDone(ArPose /*pose*/ )
{
  if (!myIsActive)
    return;
  myStatus = "Driving into dock";
  myGoalDone = true;
}

AREXPORT void ArServerModeDockTriangleBump::goalFailed(ArPose /*pose*/)
{
  if (!myIsActive || myState != DOCKING)
    return;
  gotoFailed();
  ArLog::log(ArLog::Normal, "Couldn't get to dock");
  myStatus = "Failed dock";
  // MPL commented out this line for shutting down change
  //switchState(UNDOCKED);
  // if we're not forced see if anything else wants to go
  if (!myForcedDock)
    resumeInterrupted(false);
  
  // if we're still active try to redock
  if (isActive())
  {
    ArLog::log(ArLog::Normal, "Retrying dock");
    // MPL added this line for shutting down change
    myRetryDock = true;
    myNeedToPathPlanToDock = true;
    switchState(UNDOCKED); 
    //dock();
  }
  // MPL added these two lines for shutting down change
  else
  {
    myNeedToPathPlanToDock = false;
    switchState(UNDOCKED);
  }
}



AREXPORT ArServerModeDockPatrolBot::ArServerModeDockPatrolBot(
	ArServerBase *serverBase, ArRobot *robot,
	ArLocalizationTask *locTask, ArPathPlanningTask *pathTask,
	ArFunctor *shutdownFunctor) :
  ArServerModeDockTriangleBump(serverBase, robot, locTask, pathTask, true,
			       300, 300, shutdownFunctor)
{
  //myRobot->requestIOPackets();
  myDockType = "DockPatrolBot";
}

AREXPORT ArServerModeDockPatrolBot::~ArServerModeDockPatrolBot()
{

}

AREXPORT bool ArServerModeDockPatrolBot::isDocked(void)
{
  //unsigned char powerbits = myRobot->getIODigIn(4);
  // powerbits & ArUtil::BIT3);
  return (myRobot->getFlags() & ArUtil::BIT10);
}

AREXPORT void ArServerModeDockPatrolBot::enableDock(void)
{
  myRobot->comInt(68, 1);
  return;
}

AREXPORT void ArServerModeDockPatrolBot::disableDock(void)
{
  myRobot->comInt(68, 0);
  return;
}

AREXPORT void ArServerModeDockPatrolBot::checkDock(void)
{
  // first we see if it looks like we might be docked (laser readings in the robot)
  ArRangeDevice *laser;
  const std::list<ArSensorReading *> *rawReadings;
  std::list<ArSensorReading *>::const_iterator rawReadingsIt;
  const ArSensorReading *reading;

  int numClose, numReadings;

  if ((laser = myRobot->findLaser(1)) != NULL &&
      (rawReadings = laser->getRawReadings()) != NULL)
  {
    for (rawReadingsIt = rawReadings->begin(), numClose = 0, numReadings = 0; 
	 rawReadingsIt != rawReadings->end();
	 rawReadingsIt++)
    {
      reading = (*rawReadingsIt);
      if (!reading->getIgnoreThisReading())
      {
	numReadings++;
	if (reading->getRange() < myRobot->getRobotWidth() / 2)
	  numClose++;
      }
    }

    if (numReadings == 0)
    {
      ArLog::log(ArLog::Normal, "No laser readings yet, checking dock");
    }

    if (numClose < .20 * numReadings)
    {
      ArLog::log(ArLog::Normal, "Not checking for dock since not enough laser readings to be in dock");
      return;
    }
  }
  else
  {
    ArLog::log(ArLog::Normal, "No laser, will check for dock");
  }
  ArServerModeDockTriangleBump::checkDock();
  if (isActive())
  {
    ArLog::log(ArLog::Verbose, "Setting ignore illegal pose flag (cd)");
    myLocTask->setIgnoreIllegalPoseFlag(true);
  }
}

AREXPORT void ArServerModeDockPatrolBot::beforeDriveInCallback()
{
  ArLog::log(ArLog::Verbose, "Setting ignore illegal pose flag (bd)");
  myLocTask->setIgnoreIllegalPoseFlag(true);
}

AREXPORT void ArServerModeDockPatrolBot::afterDriveOutCallback()
{
  ArLog::log(ArLog::Verbose, "Clearing ignore illegal pose flag");;
  myLocTask->setIgnoreIllegalPoseFlag(false);
}


AREXPORT ArServerModeDockSimulator::ArServerModeDockSimulator(
	ArServerBase *serverBase, ArRobot *robot,
	ArLocalizationTask *locTask, ArPathPlanningTask *pathTask, 
	ArFunctor *shutdownFunctor) :
  ArServerModeDockTriangleBump(serverBase, robot, locTask, pathTask, false,
			       300, 300, shutdownFunctor)
{
  setStallsAsBumps(true);
  myIsSim = true;
  myDockType = "";
}

AREXPORT ArServerModeDockSimulator::~ArServerModeDockSimulator()
{

}

AREXPORT bool ArServerModeDockSimulator::isDocked(void)
{
  if (myState == UNDOCKING || myState == UNDOCKED)
    return false;
  else
    return true;
}

AREXPORT void ArServerModeDockSimulator::enableDock(void)
{
  return;
}

AREXPORT void ArServerModeDockSimulator::disableDock(void)
{
  return;
}

/**
   The simulator redefines this so that the simulated robot doesn't
   always come up docked.
 **/
AREXPORT void ArServerModeDockSimulator::checkDock(void)
{
  return;
}

// This use to have 1000, 0 for docking params
AREXPORT ArServerModeDockPowerBot::ArServerModeDockPowerBot(
	ArServerBase *serverBase, ArRobot *robot,
	ArLocalizationTask *locTask, ArPathPlanningTask *pathTask,
	bool isOldDock, ArFunctor *shutdownFunctor, bool useChargeState, 
	int oldDockAnalogPort) :
  ArServerModeDockTriangleBump(serverBase, robot, locTask, pathTask, 
			       useChargeState, 1000, 1000, shutdownFunctor)
{
  myIsOldDock = isOldDock;
  myOldDockAnalogPort = oldDockAnalogPort;
  if (myIsOldDock)
  {
    myRobot->requestIOPackets();
    myBatteryChargingGood = 2.40;
  }
  myDockType = "DockPowerBot";
}

AREXPORT ArServerModeDockPowerBot::~ArServerModeDockPowerBot()
{

}

AREXPORT void ArServerModeDockPowerBot::addToConfig(ArConfig *config)
{
  std::string section;
  section = "Docking";

  if (config == NULL)
    return;

  ArServerModeDock::addToConfig(config);

  if (myIsOldDock)
  {
    config->addParam(
	    ArConfigArg("BatteryChargingGood", &myBatteryChargingGood, 
			"This is the maximum analog voltage from the charge current sensor that this class will take as charging... the sensor should read at 2.50v with no load and every amp charging reduces this by .04... so 2.3 should be at least 5 amps... this is not always at the center and has some noise so its a parameter here but shouldn't really need to be adjusted",
			0, 5),
	    section.c_str(), ArPriority::TRIVIAL);
  }
  
}
AREXPORT bool ArServerModeDockPowerBot::isDocked(void)
{
  //ArLog::log(ArLog::Normal, "## %d %d", myIsOldDock, (myRobot->getFlags() & ArUtil::BIT10));
  // if its the new dock check the flags
  if (!myIsOldDock && (myRobot->getFlags() & ArUtil::BIT10))
    return true;
  // if its the old dock check the current
  else if (myIsOldDock && 
      myRobot->getIOAnalogVoltage(myOldDockAnalogPort) < myBatteryChargingGood)
    return true;
  else
    return false;
}

AREXPORT void ArServerModeDockPowerBot::enableDock(void)
{
  ArLog::log(ArLog::Normal, "comInt 68, 1");
  myRobot->comInt(68, 1);
  return;
}

AREXPORT void ArServerModeDockPowerBot::disableDock(void)
{
  ArLog::log(ArLog::Normal, "comInt 68, 0");
  myRobot->comInt(68, 0);
  return;
}

AREXPORT void ArServerModeDockPowerBot::backoutCallback(void)
{
  ArLog::log(ArLog::Normal, "comInt 68, 0");
  myRobot->comInt(68, 0);
  return;
}

