/*

  Copyright (C) 2004,2005, ActivMedia Robotics, LLC
  Copyright (C) 2006, 2007, 2008, 2009 MobileRobots Inc.

     This program is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published by
     the Free Software Foundation; either version 2 of the License, or
     (at your option) any later version.

     This program is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with this program; if not, write to the Free Software
     Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include "ArExport.h"
#include "ariaOSDef.h"
#include "ArArgumentBuilder.h"
#include "ArArgumentParser.h"
#include "ArLog.h"
#include "ariaUtil.h"
#include <stdarg.h>

std::list<std::string> ArArgumentParser::ourDefaultArgumentLocs;
std::list<bool> ArArgumentParser::ourDefaultArgumentLocIsFile;
/**
   @param argc a pointer to the argc used
   @param argv argv
**/
AREXPORT ArArgumentParser::ArArgumentParser(int *argc, char **argv)
{
  myArgc = argc;
  myArgv = argv;
  myUsingBuilder = false;
  myBuilder = NULL;
  myOwnBuilder = false;
  
  myEmptyArg[0] = '\0';
}

/**
   @param argc a pointer to the argc used
   @param argv argv
**/
AREXPORT ArArgumentParser::ArArgumentParser(ArArgumentBuilder *builder)
{
  myUsingBuilder = true;
  myBuilder = builder;
  myOwnBuilder = false;
  myEmptyArg[0] = '\0';
}

AREXPORT ArArgumentParser::~ArArgumentParser()
{
  if (myOwnBuilder)
  {
    delete myBuilder;
    myBuilder = NULL;
  }
}

AREXPORT bool ArArgumentParser::checkArgumentVar(const char *argument, ...)
{
  char arg[2048];
  va_list ptr;
  va_start(ptr, argument);
  vsnprintf(arg, sizeof(arg), argument, ptr);
  va_end(ptr);
  return checkArgument(arg);
}

/**
   @param argument the string to check for, if the argument is found
   its pulled from the list of arguments

   @param ... the extra string to feed into the argument for parsing
   (like printf)

   @return true if the argument was found, false otherwise
**/
AREXPORT bool ArArgumentParser::checkArgument(const char *argument)
{
  size_t i;
  std::string extraHyphen;
  extraHyphen = "-";
  extraHyphen += argument;
  for (i = 0; i < getArgc(); i++)
  {
    if (strcasecmp(argument, getArgv()[i]) == 0 ||
	strcasecmp(extraHyphen.c_str(), getArgv()[i]) == 0)
    {
      removeArg(i);
      checkArgument(argument);
      return true;
    }
  }
  return false;
}

/**
   This is like checkParameterArgument but lets you fail out if the
   argument is there but the parameter for it is not

   @param argument the string to check for, if the argument is found
   its pulled from the list of arguments

   @param dest if the parameter to the argument is found then the dest
   is set to the parameter

   @param ... the extra string to feed into the argument for
   parsing (like printf)
   
   @return true if either this argument wasn't there or if the
   argument was there with a valid parameter
**/
AREXPORT bool ArArgumentParser::checkParameterArgumentStringVar(
	bool *wasReallySet, const char **dest, const char *argument, ...)
{
  char arg[2048];
  va_list ptr;
  va_start(ptr, argument);
  vsnprintf(arg, sizeof(arg), argument, ptr);
  va_end(ptr);
  return checkParameterArgumentString(arg, dest, wasReallySet);
}

/**
   This is like checkParameterArgument but lets you fail out if the
   argument is there but the parameter for it is not

   @param argument the string to check for, if the argument is found
   its pulled from the list of arguments

   @param dest if the parameter to the argument is found then the dest
   is set to the parameter

   @return true if either this argument wasn't there or if the
   argument was there with a valid parameter
**/
AREXPORT bool ArArgumentParser::checkParameterArgumentString(const char *argument,
							     const char **dest,
							     bool *wasReallySet)
{
  char *param;
  param = checkParameterArgument(argument);

  if (param == NULL)
  {
    if (wasReallySet)
      *wasReallySet = false;
    return true;
  }
  else if (param[0] != '\0')
  {
    *dest = param;    
    if (wasReallySet)
      *wasReallySet = true;
    return true;
  }
  else
  {
    ArLog::log(ArLog::Normal, "No argument given to %s", argument);
    return false;
  }
}

/**
   This is like checkParameterArgument but lets you fail out if the
   argument is there but the parameter for it is not

   @param ... the extra string to feed into the argument for
   parsing (like printf)

   @param argument the string to check for, if the argument is found
   its pulled from the list of arguments

   @param dest if the parameter to the argument is found and is a
   valid bool (true, false, 1, 0) then dest is set to the bool value
   
   @return true if either this argument wasn't there or if the
   argument was there with a valid parameter
**/
AREXPORT bool ArArgumentParser::checkParameterArgumentBoolVar(
	bool *wasReallySet, bool *dest, const char *argument, ...)
{
  char arg[2048];
  va_list ptr;
  va_start(ptr, argument);
  vsnprintf(arg, sizeof(arg), argument, ptr);
  va_end(ptr);
  return checkParameterArgumentBool(arg, dest, wasReallySet);
}

/**
   This is like checkParameterArgument but lets you fail out if the
   argument is there but the parameter for it is not

   @param argument the string to check for, if the argument is found
   its pulled from the list of arguments

   @param dest if the parameter to the argument is found and is a
   valid bool (true, false, 1, 0) then dest is set to the bool value
   
   @return true if either this argument wasn't there or if the
   argument was there with a valid parameter
**/
AREXPORT bool ArArgumentParser::checkParameterArgumentBool(const char *argument,
							   bool *dest,
							   bool *wasReallySet)
{
  char *param;
  param = checkParameterArgument(argument);
  
  if (param == NULL)
  {
    if (wasReallySet)
      *wasReallySet = false;
    return true;
  }
  else if (param[0] != '\0')
  {
    if (strcasecmp(param, "true") == 0 || strcmp(param, "1") == 0)
    {
      *dest = true;
      if (wasReallySet)
	*wasReallySet = true;
      return true;
    }
    else if (strcasecmp(param, "false") == 0 || strcmp(param, "0") == 0)
    {
      *dest = false;
      if (wasReallySet)
	*wasReallySet = true;
      return true;
    }
    else
    {
      ArLog::log(ArLog::Normal, 
		"Argument given to %s was not a bool (true, false, 1, 0) it was the string %s",
		 argument, param);
      return false;
    }
 
  }
  else
  {
    ArLog::log(ArLog::Normal, "No argument given to %s", argument);
    return false;
  }

}

/**
   This is like checkParameterArgument but lets you fail out if the
   argument is there but the parameter for it is not

   @param argument the string to check for, if the argument is found
   its pulled from the list of arguments

   @param dest if the parameter to the argument is found and is a
   valid integer then dest is set to the integer value

   @param ... the extra string to feed into the argument for
   parsing (like printf)

   @return true if either this argument wasn't there or if the
   argument was there with a valid parameter
**/
AREXPORT bool ArArgumentParser::checkParameterArgumentIntegerVar(
	bool *wasReallySet, int *dest, const char *argument, ...)
{
  char arg[2048];
  va_list ptr;
  va_start(ptr, argument);
  vsnprintf(arg, sizeof(arg), argument, ptr);
  va_end(ptr);
  return checkParameterArgumentInteger(arg, dest, wasReallySet);
}

/**
   This is like checkParameterArgument but lets you fail out if the
   argument is there but the parameter for it is not

   @param argument the string to check for, if the argument is found
   its pulled from the list of arguments

   @param dest if the parameter to the argument is found and is a
   valid integer then dest is set to the integer value

   @return true if either this argument wasn't there or if the
   argument was there with a valid parameter
**/
AREXPORT bool ArArgumentParser::checkParameterArgumentInteger(
	const char *argument, int *dest, bool *wasReallySet)
{
  char *param;
  char *endPtr;
  int intVal;
  param = checkParameterArgument(argument);
  
  if (param == NULL)
  {
    if (wasReallySet)
      *wasReallySet = false;
    return true;
  }
  else if (param[0] != '\0')
  {
    intVal = strtol(param, &endPtr, 10);
    if (endPtr[0] == '\0')
    {
      *dest = intVal;
      if (wasReallySet)
	*wasReallySet = true;
      return true;
    }
    else
    {
      ArLog::log(ArLog::Normal, 
		"Argument given to %s was not an integer it was the string %s",
		 argument, param);
      return false;
    }
 
  }
  else
  {
    ArLog::log(ArLog::Normal, "No argument given to %s", argument);
    return false;
  }

}

/**
   @param argument the string to check for, if the argument is found
   its pulled from the list of arguments

   @param ... the extra string to feed into the argument for
   parsing (like printf)

   @return NULL if the argument wasn't found, the argument after the
   one given if the argument was found, or a string with the first
   char as '\0' again if the argument after the one given isn't there
 **/
AREXPORT char *ArArgumentParser::checkParameterArgumentVar(const char *argument, ...)
{
  char arg[2048];
  va_list ptr;
  va_start(ptr, argument);
  vsnprintf(arg, sizeof(arg), argument, ptr);
  va_end(ptr);
  return checkParameterArgument(arg);
}
/**
   @param argument the string to check for, if the argument is found
   its pulled from the list of arguments

   @return NULL if the argument wasn't found, the argument after the
   one given if the argument was found, or a string with the first
   char as '\0' again if the argument after the one given isn't there
**/
AREXPORT char * ArArgumentParser::checkParameterArgument(const char *argument)
{
  char *ret;
  char *retRecursive;
  size_t i;
  std::string extraHyphen;

  extraHyphen = "-";
  extraHyphen += argument;

  for (i = 0; i < getArgc(); i++)
  {
    if (strcasecmp(argument, getArgv()[i]) == 0 ||
	strcasecmp(extraHyphen.c_str(), getArgv()[i]) == 0)
    {
      // see if we have a ret, we don't if the ret would be beyond argc
      if (getArgc() > i+1)
      {
	ret = getArgv()[i+1];
      }
      else
      {
	ret = myEmptyArg;
      }
      // remove our argument
      removeArg(i);
      // if we have a return remove that one too
      if (ret != NULL && ret != myEmptyArg)
	removeArg(i);
      // now see if there are any more, if so return that
      if ((retRecursive = checkParameterArgument(argument)) != NULL)
      {
	return retRecursive;
      }
      // otherwise return what we found
      else
      {
	return ret;
      }
    }
  }
  return NULL;
}

void ArArgumentParser::removeArg(size_t which)
{
  if (which >= getArgc())
  {
    ArLog::log(ArLog::Terse, "ArArgumentParser::removeArg: %d is greater than the number of arguments which is %d", which, getArgc());
    return;
  }
  if (myUsingBuilder)
    {
      myBuilder->removeArg(which);
    }
  else
    {
      size_t i;
      for (i = which; i < getArgc() - 1; i++)
	myArgv[i] = myArgv[i+1];
      *myArgc -= 1;
    }
}

AREXPORT size_t ArArgumentParser::getArgc(void) const
{
  if (myUsingBuilder)
    return myBuilder->getArgc();
  else
    return *myArgc;
}

AREXPORT char** ArArgumentParser::getArgv(void) const
{
  if (myUsingBuilder)
    return myBuilder->getArgv();
  else
    return myArgv;
}

AREXPORT const char* ArArgumentParser::getArg(size_t whichArg) const
{
  if (whichArg >= getArgc())
    return NULL;
  else
    return getArgv()[whichArg];
}

AREXPORT void ArArgumentParser::log(void) const
{
  size_t i;
  ArLog::log(ArLog::Terse, "Num arguments: %d", getArgc());
  for (i = 0; i < getArgc(); ++i)
    ArLog::log(ArLog::Terse, "Arg %d: %s", i, getArgv()[i]);
}

AREXPORT void ArArgumentParser::addDefaultArgument(const char *argument)
{
  if (!myUsingBuilder)
    {
      myBuilder = new ArArgumentBuilder;
      myBuilder->addStrings(myArgv, *myArgc);
      myOwnBuilder = true;
      myUsingBuilder = true;
    }
  myBuilder->add(argument);
}

/**
   NOTE, if you use this function your normal argc (from main) won't
   reflect reality anymore, you'll have to use the parser.getArgc()
   to get the actual argument count.  This is a little wierd but is 
   this way so lots of people don't have to change lots of code.   

   This goes through the list of default argument locations, if the
   string is an environmental variable it adds the value of the
   variable to the list of defaults, otherwise it tries to load a
   string from the file and add that to the list.  You can add more
   default locations with addDefaultArgumentLocation.  
 **/
AREXPORT void ArArgumentParser::loadDefaultArguments(void)
{
  std::list<std::string>::iterator it;
  std::list<bool>::iterator bIt;
  const char *str;
  char *argumentsPtr;
  char arguments[1024];

  if (!myUsingBuilder)
    {
      myBuilder = new ArArgumentBuilder;
      myBuilder->addStrings(myArgv, *myArgc);
      myOwnBuilder = true;
      myUsingBuilder = true;
    }

  for (it = ourDefaultArgumentLocs.begin(), 
        bIt = ourDefaultArgumentLocIsFile.begin();
       it != ourDefaultArgumentLocs.end(); 
       it++, bIt++)
  {
    str = (*it).c_str();
    // see if its an environmental variable
    if (!(*bIt) && (argumentsPtr = getenv(str)) != NULL)
    {
      myBuilder->addPlain(argumentsPtr, 1);
      ArLog::log(ArLog::Normal, 
		 "Added arguments from environmental variable '%s'", str);
    }
    // see if we have a file
    else if ((*bIt) && 
	     ArUtil::getStringFromFile(str, arguments, sizeof(arguments)))
    {
      myBuilder->addPlain(arguments, 1);
      ArLog::log(ArLog::Normal, "Added arguments from file '%s'", 
		 str);
    }
    // the file or env didn't exit
    // this'll return true otherwise it'll return false)
    else
    {
      ArLog::log(ArLog::Verbose, 
		 "Could not load from environmental variable or file '%s'", 
		 str);
    }
  }
}

/**
   This adds a file to the list of default argument locations.
 **/
AREXPORT void ArArgumentParser::addDefaultArgumentFile(const char *file)
{
  ourDefaultArgumentLocs.push_back(file);
  ourDefaultArgumentLocIsFile.push_back(true);
}


/**
   This adds an environmental variable to the list of default argument
   locations.
 **/
AREXPORT void ArArgumentParser::addDefaultArgumentEnv(const char *env)
{
  ourDefaultArgumentLocs.push_back(env);
  ourDefaultArgumentLocIsFile.push_back(false);
}

AREXPORT void ArArgumentParser::logDefaultArgumentLocations(void)
{
  std::list<std::string>::iterator it;
  std::list<bool>::iterator bIt;

  ArLog::log(ArLog::Normal, 
	     "Default argument files or environmental variables:");
  for (it = ourDefaultArgumentLocs.begin(), 
        bIt = ourDefaultArgumentLocIsFile.begin();
       it != ourDefaultArgumentLocs.end(); 
       it++, bIt++)
  {
    if (*bIt)
      ArLog::log(ArLog::Normal, "%10s%-10s%s", "", "file", (*it).c_str());
    else
      ArLog::log(ArLog::Normal, "%10s%-10s%s", "", "envVar", (*it).c_str());
  }
}
/**
   This function returns false if there was a help request in the
   parser (so the program should show the help and exit) and also
   warns about unhandled arguments (it doesn't mind them, just says
   they're around).
 **/
AREXPORT bool ArArgumentParser::checkHelpAndWarnUnparsed(
	unsigned int numArgsOkay)
{
  if (checkArgument("-help") || checkArgument("-h") || checkArgument("/?") ||
      checkArgument("/h"))
    return false;
      
  if (getArgc() <= 1 + numArgsOkay)
    return true;

  size_t i;
  char buf[2048];
  sprintf(buf, "Unhandled arguments to program:");
  for (i = 1 + (int)numArgsOkay; i < getArgc(); i++)
    sprintf(buf, "%s %s", buf, getArg(i));
  ArLog::log(ArLog::Normal, buf);
  ArLog::log(ArLog::Normal, 
	   "Program will continue but to see the help listing type '%s -help'",
	     getArg(0));
  return true;
}

