/*

  Copyright (C) 2004, 2005, ActivMedia Robotics, LLC
  Copyright (C) 2006, 2007, 2008, 2009 MobileRobots Inc.

     This program is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published by
     the Free Software Foundation; either version 2 of the License, or
     (at your option) any later version.

     This program is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with this program; if not, write to the Free Software
     Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include "ArExport.h"
#include "ariaOSDef.h"
#include "ArBasePacket.h"
#include <stdio.h>
#include <string.h>

/**
   @param bufferSize size of the buffer
   @param headerLength length of the header
   @param buf buffer packet uses, if NULL, instance will allocate memory
*/
AREXPORT ArBasePacket::ArBasePacket(ArTypes::UByte2 bufferSize, 
				    ArTypes::UByte2 headerLength,
				    char * buf,
				    ArTypes::UByte2 footerLength) 
{
  if (buf == NULL && bufferSize > 0) 
  {
    myOwnMyBuf = true;
    myBuf = new char[bufferSize];
  } 
  else 
  {
    myOwnMyBuf = false;
    myBuf = buf;
  }
  myHeaderLength = headerLength;
  myFooterLength = footerLength;
  myReadLength = myHeaderLength;
  myMaxLength = bufferSize;
  myLength = myHeaderLength;
}

AREXPORT ArBasePacket::~ArBasePacket()
{
  if (myOwnMyBuf && myBuf != NULL)
    delete[] myBuf;
}


AREXPORT void ArBasePacket::setBuf(char *buf, ArTypes::UByte2 bufferSize)
{
  if (myOwnMyBuf) 
  {
    delete[] myBuf;
    myOwnMyBuf = false;
  } 
  myBuf = buf;
  myMaxLength = bufferSize;
}

AREXPORT void ArBasePacket::setMaxLength(ArTypes::UByte2 bufferSize)
{
  if (myMaxLength >= bufferSize)
    return;
  if (myOwnMyBuf) 
  {
    delete[] myBuf;
    myOwnMyBuf = false;
  } 
  myBuf = new char[bufferSize];
  myMaxLength = bufferSize;
  myOwnMyBuf = true;
}

AREXPORT bool ArBasePacket::setLength(ArTypes::UByte2 length)
{
  if (myOwnMyBuf && length > myMaxLength)
    return false;
  
  myLength = length;
  return true;
}

AREXPORT void ArBasePacket::setReadLength(ArTypes::UByte2 readLength)
{
  myReadLength = readLength;
}

AREXPORT bool ArBasePacket::setHeaderLength(ArTypes::UByte2 length)
{
  if (myOwnMyBuf && length > myMaxLength)
    return false;
  
  myHeaderLength = length;
  return true;
}

/** 
    Sets the length read back to the header length so the packet can be
    reread using the other methods
*/

AREXPORT void ArBasePacket::resetRead(void)
{
  myReadLength = myHeaderLength;
}

/**
   Sets the packet length back to be the packets header length again
 */

AREXPORT void ArBasePacket::empty(void)
{
  myLength = myHeaderLength;
}

AREXPORT bool ArBasePacket::isNextGood(int bytes)
{
  if (bytes <= 0)
    return false;
  
  // make sure it comes in before the header
  if (myReadLength + bytes <= myLength - myFooterLength)
    return true;
  
  return false;
}

AREXPORT const char *ArBasePacket::getBuf(void)
{
  return myBuf;
}

AREXPORT void ArBasePacket::byteToBuf(ArTypes::Byte val)
{
  memcpy(myBuf+myLength, &val, 1);
  myLength += 1;
}

AREXPORT void ArBasePacket::byte2ToBuf(ArTypes::Byte2 val)
{
  unsigned char c;
  c = (val >> 8) & 0xff;
  memcpy(myBuf+myLength+1, &c, 1);
  c = val & 0xff;
  memcpy(myBuf+myLength, &c, 1);
  myLength += 2;
}

AREXPORT void ArBasePacket::byte4ToBuf(ArTypes::Byte4 val)
{
  unsigned char c;
  c = (val >> 24) & 0xff;
  memcpy(myBuf+myLength+3, &c, 1);
  c = (val >> 16) & 0xff;
  memcpy(myBuf+myLength+2, &c, 1);
  c = (val >> 8) & 0xff;
  memcpy(myBuf+myLength+1, &c, 1);
  c = val & 0xff;
  memcpy(myBuf+myLength, &c, 1);
  myLength += 4;
}

AREXPORT void ArBasePacket::uByteToBuf(ArTypes::UByte val)
{
  memcpy(myBuf+myLength, &val, 1);
  myLength += 1;
}

AREXPORT void ArBasePacket::uByte2ToBuf(ArTypes::UByte2 val)
{
  unsigned char c;
  c = (val >> 8) & 0xff;
  memcpy(myBuf+myLength+1, &c, 1);
  c = val & 0xff;
  memcpy(myBuf+myLength, &c, 1);
  myLength += 2;
}

AREXPORT void ArBasePacket::uByte4ToBuf(ArTypes::UByte4 val)
{
  memcpy(myBuf+myLength, &val, 4);
  myLength += 4;
}

/**
   @param str string to copy into buffer
*/
AREXPORT void ArBasePacket::strToBuf(const char *str)
{
  memcpy(myBuf+myLength, str, strlen(str) + 1);
  myLength+=strlen(str)+1;
}

/**
   @param str character array to copy into the packet buffer
   @param length how many characters to copy from str into the packet buffer
*/
AREXPORT void ArBasePacket::strNToBuf(const char *str, int length)
{
  //byte4ToBuf(length);
  memcpy(myBuf+myLength, str, length);
  myLength+=length;
}

/**
   If string ends before length it pads the string with NUL ('\0') characters.
   @param str character array to copy into buffer
   @param length how many bytes to copy from the str into packet
*/
AREXPORT void ArBasePacket::strToBufPadded(const char *str, int length)
{
  ArTypes::UByte2 len;

  len=strlen(str);
  if (len >= length) {
    memcpy(myBuf+myLength, str, length);
    myLength += length;
  }
  else
  {
    memcpy(myBuf+myLength, str, len);
    myLength+=len;
    memset(myBuf+myLength, 0, length-len);
    myLength+=length-len;
  }
}

/**
   @param data chacter array to copy into buffer
   @param length how many bytes to copy from data into packet
*/
AREXPORT void ArBasePacket::dataToBuf(const char *data, int length)
{
  memcpy(myBuf+myLength, data, length);
  myLength+=length;
}

AREXPORT ArTypes::Byte ArBasePacket::bufToByte(void)
{
  ArTypes::Byte ret=0;

  if (isNextGood(1))
  {
    memcpy(&ret, myBuf+myReadLength, 1);
    myReadLength+=1;
  }

  return(ret);
}

AREXPORT ArTypes::Byte2 ArBasePacket::bufToByte2(void)
{
  ArTypes::Byte2 ret=0;
  unsigned char c1, c2;

  if (isNextGood(2))
  {
    memcpy(&c1, myBuf+myReadLength, 1);
    memcpy(&c2, myBuf+myReadLength+1, 1);
    ret = (c1 & 0xff) | (c2 << 8);
    myReadLength+=2;
  }

  return ret;
}

AREXPORT ArTypes::Byte4 ArBasePacket::bufToByte4(void)
{
  ArTypes::Byte4 ret=0;
  unsigned char c1, c2, c3, c4;

  if (isNextGood(4))
  {
    memcpy(&c1, myBuf+myReadLength, 1);
    memcpy(&c2, myBuf+myReadLength+1, 1);
    memcpy(&c3, myBuf+myReadLength+2, 1);
    memcpy(&c4, myBuf+myReadLength+3, 1);
    ret = (c1 & 0xff) | (c2 << 8) | (c3 << 16) | (c4 << 24);
    myReadLength+=4;
  }

  return ret;
}

AREXPORT ArTypes::UByte ArBasePacket::bufToUByte(void)
{
  ArTypes::UByte ret=0;

  if (isNextGood(1))
  {
    memcpy(&ret, myBuf+myReadLength, 1);
    myReadLength+=1;
  }

  return(ret);
}

AREXPORT ArTypes::UByte2 ArBasePacket::bufToUByte2(void)
{
  ArTypes::UByte2 ret=0;
  unsigned char c1, c2;

  if (isNextGood(2))
  {
    memcpy(&c1, myBuf+myReadLength, 1);
    memcpy(&c2, myBuf+myReadLength+1, 1);
    ret = (c1 & 0xff) | (c2 << 8);
    myReadLength+=2;
  }

  return ret;
}

AREXPORT ArTypes::UByte4 ArBasePacket::bufToUByte4(void)
{
  ArTypes::Byte4 ret=0;
  unsigned char c1, c2, c3, c4;

  if (isNextGood(4))
  {
    memcpy(&c1, myBuf+myReadLength, 1);
    memcpy(&c2, myBuf+myReadLength+1, 1);
    memcpy(&c3, myBuf+myReadLength+2, 1);
    memcpy(&c4, myBuf+myReadLength+3, 1);
    ret = (c1 & 0xff) | (c2 << 8) | (c3 << 16) | (c4 << 24);
    myReadLength+=4;
  }

  return ret;
}

/** 
    Copy a string from the packet buffer into the given buffer, stopping when
    the end of the packet buffer is reached, the given length is reached,
    or a NUL character ('\0') is reached.
    @param buf Destination buffer
    @param len Maximum number of characters to copy into the destination buffer
*/
AREXPORT void ArBasePacket::bufToStr(char *buf, int len)
{
  int i;

  buf[0] = '\0';
  // see if we can read
  if (isNextGood(1))
  {
    // while we can read copy over those bytes
    for (i = 0; 
        isNextGood(1) && i < len && myBuf[myReadLength] != '\0';
	 ++myReadLength, ++i)
      buf[i] = myBuf[myReadLength];
    // if we stopped because of a null then copy that one too
    if (myBuf[myReadLength] == '\0')
    {
      buf[i] = myBuf[myReadLength];
      myReadLength++;
    }
  }
}

/**
   copies length bytes from the buffer into data, length is passed in, not read
   from packet
   @param data character array to copy the data into
   @param length number of bytes to copy into data
*/
AREXPORT void ArBasePacket::bufToData(char *data, int length)
{
  if (isNextGood(length))
  {
    memcpy(data, myBuf+myReadLength, length);
    myReadLength += length;
  }
}


/**
   Copies the given packets buffer into the buffer of this packet, also
   sets this length and readlength to what the given packet has
   @param packet the packet to duplicate
*/
AREXPORT void ArBasePacket::duplicatePacket(ArBasePacket *packet)
{
  myLength = packet->getLength();
  myReadLength = packet->getReadLength();
  memcpy(myBuf, packet->getBuf(), myLength);
}

AREXPORT void ArBasePacket::log(void)
{
  int i;
  printf("Packet: ");
  for (i = 0; i < myLength; i++)
    printf(" %d ", (unsigned char) myBuf[i]);
  printf("\n");
}

AREXPORT void ArBasePacket::printHex(void)
{
  int i;
  printf("Packet: ");
  for (i = 0; i < myLength; i++)
    printf(" 0x%x ", (unsigned char) myBuf[i]);
  printf("\n");
}

