/*

  Copyright (C) 2004,2005, ActivMedia Robotics, LLC
  Copyright (C) 2006, 2007, 2008, 2009 MobileRobots Inc.

     This program is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published by
     the Free Software Foundation; either version 2 of the License, or
     (at your option) any later version.

     This program is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with this program; if not, write to the Free Software
     Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include "ArExport.h"
#include "ariaOSDef.h"
#include "ArConfig.h"
#include "ArArgumentBuilder.h"
#include "ArLog.h"

/**
   @param baseDirectory the directory to load fomr
   @param noBlanksBetweenParams if there should be blanks between params in output
   @param ignoreBounds if this is true bounds checking will be ignored when the file is read in this should ONLY be used by developers debugging
   @param failOnBadSections if this is true and there is a bad section the parseFile will fail
 **/
AREXPORT ArConfig::ArConfig(const char *baseDirectory,
			    bool noBlanksBetweenParams,
			    bool ignoreBounds, 
			    bool failOnBadSection) : 
  myProcessFileCBList(), 
  myNoBlanksBetweenParams(noBlanksBetweenParams),
  myBaseDirectory(),
  myParser(NULL),
  mySections(),
  myParserCB(this, &ArConfig::parseArgument),
  mySectionCB(this, &ArConfig::parseSection)
{
  if (!myParser.addHandlerWithError("section", &mySectionCB))
  {
    ArLog::log(ArLog::Normal, "Could not add section to file parser, horrific failure");
  }
  myArgumentParser = NULL;
  setBaseDirectory(baseDirectory);
  myIgnoreBounds = ignoreBounds;
  myFailOnBadSection = failOnBadSection;
  myProcessFileCallbacksLogLevel = ArLog::Verbose;
  myUsingSections = false;
  mySectionBroken = false;
  myDuplicateParams = false;
  mySection = "";
}

AREXPORT ArConfig::~ArConfig()
{
  clearSections();
}


AREXPORT ArConfig::ArConfig(const ArConfig &config) :
  myProcessFileCBList(),    // TODO: Copy? (don't think we need to)
  myNoBlanksBetweenParams(config.myNoBlanksBetweenParams),
  myBaseDirectory(),
  myParser(NULL),  
  mySections(),
  myParserCB(this, &ArConfig::parseArgument),
  mySectionCB(this, &ArConfig::parseSection) 
{
  myArgumentParser = NULL;
  setBaseDirectory(config.getBaseDirectory());
  myIgnoreBounds = config.myIgnoreBounds;
  myFailOnBadSection = config.myFailOnBadSection;
  myProcessFileCallbacksLogLevel = config.myProcessFileCallbacksLogLevel;
  mySectionBroken = config.mySectionBroken;
  mySection = config.mySection;
  myUsingSections = config.myUsingSections;
  myDuplicateParams = config.myDuplicateParams;

  std::list<ArConfigSection *>::const_iterator it;
  for (it = config.mySections.begin(); 
       it != config.mySections.end(); 
       it++) 
  {
    mySections.push_back(new ArConfigSection(*(*it)));
  }
  myParser.remHandler(&myParserCB);
  myParser.remHandler(&mySectionCB);
  addParserHandlers();

}


AREXPORT ArConfig &ArConfig::operator=(const ArConfig &config)
{
  if (this != &config) {
    
    // TODO: The following attributes also need to be copied (or 
    // something) -- ditto for copy ctor:
    //     myProcessFileCBList
    //     myParser
    //     myParserCB
    //     mySectionCB
    myArgumentParser = NULL;
    setBaseDirectory(config.getBaseDirectory());
    myNoBlanksBetweenParams = config.myNoBlanksBetweenParams;
    myIgnoreBounds = config.myIgnoreBounds;    
    myFailOnBadSection = config.myFailOnBadSection;
    mySection = config.mySection;
    mySectionBroken = config.mySectionBroken;
    myUsingSections = config.myUsingSections;
    myDuplicateParams = config.myDuplicateParams;
    mySections.clear();
    std::list<ArConfigSection *>::const_iterator it;
    for (it = config.mySections.begin(); 
	 it != config.mySections.end(); 
	 it++) 
    {
      mySections.push_back(new ArConfigSection(*(*it)));
    }
    myParser.remHandler(&myParserCB);
    myParser.remHandler(&mySectionCB);
    addParserHandlers();
    
  }
  return *this;

}

AREXPORT void ArConfig::clearSections(void)
{
  while (mySections.begin() != mySections.end())
  {
    delete mySections.front();
    mySections.pop_front();
  }
}

AREXPORT void ArConfig::clearAll(void)
{
  clearSections();
  myProcessFileCBList.clear();
}

void ArConfig::addParserHandlers(void)
{
  std::list<ArConfigSection *>::const_iterator it;
  std::list<ArConfigArg> *params;
  std::list<ArConfigArg>::iterator pit;

  if (!myParser.addHandlerWithError("section", &mySectionCB))
    ArLog::log(ArLog::Verbose, 
	      "ArConfig: Could not add section parser (probably unimportant)");

  for (it = mySections.begin(); 
       it != mySections.end(); 
       it++) 
  {
    params = (*it)->getParams();
    if (params == NULL)
      continue;
    for (pit = params->begin(); pit != params->end(); pit++)
    {
      if (!myParser.addHandlerWithError((*pit).getName(), &myParserCB))
	ArLog::log(ArLog::Verbose, 
		   "ArConfig: Could not add keyword %s (probably unimportant)",
		   (*pit).getName());
    }
  }

}

/**
   This sets the comment on a section, if the section doesn't exist it
   is created.
**/
AREXPORT void ArConfig::setSectionComment(const char *sectionName, 
					  const char *comment)
{
  ArConfigSection *section = findSection(sectionName);

  if (section == NULL)
  {
    ArLog::log(ArLog::Verbose, "Making new section '%s'", sectionName);
    section = new ArConfigSection(sectionName, comment);
    mySections.push_back(section);
  }
  else
    section->setComment(comment);
}

AREXPORT bool ArConfig::addParam(const ArConfigArg &arg, const char *sectionName,
				                 ArPriority::Priority priority)
{
  ArConfigSection *section = findSection(sectionName);

  //printf("SECTION '%s' name '%s' desc '%s'\n", sectionName, arg.getName(), arg.getDescription());
  if (section == NULL)
  {
    ArLog::log(ArLog::Verbose, "Making new section '%s'", sectionName);
    section = new ArConfigSection(sectionName);
    mySections.push_back(section);
  }
   
  std::list<ArConfigArg> *params = section->getParams();

  if (params == NULL)
  {
    ArLog::log(ArLog::Terse, "Something has gone hideously wrong in ArConfig::addParam");
    return false;
  }
  

  // see if we have this parameter in another section so we can require sections
  std::list<ArConfigSection *>::iterator sectionIt;
  
  for (sectionIt = mySections.begin(); 
       sectionIt != mySections.end(); 
       sectionIt++)
  {
    // if we have an argument of this name but we don't have see if
    // this section is our own, if its not then note we have
    // duplicates
    if (strlen(arg.getName()) > 0 && 
	(*sectionIt)->findParam(arg.getName()) != NULL && 
	strcasecmp((*sectionIt)->getName(), section->getName()) != 0)
    {
      ArLog::log(ArLog::Normal, 
		 "Parameter %s duplicated in section %s and %s",
		 arg.getName(), (*sectionIt)->getName(), section->getName());
      myDuplicateParams = true;
    }
  }
  
  // now make sure we can add it to the file parser
  if (!myParser.addHandlerWithError(arg.getName(), &myParserCB))
  {
    ArLog::log(ArLog::Verbose, "Could not add parameter '%s' to file parser, probably already there.", 
	       arg.getName());
    //return false;
  }

  // we didn't have a parameter with this name so add it
  params->push_back(arg);
  params->back().setConfigPriority(priority);
  params->back().setIgnoreBounds(myIgnoreBounds);

  ArLog::log(ArLog::Verbose, "Added parameter '%s'", arg.getName());
  //arg.log();
  return true;
}

/**
   @param comment 
**/
AREXPORT bool ArConfig::addComment(const char *comment, const char *sectionName, 
				   ArPriority::Priority priority)
{
  return addParam(ArConfigArg(comment), sectionName, priority);
}



/**
   The extra string of the parser should be set to the 'section'
   command while the rest of the arg should be the arguments to the
   section command. Its case insensitive.
   
   @param errorBuffer if this is NULL it is ignored, otherwise the
   string for the error is put into the buffer, the first word should
   be the parameter that has trouble
   
   @param errorBufferLen the length of the error buffer
 **/
AREXPORT bool ArConfig::parseSection(ArArgumentBuilder *arg,
				      char *errorBuffer,
				      size_t errorBufferLen)
{
  if (myFailOnBadSection && errorBuffer != NULL)
    errorBuffer[0] = '\0';

  std::list<ArConfigSection *>::iterator sectionIt;
  ArConfigSection *section = NULL;
  
  if (myFailOnBadSection && errorBuffer != NULL)
    errorBuffer[0] = '\0';
  for (sectionIt = mySections.begin(); 
       sectionIt != mySections.end(); 
       sectionIt++)
  {
    section = (*sectionIt);
    if (ArUtil::strcasecmp(section->getName(), arg->getFullString()) == 0)
    {
      ArLog::log(ArLog::Verbose, "Config switching to section '%s'", 
		 arg->getFullString());
      //printf("Config switching to section '%s'\n", 
      //arg->getFullString());
      mySection = arg->getFullString();
      mySectionBroken = false;
      myUsingSections = true;
      return true;
    }
  }

  mySection = "";
  mySectionBroken = true;
  
  if (myFailOnBadSection)
  {
    snprintf(errorBuffer, errorBufferLen,  "ArConfig: Could not find section '%s'", 
	     arg->getFullString());
    
    ArLog::log(ArLog::Terse,  
	       "ArConfig: Could not find section '%s', failing", 
	       arg->getFullString());
    return false;
  }
  else
  {
    ArLog::log(ArLog::Normal,  
	       "ArConfig: Ignoring section '%s'", 
	       arg->getFullString());
    return true;
  }
}

/**
   The extra string of the parser should be set to the command wanted,
   while the rest of the arg should be the arguments to the command.
   Its case insensitive.
   
   @param errorBuffer if this is NULL it is ignored, otherwise the
   string for the error is put into the buffer, the first word should
   be the parameter that has trouble
   
   @param errorBufferLen the length of the error buffer
 **/
AREXPORT bool ArConfig::parseArgument(ArArgumentBuilder *arg, 
				      char *errorBuffer,
				      size_t errorBufferLen)
{
  std::list<ArConfigSection *>::iterator sectionIt;
  std::list<ArConfigArg>::iterator paramIt;
  ArConfigSection *section = NULL;
  std::list<ArConfigArg> *params = NULL;
  ArConfigArg *param = NULL;
  int valInt = 0;
  double valDouble = 0;
  bool valBool = false;
  bool ret = true;

  if (mySectionBroken)
  {
    ArLog::log(ArLog::Verbose, "Skipping parameter %s because section broken",
	       arg->getExtraString());
    if (myFailOnBadSection)
    {
      snprintf(errorBuffer, errorBufferLen, 
	       "Failed because broken config section");
      return false;
    }
    else
    {
      return true;
    }
  }

  // if we have duplicate params and don't have sections don't trash anything
  if (myDuplicateParams && myUsingSections && mySection.size() <= 0)
  {
    snprintf(errorBuffer, errorBufferLen, 
	     "%s not in section, client needs an upgrade",
	     arg->getExtraString());

    ArLog::log(ArLog::Normal, 
	       "%s not in a section, client needs an upgrade",
	       arg->getExtraString());
    return false;
  }

  if (errorBuffer != NULL)
    errorBuffer[0] = '\0';
  for (sectionIt = mySections.begin(); 
       sectionIt != mySections.end(); 
       sectionIt++)
  {
    section = (*sectionIt);
    // if we have a section make sure we're in it, otherwise do the
    // normal thing
    if (mySection.size() > 0 && 
	ArUtil::strcasecmp(mySection, section->getName()))
      continue;
    params = section->getParams();
    // find this parameter
    for (paramIt = params->begin(); paramIt != params->end(); paramIt++)
    {
      if (strcasecmp((*paramIt).getName(), arg->getExtraString()) == 0)
      {
	param = &(*paramIt);
	if (param->getType() != ArConfigArg::STRING &&
	    param->getType() != ArConfigArg::FUNCTOR &&
	    arg->getArg(0) == NULL)
	{
	  ArLog::log(ArLog::Verbose, "ArConfig: parameter '%s' has no argument.",
		     param->getName());
	  continue;
	}
	if (param->getType() == ArConfigArg::DESCRIPTION_HOLDER)
	{

	}
	// see if we're an int
	else if (param->getType() == ArConfigArg::INT)
	{
	  // if the param isn't an int fail
	  if (!arg->isArgInt(0))
	  {
	    ArLog::log(ArLog::Terse, "ArConfig: parameter '%s' is an integer parameterbut was given non-integer argument of '%s'", param->getName(), arg->getArg(0));
	    ret = false;
	    if (errorBuffer != NULL)
	      snprintf(errorBuffer, errorBufferLen, "%s is an integer parameter but was given non-integer argument of '%s'", param->getName(), arg->getArg(0));
	    continue;
	  }
	  valInt = arg->getArgInt(0);
	  if (param->setInt(valInt, errorBuffer, errorBufferLen))
	  {
	    ArLog::log(ArLog::Verbose, "Set parameter '%s' to '%d'",
		       param->getName(), valInt);
	    continue;
	  }
	  else
	  {
	    ArLog::log(ArLog::Verbose, "Could not parameter '%s' to '%d'",
		       param->getName(), valInt);
	    ret = false;
	    continue;
	  }
	}
	else if (param->getType() == ArConfigArg::DOUBLE)
	{
	  // if the param isn't an in tfail
	  if (!arg->isArgDouble(0))
	  {
	    ArLog::log(ArLog::Terse, "ArConfig: parameter '%s' is a double parameter but was given non-double argument of '%s'", param->getName(), arg->getArg(0));
	    if (errorBuffer != NULL)
	      snprintf(errorBuffer, errorBufferLen, "%s is a double parameter but was given non-double argument of '%s'", param->getName(), arg->getArg(0));

	    ret = false;
	    continue;
	  }
	  valDouble = arg->getArgDouble(0);
	  if (param->setDouble(valDouble, errorBuffer, errorBufferLen))
	  {
	    ArLog::log(ArLog::Verbose, "Set parameter '%s' to '%.10f'",
		       param->getName(), valDouble);
	    continue;
	  }
	  else
	  {
	    ArLog::log(ArLog::Verbose, "Could not set parameter '%s' to '%.10f'",
		       param->getName(), valDouble);
	    ret = false;
	    continue;
	  }
	}
	else if (param->getType() == ArConfigArg::BOOL)
	{
	  // if the param isn't an in tfail
	  if (!arg->isArgBool(0))
	  {
	    ArLog::log(ArLog::Terse, "ArConfig: parameter '%s' is a bool parameter but was given non-bool argument of '%s'", param->getName(), arg->getArg(0));
	    ret = false;
	    if (errorBuffer != NULL)
	      snprintf(errorBuffer, errorBufferLen, "%s is a bool parameter but was given non-bool argument of '%s'", param->getName(), arg->getArg(0));
	    continue;
	  }
	  valBool = arg->getArgBool(0);
	  if (param->setBool(valBool, errorBuffer, errorBufferLen))
	  {
	    ArLog::log(ArLog::Verbose, "Set parameter '%s' to %s",
		       param->getName(), valBool ? "true" : "false" );
	    continue;
	  }
	  else
	  {
	    ArLog::log(ArLog::Verbose, "Could not set parameter '%s' to %s",
		       param->getName(), valBool ? "true" : "false" );
	    ret = false;
	    continue;
	  }
	}
	else if (param->getType() == ArConfigArg::STRING)
	{
	  if (param->setString(arg->getFullString()))
	  {
	    ArLog::log(ArLog::Verbose, "Set parameter string '%s' to '%s'",
		       param->getName(), param->getString());
	    continue;
	  }
	  else
	  {
	    ArLog::log(ArLog::Verbose, "Could not set string parameter '%s' to '%s'",
		       param->getName(), param->getString());
	    if (errorBuffer != NULL && errorBuffer[0] == '\0')
	      snprintf(errorBuffer, errorBufferLen, "%s could not be set to '%s'.", param->getName(), arg->getFullString());

	    ret = false;
	    continue;
	  }
	}
	else if (param->getType() == ArConfigArg::FUNCTOR)
	{
	  if (param->setArgWithFunctor(arg))
	  {
	    ArLog::log(ArLog::Verbose, "Set arg '%s' with '%s'",
		       param->getName(), arg->getFullString());
	    continue;
	  }
	  else
	  {
	    ArLog::log(ArLog::Verbose, "Could not set parameter '%s' to '%s'",
		       param->getName(), arg->getFullString());
	    // if it didn't put in an error message make one
	    if (errorBuffer != NULL && errorBuffer[0] == '\0')
	      snprintf(errorBuffer, errorBufferLen, "%s could not be set to '%s'.", param->getName(), arg->getFullString());
	    ret = false;
	    continue;
	  }
	}
	else
	{
	  ArLog::log(ArLog::Terse, "Have no argument type for config '%s', got string '%s'.\n", param->getName(), arg->getFullString());
	}
      }
    }
  }
  return ret;
}

/**
   @param fileName the file to load

   @param continueOnErrors whether to continue parsing if we get
   errors (or just bail)

   @param noFileNotFoundMessage if the file isn't found and this param
   is true it won't complain, otherwise it will

   @param errorBuffer buffer to pass error information into
   
   @param errorBufferLen the length of the errorBuffer
 **/
AREXPORT bool ArConfig::parseFile(const char *fileName, bool continueOnErrors,
				  bool noFileNotFoundMessage, 
				  char *errorBuffer,
				  size_t errorBufferLen)
{
  bool ret = true;

  myFileName = fileName;
  if (errorBuffer != NULL)
    errorBuffer[0] = '\0';

  // parse the file (errors will go into myErrorBuffer from the functors)
  ret = myParser.parseFile(fileName, continueOnErrors, noFileNotFoundMessage);
  // set our pointers so we don't copy anymore into/over it
  if (errorBuffer != NULL && errorBuffer[0] != '\0')
  {
    errorBuffer = NULL;
    errorBufferLen = 0;
  }
  //printf("file %s\n", ArUtil::convertBool(ret));

  // if we have a parser and haven't failed (or we continue on errors)
  // then parse the arguments from the parser
  if (myArgumentParser != NULL && (ret || continueOnErrors))
    ret = ret && parseArgumentParser(myArgumentParser, continueOnErrors, 
				     errorBuffer, errorBufferLen);

  // set our pointers so we don't copy anymore into/over it
  if (errorBuffer != NULL && errorBuffer[0] != '\0')
  {
    errorBuffer = NULL;
    errorBufferLen = 0;
  }
  //printf("parser %s\n", ArUtil::convertBool(ret));
  
  // if we haven't failed (or we continue on errors) then call the
  // process file callbacks
  if (ret || continueOnErrors)
    ret = ret && callProcessFileCallBacks(continueOnErrors, errorBuffer,
					  errorBufferLen);
  // copy our error if we have one and haven't copied in yet
  // set our pointers so we don't copy anymore into/over it
  if (errorBuffer != NULL && errorBuffer[0] != '\0')
  {
    errorBuffer = NULL;
    errorBufferLen = 0;
  }
  //printf("callbacks %s\n", ArUtil::convertBool(ret));
  return ret;
}

/**
   @param fileName the fileName to write out

   @param append if true then it'll append, otherwise it'll overwrite

   @param alreadyWritten a list of strings that have already been
   written out, don't write again if its in this list... when you
   write something put it into this list (if its not NULL)

   @param writePriorities if this is true priorities will be written, if it
   is false they will not be
 **/
AREXPORT bool ArConfig::writeFile(const char *fileName, bool append, 
				  std::set<std::string> *alreadyWritten,
				  bool writePriorities)
{
  FILE *file;

  // holds each line
  char line[1024];
  // holds the fprintf
  char startLine[128];

  std::set<std::string> writtenSet;
  std::set<std::string> *written;
  if (alreadyWritten != NULL)
    written = alreadyWritten;
  else
    written = &writtenSet;
  //std::list<ArConfigArg>::iterator it;
  ArConfigArg *param = NULL;

  std::list<ArArgumentBuilder *>::const_iterator argIt;
  const std::list<ArArgumentBuilder *> *argList = NULL;

  bool commented = false;
  // later this'll have a prefix
  std::string realFileName;
  if (fileName[0] == '/' || fileName[0] == '\\')
  {
    realFileName = fileName;
  }
  else
  {
    realFileName = myBaseDirectory;
    realFileName += fileName;
  }

  unsigned int startCommentColumn = 25;
  std::string mode;

  if (append)
    mode = "a";
  else
    mode = "w";

  if ((file = ArUtil::fopen(realFileName.c_str(), mode.c_str())) == NULL)
  {
    ArLog::log(ArLog::Terse, "ArConfig: Cannot open file '%s' for writing",
	       realFileName.c_str());
    return false;
  }

  bool firstSection = true;
  std::list<ArConfigSection *>::iterator sectionIt;
  std::list<ArConfigArg>::iterator paramIt;
  ArConfigSection *section = NULL;
  std::list<ArConfigArg> *params = NULL;
  for (sectionIt = mySections.begin(); 
       sectionIt != mySections.end(); 
       sectionIt++)
  {
    section = (*sectionIt);
    /// clear out our written ones between sections
    written->clear();
    //printf("Section '%s' %p\n", section->getName(), section);
    if (!firstSection)
      fprintf(file, "\n");
    firstSection = false;
    
    if (section->getName() != NULL && strlen(section->getName()) > 0)
    {
      //fprintf(file, "; %s\n", section->getName());
      fprintf(file, "Section %s\n", section->getName());
    }
    sprintf(line, "; ");
    if (section->getComment() != NULL && strlen(section->getComment()) > 0)
    {
      ArArgumentBuilder descr;
      descr.add(section->getComment());
      for (unsigned int i = 0; i < descr.getArgc(); i++)
      {
	// see if we're over, if we are write this line out and start
	// the next one
	if (strlen(line) + strlen(descr.getArg(i)) > 78)
	{
	  fprintf(file, "%s\n", line);
	  sprintf(line, "; ");
	  sprintf(line, "%s %s", line, descr.getArg(i));
	}
	// if its not the end of the line toss this word in
	else
	{
	  sprintf(line, "%s %s", line, descr.getArg(i));
	}
      }
      // put the last line into the file
      fprintf(file, "%s\n", line);
    }

    params = section->getParams();
    // find this parameter
    for (paramIt = params->begin(); paramIt != params->end(); paramIt++)
    {
      commented = false;
      param = &(*paramIt);

      if (written != NULL && 
	  param->getType() != ArConfigArg::DESCRIPTION_HOLDER &&
	  written->find(param->getName()) != written->end())
	continue;
      else if (written != NULL && 
	       param->getType() != ArConfigArg::DESCRIPTION_HOLDER)
      {
	written->insert(param->getName());
      }
      
      // if the type is a functor then we need to handle all of it up
      // here since its a special case both in regards to comments and values
      if (param->getType() == ArConfigArg::FUNCTOR)
      {
	// put the comments in the file first
	sprintf(line, "; ");
	if (param->getDescription() != NULL && 
	    strlen(param->getDescription()) > 0)
	{
	  ArArgumentBuilder descr;
	  descr.add(param->getDescription());
	  for (unsigned int i = 0; i < descr.getArgc(); i++)
	  {
	    // see if we're over, if we are write this line out and start
	    // the next one
	    if (strlen(line) + strlen(descr.getArg(i)) > 78)
	    {
	      fprintf(file, "%s\n", line);
	      sprintf(line, "; ");
	      sprintf(line, "%s %s", line, descr.getArg(i));
	    }
	    // if its not the end of the line toss this word in
	    else
	    {
	      sprintf(line, "%s %s", line, descr.getArg(i));
	    }
	  }
	  // put the last line into the file
	  fprintf(file, "%s\n", line);
	}
	// now put in the values
	argList = param->getArgsWithFunctor();
	if (argList != NULL)
	  for (argIt = argList->begin(); argIt != argList->end(); argIt++)
	    fprintf(file, "%s %s\n", param->getName(), (*argIt)->getFullString());
	
	// okay, we did it all, now continue
	continue;
      }
      
      sprintf(line, "%s", param->getName());
      if (param->getType() == ArConfigArg::INT)
      {
	sprintf(line, "%s %d", line, param->getInt());
      }
      else if (param->getType() == ArConfigArg::DOUBLE)
      {
	sprintf(line, "%s %g", line, param->getDouble());
      }
      else if (param->getType() == ArConfigArg::STRING)
      {
	sprintf(line, "%s %s", line, param->getString());
      }
      else if (param->getType() == ArConfigArg::BOOL)
      {
	sprintf(line, "%s %s", line, param->getBool() ? "true" : "false"); 
      }
      else if (param->getType() == ArConfigArg::FUNCTOR)
      {
      }
      else if (param->getType() == ArConfigArg::DESCRIPTION_HOLDER)
      {
	if (strlen(param->getDescription()) == 0)
	{
	  fprintf(file, "\n");
	  continue;
	}
      }
      else
      {
	ArLog::log(ArLog::Terse, "ArConfig no argument type");
      }
      // configure our start of line part
      if (param->getType() == ArConfigArg::DESCRIPTION_HOLDER)
	sprintf(startLine, "; ");
      else
	sprintf(startLine, "%%-%ds;", startCommentColumn);
      // if our line is already longer then where we want to go put in
      // an extra space
      if (strlen(line) >= startCommentColumn)
	sprintf(line, "%-s ;", line);
      // if its not then just put the start in
      else
	sprintf(line, startLine, line);
      // if its an int we want to put in ranges if there are any
      if (param->getType() == ArConfigArg::INT)
      {
	if (param->getMinInt() != INT_MIN && param->getMaxInt() != INT_MAX)
	{
	  sprintf(line, "%s range [%d, %d], ", line, 
		  param->getMinInt(), param->getMaxInt());
	}
	else if (param->getMinInt() != INT_MIN)
	  sprintf(line, "%s minumum %d, ", line, param->getMinInt());
	else if (param->getMaxInt() != INT_MAX)
	  sprintf(line, "%s maximum %d, ", line, param->getMaxInt());
      }
      if (param->getType() == ArConfigArg::DOUBLE)
      {
	if (param->getMinDouble() != -HUGE_VAL && 
	    param->getMaxDouble() != HUGE_VAL)
	  sprintf(line, "%s range [%g, %g], ", line, param->getMinDouble(), 
		  param->getMaxDouble());
	else if (param->getMinDouble() != -HUGE_VAL)
	  sprintf(line, "%s minimum %g, ", line, 
		  param->getMinDouble());
	else if (param->getMaxDouble() != HUGE_VAL)
	  sprintf(line, "%s Maximum %g, ", line, 
		  param->getMaxDouble());
      }
      
      // if we have a description to put in, put it in with word wrap
      if (param->getDescription() != NULL && 
	  strlen(param->getDescription()) > 0)
      {
	ArArgumentBuilder descr;
	descr.add(param->getDescription());
	for (unsigned int i = 0; i < descr.getArgc(); i++)
	{
	  // see if we're over, if we are write this line out and start
	  // the next one
	  if (strlen(line) + strlen(descr.getArg(i)) > 78)
	  {
	    fprintf(file, "%s\n", line);
	    sprintf(line, startLine, "");
	    sprintf(line, "%s %s", line, descr.getArg(i));
	  }
	  // if its not the end of the line toss this word in
	  else
	  {
	    sprintf(line, "%s %s", line, descr.getArg(i));
	  }
	}
	// put the last line into the file
	fprintf(file, "%s\n", line);
      }
      // if they're no description just end the line
      else 
	fprintf(file, "%s\n", line);
      // if they have a config priority put that on its own line
      if (writePriorities)
      {
	sprintf(line, startLine, "");
	fprintf(file, "%s Priority: %s\n", line,
		ArPriority::getPriorityName(param->getConfigPriority()));
      }
      // now put a blank line between params if we should
      if (!myNoBlanksBetweenParams)
	fprintf(file, "\n");
    }
  }
  fclose(file);
  return true;
}

AREXPORT const char *ArConfig::getBaseDirectory(void) const
{
  return myBaseDirectory.c_str();
}

AREXPORT void ArConfig::setBaseDirectory(const char *baseDirectory)
{
  if (baseDirectory != NULL && strlen(baseDirectory) > 0)
    myBaseDirectory = baseDirectory;
  else
    myBaseDirectory = "";
   
  myParser.setBaseDirectory(baseDirectory);
}

AREXPORT const char *ArConfig::getFileName(void) const
{
  return myFileName.c_str();
}

/**
   After a file has been read all the way these processFileCBs are
   called in the priority (higher numbers first)... these are only
   called if there were no errors parsing the file or
   continueOnError was set to false when parseFile was called

   The functor should return true if the config parsed was good
   (parseFile will return true) false if the config parsed wasn't
   (parseFile will return false)

   @param functor the functor to call 

   @param priority the functors are called in descending order, if two
   things have the same number the first one added is the first one
   called
**/
AREXPORT void ArConfig::addProcessFileCB(ArRetFunctor<bool> *functor,
					 int priority)
{
  myProcessFileCBList.insert(
	  std::pair<int, ProcessFileCBType *>(-priority, 
					      new ProcessFileCBType(functor)));
}

/** 
    Removes a processFileCB, see addProcessFileCB for details
 **/
AREXPORT void ArConfig::remProcessFileCB(ArRetFunctor<bool> *functor)
{
  std::multimap<int, ProcessFileCBType *>::iterator it;
  ProcessFileCBType *cb;

  for (it = myProcessFileCBList.begin(); it != myProcessFileCBList.end(); ++it)
  {
    if ((*it).second->haveFunctor(functor))
    {
      cb = (*it).second;
      myProcessFileCBList.erase(it);
      delete cb;
      remProcessFileCB(functor);
    }
  }
}

/**
   This function has a different name than addProcessFileCB just so
   that if you mean to get this function but have the wrong functor
   you'll get an error.  The rem's are the same though since that
   shouldn't matter.

   After a file has been read all the way these processFileCBs are
   called in the priority (higher numbers first)... these are only
   called if there were no errors parsing the file or
   continueOnError was set to false when parseFile was called

   The functor should return true if the config parsed was good
   (parseFile will return true) false if the config parsed wasn't
   (parseFile will return false)

   @param functor the functor to call (the char * and unsigned int
   should be used by the functor to put an error message into that
   buffer)

   @param priority the functors are called in descending order, if two
   things have the same number the first one added is the first one
   called
**/
AREXPORT void ArConfig::addProcessFileWithErrorCB(
	ArRetFunctor2<bool, char *, size_t> *functor,
	int priority)
{
  myProcessFileCBList.insert(
	  std::pair<int, ProcessFileCBType *>(-priority, 
				      new ProcessFileCBType(functor)));
}

/** 
    Removes a processFileCB, see addProcessFileCB for details
 **/
AREXPORT void ArConfig::remProcessFileCB(
	ArRetFunctor2<bool, char *, size_t> *functor)
{
  std::multimap<int, ProcessFileCBType *>::iterator it;
  ProcessFileCBType *cb;

  for (it = myProcessFileCBList.begin(); it != myProcessFileCBList.end(); ++it)
  {
    if ((*it).second->haveFunctor(functor))
    {
      cb = (*it).second;
      myProcessFileCBList.erase(it);
      delete cb;
      remProcessFileCB(functor);
    }
  }
}

AREXPORT bool ArConfig::callProcessFileCallBacks(bool continueOnErrors,
						 char *errorBuffer,
						 size_t errorBufferLen)
{
  bool ret = true;
  std::multimap<int, ProcessFileCBType *>::iterator it;
  ProcessFileCBType *callback;
  ArLog::LogLevel level = myProcessFileCallbacksLogLevel;

  // reset our section to nothing again
  mySection = "";

  // in windows, can't simply declare an array of errorBufferLen -- so
  // allocate one.
  
  // empty the buffer, we're only going to put the first error in it
  if (errorBuffer != NULL)
    errorBuffer[0] = '\0';

  ArLog::log(level, "ArConfig: Processing file");
  for (it = myProcessFileCBList.begin(); 
       it != myProcessFileCBList.end(); 
       ++it)
  {
    callback = (*it).second;
    if (callback->getName() != NULL && callback->getName()[0] != '\0')
      ArLog::log(level, "ArConfig: Processing functor '%s' (%d)", 
		 callback->getName(), -(*it).first);
    else
      ArLog::log(level, "ArConfig: Processing unnamed functor (%d)", -(*it).first);
    if (!(*it).second->call(errorBuffer, errorBufferLen))
    {
      //printf("# %s\n", scratchBuffer); 

      // if there is an error buffer and it got filled get rid of our
      // pointer to it
      if (errorBuffer != NULL && errorBuffer[0] != '\0')
      {
	errorBuffer = NULL;
	errorBufferLen = 0;
      }
      ret = false;
      if (!continueOnErrors)
      {
	if (callback->getName() != NULL && callback->getName()[0] != '\0')
	  ArLog::log(ArLog::Normal, "ArConfig: Failed, stopping because the '%s' process file callback failed", 
		     callback->getName());
	else
	  ArLog::log(ArLog::Normal, "ArConfig: Failed, stopping because unnamed process file callback failed");
	break;
      }
      else
      {
	if (callback->getName() != NULL && callback->getName()[0] != '\0')
	  ArLog::log(ArLog::Normal, "ArConfig: Failed but continuing, the '%s' process file callback failed", 
		     callback->getName());
	else
	  ArLog::log(ArLog::Normal, "ArConfig: Failed but continuing, an unnamed process file callback failed");
      }

    }
  }
  if (ret || continueOnErrors)
  {
    ArLog::log(level, "ArConfig: Processing with own processFile");
    if (!processFile())
      ret = false;
  }
  ArLog::log(level, "ArConfig: Done processing file, ret is %s", 
	     ArUtil::convertBool(ret));

  return ret;
}


AREXPORT std::list<ArConfigSection *> *ArConfig::getSections(void)
{
  return &mySections;
}

AREXPORT void ArConfig::setNoBlanksBetweenParams(bool noBlanksBetweenParams)
{
  myNoBlanksBetweenParams = noBlanksBetweenParams;
}

AREXPORT bool ArConfig::getNoBlanksBetweenParams(void)
{
  return myNoBlanksBetweenParams;
}

/**
   This argument parser the arguments in to check for parameters of
   this name, note that ONLY the first parameter of this name will be
   used, so if you have duplicates only the first one will be set.
 **/
AREXPORT void ArConfig::useArgumentParser(ArArgumentParser *parser)
{
  myArgumentParser = parser;
}

AREXPORT bool ArConfig::parseArgumentParser(ArArgumentParser *parser, 	
					    bool continueOnError,
					    char *errorBuffer,
					    size_t errorBufferLen)
{
  std::list<ArConfigSection *>::iterator sectionIt;
  std::list<ArConfigArg>::iterator paramIt;
  ArConfigSection *section = NULL;
  ArConfigArg *param = NULL;
  std::list<ArConfigArg> *params = NULL;

  bool ret;
  size_t i;
  std::string strArg;
  std::string strUndashArg;
  ArArgumentBuilder builder;
  bool plainMatch;
  
  for (i = 0; i < parser->getArgc(); i++)
  {
    if (parser->getArg(i) == NULL)
    {
      ArLog::log(ArLog::Terse, "ArConfig set up wrong (parseArgumentParser broken).");
      if (errorBuffer != NULL)
	strncpy(errorBuffer, 
		"ArConfig set up wrong (parseArgumentParser broken).", 
		errorBufferLen);
      return false;
    }
    strArg = parser->getArg(i); 
    if (parser->getArg(i)[0] == '-')
      strUndashArg += &parser->getArg(i)[1]; 
    else
      strUndashArg = "";
    //printf("normal %s undash %s\n", strArg.c_str(), strUndashArg.c_str());
    for (sectionIt = mySections.begin(); 
	 sectionIt != mySections.end(); 
	 sectionIt++)
    {
      section = (*sectionIt);
      params = section->getParams();
      for (paramIt = params->begin(); paramIt != params->end(); paramIt++)
      {
	param = &(*paramIt);
	/*
	printf("### normal %s undash %s %d %d\n", strArg.c_str(), 
	       strUndashArg.c_str(),
	       ArUtil::strcasecmp(param->getName(), strArg),
	       ArUtil::strcasecmp(param->getName(), strUndashArg));
	*/
	if (strlen(param->getName()) > 0 &&
	    ((plainMatch = ArUtil::strcasecmp(param->getName(),strArg)) == 0 ||
	     ArUtil::strcasecmp(param->getName(), strUndashArg) == 0))
	{
	  if (plainMatch == 0)
	    builder.setExtraString(strArg.c_str());
	  else
	    builder.setExtraString(strUndashArg.c_str());
	  if (i+1 < parser->getArgc())
	  {
	    builder.add(parser->getArg(i+1));
	    parser->removeArg(i+1);
	  }
	  parser->removeArg(i);

	  // set us to use the section this is in and then parse the argument
	  std::string oldSection = mySection;
	  bool oldSectionBroken = mySectionBroken;
	  bool oldUsingSections = myUsingSections;
	  bool oldDuplicateParams = myDuplicateParams;
	  mySection = section->getName();
	  mySectionBroken = false;
	  myUsingSections = true;
	  myDuplicateParams = false;
	  ret = parseArgument(&builder, errorBuffer, errorBufferLen);
	  mySection = oldSection;
	  mySectionBroken = oldSectionBroken;
	  myUsingSections = oldUsingSections;
	  myDuplicateParams = oldDuplicateParams;

	  // if we parsed the argument right or are continuing on
	  // errors call ourselves again (so we don't hose iterators up above)
	  if (ret || continueOnError)
	  {
	    //printf("Ret %s\n", ArUtil::convertBool(ret));
	    return ret && parseArgumentParser(parser, continueOnError, 
					     errorBuffer, errorBufferLen);
	  }
	  else
	    return false;
	}
      }
    }
  }

  return true;
}

AREXPORT ArConfigSection::ArConfigSection(const char *name, 
					  const char *comment)
{
  myName = name;
  if (comment != NULL)
    myComment = comment;
  else
    myComment = "";
}


AREXPORT ArConfigSection::ArConfigSection(const ArConfigSection &section) 
{
  myName = section.myName;
  myComment = section.myComment;
  for (std::list<ArConfigArg>::const_iterator it = section.myParams.begin(); 
       it != section.myParams.end(); 
       it++) 
  {
    myParams.push_back(*it);
  }
}

AREXPORT ArConfigSection &ArConfigSection::operator=(const ArConfigSection &section)
{
  if (this != &section) 
  {
    
    myName = section.getName();
    myComment = section.getComment();
    
    myParams.clear();
    
    for (std::list<ArConfigArg>::const_iterator it = section.myParams.begin(); 
	 it != section.myParams.end(); 
	 it++) 
    {
      myParams.push_back(*it);
    }
  }
  return *this;
}



AREXPORT ArConfigSection::~ArConfigSection()
{

}


AREXPORT ArConfigSection *ArConfig::findSection(const char *sectionName) const
{
  ArConfigSection *section = NULL;
  ArConfigSection *tempSection = NULL;

  for (std::list<ArConfigSection *>::const_iterator sectionIt = mySections.begin(); 
       sectionIt != mySections.end(); 
       sectionIt++)
  {
    tempSection = (*sectionIt);
    if (ArUtil::strcasecmp(tempSection->getName(), sectionName) == 0)
    {
      section = tempSection;
    }
  }
  return section;

} // end method findSection

AREXPORT ArConfigArg *ArConfigSection::findParam(const char *paramName)
{
  ArConfigArg *param = NULL;
  ArConfigArg *tempParam = NULL;

  for (std::list<ArConfigArg>::iterator pIter = myParams.begin(); 
       pIter != myParams.end(); 
       pIter++)
  {
    // TODO: Does this do what I'm hoping it does?
    tempParam = &(*pIter);
    if (ArUtil::strcasecmp(tempParam->getName(), paramName) == 0)
    {
      param = tempParam;
    }
  }
  return param;

} // end method findParam



