/*

  Copyright (C) 2004,2005, ActivMedia Robotics, LLC
  Copyright (C) 2006,2007,2008,2009, MobileRobots Inc.

     This program is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published by
     the Free Software Foundation; either version 2 of the License, or
     (at your option) any later version.

     This program is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with this program; if not, write to the Free Software
     Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#ifndef ARCONFIG_H
#define ARCONFIG_H

#include "ArConfigArg.h"
#include "ArFileParser.h"
#include <set>

class ArArgumentBuilder;
class ArConfigSection;

/// Classes dealing with config files can inherit from this one
/**
   
 **/
class ArConfig
{
public:
  /// Constructor
  AREXPORT ArConfig(const char *baseDirectory = NULL, 
		    bool noBlanksBetweenParams = false,
		    bool ignoreBounds = false,
		    bool failOnBadSection = false);
  /// Copy constructor
  AREXPORT ArConfig(const ArConfig &config);

  AREXPORT ArConfig &operator=(const ArConfig &config);

  /// Destructor
  AREXPORT virtual ~ArConfig();
  /// Parse a config file
  AREXPORT bool parseFile(const char *fileName, bool continueOnError = false,
			  bool noFileNotFoundMessage = false, 
			  char *errorBuffer = NULL,
			  size_t errorBufferLen = 0);
  /// Write out a config file with defaults
  AREXPORT bool writeFile(const char *fileName, bool append = false,
			  std::set<std::string> *alreadyWritten = NULL,
			  bool writePriorities = false);
  /// Command to add a parameter to the given section with given priority
  AREXPORT bool addParam(const ArConfigArg &arg, const char *sectionName = "", 
			 ArPriority::Priority priority = ArPriority::NORMAL);
  /// Command to add a new comment to the given section with given priority
  AREXPORT bool addComment(const char *comment, const char *sectionName = "", 
			   ArPriority::Priority priority = ArPriority::NORMAL);
  /// Sets the comment for a section
  AREXPORT void setSectionComment(const char *sectionName, 
				  const char *comment);
  /// Uses this argument parser after it parses a file before it processes
  AREXPORT void useArgumentParser(ArArgumentParser *parser);
  /// for inheritors this is called after the file is processed
  /**
     For classes that inherit from ArConfig this function is called
     after parseFile and all of the processFileCBs are called... If
     you want to call something before the processFileCBs then just
     add a processFileCB... this is only called if there were no
     errors parsing the file or continueOnError was set to false when
     parseFile was called

     @return true if the config parsed was good (parseFile will return
     true) false if the config parsed wasn't (parseFile will return false)
   **/
  AREXPORT virtual bool processFile(void) { return true; }
  /// Adds a processedFile callback
  AREXPORT void addProcessFileCB(ArRetFunctor<bool> *functor, 
				 int priority = 0);
  /// Adds a processedFile callback with error messages
  AREXPORT void addProcessFileWithErrorCB(
	  ArRetFunctor2<bool, char *, size_t> *functor, 
	  int priority = 0);
  /// Removes a processedFile callback
  AREXPORT void remProcessFileCB(ArRetFunctor<bool> *functor);
  /// Removes a processedFile callback
  AREXPORT void remProcessFileCB(
	  ArRetFunctor2<bool, char *, size_t> *functor);
  /// Call the processFileCBs
  AREXPORT bool callProcessFileCallBacks(bool continueOnError,
					 char *errorBuffer,
					 size_t errorBufferLen);
  /// This parses the argument given (for parser or other use)
  AREXPORT bool parseArgument(ArArgumentBuilder *arg, 
			      char *errorBuffer = NULL,
			      size_t errorBufferLen = 0);
  /// This parses the section change (for parser or other use)
  AREXPORT bool parseSection(ArArgumentBuilder *arg, 
			      char *errorBuffer = NULL,
			      size_t errorBufferLen = 0);
  /// Gets the base directory
  AREXPORT const char *getBaseDirectory(void) const;
  /// Sets the base directory
  AREXPORT void setBaseDirectory(const char *baseDirectory);
  /// Gets the file name we loaded
  AREXPORT const char *getFileName(void) const;
  /// Set whether we have blanks between the params or not
  AREXPORT void setNoBlanksBetweenParams(bool noBlanksBetweenParams);

  /// Get whether we have blanks between the params or not
  AREXPORT bool getNoBlanksBetweenParams(void);

  /// Uses an argument parser to change the config
  AREXPORT bool parseArgumentParser(ArArgumentParser *parser,
				    bool continueOnError = false,
				    char *errorBuffer = NULL,
				    size_t errorBufferLen = 0);

  /// Gets the sections themselves (use only if you know what to do)
  AREXPORT std::list<ArConfigSection *> *getSections(void);

  /// Finds the section with the given name.  Returns NULL if not found.
  AREXPORT ArConfigSection *findSection(const char *sectionName) const;
  /// Sets the level we log our map changed callback at
  AREXPORT void setProcessFileCallbacksLogLevel(ArLog::LogLevel level) 
    {  myProcessFileCallbacksLogLevel = level; }
  /// Gets the level we log our map changed callback at
  AREXPORT ArLog::LogLevel getProcessFileCallbacksLogLevel(void)
    {  return myProcessFileCallbacksLogLevel; }
  /// Clears out all the section information
  AREXPORT void clearSections(void);
  /// Clears out all the section information and the processFileCBs
  AREXPORT void clearAll(void);
protected:
  /**
     This class's job is to make the two functor types largely look
     like the same one from the code's perspective, this is so we can
     store them both in the same map for order of operations purposes.
     
     The funkiness with the constructor is because the retfunctor2
     looks like the retfunctor and winds up always falling into that
     constructor.
  **/
  class ProcessFileCBType
  {
    public:
    ProcessFileCBType(
	    ArRetFunctor2<bool, char *, size_t> *functor)
    {
      myCallbackWithError = functor;
      myCallback = NULL;
    }
    ProcessFileCBType(ArRetFunctor<bool> *functor)
    {
      myCallbackWithError = NULL;
      myCallback = functor;
    }
    ~ProcessFileCBType() {}
    bool call(char *errorBuffer, size_t errorBufferLen) 
    { 
      if (myCallbackWithError != NULL) 
	return myCallbackWithError->invokeR(errorBuffer, errorBufferLen);
      else if (myCallback != NULL) 
	return myCallback->invokeR(); 
      // if we get here there's a problem
      ArLog::log(ArLog::Terse, "ArConfig: Horrible problem with process callbacks");
      return false;
    }
    bool haveFunctor(ArRetFunctor2<bool, char *, size_t> *functor)
    { 
      if (myCallbackWithError == functor) 
	return true; 
      else 
	return false; 
    }
    bool haveFunctor(ArRetFunctor<bool> *functor)
    { 
      if (myCallback == functor) 
	return true; 
      else 
	return false; 
    }
    const char *getName(void) 
    { 
      if (myCallbackWithError != NULL)
	return myCallbackWithError->getName();
      else if (myCallback != NULL)
	return myCallback->getName();
      // if we get here there's a problem
      ArLog::log(ArLog::Terse, "ArConfig: Horrible problem with process callback names");
      return NULL;
    }
    protected:
    ArRetFunctor2<bool, char *, size_t> *myCallbackWithError;
    ArRetFunctor<bool> *myCallback;
  };
  void addParserHandlers(void);
  ArArgumentParser *myArgumentParser;
  std::multimap<int, ProcessFileCBType *> myProcessFileCBList;
  bool myNoBlanksBetweenParams;
  std::string mySection;
  bool mySectionBroken;
  bool myUsingSections;
  std::string myFileName;
  std::string myBaseDirectory;
  ArFileParser myParser;
  bool myIgnoreBounds;
  bool myFailOnBadSection;
  bool myDuplicateParams;
  ArLog::LogLevel myProcessFileCallbacksLogLevel;
  // our list of sections which has in it the argument list for each
  std::list<ArConfigSection *> mySections;
  // callback for the file parser
  ArRetFunctor3C<bool, ArConfig, ArArgumentBuilder *, char *, size_t> myParserCB;
  // callback for the section in the file parser
  ArRetFunctor3C<bool, ArConfig, ArArgumentBuilder *, char *, size_t> mySectionCB;
};


/** Represents a section in the configuration. Sections are used to
 *  group items used by seperate parts of Aria.
 */
class ArConfigSection
{
public:
  AREXPORT ArConfigSection(const char *name = NULL, 
						   const char *comment = NULL);
  AREXPORT virtual ~ArConfigSection();
  AREXPORT ArConfigSection(const ArConfigSection &section);
  AREXPORT ArConfigSection &operator=(const ArConfigSection &section);

  /// @return The name of this section
  const char *getName(void) const { return myName.c_str(); }

  /// @return A comment describing this section
  const char *getComment(void) const { return myComment.c_str(); }
  std::list<ArConfigArg> *getParams(void) { return &myParams; }
  void setName(const char *name) { myName = name; }
  void setComment(const char *comment) { myComment = comment; }

  /// Finds a parameter item in this section with the given name.  Returns NULL if not found.
  AREXPORT ArConfigArg *findParam(const char *paramName); 

protected:
  std::string myName;
  std::string myComment;
  std::list<ArConfigArg> myParams;
};

#endif // ARCONFIG
