/*

  Copyright (C) 2004,2005, ActivMedia Robotics, LLC
  Copyright (C) 2006, 2007, 2008, 2009 MobileRobots Inc.

     This program is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published by
     the Free Software Foundation; either version 2 of the License, or
     (at your option) any later version.

     This program is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with this program; if not, write to the Free Software
     Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include "ArExport.h"
#include "ariaOSDef.h"
#include "ArConfigArg.h"
#include "ArLog.h"
#include "ArArgumentBuilder.h"

AREXPORT ArConfigArg::ArConfigArg()
{
  clear();
}

AREXPORT ArConfigArg::ArConfigArg(const char * name, int *pointer, 
		      const char * description, int minInt, int maxInt) 
{ 
  clear();
  myType = INT;
  myIntType = INT_INT;
  myName = name; 
  myDescription = description;
  myMinInt = minInt;
  myMaxInt = maxInt;
  myIntPointer = pointer;
}

AREXPORT ArConfigArg::ArConfigArg(const char * name, short *pointer, 
		      const char * description, int minInt, int maxInt) 
{ 
  clear();
  myType = INT;
  myIntType = INT_SHORT;
  myName = name; 
  myDescription = description;
  myMinInt = minInt;
  myMaxInt = maxInt;
  myIntShortPointer = pointer;
}

AREXPORT ArConfigArg::ArConfigArg(const char * name, unsigned short *pointer, 
		      const char * description, int minInt, int maxInt) 
{ 
  clear();
  myType = INT;
  myIntType = INT_UNSIGNED_SHORT;
  myName = name; 
  myDescription = description;
  myMinInt = minInt;
  myMaxInt = maxInt;
  myIntUnsignedShortPointer = pointer;
}

AREXPORT ArConfigArg::ArConfigArg(const char * name, unsigned char *pointer, 
		      const char * description, int minInt, int maxInt) 
{ 
  clear();
  myType = INT;
  myIntType = INT_UNSIGNED_CHAR;
  myName = name; 
  myDescription = description;
  myMinInt = minInt;
  myMaxInt = maxInt;
  myIntUnsignedCharPointer = pointer;
}

AREXPORT ArConfigArg::ArConfigArg(const char * name, double *pointer,
		      const char * description, double minDouble, 
		      double maxDouble) 
{ 
  clear();
  myType = DOUBLE;
  myName = name; 
  myDescription = description;
  myMinDouble = minDouble;
  myMaxDouble = maxDouble;
  myDoublePointer = pointer;
}

AREXPORT ArConfigArg::ArConfigArg(const char * name, bool *pointer, 
		      const char * description) 
{ 
  clear();
  myType = BOOL;
  myName = name; 
  myDescription = description;
  myBoolPointer = pointer;
}


AREXPORT ArConfigArg::ArConfigArg(const char * name, int val, 
		      const char * description, int minInt, int maxInt) 
{ 
  clear();
  myType = INT;
  myIntType = INT_INT;
  myName = name; 
  myDescription = description;
  myMinInt = minInt;
  myMaxInt = maxInt;
  myIntPointer = new int;
  *myIntPointer = val;
  myOwnPointedTo = true;
}

AREXPORT ArConfigArg::ArConfigArg(const char * name, double val,
		      const char * description, double minDouble, 
		      double maxDouble) 
{ 
  clear();
  myType = DOUBLE;
  myName = name; 
  myDescription = description;
  myMinDouble = minDouble;
  myMaxDouble = maxDouble;
  myDoublePointer = new double;
  *myDoublePointer = val;
  myOwnPointedTo = true;
}

AREXPORT ArConfigArg::ArConfigArg(const char * name, bool val, 
		      const char * description) 
{ 
  clear();
  myType = BOOL;
  myName = name; 
  myDescription = description;
  myBoolPointer = new bool;
  *myBoolPointer = val;
  myOwnPointedTo = true;
}

/**
   This one is tricky because it needs to handle both the pointer case
   and value case which is the same for strings... and so if the
   maxStrLen is given as 0 here then the ArConfigArg class will assume it 
   owns the string and use its own memory for it.
 **/
AREXPORT ArConfigArg::ArConfigArg(const char * name, char *str, 
		      const char * description, size_t maxStrLen) 
{ 
  clear();
  myType = STRING;
  myName = name; 
  myDescription = description;
  if (maxStrLen == 0)
  {
    myUsingOwnedString = true;
    myString = str;
  }
  else
  {
    myStringPointer = str;
    myMaxStrLen = maxStrLen;
  }
}

/**
   This constructor is for the functor type of argument, this is for
   cases that need to be complicated and have more than one argument
   per name... such as the sonar in a config file.  Where this data
   needs to be used to construct internal data structures.
   
   @param setFunctor when an argument is read it is passed to this
   functor which should set up whatever it needs to from the data
   
   @param getFunctor since parameter files need to be written too,
   this get functor will get a list of strings to be written to the file
**/
AREXPORT ArConfigArg::ArConfigArg(const char *name, 
		      ArRetFunctor1<bool, ArArgumentBuilder *> *setFunctor, 
	      ArRetFunctor<const std::list<ArArgumentBuilder *> *> *getFunctor,
		      const char *description)
{
  clear();
  myType = FUNCTOR;
  myName = name;
  myDescription = description;
  mySetFunctor = setFunctor;
  myGetFunctor = getFunctor;
}

AREXPORT ArConfigArg::ArConfigArg(const char * description)
{ 
  clear();
  myType = DESCRIPTION_HOLDER;
  myDescription = description;
}

AREXPORT ArConfigArg::ArConfigArg(const ArConfigArg & arg) 
{
  copy(arg);
}

AREXPORT ArConfigArg &ArConfigArg::operator=(const ArConfigArg & arg) 
{
  if (this != &arg) 
  {
    copy(arg);
  }
  return *this;
}

void ArConfigArg::copy(const ArConfigArg &arg)
{
  clear();
  myType = arg.myType;
  myIntType = arg.myIntType;
  myName = arg.myName;
  myDescription = arg.myDescription;
  myOwnPointedTo = arg.myOwnPointedTo;
  if (arg.myOwnPointedTo && arg.myIntPointer != NULL)
  {
    myIntPointer = new int;
    *myIntPointer = *arg.myIntPointer;
  }
  else
  {
    myIntPointer = arg.myIntPointer;
  }
  if (arg.myOwnPointedTo && arg.myIntShortPointer != NULL)
  {
    myIntShortPointer = new short;
    *myIntShortPointer = *arg.myIntShortPointer;
  }
  else
  {
    myIntShortPointer = arg.myIntShortPointer;
  }
  if (arg.myOwnPointedTo && arg.myIntUnsignedShortPointer != NULL)
  {
    myIntUnsignedShortPointer = new unsigned short;
    *myIntUnsignedShortPointer = *arg.myIntUnsignedShortPointer;
  }
  else
  {
    myIntUnsignedShortPointer = arg.myIntUnsignedShortPointer;
  }
  if (arg.myOwnPointedTo && arg.myIntUnsignedCharPointer != NULL)
  {
    myIntUnsignedCharPointer = new unsigned char;
    *myIntUnsignedCharPointer = *arg.myIntUnsignedCharPointer;
  }
  else
  {
    myIntUnsignedCharPointer = arg.myIntUnsignedCharPointer;
  }
  if (arg.myOwnPointedTo && arg.myDoublePointer != NULL) 
  {
    myDoublePointer = new double;
    *myDoublePointer = *arg.myDoublePointer;
  }
  else
  {
    myDoublePointer = arg.myDoublePointer;
  }
  if (arg.myOwnPointedTo && arg.myBoolPointer != NULL)
  {
    myBoolPointer = new bool;
    *myBoolPointer = *arg.myBoolPointer;
  }
  else
  {
    myBoolPointer = arg.myBoolPointer;
  }
  myStringPointer = arg.myStringPointer;
  myMinInt = arg.myMinInt;
  myMaxInt = arg.myMaxInt;
  myMinDouble = arg.myMinDouble;
  myMaxDouble = arg.myMaxDouble;
  myMaxStrLen = arg.myMaxStrLen;
  mySetFunctor = arg.mySetFunctor;
  myGetFunctor = arg.myGetFunctor;    
  myUsingOwnedString = arg.myUsingOwnedString;
  myString = arg.myString;
  myConfigPriority = arg.myConfigPriority;
  myIgnoreBounds = arg.myIgnoreBounds;
}

AREXPORT ArConfigArg::~ArConfigArg()
{
  clear();
}

void ArConfigArg::clear(void)
{
  myType = INVALID;
  myIntType = INT_NOT;
  myName = "";
  myDescription = "";
  myOwnPointedTo = false;
  if (myOwnPointedTo && myIntPointer != NULL)
    delete myIntPointer;
  myIntPointer = NULL;
  if (myOwnPointedTo && myIntShortPointer != NULL)
    delete myIntShortPointer;
  myIntShortPointer = NULL;
  if (myOwnPointedTo && myIntUnsignedShortPointer != NULL)
    delete myIntUnsignedShortPointer;
  myIntUnsignedShortPointer = NULL;
  if (myOwnPointedTo && myIntUnsignedCharPointer != NULL)
    delete myIntUnsignedCharPointer;
  myIntUnsignedCharPointer = NULL;
  if (myOwnPointedTo && myDoublePointer != NULL)
    delete myDoublePointer;
  myDoublePointer = NULL;
  if (myOwnPointedTo && myBoolPointer != NULL)
    delete myBoolPointer;
  myBoolPointer = NULL;
  myStringPointer = NULL;
  myUsingOwnedString = false;
  myString = "";
  myMinInt = INT_MIN;
  myMaxInt = INT_MAX;
  myMinDouble = -HUGE_VAL;
  myMaxDouble = HUGE_VAL;
  myMaxStrLen = 0;
  mySetFunctor = NULL;
  myGetFunctor = NULL;  
  myConfigPriority = ArPriority::NORMAL;
  myIgnoreBounds = false;
}

/**
   @see INVALID
   @see INT
   @see DOUBLE
   @see BOOL
   @see POSE */
AREXPORT ArConfigArg::Type ArConfigArg::getType(void) const
{
  return myType;
}

AREXPORT int ArConfigArg::getMinInt(void) const
{
  return myMinInt;
}

AREXPORT int ArConfigArg::getMaxInt(void) const
{
  return myMaxInt;
}

AREXPORT double ArConfigArg::getMinDouble(void) const
{
  return myMinDouble;
}

AREXPORT double ArConfigArg::getMaxDouble(void) const
{
  return myMaxDouble;
}

AREXPORT const char *ArConfigArg::getName(void) const
{
  return myName.c_str();
}

AREXPORT const char *ArConfigArg::getDescription(void) const
{
  return myDescription.c_str();
}

AREXPORT int ArConfigArg::getInt(void) const
{ 
  // only one of these will be valid
  if (myIntPointer != NULL)
    return *myIntPointer;
  else if (myIntShortPointer != NULL)
    return *myIntShortPointer;
  else if (myIntUnsignedShortPointer != NULL)
    return *myIntUnsignedShortPointer;
  else if (myIntUnsignedCharPointer != NULL)
    return *myIntUnsignedCharPointer;
  else
    return 0;
}

AREXPORT double ArConfigArg::getDouble(void) const 
{
  if (myDoublePointer != NULL)
    return *myDoublePointer; 
  else
    return 0;
}

AREXPORT bool ArConfigArg::getBool(void) const
{
  if (myBoolPointer != NULL)
    return *myBoolPointer;
  else
    return false;
}

AREXPORT const char *ArConfigArg::getString(void) const
{
  if (myUsingOwnedString)
    return myString.c_str();
  else if (myStringPointer != NULL)
    return myStringPointer;
  else
    return NULL;
}

AREXPORT const std::list<ArArgumentBuilder *> *ArConfigArg::getArgsWithFunctor(void) const
{
  if (myGetFunctor == NULL)
    return NULL;
  else
    return myGetFunctor->invokeR();
}

AREXPORT bool ArConfigArg::setInt(int val, char *errorBuffer, 
				  size_t errorBufferLen)
{
  
  if (!myIgnoreBounds && val < myMinInt)
  {
    ArLog::log(ArLog::Normal, "ArConfigArg of %s: setInt value %d below range [%d, %d]", getName(), val, myMinInt, myMaxInt);
    if (errorBuffer != NULL)
      snprintf(errorBuffer, errorBufferLen, "%s value of %d is below minimum of %d.", getName(), val, myMinInt);
    return false;
  }
  if (!myIgnoreBounds && val > myMaxInt)
  {
    ArLog::log(ArLog::Normal, "ArConfigArg of %s: setInt value %d above range [%d, %d]", getName(), val, myMinInt, myMaxInt);
    if (errorBuffer != NULL)
      snprintf(errorBuffer, errorBufferLen, "%s value of %d is above maximum of %d.", getName(), val, myMaxInt);
    return false;
  }
  if ((myIntType == INT_INT && myIntPointer == NULL) || 
      (myIntType == INT_SHORT && myIntShortPointer == NULL) || 
      (myIntType == INT_UNSIGNED_SHORT && 
       myIntUnsignedShortPointer == NULL) || 
      (myIntType == INT_UNSIGNED_CHAR && myIntUnsignedCharPointer == NULL))
  {
    ArLog::log(ArLog::Normal, "ArConfigArg of %s: setInt called with NULL int pointer.", getName());
    if (errorBuffer != NULL)
      snprintf(errorBuffer, errorBufferLen, "%s pointer is NULL.", getName());
    return false;
  }

  if (myIntType == INT_INT)
    *myIntPointer = val;
  else if (myIntType == INT_SHORT) 
    *myIntShortPointer = val;
  else if (myIntType == INT_UNSIGNED_SHORT) 
    *myIntUnsignedShortPointer = val;
  else if (myIntType == INT_UNSIGNED_CHAR) 
    *myIntUnsignedCharPointer = val;
  else
  {
    ArLog::log(ArLog::Normal, "ArConfigArg of %s: int is bad type.", getName());
    if (errorBuffer != NULL)
      snprintf(errorBuffer, errorBufferLen, "%s int is bad type (%d).", getName(), myIntType);
    return false;
  }
  return true;
}

AREXPORT bool ArConfigArg::setDouble(double val, char *errorBuffer,
				     size_t errorBufferLen)
{ 
  if (!myIgnoreBounds && val < myMinDouble)
  {
    ArLog::log(ArLog::Normal, "ArConfigArg of %s: setDouble value %g below range [%g, %g]", getName(), val, myMinDouble, myMaxDouble);
    if (errorBuffer != NULL)
      snprintf(errorBuffer, errorBufferLen, "%s value of %g is below minimum of %g.", getName(), val, myMinDouble);
    return false;
  }
  if (!myIgnoreBounds && val > myMaxDouble)
  {
    ArLog::log(ArLog::Normal, "ArConfigArg of %s: setDouble value %g above range [%g, %g]", getName(), val, myMinDouble, myMaxDouble);
    if (errorBuffer != NULL)
      snprintf(errorBuffer, errorBufferLen, "%s value of %g is above maximum of %g.", getName(), val, myMaxDouble);
    return false;
  }
  if (myDoublePointer == NULL)
  {
    ArLog::log(ArLog::Normal, "ArConfigArg of %s: setDouble called with NULL pointer.", getName());
    if (errorBuffer != NULL)
      snprintf(errorBuffer, errorBufferLen, "%s pointer is NULL.", getName());
    return false;
  }
  // if we got to here we're good
  *myDoublePointer = val;
  return true;
}


AREXPORT bool ArConfigArg::setBool(bool val, char *errorBuffer,
				   size_t errorBufferLen)
{
  if (myBoolPointer == NULL)
  {
    ArLog::log(ArLog::Normal, "ArConfigArg of %s: setBool called with NULL pointer.", getName());
    if (errorBuffer != NULL)
      snprintf(errorBuffer, errorBufferLen, "%s pointer is NULL.", getName());
    return false;
  }
  *myBoolPointer = val;
  return true;
}

AREXPORT bool ArConfigArg::setString(const char *str, char *errorBuffer,
				     size_t errorBufferLen)
{
  size_t len;
  if (myUsingOwnedString)
  {
    myString = str;
    return true;
  }
  if (myStringPointer == NULL)
  {
    ArLog::log(ArLog::Normal, "ArConfigArg of %s: setString called with NULL pointer.", getName());
    if (errorBuffer != NULL)
      snprintf(errorBuffer, errorBufferLen, "%s pointer is NULL.", getName());
    return false;
  }
  // this is >= so that if it wouldn't have room with NULL that's
  // taken care of too
  if ((len = strlen(str)) >= myMaxStrLen)
  {
    ArLog::log(ArLog::Normal, "ArConfigArg of %s: setString called with argument %d long, when max length is %d.", getName(), len, myMaxStrLen);
    if (errorBuffer != NULL)
      snprintf(errorBuffer, errorBufferLen, "%s string is %d long when max length is %d.", getName(), len, myMaxStrLen);
    return false;
  }
  strcpy(myStringPointer, str);
  return true;
}

AREXPORT bool ArConfigArg::setArgWithFunctor(ArArgumentBuilder *argument, 
					     char *errorBuffer,
					     size_t errorBufferLen)
{
  if (mySetFunctor == NULL)
  {
    ArLog::log(ArLog::Normal, "ArConfigArg of %s: setArgWithFunctor called with NULL pointer.", getName());
    if (errorBuffer != NULL)
      snprintf(errorBuffer, errorBufferLen, "%s pointer is NULL.", getName());
    return false;
  }
  return mySetFunctor->invokeR(argument);
}


AREXPORT void ArConfigArg::log(bool verbose) const
{
  std::list<ArArgumentBuilder *>::const_iterator it;
  const std::list<ArArgumentBuilder *> *argList;
  std::string intType;

  switch (getType()) 
  {
  case ArConfigArg::INVALID:
    ArLog::log(ArLog::Terse, 
	       "\tType: %10s.  This argument was not created properly.", 
	       "invalid");
  case ArConfigArg::INT:
    if (myIntType == INT_NOT)
      intType = "Not";
    else if (myIntType == INT_INT)
      intType = "Int";
  else if (myIntType == INT_SHORT)
      intType = "Short";
    else if (myIntType == INT_UNSIGNED_SHORT)
      intType = "Unsigned Short";
    else if (myIntType == INT_UNSIGNED_CHAR)
      intType = "Unsigned Short";
    else
      intType = "Unknown";
    ArLog::log(ArLog::Terse, "\tType: %10s name: %12s value: %d intType: %s",
	       "int", getName(), getInt(), intType.c_str());
    if (strlen(getDescription()) != 0)
      ArLog::log(ArLog::Terse, "\t\tDescription: %s",
		 getDescription());
    if (verbose)
      ArLog::log(ArLog::Terse, "\t\tMin: %10d     Max: %10d", 
		 myMinInt, myMaxInt);
    break;
  case ArConfigArg::DOUBLE:
    ArLog::log(ArLog::Terse, "\tType: %10s name: %12s value: %f", "double",
	       getName(), getDouble());
    if (strlen(getDescription()) != 0)
      ArLog::log(ArLog::Terse, "\t\tDescription: %s",
		 getDescription());
    if (verbose)
      ArLog::log(ArLog::Terse, "\t\tMin: %10g     Max: %10g", 
		 myMinDouble, myMaxDouble);
    break; 
  case ArConfigArg::STRING:
    ArLog::log(ArLog::Terse, "\tType: %10s name: %12s value: %s", "string", 
               getName(), getString());
    if (strlen(getDescription()) != 0)
      ArLog::log(ArLog::Terse, "\t\tDescription: %s",
                 getDescription());
    if (verbose)
      ArLog::log(ArLog::Terse, "\t\tLength: %d", myMaxStrLen);
    break;
  case ArConfigArg::BOOL:
    ArLog::log(ArLog::Terse, "\tType: %10s name: %12s value: %d", "bool",
	       getName(), getBool());
    if (strlen(getDescription()) != 0)
      ArLog::log(ArLog::Terse, "\t\tDescription: %s",
		 getDescription());
    break;
  case ArConfigArg::FUNCTOR:
    ArLog::log(ArLog::Terse, "\tType: %10s name: %12s", 
	       "functor", getName());
    if (strlen(getDescription()) != 0)
      ArLog::log(ArLog::Terse, "\t\tDescription: %s",
		 getDescription());
    ArLog::log(ArLog::Terse, "\t\tValues:");
    argList = myGetFunctor->invokeR();
    for (it = argList->begin(); it != argList->end(); it++)
      ArLog::log(ArLog::Terse, "\t\t\t%s", (*it)->getFullString());
    break;
  case ArConfigArg::DESCRIPTION_HOLDER:
    ArLog::log(ArLog::Terse, "\tType: %20s Description: %s", 
	       "description_holder", getDescription());

  default:
    ArLog::log(ArLog::Terse, 
	       "\tType: %10s.  This type doesn't have a case in ArConfigArg::print.",
	       "unknown");
    break;
  }

  ArLog::log(ArLog::Terse, "\t\tPriority: %s", 
	     ArPriority::getPriorityName(myConfigPriority));
}

/**
   The priority of this argument when used in ArConfig.
 **/
AREXPORT ArPriority::Priority ArConfigArg::getConfigPriority(void) const
{
  return myConfigPriority;
}

/**
   The priority of this argument when used in ArConfig.
 **/

AREXPORT void ArConfigArg::setConfigPriority(ArPriority::Priority priority)
{
  myConfigPriority = priority;
}

/**
   This is for debugging and will prevent the bounds checking from
   happening, you shouldn't normally use it
 **/
AREXPORT void ArConfigArg::setIgnoreBounds(bool ignoreBounds)
{
  myIgnoreBounds = ignoreBounds;
}
