/*

  Copyright (C) 2004,2005, ActivMedia Robotics, LLC
  Copyright (C) 2006,2007,2008,2009, MobileRobots Inc.

     This program is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published by
     the Free Software Foundation; either version 2 of the License, or
     (at your option) any later version.

     This program is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with this program; if not, write to the Free Software
     Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#ifndef ARMAP_H
#define ARMAP_H

#include "ariaTypedefs.h"
#include "ariaUtil.h"
#include "ArFunctor.h"
#include "ArArgumentBuilder.h"
#include "ArMutex.h"

#include <vector>

class ArFileParser;

class ArMapObject;

/// This is a class for point maps made with ScanStudio and LaserMapper
/**
   This map class has a lock and unlock associated with it, if you use
   the addReadCB and add a callback the map will be locked when the
   callback is called but if you want to get map data not from the
   read callbacks you should lock the map before you get data and
   unlock it when you're done.  The readFile and writeFile
   automatically lock the map while they read and write.
   
   If you are going to use the setMapObjects, setPoints, setLines, or
   setMapInfo then you should lock the map before you do these, do
   them, and then call mapChanged to call the callbacks (it'll only
   call them if needed) and then unlock it when you're done...
 **/
class ArMap
{
  public:
  /// Constructor
  AREXPORT ArMap(const char *baseDirectory = "./");
  /// Destructor
  AREXPORT virtual ~ArMap(void);
  /// Reads a map
  AREXPORT bool readFile(const char *fileName, 
			 char *errorBuffer = NULL, size_t errorBufferLen = 0);

  /// Write out a map file
  AREXPORT bool writeFile(const char *fileName);
  /// Adds a callback called when the map is changed
  AREXPORT void addMapChangedCB(ArFunctor *functor, 
				ArListPos::Pos position = ArListPos::LAST);
  /// Removes a callback called when the map is changed
  AREXPORT void remMapChangedCB(ArFunctor *functor);
  /// Gets the base directory
  AREXPORT const char *getBaseDirectory(void) const;
  /// Gets the fileName that was loaded
  AREXPORT const char *getFileName(void) const;
#if 0
  /// Sets whether we ignore empty file names or fail if we encounter one
  void setIgnoreEmptyFileName(bool ignore) { myIgnoreEmptyFileName = ignore; }
  /// Gets whether we ignore empty file names or fail if we encounter one
  bool getIgnoreEmptyFileName(void) { return myIgnoreEmptyFileName; }
#endif
  /// Sets the base directory
  AREXPORT void setBaseDirectory(const char *baseDirectory);
  /// Gets the map objects
  std::list<ArMapObject *> *getMapObjects(void) { return &myMapObjects; }
  /// Gets the map info strings
  std::list<ArArgumentBuilder *> *getMapInfo(void) { return &myMapInfo; }
  /// Gets the map points
  std::vector<ArPose> *getPoints(void) { return &myPoints; }
  /// Gets the lines
  std::vector<ArLineSegment> *getLines(void) { return &myLines; }
  /// Gets the lower left point of the points
  ArPose getMinPose(void) { return myMin; }
  /// Gets the upper right point of the points
  ArPose getMaxPose(void) { return myMax; }
  /// Gets the lower left point of the lines
  ArPose getLineMinPose(void) { return myLineMin; }
  /// Gets the upper right point of the lines
  ArPose getLineMaxPose(void) { return myLineMax; }
  /// Gets the resolution (-1 if none specified)
  int getResolution(void) { return myResolution; }
  /// Gets the first map object of given name and type if it exists
  AREXPORT ArMapObject *findFirstMapObject(const char *name, 
				      const char *type = NULL);
  /// Gets the last time the map objects were changed
  ArTime getMapObjectsChanged(void) { return myMapObjectsChanged; }
  /// Gets the last time the points were changed
  ArTime getPointsChanged(void) { return myPointsChanged; }
  /// Gets the last time the lines were changed
  ArTime getLinesChanged(void) { return myLinesChanged; }
  /// Gets the last time the map info was changed
  ArTime getMapInfoChanged(void) { return myMapInfoChanged; }
  /// Sets the map objects (copies those passed in)
  AREXPORT void setMapObjects(const std::list<ArMapObject *> *mapObjects); 
  /// Sets the points (copies those passed in)
  AREXPORT void setPoints(const std::vector<ArPose> *points); 
  /// Sets the lines (copies those passed in)
  AREXPORT void setLines(const std::vector<ArLineSegment> *points); 
  /// Sets the map info (copies those passed in)
  AREXPORT void setMapInfo(const std::list<ArArgumentBuilder *> *mapInfo); 
  /// Function that will call the map changed CBs if needed
  AREXPORT void mapChanged(void);

  /// Writes all of the map to a functor instead of a to a file
  AREXPORT void writeToFunctor(ArFunctor1<const char *> *functor, 
			       const char *endOfLineChars);


  /// Parses a map line
  AREXPORT bool parseLine(char *line);
  /// Says that the parsing by lines is done and to use the parsed data
  AREXPORT void parsingComplete(void);

  /// Lock the map instance
  AREXPORT int lock() {return(myMutex.lock());}
  /// Try to lock the map instance without blocking
  AREXPORT int tryLock() {return(myMutex.tryLock());}
  /// Unlock the map instance
  AREXPORT int unlock() {return(myMutex.unlock());}

  

  // When loading a map, returns whether all header, objects, and lines have completed loading.
  /**
   * This value returns true once the DATA tag has been reached.  
   * The rest of the map contains data points.
  **/
  AREXPORT bool isLoadingDataStarted() {return myLoadingDataStarted;}

  // When loading a map, returns whether all header and objects have completed loading.
  /**
   * This value returns true once the LINES tag has been reached.  
   * The rest of the map contains data points.
  **/
  AREXPORT bool isLoadingLinesAndDataStarted() 
               {return myLoadingLinesAndDataStarted;}

  /// Writes the map header information and objects to a text-based functor.
  /**
   * This method writes everything up to and including the DATA tag
   * to the given functor.  The data points are not written.
  **/
  AREXPORT void writeObjectsToFunctor
					(ArFunctor1<const char *> *functor, 
			         const char *endOfLineChars);

  /// Writes the map data points to a functor.
  /**
   * A pointer to the entire data point vector is passed directly to the 
   * functor in order to improve performance.  The functor should not
   * modify the vector's contents.
  **/
  AREXPORT void writePointsToFunctor
			(ArFunctor2<int, std::vector<ArPose> *> *functor);
  /// Writes the map line segments to a functor.
  /**
   * A pointer to the entire line segment vector is passed directly to the 
   * functor in order to improve performance.  The functor should not
   * modify the vector's contents.
  **/
  AREXPORT void writeLinesToFunctor
		(ArFunctor2<int, std::vector<ArLineSegment> *> *functor);

  /// Reads a data point from the given line, and adds it to the loading points.
  AREXPORT bool readDataPoint( char *line);
  /// Reads a line segment from the given line, and adds it to the loading lines.
  AREXPORT bool readLineSegment( char *line);
  /// Adds the specified data point to the loading points.
  AREXPORT void loadDataPoint(double x, double y);
  /// Adds the specified line segment to the loading lines.
  AREXPORT void loadLineSegment(double x1, double y1, double x2, double y2);
#if 0
  /// Sets the level we log our map changed callback at
  AREXPORT void setMapChangedLogLevel(ArLog::LogLevel level) 
    {  myMapChangedLogLevel = level; }
  /// Gets the level we log our map changed callback at
  AREXPORT ArLog::LogLevel getMapChangedLogLevel(void)
    {  return myMapChangedLogLevel; }
#endif
protected:

#if 0
  // Function for processing the config file
  bool processFile(char *errorBuffer, size_t errorBufferLen);
#endif
  // Function to read the 2D-Map
  bool handle2DMap(ArArgumentBuilder *arg);
  // Function to read the minimum pos
  bool handleMinPos(ArArgumentBuilder *arg);
  // Function to read the maximum pos
  bool handleMaxPos(ArArgumentBuilder *arg);
  // Function to read the number of points
  bool handleNumPoints(ArArgumentBuilder *arg);
  // Function to read the line minimum pos
  bool handleLineMinPos(ArArgumentBuilder *arg);
  // Function to read the line maximum pos
  bool handleLineMaxPos(ArArgumentBuilder *arg);
  // Function to read the number of lines
  bool handleNumLines(ArArgumentBuilder *arg);
  // Function to handle the resolution
  bool handleResolution(ArArgumentBuilder *arg);
  // Function to handle the cairns
  bool handleMapObject(ArArgumentBuilder *arg);
  // Function to handle the information about hte file
  bool handleInfo(ArArgumentBuilder *arg);
  // Function to catch the LINES line signifying data
  bool handleLines(ArArgumentBuilder *arg);
  // Function to catch the DATA line signifying data
  bool handleData(ArArgumentBuilder *arg);
  // Function to snag the map points (not used now)
  //bool handlePoint(ArArgumentBuilder *arg);

  // lock for our data
  ArMutex myMutex;
  // resets the parser functions and variables (true if good, false otherwise)
  AREXPORT bool reset(void);
  std::string myBaseDirectory;
  std::string myFileName;
  struct stat myReadFileStat;
  ArFileParser *myLoadingParser;
  // vars for if we got some important info, the other important info
  // is taken care of by the adding of callbacks
  bool myLoadingGot2DMap;
  bool myLoadingGotMaxPos;
  bool myLoadingGotMinPos;
  bool myLoadingGotLineMaxPos;
  bool myLoadingGotLineMinPos;
  ArPose myLoadingMaxFromFile;
  ArPose myLoadingMinFromFile;
  int myLoadingPointsRead;
  ArPose myLoadingLineMaxFromFile;
  ArPose myLoadingLineMinFromFile;
  int myLoadingLinesRead;

//  std::string myConfigParam;
//  bool myIgnoreEmptyFileName;
  // data from the file
  std::list<ArMapObject *> myLoadingMapObjects;
  std::vector<ArPose> myLoadingPoints;
  std::vector<ArLineSegment> myLoadingLines;
  std::list<ArArgumentBuilder *> myLoadingMapInfo;
  int myLoadingNumPoints;
  int myLoadingNumLines;
  int myLoadingResolution;
  ArPose myLoadingMax;
  ArPose myLoadingMin;
  ArPose myLoadingLineMax;
  ArPose myLoadingLineMin;

  // our good data in memory (could be the same thing)
  std::list<ArMapObject *> myMapObjects;
  ArTime myMapObjectsChanged;
  std::vector<ArPose> myPoints;
  ArTime myPointsChanged;

  std::vector<ArLineSegment> myLines;
  ArTime myLinesChanged;

  std::list<ArArgumentBuilder *> myMapInfo;
  ArTime myMapInfoChanged;

  int myNumPoints;
  int myNumLines;
  int myResolution;
  ArPose myMax;
  ArPose myMin;
  ArPose myLineMax;
  ArPose myLineMin;


  std::list<ArFunctor *> myMapChangedCBList;
  ArTime myMapChangedMapObjects;
  ArTime myMapChangedPoints;
  ArTime myMapChangedMapInfo;

#if 0
  // things for our config
  bool myConfigProcessedBefore;
  char myConfigMapName[512];
#endif

  bool myLoadingDataStarted;
  bool myLoadingLinesAndDataStarted;
  ArLog::LogLevel myMapChangedLogLevel;
  // callbacks
  ArRetFunctor1C<bool, ArMap, ArArgumentBuilder *> my2DMapCB;
  ArRetFunctor1C<bool, ArMap, ArArgumentBuilder *> myMinPosCB;
  ArRetFunctor1C<bool, ArMap, ArArgumentBuilder *> myMaxPosCB;
  ArRetFunctor1C<bool, ArMap, ArArgumentBuilder *> myNumPointsCB;
  ArRetFunctor1C<bool, ArMap, ArArgumentBuilder *> myLineMinPosCB;
  ArRetFunctor1C<bool, ArMap, ArArgumentBuilder *> myLineMaxPosCB;
  ArRetFunctor1C<bool, ArMap, ArArgumentBuilder *> myNumLinesCB;
  ArRetFunctor1C<bool, ArMap, ArArgumentBuilder *> myResolutionCB;
  ArRetFunctor1C<bool, ArMap, ArArgumentBuilder *> myMapObjectCB;
  ArRetFunctor1C<bool, ArMap, ArArgumentBuilder *> myInfoCB;
  ArRetFunctor1C<bool, ArMap, ArArgumentBuilder *> myDataCB;
  ArRetFunctor1C<bool, ArMap, ArArgumentBuilder *> myLinesCB;
  //ArRetFunctor1C<bool, ArMap, ArArgumentBuilder *> myPointCB;
#if 0
  ArRetFunctor2C<bool, ArMap, char *, size_t> myProcessFileCB;
#endif
};

/// This is a class for objects within an ArMap
class ArMapObject
{
public: 
  /// Constructor
  AREXPORT ArMapObject(const char *type, ArPose pose, const char *fileName,
		       const char *iconName, const char *name,
		       bool hasFromTo, ArPose fromPose, ArPose toPose)
    {
      if (type != NULL) myType = type; 
      if (name != NULL) myName = name; 
      myPose = pose; 
      if (iconName != NULL) myIconName = iconName;
      if (fileName != NULL) myFileName = fileName;
      myHasFromTo = hasFromTo; myFromPose = fromPose;  myToPose = toPose;
    }
  /// Copy constructor
  AREXPORT ArMapObject(const ArMapObject &mapObject) 
    {
      myType = mapObject.myType; myName = mapObject.myName; 
      myPose = mapObject.myPose; myIconName = mapObject.myIconName;
      myFileName = mapObject.myFileName;  myHasFromTo = mapObject.myHasFromTo;
      myFromPose = mapObject.myFromPose; myToPose = mapObject.myToPose;
    }

  /// Destructor
  AREXPORT virtual ~ArMapObject() {}
  /// Gets the type of the object
  const char *getType(void) const { return myType.c_str(); }
  /// Gets the pose of the object 
  ArPose getPose(void) const { return myPose; }
  /// Gets the fileName of the object (probably never used for maps)
  const char *getFileName(void) const { return myFileName.c_str(); }
  /// Gets the icon string of the object 
  const char *getIconName(void) const { return myIconName.c_str(); }
  /// Gets the name of the object (if any)
  const char *getName(void) const { return myName.c_str(); }
  /// Gets the addition args of the object
  bool hasFromTo(void) const { return myHasFromTo; }
  /// Gets the from pose (could be for line or box, depending)
  ArPose getFromPose(void) const { return myFromPose; }
  /// Gets the to pose (could be for line or box, depending)
  ArPose getToPose(void) const { return myToPose; }
  void log(void) { 
    if (myHasFromTo)
      ArLog::log(ArLog::Terse, 
		 "Cairn: %s %g %g %g \"%s\" %s \"%s\" %d %d %d %d",
		 getType(), myPose.getX(), myPose.getY(), myPose.getTh(), 
		 getFileName(), getIconName(), getName(), myFromPose.getX(),
		 myFromPose.getY(), myToPose.getX(), myToPose.getY());
    else
      ArLog::log(ArLog::Terse, "Cairn: %s %g %g %g \"%s\" %s \"%s\"",
		 getType(), myPose.getX(), myPose.getY(), myPose.getTh(), 
		 getFileName(), getIconName(), getName());
  }
protected:
  std::string myType;
  std::string myName;
  ArPose myPose;
  std::string myFileName;
  std::string myIconName;
  bool myHasFromTo;
  ArPose myFromPose;
  ArPose myToPose;
};


#endif // ARMAP_H
