/* 
 *
 *  Copyright (C) 2005, ActivMedia Robotics
 *  Copyright (C) 2006, 2007, 2008, 2009 MobileRobots, Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include "EmulatePioneer.hh"

#include "MobileSim.hh"

#include <pthread.h>
#include <unistd.h>
#include <string.h>
#include <vector>

#include <stdint.h>
#include <string>

#ifdef _WIN32
#include <winsock2.h>
#else
#include <netinet/in.h>
#endif

#include <signal.h>


/* Some useful classes from Aria */
#include "ArCommands.h"
#include "ArSocket.h"
#include "ArFunctor.h"
#include "ArTcpConnection.h"
#include "ArRobotPacket.h"
#include "ArRobotPacketSender.h"
#include "ArRobotPacketReceiver.h"
#include "ArMutex.h"
#include "ArScopedLock.h"

#include "MobileSim.hh"

/* Turn these on to get debugging messages: */
//#define DEBUG_COMMANDS_RECIEVED 1
//#define DEBUG_SIP_VALUES 1  // Every SIP! Very frequent!
//#define DEBUG_SIP_PACKET_CONTENTS 1 // Every SIP! Very frequent, lots of data!
//#define DEBUG_SIP_SONAR_DATA 1
//#define DEBUG_CLIENT_CONNECTION 1

// Maximum number of sonar readings to include in one SIP. Note, we don't 
// simulated sonar polling timing.  This number must be small enough to
// avoid sending a SIP larger than 200 bytes. 
#define MAX_SONAR_READINGS_PER_SIP 16

#define TCP_PORT_RETRY_TIME 3000 //ms
#define TCP_PORT_RETRY_LIMIT 3

#define CLIENT_CONNECTION_TIMEOUT 20000 // ms (20 sec.)

// There are four ways to deal with the sync handshake:
// First is passive, just echo back what the client sends, but warn on out of
// sequence packets. This is done if no preprocessor symbol is defined.
// The last three attempt to enforce correct sequence.
// RESTART_ON_SYNC_ERROR means to restart the handshake over at 0 (this is the spec'ed behavior that 
// ARCOS does, and ARIA should deal with.), TOLERATE_SYNC_ERROR is to just try the same sync
// sequence number again, DISCONNECT_ON_SYNC_ERROR is to close the client connection (which is the
// default if neither is defined). In any case, we disconnect if the handshake
// hasn't been successful after 10 seconds (TOLERATE) or iterations (RESTART or
// ignore error) of trying.
//#define RESTART_ON_SYNC_ERROR 1
//#define TOLERATE_SYNC_ERROR 1
//#define DISCONNECT_ON_SYNC_ERROR 1

using namespace std;

unsigned long EmulatePioneer::lifetimeThreadCount = 0;
unsigned long EmulatePioneer::lifetimeConnectionCount = 0;
unsigned long EmulatePioneer::lifetimeFailedConnectionCount = 0;
unsigned long EmulatePioneer::currentEmulatorCount = 0;


inline double min(double x, double y) { return ((x <= y) ? x : y); }
inline double max(double x, double y) { return ((x >= y) ? x : y); }
inline int min(int x, int y) { return ((x <= y) ? x : y); }
inline int max(int x, int y) { return ((x >= y) ? x : y); }

#define WARN_DEPRECATED(oldname, oldnum, newname, newnum) { \
  if(myVerbose) robotInterface->warn("Deprecated simulator %s command (%d) received. Use %s (%d) instead.", oldname, oldnum, newname, newnum); \
}

#define WARN_DEPRECATED_QUIETLY(oldname, oldnum, newname, newnum) { \
  if(myVerbose) robotInterface->log("Deprecated simulator %s command (%d) received. Use %s (%d) instead in future versions.", oldname, oldnum, newname, newnum); \
}


std::string toPrintable(unsigned char c)
{
  char buf[12];
  if(c >= 32 && c <= 126)
  {
    snprintf(buf, 11, "%02x('%c') ", c, c);
    return std::string(buf);
  }
  if(c == 0x0d) return "0d(CR) ";
  if(c == 0x0a) return "0a(LF) ";
  if(c == 0x04) return "04(EOF) ";
  snprintf(buf, 11, "%02x ", c);
  return std::string(buf);
}


/** The first byte of a command indicates the type of the following argument */
class ArgTypes
{
public:
  enum {
    POSINT = 0x3B,
    NEGINT = 0x1B,
    STRING = 0x2B
  };
};

/** Some commands for partial compatability with SRISim */
enum
{
  OLD_LRF_ENABLE = 35,
  OLD_LRF_CFG_START = 36,
  OLD_LRF_CFG_END = 37,
  OLD_LRF_CFG_INC = 38,
  OLD_END_SIM = 62,
  OLD_SET_TRUE_X = 66,
  OLD_SET_TRUE_Y = 67,
  OLD_SET_TRUE_TH = 68,
  OLD_RESET_TO_ORIGIN = 69
};


/** Base class for packet generators */
class PacketGenerator 
{
protected:
  RobotInterface* robotInterface; ///< Robot interface to obtain data from
  RobotParams* params; ///< Pointer to global robot paremeters struct
  bool started;        ///< Whether we should return packets or NULL
  ArRobotPacket pkt;   ///< A packet object to set data fields in, then return in getPacket()
  unsigned short deviceIndex; ///< If there is more than one packet generator, this identifies them.

public:

  /** Constructor, initialize driver table and parameters, and initialize
   * 'started' to false. */
  PacketGenerator(RobotInterface* _iface, RobotParams* _params, unsigned short _deviceIndex = 0) :
    robotInterface(_iface),
    params(_params),
    started(false),
    deviceIndex(_deviceIndex)
  {}

  virtual ~PacketGenerator()
  {}

  /** set started to true. */
  void start() {
    started = true;
  }

  /** Set started to false. */
  void stop() {
    started = false;
  }

  /** Return value of started. */
  bool isStarted() const { return started; }

  /** Fill in a packet (a member object, pkt) with new data and return its
   * pointer (so please don't deallocate the returned object!)
   */
  virtual ArRobotPacket* getPacket() = 0;

  RobotInterface* getRobotInterface() const { return robotInterface; }

  unsigned short getDeviceIndex() const { return deviceIndex; }
};

/** Packet generator for standard SIPs.
 *  Create one instance of this class per client session.
 *
 *  @todo One possible optimization is instead of recreating the SIP all the time from scratch,
 *  use a quickly updatable SIP: keep pointers into its buffer for the things
 *  you could change (However, you will still need to truncate it back and add
 *  the sonar data in each time, then add the stuff that comes after it).
 */
class SIPGenerator  : public virtual PacketGenerator
{
protected:
  ArTypes::Byte2 xPosAccum, yPosAccum;
  int firstSonarReadingToSend;

public:
  SIPGenerator(RobotInterface* _robotInterface, RobotParams* _params) :
    PacketGenerator(_robotInterface, _params),
    xPosAccum(0), yPosAccum(0), 
    firstSonarReadingToSend(0)
  {
  }



  /** Get a finalized SIP with current data.
   *  @return packet filled with current data, or NULL if we haven't been
   *  "started" yet.
   */
  virtual ArRobotPacket* getPacket();

};



/** Generate laser data packets */
class LaserPacketGenerator  : public virtual PacketGenerator
{
protected:
  size_t currentReading;
  bool extendedInfoFormat;
public:
  LaserPacketGenerator(RobotInterface* _interface, RobotParams* _params, int deviceIndex = 0) :
    PacketGenerator(_interface, _params, deviceIndex),
    currentReading(0), extendedInfoFormat(false)
  {}

  /* Construct one laser data packet, if available. This will only contain some of the laser
   * readings: there are too many to put in one packet.  After a call to this
   * method returns the last packet in a group, it returns NULL. The next call
   * will return a packet with the first group of packets.   
   * If there is no laser data available, or we haven't been started with
   * start(), then return NULL.
   * If the ExtraReadingInfo flag is set, then the packet ID
   * will be 0x61 instead of 0x60, and each reading in the packet will be followed 
   * by two bytes of extra info. The first 4 bits of the first byte contains a 
   * reflectance value (0 for normal).
   *
   * So, you can use this method like this:
   * @code
   * ArRobotPacket *laserPacket;
   * while(laserPacket = laserGenerator.getPacket())
   *     send(laserPacket);
   * @endcode
   */
  virtual ArRobotPacket* getPacket();

  virtual void start(bool enableExtraReadingInfo = false) {
    extendedInfoFormat = enableExtraReadingInfo;
    robotInterface->openLaser();
    PacketGenerator::start();
  }

  virtual void stop() {
    robotInterface->closeLaser();
    PacketGenerator::stop();
  }

  /** A debugging tool, assert that this is a laser packet, then print it out */
  void printLaserPacket(ArRobotPacket* pkt) const;
};




bool EmulatePioneer::mapMasterEnabled = false;
ArMutex EmulatePioneer::mapMasterMutex;


// Constructor:
// Retrieve options from the configuration file, and get ready to open
// TCP socket to listen on.
EmulatePioneer::EmulatePioneer(RobotInterface* rif, std::string robotName, int port, bool endThreadOnDisconnect, bool deleteRobotInterface, bool trySubsequentPorts, bool verbose) :
      shouldThreadExit(false),
      threadRunning(false),
      threadCreated(false),
      robotInterface(rif),
      myTCPPort(port),
      myRequestedTCPPort(port),
      myClientSocket(NULL),
      myPortOpened(false),
      myClientConnected(false),
      myClientDisconnectCB(this, &EmulatePioneer::clientDisconnected),
      myEndThreadOnDisconnect(endThreadOnDisconnect),
      haveInitialPose(false), requestedOpenSonar(false),
      myApplicationName(""), myApplicationVersion(""),
      myDeleteRobotInterfaceOnDisconnect(deleteRobotInterface),
      myDeleteClientSocketOnDisconnect(false),
      myTrySubsequentPorts(trySubsequentPorts),
      myVerbose(verbose),
      inWatchdogState(false),
      myListenAddress(NULL),
      myThreadError(0),
      inEstop(false),
      sendingSimstats(false)
{
  memset(params.RobotName, 0, ROBOT_IDENTIFIER_LEN);
  strncpy(params.RobotName, robotName.c_str(), ROBOT_IDENTIFIER_LEN);
#ifdef DEBUG_CLIENT_CONNECTION
  myClientConnectedMutex.setLogName("EmulatePioneer client connected flag mutex");
  myClientConnectedMutex.setLog(true);
#endif
  ArSocket::init();
  ++currentEmulatorCount;
}

// Constructor: just set our client socket, assumed to already be connected
// to a client.
EmulatePioneer::EmulatePioneer(RobotInterface *rif, std::string robotName, ArSocket *clientSocket, bool endThreadOnDisconnect, bool deleteRobotInterfaceOnDisconnect, bool deleteClientSocketOnDisconnect, bool verbose) :
      shouldThreadExit(false),
      threadRunning(false),
      threadCreated(false),
      robotInterface(rif),
      myTCPPort(-1),
      myRequestedTCPPort(-1),
      myClientSocket(clientSocket),
      myPortOpened(false),
      myClientConnected(false),
      myClientDisconnectCB(this, &EmulatePioneer::clientDisconnected),
      myEndThreadOnDisconnect(endThreadOnDisconnect),
      haveInitialPose(false), requestedOpenSonar(false),
      myApplicationName(""), myApplicationVersion(""),
      myDeleteRobotInterfaceOnDisconnect(deleteRobotInterfaceOnDisconnect),
      myDeleteClientSocketOnDisconnect(deleteClientSocketOnDisconnect),
      myTrySubsequentPorts(false), // irrelevant in this mode, actually.
      myVerbose(verbose),
      inWatchdogState(false),
      myListenAddress(NULL),
      myThreadError(0)
{
  memset(params.RobotName, 0, ROBOT_IDENTIFIER_LEN);
  strncpy(params.RobotName, robotName.c_str(), ROBOT_IDENTIFIER_LEN);
#ifdef DEBUG_CLIENT_CONNECTION
  myClientConnectedMutex.setLogName("EmulatePioneer client connected flag mutex");
  myClientConnectedMutex.setLog(true);
#endif
  ++currentEmulatorCount;
}



EmulatePioneer::~EmulatePioneer()
{
  // std::string name =  robotInterface->getRobotName();
  // printf("*** ~EmulatePioneer %s...\n", name.c_str()); fflush(stdout);
  if(threadRunning)
  { 
  //  printf("*** ~EmulatePioneer %s cancelling thread!\n", name.c_str()); fflush(stdout);
    pthread_cancel(thread);

  }
  threadStopped();
  myClientConnected = false;
  if(myDeleteRobotInterfaceOnDisconnect && robotInterface)
  {
    //puts(" ~~~ Deleting robot interface... ~~~ "); fflush(stdout);
    delete robotInterface;
    //puts(" ~~~ Deleted robot interface. ~~~ "); fflush(stdout);
  }
  if(myDeleteClientSocketOnDisconnect && myClientSocket)
  {
    delete myClientSocket;
  }
  removeStoredEmulator(this);

  --currentEmulatorCount;
  // printf("*** ~EmulatePioneer %s done!\n", name.c_str()); fflush(stdout);
}


static void* runEmulatePioneerMain(void* ptr)
{
  ((EmulatePioneer*)ptr)->Main();
  ((EmulatePioneer*)ptr)->threadStopped();
  return 0;
}

static void* runEmulatePioneerMainThenDelete(void *ptr)
{
  runEmulatePioneerMain(ptr);
  EmulatePioneer *ep = (EmulatePioneer*)ptr;
  delete ep;
  return 0;
}


bool EmulatePioneer::runThread( bool deleteSelfOnThreadExit, bool /*deleteRobotInterface*/ ) 
{
  myThreadError = 0;
  if(deleteSelfOnThreadExit)
    myThreadError = pthread_create(&thread, NULL, &runEmulatePioneerMainThenDelete, this);
  else
    myThreadError = pthread_create(&thread, NULL, &runEmulatePioneerMain, this);
  if(myThreadError != 0)
  {
    fprintf(stderr, "*** Pioneer emulation: Critical error creating new thread (have created %lu threads so far, have %lu EmulatePioneer objects currently): ", lifetimeThreadCount, currentEmulatorCount);
    fflush(stderr);
    if(myThreadError == EAGAIN) { 
      fputs("system thread resources exhausted (EAGAIN).\n", stderr);
    } else if(myThreadError == EINVAL) {
      fputs("EINVAL\n", stderr);
    } else if(myThreadError == EPERM) {
      fputs("EPERM\n", stderr);
    } else if(myThreadError == ENOMEM) {
      fputs("out of memory (ENOMEM).\n", stderr);
    } else {
      fprintf(stderr, "unknown error: %d\n", myThreadError);
    }
    fprintf(stderr, "*** Pioneer emulation: Currently have %d RobotInterface objects, %d EmulatePioneer objects, %d RobotFactory objects.\n", robotInterfaces.size(), emulators.size(), factories.size());
    fflush(stderr);
    abort();
    //return false;
  }
  threadCreated = true;
  pthread_detach(thread);  // so thread is deallocated when it exits
  ++lifetimeThreadCount;
  return true;
}




int EmulatePioneer::getPort() 
{
  return myTCPPort;
}

void EmulatePioneer::Main() 
{
  ArTime startThread;

  class ScopeWatch
  {
    bool *var;
  public:
    ScopeWatch(bool *vp) : var(vp) {
      *var = true;
    }
    ~ScopeWatch() {
      *var = false;
    }
  };
  ScopeWatch w(&threadRunning);

  while(true)
  {
    // Each iteraton of this loop corresponds to one client connection and
    // session.
    
    ArTime timer;
    pthread_testcancel();
    if(shouldThreadExit) return;

    ArTcpConnection myTCPConnection;
    ArRobotPacketReceiver myPacketReceiver;
    ArRobotPacketSender myPacketSender;

    robotInterface->log("(debug) next connection will be the %lu'th connection (with %lu failed connections also previously). There are %lu robots currently.", lifetimeConnectionCount+1, lifetimeFailedConnectionCount, currentEmulatorCount);
    /* New session, reconfigure robot interface and create new object for
     * session settings. */
    robotInterface->setup(&params);
    robotInterface->setOdom(0, 0, 0);
    CurrentSettings settings(&params);

    if(!haveInitialPose)
    {
      // Only set these once, but we need to have set up the robot interface
      // before using it.
      long z;
      robotInterface->getSimulatorPose(initialPoseX, initialPoseY, z, initialPoseTheta);
      haveInitialPose = true;
    }


    /* If a client socket wasn't provided, then listen on a new port and
     * wait for a client to connect.
     */
    ArSocket newClientSocket;
    bool madeOwnClientSocket = false;
    if(!myClientSocket)
    {
      //myListenSocket.setNonBlock(false);
      // First try a few times to open the requested port (it may still be open
      // from a very recent run of MobileSim)
      for(size_t i = 0; i < TCP_PORT_RETRY_LIMIT-1; ++i)
      {
        if(myListenSocket.open(myTCPPort, ArSocket::TCP, myListenAddress))
        {
          myPortOpened = true;
          break;
        }
        robotInterface->inform("TCP port %d unavailable. Trying again in %d seconds...", myTCPPort, TCP_PORT_RETRY_TIME/1000);
        ArUtil::sleep(TCP_PORT_RETRY_TIME);
      }

      // keep trying until we get an open port or hit the
      // upper limit for TCP ports
      if(!myPortOpened)
      {
        for(myRequestedTCPPort = myTCPPort; myTrySubsequentPorts && myTCPPort <= 65535; ++myTCPPort)
        {
          if(myListenSocket.open(myTCPPort, ArSocket::TCP, myListenAddress) )
          {
            myPortOpened = true;
            break;
          }
        }
      }

      if(!myPortOpened) {
        robotInterface->error("Could not open a TCP port!");
        return;
      }

      if(myRequestedTCPPort != myTCPPort)
        robotInterface->warn("Requested TCP port %d was unavailable. Using %d instead.", myRequestedTCPPort, myTCPPort);
      if(myListenAddress != NULL)
        robotInterface->inform("Ready for a client to connect to address %s on TCP port %d.", myListenAddress, myTCPPort);
      else
        robotInterface->inform("Ready for a client to connect on TCP port %d.", myTCPPort);

      myListenSocket.setReuseAddress();
      if(!myListenSocket.accept(&newClientSocket))
      {
        robotInterface->warn("Error accepting a client connection!");
        continue;
      }

      if(newClientSocket.getFD() < 0)
      {
        // Try again
        continue;
      }

    
      // Close listening socket to prevent other clients from screwing it up by
      // trying to connect.
      myListenSocket.close();

      myClientSocket = &newClientSocket;
      madeOwnClientSocket = true;

#ifdef DEBUG_CLIENT_CONNECTION
      printf("Accepted a client. on the TCP socket %d.\n", newClientSocket.getFD());
#endif
    // end setting up our own socket
    }
    
    /*
    if(!myClientSocket->setNoDelay())
    {
      robotInterface->log("Note, unable to set No Delay option on client socket (this is an optimization, no harm, but it ought to have worked.)");
    }
    */
    myClientSocket->setReuseAddress();
    myTCPConnection.setSocket(myClientSocket);
    myTCPConnection.setStatus(ArDeviceConnection::STATUS_OPEN);
    myPacketReceiver.setDeviceConnection(&myTCPConnection);
    myPacketSender.setDeviceConnection(&myTCPConnection);

    //timer.setToNow();
    myClientConnectedMutex.lock(); 
    myClientConnected = true;
    myClientConnectedMutex.unlock(); 
    //robotInterface->log("%d ms to lock client connected mutex.", timer.mSecSince());
    myClientSocket->setCloseCallback(&myClientDisconnectCB);
    

    /* Handshake with the client: */
    
    int syncSeq = 0;
    int attempt = 0;
    ArTime syncStart;
    while(syncSeq < 3 && clientIsConnected() && attempt++ < 10 
#ifdef TOLERATE_SYNC_ERROR
      && syncStart.mSecSince() <= 10000
#endif
    )
    {
      pthread_testcancel();
      if(shouldThreadExit) return;

      if(myVerbose) robotInterface->log("Waiting for SYNC %d...",  syncSeq);
      ArTime rectime;
      ArRobotPacket *pkt = NULL;
      /*
      if(attempt == 1)
      {
        robotInterface->log("first attempt, flushing any buffered packets.");
        ArRobotPacket *p = myPacketReceiver.receivePacket(SYNC_READ_TIMEOUT);
        while(p != NULL)
        {
          p = myPacketReceiver.receivePacket(0);
          if(p) {
            pkt = p;
            robotInterface->log("read a buffered packet (%d)...", p->getID());
          }
        }
      }
      else
      */
      {
        pkt = myPacketReceiver.receivePacket(SYNC_READ_TIMEOUT);
      }

      pthread_testcancel();
      if(shouldThreadExit) return;

      if(!pkt)
      {
        robotInterface->warn("Error or timeout (timeout is %d, and it took %d msec)  waiting for SYNC%d packet from client. Start again from SYNC0...", SYNC_READ_TIMEOUT, rectime.mSecSince(), syncSeq);
        clientDisconnected();
        continue;
      }

      if(myVerbose) robotInterface->log("\t SYNC %d received.", pkt->getID());

      bool correctSeq = pkt->getID() == syncSeq;
      bool correctChecksum = pkt->verifyCheckSum();

      if(!correctSeq || !correctChecksum)
      {
        if(pkt->getID() != syncSeq)
        {
  #ifdef RESTART_ON_SYNC_ERROR
          robotInterface->warn("Recieved out of sequence SYNC packet: got %d, expecting %d. Start again from SYNC0...", pkt->getID(), syncSeq);
          syncSeq = 0;
          continue;
  #elif defined(TOLERATE_SYNC_ERROR)
          robotInterface->warn("Recieved out of sequence SYNC packet: got %d, expecting %d. Sending %d again...", pkt->getID(), syncSeq, syncSeq);
          myPacketSender.com(syncSeq);
          continue;
  #elif defined(DISCONNECT_ON_SYNC_ERROR)
          // Drop connection instead
          robotInterface->warn("Recieved out of sequence SYNC packet: got %d, expecting %d. Closing connection.", pkt->getID(), syncSeq);
          ++lifetimeFailedConnectionCount;
          return;
  #else
          // just warn and use the client's sequence.
          robotInterface->warn("Recieved out of sequence SYNC packet: got %d, expecting %d.", pkt->getID(), syncSeq);
          syncSeq = pkt->getID();
  #endif
        }

        if(!pkt->verifyCheckSum())
        {
  #ifdef RESTART_ON_SYNC_ERROR
          robotInterface->warn("Recieved SYNC%d packet with bad checksum. Start again from SYNC0...", pkt->getID());
          syncSeq = 0;
          continue;
  #elif defined(TOLERATE_SYNC_ERROR)
          robotInterface->warn("Recieved SYNC%d packet with bad checksum. Sending %d again...", pkt->getID(), syncSeq);
          myPacketSender.com(syncSeq);
          continue;
  #elif defined (DISCONNECT_ON_SYNC_ERROR)
          // Drop connection instead
          robotInterface->warn("Recieved SYNC%d packet with bad checksum. Closing connection.", pkt->getID());
          ++lifetimeFailedConnectionCount;
          return;
  #else
          robotInterface->warn("Recieved SYNC%d packet with bad checksum.", pkt->getID());
  #endif
        }

      }

      // OK, reply 
      if(syncSeq == 2)
      {
        robotInterface->log("Sending SYNC2 with robot config info.");
        ArRobotPacket pkt;
        pkt.setID(2);
        pkt.strToBuf("MobileSim");
        pkt.strToBuf(params.RobotClass);
        pkt.strToBuf(params.RobotSubclass);
        pkt.finalizePacket();
        if(myTCPConnection.write(pkt.getBuf(), pkt.getLength()) == -1)
        {
          robotInterface->warn("Error sending robot indentification strings to the client. Closing connection.");
          clientDisconnected();
          if(myEndThreadOnDisconnect) return;
          continue;
        }
      }
      else
      {
        robotInterface->log("Sending SYNC%d...", syncSeq);
        myPacketSender.com(syncSeq);
      }

      ++syncSeq;

    }

    if(attempt >= 10)
    {
      robotInterface->warn("Failed to handshake after 10 iterations of trying! Closing connection.");
      ++lifetimeFailedConnectionCount;
      return;
    }


#ifdef TOLERATE_SYNC_ERRORS
    if(syncStart.mSecSince() >= 10000)
    {
      robotInterface->warn("Failed to handshake after 10 sec of trying! Closing connection.",);
      ++lifetimeFailedConnectionCount;
      return;
    }
#endif
      

    if(!clientIsConnected())
    {
      robotInterface->warn("Client disconnected during SYNC%d. Closing connection.", syncSeq);
      ++lifetimeFailedConnectionCount;
      //if(myEndThreadOnDisconnect) return;
      //continue;
      return;
    }

    robotInterface->inform("Client connected.");

    ++lifetimeConnectionCount;

    /* New SIP and laser packet generators */
    SIPGenerator sipGen(robotInterface, &params);
    ArTime sentLastSIP;
    sentLastSIP.setToNow();

    // ARIA assumes sonar are on at start:
    robotInterface->openSonar();

    LaserPacketGenerator laserGen(robotInterface, &params);

    /* Used to send a reply packet (e.g. CONFIG, SIMSTAT) */
    ArRobotPacket replyPkt;
 
    /* Reset some state */
    gotLastCommand.setToNow();
    gotLastValidCommand.setToNow();
    inEstop = false;
    sendingSimstats = false;
    
    const double defaultBatteryCharge = 13.0; // todo make this default value configurable in world file.
    robotInterface->setBatteryVoltage(defaultBatteryCharge); // todo make this default value configurable in world file.
    robotInterface->setHaveTemperature(false); // todo make this default value configurable in world file.

    /* Process commands */
    while(clientIsConnected())
    {
      // This loop's scope comprises a client's session. During one loop,
      // it sends packets to the client when it's time, checks for 
      // incoming commands, and does other misc. stuff.
      // To end a client's session, do this:
      //    clientDisconnected();
      //    if(myEndThreadOnDisconnect) return;
      //    continue;
      // clientDisconnected() does various cleanup, and sets a flag
      // that will cause clientIsConnected() to return false after continuing
      // to the next loop iteration.  Return will exit this function which will
      // end the thread and perhaps cause the EmulatePioneer object to be 
      // destroyed. (desired behavior if we were created by a RobotFactory)
    
      pthread_testcancel();
      if(shouldThreadExit) return;

      // If it's been too long since we received a command, stop the robot and
      // warn (like the robot's watchdog behavior).  If it's been way too long,
      // exit.
      if(params.WatchdogTime != 0 && gotLastValidCommand.mSecSince() > params.WatchdogTime && !inWatchdogState)
      {
        robotInterface->warn("Have not received a vaild command packet in %d msec. (Watchdog timeout is %d msec.) [no action taken other than this warning]", gotLastValidCommand.mSecSince(), params.WatchdogTime);
        //robotInterface->stop();
        // BUG need to resume last motion command when communications are
        // restored
        inWatchdogState = true;
      }

      if(gotLastCommand.mSecSince() > CLIENT_CONNECTION_TIMEOUT)
      {
        robotInterface->warn("Have not received any data from client in %d msec, ending connection.", gotLastCommand.mSecSince());
        myTCPConnection.close();
        clientDisconnected();
        if(myEndThreadOnDisconnect) return;
        continue;
      }

      // check if we should leave estop state:
      if(inEstop)
      {
        int x, y, theta;
        robotInterface->getVelocity(x, y, theta);
        if(x == 0 && y == 0 && theta == 0)
        {
          if(myVerbose)
            robotInterface->inform("Done ESTOPping");
          inEstop = false;
        }
      }

      // If it's been too long send data packets:
      // TODO: optionally check stage simulation time with stg_world_get_time()
      // instead of real time. (so that clients can be synchronized to
      // simulation time, if it's not running at real time)
      if(sentLastSIP.mSecSince() >= params.SIPFreq)
      {
        // Send SIMSTAT if requested
        if(sendingSimstats)
        {
          if(!sendSIMSTAT(&myTCPConnection))
          {
             robotInterface->warn("Error sending SIMSTAT packet to a client. Stopping.");
             sendingSimstats = false;
          }
        }

        ArRobotPacket* sip = sipGen.getPacket();
          
        // NOTE if sipGen returns NULL because SIP packets were disabled by
        // calling sipGen.stop(), then we may never know that a client is
        // disconnected because we will never get an error on write. This is OK
        // if the only time sipGen.stop() is called, is from a CLOSE command,
        // since the client connection is forced closed then anyway.
        if(sip)
        {
          if(sip->getLength() > 200)
            robotInterface->warn("I'm sending a SIP that's more than 200 bytes long (%d bytes)! (This violates the protocol and ought not happen.)", sip->getLength());

          if(myTCPConnection.write(sip->getBuf(), sip->getLength()) < 0)
          {
            robotInterface->warn("Error sending SIP to client. Closing connection.");
            clientDisconnected();       
            if(myEndThreadOnDisconnect) return;
            continue;
          }
        }
        sentLastSIP.setToNow();

        // Send all pending laser packets (only some data is in each packet)
        while(ArRobotPacket* laser = laserGen.getPacket())
        {
          if(myTCPConnection.write(laser->getBuf(), laser->getLength()) == -1)
          {
            robotInterface->warn("Error sending laser packet to client. Closing connection.");
            clientDisconnected();       
            if(myEndThreadOnDisconnect) return;
            continue;
          }
        }


      }


      // Get packet from client and act on it:
      ArRobotPacket* pkt = myPacketReceiver.receivePacket(params.SIPFreq/2);


      if(!pkt) continue; // no packet. 

      gotLastCommand.setToNow();

      if(!pkt->verifyCheckSum())
      {
        robotInterface->warn("Recieved a packet (id=%d) with an incorrect checksum; ignoring this packet.", pkt->getID());
        continue;
      }

      gotLastValidCommand.setToNow();
      if(inWatchdogState)
      {
        robotInterface->inform("Data reception restored, watchdog warning reset.");
        inWatchdogState = false;
        // BUG if were previously inWatchdogState, ought to resume previous motion
        // commands.
      }

      if(myCommandsToIgnore.find(pkt->getID()) != myCommandsToIgnore.end())
      {
        if(myVerbose) robotInterface->warn("Ignoring command %d, as requested in program options.", pkt->getID());
        continue;
      }


      

      char argType;
      ArTypes::Byte2 intVal;
      int x, y, th;
      char byte1, byte2, byte3;
      double left, right;
      const int charbuf_maxlen = 128;
      char charbuf[charbuf_maxlen];
      char c;
      char len;
      ///@todo make this more robust against bad commands and warn (useful for
      //debugging bugs in client programs too). check arg type, check packet
      //size. might need a fancier command->function map than just switch soon.
      switch(pkt->getID())
      { 
        case ArCommands::PULSE: 
           break;
        case ArCommands::OPEN:
           sipGen.start();
           break;
        case ArCommands::CLOSE:
            robotInterface->inform("Got CLOSE command, closing connection.");
            sipGen.stop();
            clientDisconnected();
            if(myEndThreadOnDisconnect) return;
            continue;
           break;

        case ArCommands::SONAR:
           argType = pkt->bufToByte();
           intVal = pkt->bufToByte2();
           if(intVal) {
             //robotInterface->inform("Sonar on");
             robotInterface->openSonar();
             requestedOpenSonar = true;
           } else {
             //robotInterface->inform("Sonar off");
             robotInterface->closeSonar();
             requestedOpenSonar = false;
           }
           break;

        case ArCommands::ENABLE:
           argType = pkt->bufToByte();
           intVal = pkt->bufToByte2();
           robotInterface->enableMotors(intVal);
           break;



        case ArCommands::VEL:
          if(inEstop) break;
           intVal = getIntFromPacket(pkt);
#ifdef DEBUG_COMMANDS_RECIEVED
     fprintf(stderr, "Pioneer emulation: <command>   VEL   = %4d mm/s\n", intVal);
     fflush(stderr);
#endif
           robotInterface->transVel(intVal);
           break;

        case ArCommands::VEL2:
          if(inEstop) break;
           // The byte parameters are signed, but the argument type still
           // applies to each, I guess. Consistent, but confusing. The fact
           // that it's right,left instead of left,right is just plain
           // confusing :)
           intVal = getIntFromPacket(pkt);
           left = (double)((char)intVal) * params.Vel2DivFactor;       // low 8 bits
           right = (double)((char)(intVal>>8)) * params.Vel2DivFactor;   // high 8 bits
#ifdef DEBUG_COMMANDS_RECIEVED
           byte2 = (char)intVal;      // low 8 bits
           byte1 = (char)(intVal >> 8);  // high 8 bits
           fprintf(stderr, "Pioneer emulation: <command>   VEL2  = L: %d (=> %.4f mm/s), R: %d (=> %.4f mm/s) [arg type: %d; Vel2Div=%f]\n", byte2, left, byte1, right, argType, params.Vel2DivFactor);
           printf("setting rotVel:%f deg/s, transVel:%f mm/s\n", (float)(ArMath::radToDeg((right - left) / 2.0 ) * params.DiffConvFactor), (float)( (left + right) / 2.0 ));
           fflush(stderr);
#endif
           robotInterface->rotVel( (int) ( ArMath::radToDeg((right - left) / 2.0) * params.DiffConvFactor ) );
           robotInterface->transVel( (int) ( (left + right) / 2.0 ) );
           break;


        case ArCommands::RVEL:
        case ArCommands::ROTATE:
          if(inEstop) break;
           intVal = getIntFromPacket(pkt);
#ifdef DEBUG_COMMANDS_RECIEVED
           fprintf(stderr, "Pioneer emulation: <command>   RVEL  =   %4d deg/s\n", intVal);
           fflush(stderr);
#endif
          robotInterface->rotVel(intVal);
          break;

        case ArCommands::LATVEL: //110
          if(inEstop) break;
          intVal = getIntFromPacket(pkt);
#ifdef DEBUG_COMMANDS_RECIEVED
          fprintf(stderr, "Pioneer emulation: <command>  LATVEL =  %4d mm/s\n", intVal);
          fflush(stderr);
#endif
          robotInterface->latVel(intVal);

          break;

        case ArCommands::MOVE:
          if(inEstop) break;
          intVal = getIntFromPacket(pkt);
#ifdef DEBUG_COMMANDS_RECIEVED
          fprintf(stderr, "Pioneer emulation: <command>   MOVE  =   %4d mm\n", intVal);
          fflush(stderr);
#endif
          robotInterface->move(intVal);
          break;


        case ArCommands::HEAD:  // absolute heading
           if(inEstop) break;
           argType = pkt->bufToByte();
           intVal = pkt->bufToByte2();
           if(argType == ArgTypes::NEGINT)
             intVal = -intVal;
#ifdef DEBUG_COMMANDS_RECIEVED
     fprintf(stderr, "Pioneer emulation: <command>   HEAD  =  %4d deg      speed = %d deg/s\n", intVal, settings.RotVelMax);
           fflush(stderr);
#endif
           robotInterface->heading(intVal);
           break;

        case ArCommands::DHEAD: // relative heading
          if(inEstop) break;
           intVal = getIntFromPacket(pkt);
#ifdef DEBUG_COMMANDS_RECIEVED
     fprintf(stderr, "Pioneer emulation: <command>   DHEAD =  %4d deg      speed = %d deg/s\n", intVal, settings.RotVelMax);
           fflush(stderr);
#endif
           robotInterface->deltaHeading(intVal);
           break;

        case ArCommands::STOP:
          if(inEstop) break;
#ifdef DEBUG_COMMANDS_RECIEVED
          fprintf(stderr, "Pioneer emulation: <command>   STOP\n");
          fflush(stderr);
#endif
          robotInterface->stop();
          break;

        case ArCommands::QSTOP:
#ifdef DEBUG_COMMANDS_RECIEVED
          fprintf(stderr, "Pioneer emulation: <command>   ESTOP\n");
          fflush(stderr);
#endif
          if(myVerbose)
            robotInterface->inform("Received ESTOP/QSTOP command, ESTOPping...");
          robotInterface->stop(FAKE_TRANSACCTOP, FAKE_ROTACCTOP);
          inEstop = true;
          break;


        case ArCommands::SETA:
          argType = pkt->bufToByte();
          intVal = pkt->bufToByte2();
          if(argType == ArgTypes::NEGINT)
          {
            if(intVal < 0)
            {
              robotInterface->error("negative value given for deceleration in SETA command from client! aborting");
              assert(intVal >= 0);
            }
      
            settings.TransDecel = intVal;
#ifdef DEBUG_COMMANDS_RECIEVED
            fprintf(stderr, "Pioneer emulation: <command>   SETA  = -%d mm/s2. Setting robot deceleration to %d.\n", intVal, settings.TransDecel);
            fflush(stderr);
#endif
            robotInterface->setDecel(settings.TransDecel);
          }
          else
          {
            settings.TransAccel = intVal;
#ifdef DEBUG_COMMANDS_RECIEVED
            fprintf(stderr, "Pioneer emulation: <command>   SETA  = %d mm/s2. Setting robot acceleration to %d.\n", intVal, settings.TransAccel);
            fflush(stderr);
#endif
            robotInterface->setAccel(settings.TransAccel);
          }
          break;

        case ArCommands::SETRA:
          argType = pkt->bufToByte();
          intVal = pkt->bufToByte2();
          if(argType == ArgTypes::NEGINT)
          {
            settings.RotDecel = intVal;
#ifdef DEBUG_COMMANDS_RECIEVED
            fprintf(stderr, "Pioneer emulation: <command>   SETRA = -%3d mm/s2\n", intVal);
            fflush(stderr);
#endif
            robotInterface->setRotDecel(settings.RotDecel);
          }
          else
          {
            settings.RotAccel = intVal;
#ifdef DEBUG_COMMANDS_RECIEVED
            fprintf(stderr, "Pioneer emulation: <command>   SETRA = %4d mm/s2\n", intVal);
            fflush(stderr);
#endif
            robotInterface->setRotAccel(settings.RotAccel);
          }
          break;

        case ArCommands::LATACCEL:   // 113
          argType = pkt->bufToByte();
          intVal = pkt->bufToByte2();
          if(argType == ArgTypes::NEGINT)
          {
            settings.LatDecel = intVal;
            robotInterface->setLatDecel(settings.LatDecel);
          }
          else
          {
            settings.LatAccel = intVal;
            robotInterface->setLatAccel(settings.LatAccel);
          }
          break;

        /* TODO
        case ArCommand::SETLATV:
          break;
        */

        case ArCommands::SETV:
          argType = pkt->bufToByte();
          settings.TransVelMax = pkt->bufToByte2();
#ifdef DEBUG_COMMANDS_RECIEVED
          fprintf(stderr, "Pioneer emulation: <command>   SETV  = %4d mm/s\n", settings.TransVelMax);
          fflush(stderr);
#endif
          robotInterface->setDefaultVel(settings.TransVelMax);
          break;

        case ArCommands::SETRV:
          argType = pkt->bufToByte();
          settings.RotVelMax = pkt->bufToByte2();
#ifdef DEBUG_COMMANDS_RECIEVED
          fprintf(stderr, "Pioneer emulation: <command>   SETRV = %4d mm/s\n", settings.RotVelMax);
          fflush(stderr);
#endif
          robotInterface->setDefaultRotVel(settings.RotVelMax);
          break;

        case ArCommands::BUMPSTALL:
          argType = pkt->bufToByte();
          intVal = pkt->bufToByte2();
          settings.BumpStallFront = (intVal == 1 || intVal == 3);
          settings.BumpStallRear = (intVal == 2 || intVal == 3);
          break;

        case ArCommands::CONFIG:
           replyPkt.empty();
           replyPkt.setID(0x20);
           replyPkt.strToBuf(params.RobotClass);
           replyPkt.strToBuf(params.RobotSubclass);
           replyPkt.strToBuf("SIM");  // Serial number
           replyPkt.byteToBuf(0);  // nothing (used to indicate that robot is an AT with P2OS)
           replyPkt.byte2ToBuf(FAKE_ROTVELTOP);
           replyPkt.byte2ToBuf(FAKE_TRANSVELTOP);
           replyPkt.byte2ToBuf(FAKE_ROTACCTOP);
           replyPkt.byte2ToBuf(FAKE_TRANSACCTOP);
           replyPkt.byte2ToBuf(400); // max pwm const. 
           replyPkt.strToBuf("MobileSim"); // Robot name, always MobileSim
           replyPkt.byteToBuf(params.SIPFreq); 
           replyPkt.byteToBuf(0);  // host baud, irrelevant for TCP
           replyPkt.byteToBuf(0);  // baud rate for AUX1
           replyPkt.byte2ToBuf(robotInterface->haveGripper()?1:0); // have gripper
           replyPkt.byte2ToBuf(robotInterface->haveFrontSonar()?1:0); // have front sonar
           replyPkt.byteToBuf(robotInterface->haveRearSonar()?1:0);  // have rear sonar
           replyPkt.byte2ToBuf(0); // low battery alarm (decivolts)
           replyPkt.byte2ToBuf(0); // revcount const.
           replyPkt.byte2ToBuf(params.WatchdogTime); 
           replyPkt.byteToBuf(0);  // nothing
           replyPkt.byte2ToBuf(0); // stall PWM limit const.
           replyPkt.byte2ToBuf(0); // post-stall idle time, ms
           replyPkt.byte2ToBuf(0); // joyvel const.
           replyPkt.byte2ToBuf(0); // joyrvel const.
           replyPkt.byte2ToBuf(settings.RotVelMax); // current max rot vel
           replyPkt.byte2ToBuf(settings.TransVelMax); // current max trans vel
           replyPkt.byte2ToBuf(settings.RotAccel); // current rot acc
           replyPkt.byte2ToBuf(settings.RotDecel); // current rot decel 
           replyPkt.byte2ToBuf(0); // current rot kp
           replyPkt.byte2ToBuf(0); // current rot kv 
           replyPkt.byte2ToBuf(0); // current rot ki 
           replyPkt.byte2ToBuf(settings.TransAccel); // current trans acc
           replyPkt.byte2ToBuf(settings.TransDecel); // current trans decel
           replyPkt.byte2ToBuf(0); // current trans kp 
           replyPkt.byte2ToBuf(0); // current trans kv
           replyPkt.byte2ToBuf(0); // current trans ki
           replyPkt.byteToBuf(0); // number of front bumper segments TODO
           replyPkt.byteToBuf(0); // number of rear bumper segments TODO
           replyPkt.byteToBuf(0); // charger type
           replyPkt.byteToBuf(0); // sonar cycle
           replyPkt.byteToBuf(2); // autobaud
           replyPkt.byteToBuf(0); // has gyro 
           replyPkt.byte2ToBuf(0); // driftfactor
           replyPkt.byteToBuf(0);  // AUX2 baud rate
           replyPkt.byteToBuf(0);  // AUX3 baud rate
           replyPkt.byte2ToBuf(0); // encoder ticks per mm
           replyPkt.byte2ToBuf(0); // voltage beneath which everything is shut down
           replyPkt.byteToBuf(ARCOS_VER_MAJ);
           replyPkt.byteToBuf('.');
           replyPkt.byteToBuf(ARCOS_VER_MIN);
           replyPkt.byteToBuf(0);  // ARCOS version number end string
           replyPkt.byte2ToBuf(0); // CW gyro error
           replyPkt.byte2ToBuf(0); // CCW gyro error
           replyPkt.byteToBuf(0);  // Kinematic delay TODO
           replyPkt.byte2ToBuf(FAKE_LATVELTOP);
           replyPkt.byte2ToBuf(FAKE_LATACCTOP);
           replyPkt.byte2ToBuf(settings.LatVelMax);
           replyPkt.byte2ToBuf(settings.LatAccel);
           replyPkt.byte2ToBuf(settings.LatDecel);
           replyPkt.byte2ToBuf(0); // charge threshold (powerbot)
           replyPkt.finalizePacket();
           if(!myTCPConnection.write(replyPkt.getBuf(), replyPkt.getLength()))
             robotInterface->warn("Error sending CONFIG packet to client. Continuing.");
           break;

        case ArCommands::BATTEST:
          argType = pkt->bufToByte();
          intVal = pkt->bufToByte2();
          if(intVal == 0) 
          {
            robotInterface->inform("Resetting battery voltage value to default value of %.2f volts (reset from client command %d", defaultBatteryCharge, ArCommands::BATTEST);
            robotInterface->setBatteryVoltage(defaultBatteryCharge);
          }
          else
          {
            robotInterface->inform("Setting battery voltage value to %.2f volts (override from client command %d", (double)intVal / 10.0, ArCommands::BATTEST);
            robotInterface->setBatteryVoltage( (double)intVal / 10.0 );
          }
          break;

        case 251: // DIGTEMPTEST
          argType = pkt->bufToByte();
          intVal = pkt->bufToByte2();
          robotInterface->inform("Setting temperature warning flag %s (override from client command 251)", intVal?"on":"off");
          robotInterface->setTempWarning(intVal);
          break;

        case 252: // SPOOFANALOGTEMP
          intVal = getIntFromPacket(pkt);
          if(intVal == -128) 
          {
            robotInterface->inform("Resetting temperature to disabled (from client command 252)");
            robotInterface->setHaveTemperature(false);
          }
          else 
          {
            robotInterface->inform("Setting temperature value %d (override from client command 252)", intVal);
            robotInterface->setTemperature((double)intVal);
          }
          break;

        case ArCommands::SETO:
           argType = pkt->bufToByte();
           if(pkt->getDataLength() > 0) {
             x = pkt->bufToByte2();
             y = pkt->bufToByte2();
             th = pkt->bufToByte2();
             robotInterface->inform("Client requested odometry set to %d, %d, %d with SETO command.", x, y, th);
             robotInterface->setOdom(x, y, th);
           } else {
             robotInterface->inform("Client requested odometry reset to 0,0,0 with SETO command.");
             robotInterface->setOdom(0, 0, 0);
           }
           break;

        case OLD_RESET_TO_ORIGIN:
           WARN_DEPRECATED("OLDSIM_RESETTOORIGIN", pkt->getID(), "SIM_RESET", ArCommands::SIM_RESET);
        case ArCommands::SIM_RESET:
           robotInterface->inform_s("Client requested simulator reset. Moving robot back to its initial pose.");
           robotInterface->setOdom(0, 0, 0);
           robotInterface->resetSimulatorPose();
           break;
          
        case ArCommands::SIM_SET_POSE:
          argType = pkt->bufToByte(); // not used

          // 4 bytes for each value
          x = pkt->bufToByte4();
          y = pkt->bufToByte4();
          th = pkt->bufToByte4();
          //printf("Got SIM_SET_POSE (%d, %d, %d)\n", x, y, th);
          
          // Set pose in sim:
          if(myVerbose) robotInterface->inform("Client requested move to true pose (%d mm, %d mm, %d deg).", x, y, th);
          robotInterface->setSimulatorPose(x, y, 0, th);
          break;

        case OLD_SET_TRUE_X:
        {
           WARN_DEPRECATED("SET_X", pkt->getID(), "SIM_SET_POSE", ArCommands::SIM_SET_POSE);
           argType = pkt->bufToByte();
           long x, y, z;
           robotInterface->getSimulatorPose(x, y, z, th);
           x = pkt->bufToByte2();
           robotInterface->setSimulatorPose(x, y, z, th);
           break;
        }

        case OLD_SET_TRUE_Y:
        {
           WARN_DEPRECATED("SET_Y", pkt->getID(), "SIM_SET_POSE", ArCommands::SIM_SET_POSE);
           argType = pkt->bufToByte();
           long x, y, z;
           robotInterface->getSimulatorPose(x, y, z, th);
           y = pkt->bufToByte2();
           robotInterface->setSimulatorPose(x, y, z, th);
           break;
        }
          
        case OLD_SET_TRUE_TH:
        {
           WARN_DEPRECATED("SET_THETA", pkt->getID(), "SIM_SET_POSE", ArCommands::SIM_SET_POSE);
           long x, y, z;
           robotInterface->getSimulatorPose(x, y, z, th);
           th = pkt->bufToByte();
           robotInterface->setSimulatorPose(x, y, z, th);
           break;
        }

        case OLD_LRF_ENABLE:
           WARN_DEPRECATED_QUIETLY("LRF_ENABLE", pkt->getID(), "SIM_LRF_ENABLE", ArCommands::SIM_LRF_ENABLE);
        case ArCommands::SIM_LRF_ENABLE:
           argType = pkt->bufToByte();
           intVal = pkt->bufToByte2();
           if(intVal)
             laserGen.start(intVal == 2);
           else
             laserGen.stop();
           break;

        case OLD_LRF_CFG_START:
           WARN_DEPRECATED_QUIETLY("LRF_CFG_START", pkt->getID(), "SIM_LRF_SET_FOV_START", ArCommands::SIM_LRF_SET_FOV_START);
        case ArCommands::SIM_LRF_SET_FOV_START:
           intVal = getIntFromPacket(pkt);
           robotInterface->setLaserAngles(intVal, robotInterface->getLaserEndAngle());
           break;

        case OLD_LRF_CFG_END:
           WARN_DEPRECATED_QUIETLY("LRF_CFG_END", pkt->getID(), "SIM_LRF_SET_FOV_END", ArCommands::SIM_LRF_SET_FOV_END);
        case ArCommands::SIM_LRF_SET_FOV_END:
           intVal = getIntFromPacket(pkt);
           robotInterface->setLaserAngles(robotInterface->getLaserStartAngle(), intVal);
           break;

        case OLD_LRF_CFG_INC:
           WARN_DEPRECATED_QUIETLY("LRF_CFG_INC", pkt->getID(), "SIM_LRF_SET_RES", ArCommands::SIM_LRF_SET_RES);
        case ArCommands::SIM_LRF_SET_RES:
           argType = pkt->bufToByte();
           intVal = pkt->bufToByte2();
           // ignore arg type, it can only be positive
           robotInterface->setLaserResolution((double)intVal/100.0);
           break;

        case 59:
           argType = pkt->bufToByte();
           len = pkt->bufToUByte();
           // first three chars are special codes.
           byte1 = pkt->bufToByte();  
           byte2 = pkt->bufToByte();
           byte3 = pkt->bufToByte();
           len -= 3;
           goto msg_getchars;
        case ArCommands::SIM_MESSAGE:
           argType = pkt->bufToByte();
           len = pkt->bufToByte();
        msg_getchars:
           memset(charbuf, 0, charbuf_maxlen);
           for(int i = 0; i < charbuf_maxlen && i < len && i < (MAX_PACKET_PAYLOAD_SIZE-1) &&  (c = pkt->bufToByte()); ++i)
             charbuf[i] = c;
#ifdef DEBUG_COMMAND_RECIEVED
           fprintf(stderr, "Pioneer emulation (%s): Message from client (%d bytes): %s\n", params.RobotName, len, charbuf);
#endif
           robotInterface->inform_s(charbuf);
           break;

        case ArCommands::SIM_EXIT:
           argType = pkt->bufToByte();
           intVal = pkt->bufToByte2();
           robotInterface->inform("Client requested exit with code %d (SIM_EXIT command).", intVal);
           // Wait a few seconds to let the user see the message before exiting.
           ArUtil::sleep(1500);
           robotInterface->shutdown(intVal);
           break;

        case OLD_END_SIM:
           WARN_DEPRECATED("OLDSIM_EXIT", pkt->getID(), "SIM_EXIT", ArCommands::SIM_EXIT);
           robotInterface->inform_s("Client requested exit (OLDSIM_EXIT command).");
           // Wait a few seconds to let the user see the message before exiting.
           ArUtil::sleep(1500);
           robotInterface->shutdown();
           break;

        case ArCommands::SIM_STAT:
        {
           argType = pkt->bufToByte();
           intVal = pkt->bufToByte2();
           if(intVal == 0)
            sendingSimstats = false;
           else if(intVal == 1)
           {
            if(!sendSIMSTAT(&myTCPConnection))
             robotInterface->warn("Error sending SIMSTAT packet to a client.");
           }
           else if(intVal == 2)
            sendingSimstats = true;
           else
            robotInterface->warn("Received unrecognized argument %d to SIM_STAT (expect 0, 1 or 2)", intVal);
          break;
        }
           

        case ArCommands::SIM_CTRL:
        {
          argType = pkt->bufToByte();
          intVal = pkt->bufToByte2(); 
          if(intVal == SIM_CTRL_LOAD_MAP || intVal == SIM_CTRL_MASTER_LOAD_MAP)
          {
            mapMasterMutex.lock();
            if(mapMasterEnabled && intVal != SIM_CTRL_MASTER_LOAD_MAP)
            {
                if(myVerbose) robotInterface->inform("Client requested new map file, but map master has already set it. Ignoring.");
                mapMasterMutex.unlock();
                break;
            }
            char mapfile[256];
            memset(mapfile, 0, 256);
            size_t len = pkt->bufToUByte2();
            if(len > 255) len = 255;
            pkt->bufToStr((char*)mapfile, len);
            if(intVal == SIM_CTRL_MASTER_LOAD_MAP) 
            {
                mapMasterEnabled = true;
                robotInterface->inform("Map master client requested new map file: \"%s\".", mapfile);
            } else {
                if(myVerbose) robotInterface->inform("Client requested new map file: \"%s\".", mapfile);
            }
            robotInterface->loadNewMapAsync(mapfile);
            mapMasterMutex.unlock();
            break;
          }
          else if(intVal == SIM_CTRL_CLEAR_MASTER_MAP) 
          {
            mapMasterMutex.lock();
            mapMasterEnabled = false;
            robotInterface->inform("Canceled map master mode, now any client can change the map.");
            mapMasterMutex.unlock();
            break;
          }
          else if(intVal == SIM_CTRL_ROTATE_LOGFILES)
          {
            if(!mobilesim_log_file)
            {
              robotInterface->warn("Cannot rotate log files, because we're not logging to a file.");
              break;
            }
            if(mobilesim_rotate_log_files(NULL))
            {
              if(!mobilesim_reopen_log_file())
              {
                robotInterface->warn("Error opening new log file \"%s\" after saving old files.", mobilesim_log_file);
                break;
              }
              robotInterface->inform("Saved copies of old log files and started a new log \"%s\".", mobilesim_log_file);
            }
            else
            {
              robotInterface->warn("Error rotating log files.");
            }
            break;
          }
          else if(intVal == SIM_CTRL_LOG_STATE)
          {
            robotInterface->log("-- Begin state log --");
            robotInterface->log("Current Settings: Robot type=%s, subtype=%s, name=%s.", params.RobotClass, params.RobotSubclass, params.RobotName);
            robotInterface->log("  SIPFreq=%d, WatchdogTime=%d", params.SIPFreq, params.WatchdogTime);
            robotInterface->log("  RotVelMax=%d (default %d), TransVelMax=%d (%d), LatVelMax=%d (%d)",
              settings.RotVelMax, params.RotVelMax, 
              settings.TransVelMax, params.TransVelMax, 
              settings.LatVelMax, params.LatVelMax);
            robotInterface->log("  RotAccel=%d (%d), RotDecel=%d (%d), TransAccel=%d (%d), TransDecel=%d, (%d), LatAccel=%d (%d), LatDecel=%d (%d)", 
              settings.RotAccel, params.RotAccel, 
              settings.RotDecel, params.RotDecel, 
              settings.TransAccel, params.TransAccel, 
              settings.TransDecel, params.TransDecel, 
              settings.LatAccel, params.LatAccel, 
              settings.LatDecel, params.LatDecel);
            robotInterface->log("  BumpStallFront=%d, BumpStallRear=%d", settings.BumpStallFront, settings.BumpStallRear);
            robotInterface->log("In watchdog disable state? %s, Estopping? %s", inWatchdogState?"yes":"no", inEstop?"yes":"no");
            mapMasterMutex.lock();
            if(mapMasterEnabled)
              robotInterface->log("Map master mode is enabled in MobileSim.");
            mapMasterMutex.unlock();
            robotInterface->log("Currently have %d RobotInterface objects, %d EmulatePioneer objects (have created %lu since start), %d RobotFactory objects.", robotInterfaces.size(), emulators.size(), lifetimeThreadCount, factories.size());
            std::string commandsIgnoredStr;
            char s[4];
            for(std::set<int>::const_iterator ig = myCommandsToIgnore.begin(); ig != myCommandsToIgnore.end(); ++ig)
            {
              snprintf(s, 4, "%d ", *ig);
              commandsIgnoredStr += s;
            }
            robotInterface->log("Ignored commands (as requested by user): %s", commandsIgnoredStr.c_str());
            robotInterface->logState();
            robotInterface->log("-- End state log --");
            break;
          }
          else if(intVal == SIM_CTRL_SIM_INFO)
          {
              // Send SIMINFO packet
             replyPkt.empty();
             replyPkt.setID(0x63);

             replyPkt.strToBuf(myApplicationName.c_str());
             replyPkt.strToBuf(myApplicationVersion.c_str());

             // feature flags:
             ArTypes::UByte4 flags = 0;
             if(!NonInteractive)
              flags |= ArUtil::BIT0; // have GUI
             if(RestartedAfterCrash)
              flags |= ArUtil::BIT1;
             replyPkt.uByte4ToBuf(flags);  

             // devices:
             std::vector< RobotInterface::DeviceInfo > devs = robotInterface->getDeviceInfo();
             //printf("SIMINFO: %lu devices.\n", devs.size());
             replyPkt.uByte4ToBuf(devs.size());
             for(std::vector< RobotInterface::DeviceInfo >::const_iterator i = devs.begin(); i != devs.end(); ++i)
             {
               //printf("SIMINFO: adding device %s (%s:%d)\n", i->name.c_str(), i->type.c_str(), i->which);
                replyPkt.strToBuf(i->name.c_str());
                replyPkt.strToBuf(i->type.c_str());
                replyPkt.uByteToBuf((unsigned char) i->which);
                replyPkt.uByte4ToBuf(i->status); // status
             }
             replyPkt.finalizePacket();
             if(!myTCPConnection.write(replyPkt.getBuf(), replyPkt.getLength()))
               robotInterface->warn("Error sending SIMINFO packet to a client.");
             break;
          }
          else
          {
            robotInterface->warn("Unknown SIM_CTRL code %d.", intVal);
          }
          break;
        }


        case ArCommands::TTY2:
        //Can't do anything with TTY3 which is the same as OLDSIM_SETORIGIN_X.
        case ArCommands::TTY4:
        {
          if(!myVerbose) break;

          argType = pkt->bufToByte();
          size_t len = (size_t)(pkt->bufToUByte()); 
          std::string str;
          for(unsigned int i = 0; i < len; ++i)
          {
            str += toPrintable(pkt->bufToUByte());
          }
          robotInterface->inform("For aux tty %d: %s (%u bytes)", (pkt->getID() == ArCommands::TTY2)?1:3, str.c_str(), len);
          break;
        }


        // These commands are unsupported, but common, so just log instead of warn loudly:
        case 31: // SETPBIOPORT
        {
          if(!myVerbose) break;

          argType = pkt->bufToByte();
          byte1 = pkt->bufToUByte();  // On/Off (this was the LSB of the 2-byte int)
          byte2 = pkt->bufToUByte();  // Num    (this was the MSB of the 2-byte int)
          robotInterface->log("Ignoring command: SETPBIOPORT %u %s.", byte2, byte1?"ON":"OFF");
          break;
        }


        case ArCommands::MOVINGBLINK:
        {
          if(!myVerbose) break;

          argType = pkt->bufToByte();
          intVal = pkt->bufToUByte2();
          robotInterface->log("Ignoring command: MOVINGBLINK %s.", intVal?"ON":"OFF");
          break;
        }

        case ArCommands::JOYINFO:
        {
          if(!myVerbose) break;
          argType = pkt->bufToByte();
          intVal = pkt->bufToUByte2();
          robotInterface->log("Ignoring command: JOYPAC %u.", intVal);
          break;
        }

        case ArCommands::JOYDRIVE:
        {
          if(!myVerbose) break;
          argType = pkt->bufToByte();
          intVal = pkt->bufToByte2();
          robotInterface->log("Ignoring command: JOYDRIVE %d.", intVal);
          break;
        }

        case ArCommands::IOREQUEST:
        {
          if(!myVerbose) break;
          argType = pkt->bufToByte();
          intVal = pkt->bufToUByte2();
          robotInterface->log("Ignoring command: IOREQ %u.", intVal);
          break;
        }


        // "Known unsupported", but less common or maybe important to know, so warn in the gui
        case ArCommands::SOUNDTOG:
          if(myVerbose) robotInterface->warn("Received unsupported command %d (Amigo-H8 SOUNDTOG). Ignoring.", pkt->getID());
          break;
        case ArCommands::DCHEAD:
          if(myVerbose) robotInterface->warn("Received unsupported command %d (DCHEAD). Ignoring.", pkt->getID());
          break;
        case ArCommands::SOUND:
          if(myVerbose) robotInterface->warn("Received unsupported command %d (SOUND). Ignoring.", pkt->getID());
          break;
        case ArCommands::PLAYLIST:
          if(myVerbose) robotInterface->warn("Received unsupported command %d (Amigo-H8 PLAYLIST). Ignoring.", pkt->getID());
          break;
        case ArCommands::TCM2:
          if(myVerbose) robotInterface->warn("Received unsupported command %d (TCM2). Ignoring.", pkt->getID());
          break;


        case 253:
        case 255:
            // intentionally raise a SIGABRT signal to trigger crash handler
            if(NonInteractive)
                robotInterface->log_error("Received maintainence command 255, aborting program. ABORT signal may trigger crash/debug handler if supported in this platform and installation (in noninteractive mode).");
            else
                robotInterface->log_error("Received maintainence command 255, triggering ABORT signal.");
            threadRunning = false;   // otherwise it will hang when trying to cancel the thread when the crash handler destroys EmulatePioneer objects
            abort();
            //raise(SIGABRT);
            break;

        // Unrecognized command, warn in the GUI:
        default:
          if(myVerbose) robotInterface->warn("Received unsupported command %d", pkt->getID());
          break;
      }

    }


    // Disconnected, make the robot stop, close drivers, sockets and go back to waiting for a client to
    // connect...

    robotInterface->disconnect();
    if(myClientSocket && myDeleteClientSocketOnDisconnect)
    {
      delete myClientSocket;
    }
    myClientSocket = NULL;
    if(myEndThreadOnDisconnect) { return; }

  }
  threadRunning = false;
}

void EmulatePioneer::clientDisconnected()
{
  ArScopedLock lock(&myClientConnectedMutex);
  if(!myClientConnected)
    return;
  robotInterface->inform("Client disconnected.");
  myClientConnected = false;
}



ArRobotPacket* SIPGenerator::getPacket()
{
  if(!started) return 0;

#if 0
    if(positionData)
      printf("DEBUG: position x=%d y=%d th=%d vel=%d rotvel=%d\n",
          (int16_t) revByteOrder32(positionData->xpos), 
          (int16_t) revByteOrder32(positionData->ypos), 
          (int16_t) revByteOrder32(positionData->yaw), 
          (int16_t) revByteOrder32(positionData->xspeed), 
          (int16_t) revByteOrder32(positionData->yawspeed));
    if(powerData)
      printf("DEBUG: power charge=%d\n", revByteOrder32(powerData->charge));
#endif

  // TODO, keep a data buffer ourselves so it's not reallocated so much
  // (same with config pakcets!)
 
  /* Pioneer comms is always little endian (Intel byte order) */

  pkt.empty();
  pkt.setID(0x32);



  // Get all robot motion state at once, saves us several function calls that
  // might lock:
  int x, y, theta, transVel, rotVel;
  bool stalled, motorsEnabled;
  robotInterface->getMotionState(x, y, theta, transVel, rotVel, stalled, motorsEnabled);

  if(robotInterface->havePositionData())
  {

    // different packet ID if robot is moving or stopped: (or should it be for
    // enabled vs. disabled?)  Does this even matter? What's the point?
    //if (positionData->xspeed > 0 || positionData->yawspeed > 0 ||
    //  positionData->yspeed > 0)
    //{
    //  pkt->setID(0x33);
    //} else {
    //  pkt->setID(0x32);
    //}

    // By adding the difference from the last time we checked to the 16 bit 
    // variable, we assure that
    // the pos returned wraps at the 16 bit limit just like ARIA checks for,
    // with a minimum of fuss.

    xPosAccum += (int32_t)( x/params->DistConvFactor - xPosAccum );
    pkt.byte2ToBuf(xPosAccum);

    yPosAccum += (int32_t)( y/params->DistConvFactor - yPosAccum );
    pkt.byte2ToBuf(yPosAccum);

#ifdef DEBUG_SIP_VALUES
    fprintf(stderr, "Pioneer emulation: <sip>       xpos= %4d mm, ypos= %4d, DistConv= %f\n", xPosAccum, yPosAccum, params->DistConvFactor);
#endif


    // theta, just truncated from 32 bits, stage should keep this between
    // -180,180 for us.
    pkt.byte2ToBuf( (ArTypes::Byte2) ( ArMath::degToRad(theta)/params->AngleConvFactor ) );  

    // wheel velocities (left, right):
    int16_t rightVel = (int16_t) (transVel + ArMath::degToRad((double)rotVel / params->DiffConvFactor));
    int16_t leftVel =  (int16_t) (transVel - ArMath::degToRad((double)rotVel / params->DiffConvFactor));
#ifdef DEBUG_SIP_VALUES
    fprintf(stderr, "Pioneer emulation: <sip>       leftVel=%4d     rightVel=%4d\n",  leftVel, rightVel ) ;
#endif
    pkt.byte2ToBuf((ArTypes::Byte2)leftVel);
    pkt.byte2ToBuf((ArTypes::Byte2)rightVel);
  }
  else
  {
    // Have no position data.
    //pkt->setID(0x32);
    pkt.byte2ToBuf(0);   // xpos
    pkt.byte2ToBuf(0);   // ypos
    pkt.byte2ToBuf(0);   // theta
    pkt.byte2ToBuf(0);   // lvel
    pkt.byte2ToBuf(0);   // rvel
  }
  pkt.byteToBuf( (ArTypes::Byte) (robotInterface->getBatteryVoltage() * 10.0) ); // battery decivolts, one byte

  // Stall and bumper flags. bumpers are not implemented yet.
  // Stall only if trying to translate, not rotate.
  ArTypes::UByte2 bumpStallFlags = 0;
  if(stalled)
    bumpStallFlags |= ArUtil::BIT0 | ArUtil::BIT8;
  pkt.uByte2ToBuf(bumpStallFlags); 

  pkt.byte2ToBuf(0);  // "control", whatever that is (old p1 accessory?)

  // Misc status flags:
  ArTypes::UByte2 flags = 0;
  
  // sonar status flags on: 
  if(robotInterface->sonarOpenRequested())
  {
    flags |= ArUtil::BIT1 | ArUtil::BIT2 
      | ArUtil::BIT3 | ArUtil::BIT4;
  }

  // Motor status
  if(motorsEnabled) flags |= ArUtil::BIT0;
  pkt.uByte2ToBuf(flags); 

  pkt.uByteToBuf(0);   // compass

  // Sonar values:
  if(robotInterface->sonarOpen() && robotInterface->haveSonar())
  {
    size_t n = robotInterface->numSonarReadings(); 
#ifdef DEBUG_SIP_SONAR_DATA
printf("SIP: sonar has %d readings. Max per packet is %d.\n", n, params->Sim_MaxSonarReadingsPerSIP);
fflush(stdout);
#endif
    pkt.byteToBuf((ArTypes::Byte)n);

    class PackSonarReadingFunc : public virtual RobotInterface::SonarReadingFunc {
    public:
      ArRobotPacket &pkt;
      const double &conv;
      const size_t &max;
      size_t i;
      size_t count;
      PackSonarReadingFunc(ArRobotPacket &init_pkt, const double &init_conv, const size_t &init_max, const size_t &init_i = 0) 
        : pkt(init_pkt), conv(init_conv), max(init_max), i(init_i), count(0)
        {}
      virtual bool operator()(int r) {
        pkt.byteToBuf((ArTypes::Byte)i);
        ArTypes::Byte2 convr = ArMath::roundInt(r / conv);
        pkt.byte2ToBuf(convr);

#ifdef DEBUG_SIP_SONAR_DATA
        printf("SIP: Sonar %d => %d (converted from %d) )\n", s, convr, r);
#endif

        ++i;
        if(++count > max) return false;
        return true;
      }
    };
  
    PackSonarReadingFunc func(pkt, params->RangeConvFactor, params->Sim_MaxSonarReadingsPerSIP, firstSonarReadingToSend);

    size_t numPacked = robotInterface->forEachSonarReading(func, firstSonarReadingToSend);

    if(numPacked >= n)
      firstSonarReadingToSend = 0; // start from beginning next time
    else
      firstSonarReadingToSend += numPacked;
#ifdef DEBUG_SIP_SONAR_DATA
    printf("SIP: Will start with sonar reading #%d in next SIP.\n", firstSonarReadingToSend);
#endif
  }
  else
  {
#ifdef DEBUG_SIP_SONAR_DATA
printf("SIP: sonar not enabled.\n");
fflush(stdout);
#endif
    pkt.byteToBuf(0);   
  }

  // More:
  pkt.byteToBuf(robotInterface->gripperState()); 
  pkt.byteToBuf(0);   // analog select
  pkt.byteToBuf(0);   // analog data
  pkt.byteToBuf(0xFF);   // digital in, used for IR on peoplebot (0 means triggered, 1 means not triggered)
  pkt.byteToBuf(0xFF);   // digital out (0 means triggered, 1 means not triggered)
  pkt.byte2ToBuf( (ArTypes::Byte2) (robotInterface->getBatteryVoltage()*10.0) );  // battery decivolts in two bytes, default of 13V 
  pkt.uByteToBuf(0);  // charge/dock status (0=not,1=bulk,2=over,3=float)
  pkt.byte2ToBuf( (ArTypes::Byte2) (rotVel*10.0) );  // current rot. vel
  if(robotInterface->getTempWarning())
    pkt.byte2ToBuf((ArTypes::Byte2)ArUtil::BIT1); // fault flags
  else
    pkt.byte2ToBuf(0); // fault flags
  pkt.byte2ToBuf((ArTypes::Byte2) robotInterface->yspeed());
  if(robotInterface->haveTemperature())
    pkt.byteToBuf((ArTypes::Byte) robotInterface->getTemperature());
  else
    pkt.byteToBuf((ArTypes::Byte) -127);
  pkt.finalizePacket();

#ifdef DEBUG_SIP_PACKET_CONTENTS
  printf("Pioneer emulation: ------- sending SIP: ----\n");
  pkt.printHex();
  printf("--------------------------------------------\n");
#endif 


  return &pkt;
}



ArRobotPacket* LaserPacketGenerator::getPacket()
{
  if(!started || !robotInterface->haveLaser()) return 0;

  // max. number of readings per packet.
  // (with 5 bytes per reading plus 5 bytes of metadata, this could
  // be as high as 39, but this is a nice even factor of 180 and only
  // gives us 2 extra packets)
  const int MaxReadingsPerPacket = 30;

  // Figure out how many readings to put in this packet. 

  //ArTypes::Byte2 totalReadings = robotInterface->numLaserReadings();
  size_t totalReadings = robotInterface->numLaserReadings();
  if(currentReading >= totalReadings) 
  {
    currentReading = 0;
    return NULL;
  }


  pkt.empty(); // TODO just replace the fields in the packet buffer which have changed instead of emptying and repacking

  if(extendedInfoFormat) 
  {
    pkt.setID(0x61);
  } 
  else 
  {
    pkt.setID(0x60);
    int x, y, th;
    robotInterface->getPosition(x, y, th);
    pkt.byte2ToBuf(x);
    pkt.byte2ToBuf(y);
    pkt.byte2ToBuf(th);
  }
  pkt.byte2ToBuf((ArTypes::Byte2) totalReadings);  // total range reading count the device has
  pkt.byte2ToBuf((ArTypes::Byte2) currentReading); // which reading is the first one in this packet
  pkt.uByteToBuf((ArTypes::Byte2) min(MaxReadingsPerPacket, totalReadings - currentReading));  // num. readings that follow

  class PackLaserReadingFunc_OldFormat : public virtual RobotInterface::LaserReadingFunc {
  protected:
    ArRobotPacket &pkt;
    size_t i;
    const size_t &max;
    size_t count;
  public:
    PackLaserReadingFunc_OldFormat(ArRobotPacket &set_pkt, const size_t &set_max, size_t init_i = 0) 
      : pkt(set_pkt), i(init_i), max(set_max), count(0)
      {}
    virtual ~PackLaserReadingFunc_OldFormat() {}
    virtual bool operator()(int range, int /*ref*/) {
      pkt.uByte2ToBuf(range);
      ++i;
      if(++count > max) return false;
      return true;
    }
  };

  class PackLaserReadingFunc_ExtFormat : public virtual PackLaserReadingFunc_OldFormat {
  public:
    PackLaserReadingFunc_ExtFormat(ArRobotPacket &set_pkt, const size_t &set_max, size_t init_i = 0) 
      : PackLaserReadingFunc_OldFormat(set_pkt, set_max, init_i)
      {}
    virtual ~PackLaserReadingFunc_ExtFormat() {}
    virtual bool operator()(int range, int ref) {
      bool r = PackLaserReadingFunc_OldFormat::operator()(range, ref);
      pkt.uByteToBuf(ref);
      pkt.uByteToBuf(0); // reserved for future flags
      pkt.uByteToBuf(0); // reserved for future flags
      return r;
    }
  };

  size_t numPacked = 0;

  if(extendedInfoFormat) {
    PackLaserReadingFunc_ExtFormat func(pkt, MaxReadingsPerPacket, currentReading);
    numPacked = robotInterface->forEachLaserReading(func, currentReading);
  } else {
    PackLaserReadingFunc_OldFormat func(pkt, MaxReadingsPerPacket, currentReading);
    numPacked = robotInterface->forEachLaserReading(func, currentReading);
  }


  if(numPacked > totalReadings)
    currentReading = 0;
  else
    currentReading += numPacked;

  pkt.finalizePacket();
  return &pkt;
}
  
/* Debugging tool */
void LaserPacketGenerator::printLaserPacket(ArRobotPacket* pkt) const
{
  assert(pkt->verifyCheckSum());
  assert(pkt->getID() == 0x60 || pkt->getID() == 0x61);
  printf("Laser packet length=%d: ", pkt->getLength());
  if(pkt->getID() == 0x60)
  {
    pkt->bufToByte2();
    pkt->bufToByte2();
    pkt->bufToByte2();
  }
  int total = pkt->bufToByte2();
  int first = pkt->bufToByte2();
  int here = pkt->bufToUByte();
  printf("total=%d this=%d curr=%d  readings: ", total, here, first );
  for(int i = 0; i < here; ++i)
  {
    printf("%d ", pkt->bufToUByte2());
    fflush(stdout);
  }
  if(pkt->getID() == 0x61)
  {
    printf("(R=%d) ", pkt->bufToUByte());
    fflush(stdout);
    pkt->bufToUByte();
    pkt->bufToUByte();
  }
  printf("\n");
}

ArTypes::Byte2 EmulatePioneer::getIntFromPacket(ArRobotPacket* pkt)
{
   if(pkt->bufToByte() == ArgTypes::NEGINT)
     return -1 * pkt->bufToByte2();
   else
     return pkt->bufToByte2();
}

bool EmulatePioneer::sendSIMSTAT(ArDeviceConnection *con)
{
  ArRobotPacket replyPkt;
  replyPkt.empty();
  replyPkt.setID(0x62);

  // compatability with older MobileSim versions:
  replyPkt.strToBuf("");
  replyPkt.strToBuf("");

  // status flags (reserved):
  ArTypes::UByte4 flags = 0;
  replyPkt.uByte4ToBuf(flags);

  // time statistics:
  replyPkt.uByte2ToBuf((unsigned short)robotInterface->getSimInterval());
  replyPkt.uByte2ToBuf((unsigned short)robotInterface->getRealInterval());
  replyPkt.uByte2ToBuf((unsigned short)robotInterface->getLastInterval());

  // true pose:
  long x, y, z;
  int th;
  robotInterface->getSimulatorPose(x, y, z, th);
  replyPkt.byte4ToBuf((int)x);
  replyPkt.byte4ToBuf((int)y);
  replyPkt.byte4ToBuf((int)z);
  replyPkt.byte4ToBuf(th);

  replyPkt.finalizePacket();
  return con->write(replyPkt.getBuf(), replyPkt.getLength());
}
