/*
 *  Copyright (C) 2005, ActivMedia Robotics
 *  Copyright (C) 2006, 2007, 2008, 2009 MobileRobots, Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#ifndef EP_EMULATE_PIONEER_HH_
#define EP_EMULATE_PIONEER_HH_

#include "RobotInterface.hh"

#include <assert.h>
//#include <pthread.h>
#include <string>
#include <set>

/* Some useful classes from Aria */
#include "ArSocket.h"
#include "ArFunctor.h"
#include "ArMutex.h"
#include "ArRobotPacket.h"

class ArDeviceConnection;



/* Constants required for compatability, shouldn't change these */
#define DEFAULT_PIONEER_SIM_PORT  8101
#define ROBOT_IDENTIFIER_LEN      20    // Max length of robot type, subtype and name strings.
#define MAX_PACKET_PAYLOAD_SIZE   200   // Ignore or truncate packets which claim to have more bytes of data than this

/* What version of ARCOS to claim we are in the CONFIG packet */
#define ARCOS_VER_MAJ 'S'  
#define ARCOS_VER_MIN '0'

/* Defauts for settings in the config file: */
#define DEFAULT_SIP_FREQ       100        // defaut SIP cycle time, ms.
#define SYNC_READ_TIMEOUT      1000       // Timeout on an otherwise blocking read while in handshaking stage, ms.
#define DEFAULT_WATCHDOG_TIME  2000       // ms of no activity to stop robot.
#define DEFAULT_MAX_SONAR_READINGS_PER_SIP 32  // Set to a low number like 4 to crudely simulate sonar timing delay
#define DEFAULT_ROBOT_NAME     "MobileSim"
#define DEFAULT_ROBOT_CLASS       "Pioneer"
#define DEFAULT_ROBOT_SUBCLASS    "p3dx"
#define DEFAULT_DIFFCONV          0.0056
#define DEFAULT_ANGLECONV         0.001534
#define DEFAULT_DISTCONV          0.485
#define DEFAULT_VELCONV           1.0
#define DEFAULT_RANGECONV         1.0
#define DEFAULT_VEL2DIV           20.0
#define DEFAULT_STALL_ON_ROT      false

/* These are values clients can set during a session, but not saved between. */
#define DEFAULT_TRANSVELMAX       3000
#define DEFAULT_TRANSACCEL        300
#define DEFAULT_TRANSDECEL        300
#define DEFAULT_ROTVELMAX         150
#define DEFAULT_ROTACCEL          100
#define DEFAULT_ROTDECEL          100
#define DEFAULT_LATVELMAX         3000
#define DEFAULT_LATACCEL          300
#define DEFAULT_LATDECEL          300
#define DEFAULT_BUMPSTALL_FRONT   false
#define DEFAULT_BUMPSTALL_REAR    false


/* We don't have any "Top" limits like ARCOS does, you can
 * set any acceleration you want.  But we need to tell
 * the client something, so use these values.
 */
#define FAKE_TRANSVELTOP  9999
#define FAKE_ROTVELTOP    9999
#define FAKE_ROTACCTOP    9999
#define FAKE_TRANSACCTOP  9999
#define FAKE_LATVELTOP    9999
#define FAKE_LATACCTOP    9999


/* Operation codes used in the SIM_CTRL command */
enum {
  SIM_CTRL_LOAD_MAP = 1,
  SIM_CTRL_MASTER_LOAD_MAP = 2,
  SIM_CTRL_CLEAR_MASTER_MAP = 3,
  SIM_CTRL_ROTATE_LOGFILES = 4,
  SIM_CTRL_LOG_STATE = 5,
  SIM_CTRL_SIM_INFO = 6,
  SIM_CTRL_SNAPSHOT = 7
};


/** Robot parameters. There are read from the 
 *  Stage (or other) configuration. Clients cannot change these at runtime. 
 *  @todo Read more params from model definition in Stage world. Need mechanism
 *  in Stage worldfile for adding arbitrary properties to models.
 */
class RobotParams {
public:
    char RobotName[ROBOT_IDENTIFIER_LEN];  ///< Note, this is used in the MobileSim UI, but the name is always reported to clients as "MobileSim".
    char RobotClass[ROBOT_IDENTIFIER_LEN];
    char RobotSubclass[ROBOT_IDENTIFIER_LEN];
    int SIPFreq;
    int WatchdogTime;
    double DistConvFactor;
    double DiffConvFactor;
    double AngleConvFactor;
    double RangeConvFactor;
    double Vel2DivFactor;
    double VelConvFactor;

    int RotVelMax;
    int RotAccel;
    int RotDecel;
    int TransVelMax;
    int TransAccel;
    int TransDecel;
    int LatVelMax;
    int LatAccel;
    int LatDecel;

    // Note, we don't have any "Top" limits like ARCOS does. You can set the
    // vels and accels to any value you want.

    /** Set this to a low number for a crude stand-in for sonar timing delay: */
    int Sim_MaxSonarReadingsPerSIP;

    /** Normally motor stall flag is only set if trying to drive forward and
     * backwards.  Set this to true if stall flag should be set even if 
     * there is no translational velocity; however this could make it hard
     * for robots to extract themselves from a collision by rotating.
     * (Real robots usually have enough power to scrape out of a collision
     * through rotation, their motors will only stall on a direct hit.)
     */
    bool Sim_StallOnRot;

    RobotParams() :
      SIPFreq(DEFAULT_SIP_FREQ), 
      WatchdogTime(DEFAULT_WATCHDOG_TIME),

      DistConvFactor(DEFAULT_DISTCONV),
      DiffConvFactor(DEFAULT_DIFFCONV),
      AngleConvFactor(DEFAULT_ANGLECONV),
      RangeConvFactor(DEFAULT_RANGECONV),
      Vel2DivFactor(DEFAULT_VEL2DIV),
      VelConvFactor(DEFAULT_VELCONV),

      RotVelMax(DEFAULT_ROTVELMAX),
      RotAccel(DEFAULT_ROTACCEL),
      RotDecel(DEFAULT_ROTDECEL),
      TransVelMax(DEFAULT_TRANSVELMAX),
      TransAccel(DEFAULT_TRANSACCEL),
      TransDecel(DEFAULT_TRANSDECEL),
      LatVelMax(DEFAULT_LATVELMAX),
      LatAccel(DEFAULT_LATACCEL),
      LatDecel(DEFAULT_LATDECEL),

      Sim_MaxSonarReadingsPerSIP(DEFAULT_MAX_SONAR_READINGS_PER_SIP),
      Sim_StallOnRot(DEFAULT_STALL_ON_ROT)
    {
      memset(RobotName, 0, ROBOT_IDENTIFIER_LEN);
      memset(RobotClass, 0, ROBOT_IDENTIFIER_LEN);
      memset(RobotSubclass, 0, ROBOT_IDENTIFIER_LEN);
      strncpy(RobotName, DEFAULT_ROBOT_NAME, ROBOT_IDENTIFIER_LEN);
      strncpy(RobotClass, DEFAULT_ROBOT_CLASS, ROBOT_IDENTIFIER_LEN);
      strncpy(RobotSubclass, DEFAULT_ROBOT_SUBCLASS, ROBOT_IDENTIFIER_LEN);
    }
};

/** These are values clients can set during a session, but not saved between */
class CurrentSettings {
public:
  int TransVelMax;
  int TransAccel;
  int TransDecel;
  int RotVelMax;
  int RotAccel;
  int RotDecel;
  int LatVelMax;
  int LatAccel;
  int LatDecel;
  bool BumpStallFront, BumpStallRear;
public:
  CurrentSettings(RobotParams* params = NULL) :
    TransVelMax(DEFAULT_TRANSVELMAX),
    TransAccel(DEFAULT_TRANSACCEL),
    TransDecel(DEFAULT_TRANSACCEL),
    RotVelMax(DEFAULT_ROTVELMAX),
    RotAccel(DEFAULT_ROTACCEL),
    RotDecel(DEFAULT_ROTACCEL),
    BumpStallFront(DEFAULT_BUMPSTALL_FRONT),
    BumpStallRear(DEFAULT_BUMPSTALL_REAR)
  {
    if(params)
    {
      TransVelMax = params->TransVelMax;
      TransAccel = params->TransAccel;
      TransDecel = params->TransDecel;
      RotVelMax = params->RotVelMax;
      RotAccel = params->RotAccel;
      RotDecel = params->RotDecel;
      LatVelMax = params->LatVelMax;
      LatAccel = params->LatAccel;
      LatDecel = params->LatDecel;
    }
  }
};


/** Pioneer emulator over TCP.
 *
 *  Communicates with a client over a TCP socket, recieve ARCOS command
 *  packets and call the appropriate methods in a RobotInterface object, and
 *  recieve data from the RobotInterface to return in information packets.
 *
 *  EmulatePioneer can be used in two ways.  You can either give it
 *  an already-connected client socket when constructing it (see
 *  EmulatePioneer::EmulatePioneer(RobotInterface*, std::string, ArTCPSocket*)), 
 *  or it can listen for new clients itself on the first available port number
 *  after the default (see EmulatePioneer::EmulatePioneer(RobotInterface*,
 *  std::string, int)
 *
 *  @todo make this not be a thread, just include a function to do I/O if
 *  neccesary, and run in main stage thread with a list of emulators to check
 *  (with timeout to make sure it doesn't take too long, etc.).  Then need a 
 *  way for StageMapLoader to work in a new thread (so need to still do
 *  I/O even if world is locked -- use trylock, or do I/O in one seperate thread
 *  rather than many).  The main issue is that it's possile to run out of
 *  threads.
 * 
 */
class EmulatePioneer 
{
  public:

    /** Create a new EmulatePioneer using the given RobotInterface and robot
     *  model, and use the given client TCP socket for communication. 
     *  The socket must already be connected to the client. open() will
     *  not reopen it.
     */
    EmulatePioneer(RobotInterface *rif, std::string robotModel, ArSocket *clientSocket, bool exitThreadOnDisconnect = true, bool deleteRobotInterfaceOnDisconnect = false, bool deleteClientSocketOnDisconnect = true, bool verbose = false);

    /** Create a new EmulatePioneer using the given RobotInterface for the given
     *  robot model. Use @a port as the default when opening a new listening
     *  socket in open() or runThread()
     */
    EmulatePioneer(RobotInterface *rif, std::string robotModel, int port = DEFAULT_PIONEER_SIM_PORT, bool exitThreadOnDisconnect = false, bool deleteRobotInterface = false, bool trySubsequentPorts = true, bool verbose = false);

    ~EmulatePioneer();

    /** Set identifier strings which may be returned in the SIMSTAT packet. */
    void setSimulatorIdentification(const char* appName, const char* appVersion) {
      myApplicationName = appName;
      myApplicationVersion = appVersion;
    }

    void setVerbose(bool v) { 
      myVerbose = v;
    }

    void setListenAddress(const char *addr) {
      myListenAddress = addr;
    }

    /** Access list of commands to ignore (returns reference to actual stored
     * set, so you can just change or replace it directly)
     */
    std::set<int>& commandsToIgnore() { return myCommandsToIgnore; }

    void setCommandsToIgnore(std::set<int>& ig) {
      myCommandsToIgnore = ig;
    }

    /** Loop indefinitely, either waiting for clients to connect or communicating
     * with one client.  If any fatal errors occur, etiher disconnect the
     * current client session and begin waiting for a new one, or abort the
     * program.
     * @internal
     */
    void Main();

    /** Get the TCP port either in use if open() has been called, or the
     *  port that open() will start with when it is called, or -1 if
     *  not listening for clients.
     */
    int getPort();

    /** Start in a new background thread. Creates a new pthread which calls
     * Main().  If an error occurs creating the thread, print a message and return false.
     */
    bool runThread(bool deleteSelfOnThreadExit = false, bool deleteRobotInterface = false);

    /* Stop the thread. */
    // TODO maybe remove in favor of just pthread_cancel()?
    void stopThread() { shouldThreadExit = true; }

    /** Get reference to Params. Your should not modify them. */
    const RobotParams* getParams() { return &params; }

    /** Get robot interface. */
    RobotInterface* getRobotInterface() { return robotInterface; }

    void threadStopped() { threadRunning = false; }

    int getThreadError() { return myThreadError; }

  private:

    pthread_t thread;
    bool shouldThreadExit; // TODO maybe remove in favor of just pthread_test_cancel() and pthread_cancel()?
    bool threadRunning;
    bool threadCreated;

  protected:
    RobotInterface* robotInterface;
    RobotParams params;
    
    int myTCPPort, myRequestedTCPPort;
    ArSocket myListenSocket;
    ArSocket *myClientSocket;
    bool myPortOpened;

    bool myClientConnected;
    ArMutex myClientConnectedMutex;
    inline bool clientIsConnected() {
      bool r;
      myClientConnectedMutex.lock();
      r = myClientConnected;
      myClientConnectedMutex.unlock();
      return r;
    }
    ArFunctorC<EmulatePioneer> myClientDisconnectCB;
    void clientDisconnected(); 

    bool myEndThreadOnDisconnect;

    /** Get a signed short integer from a command packet. 
     *  Unpack three bytes from the packet's data buffer. The first byte 
     *  determines the sign of the integer, the second and third bytes are
     *  the absolute value of the integer. 
     *  No check is made that the packet's buffer contains enough bytes.
     */
    ArTypes::Byte2 getIntFromPacket(ArRobotPacket* pkt); 

    long initialPoseX, initialPoseY; 
    int initialPoseTheta;
    bool haveInitialPose;

    bool requestedOpenSonar;

    std::string myApplicationName;
    std::string myApplicationVersion;

    bool myDeleteRobotInterfaceOnDisconnect;
    bool myDeleteClientSocketOnDisconnect;

    std::set<int> myCommandsToIgnore;

    bool myTrySubsequentPorts;

    bool myVerbose;

    static bool mapMasterEnabled;
    static ArMutex mapMasterMutex;

    ArTime gotLastCommand;
    ArTime gotLastValidCommand;
    bool inWatchdogState;

    const char *myListenAddress;

    int myThreadError;
    static unsigned long lifetimeThreadCount;
    static unsigned long lifetimeConnectionCount;
    static unsigned long lifetimeFailedConnectionCount;
    static unsigned long currentEmulatorCount;

    bool inEstop;
    bool sendingSimstats;

    bool sendSIMSTAT(ArDeviceConnection *con);
};



#endif

