
/*  
    (C) Copyright 2005, ActivMedia Robotics LLC <http://www.activmedia.com>
    (C) Copyright 2006, 2007, 2008, 2009 MobileRobots, Inc. <http://www.mobilerobots.com>

    This is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This software is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this software; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <errno.h>

#include "RobotFactory.hh"
#include "RobotInterface.hh"
#include "EmulatePioneer.hh"

#include "MobileSim.hh"


/* Notes for improvement:
 *   Creating and destroying stage models from multiple threads is hard, since
 *   hardly anything in stage or gtk is threadsafe.
 *   Maybe instead we should queue up pending clients and let the main thread 
 *   actually perform the creation/deletion? (This steals time from  the simulation and gui a bit
 *   though). The alternative is to spend more time getting locking right
 *   within stage...
 */

static void *runRobotFactoryListenThread(void *ptr)
{
  ((RobotFactory*)ptr)->listen();
  return 0;
}

RobotFactory::RobotFactory(const std::string& modelName, bool verbose, const char *listenAddress) :
  myModelName(modelName),
  myPort(8101),
  myVerbose(verbose),
  myListenAddress(listenAddress)
{
}

ArSocket* RobotFactory::open(int port, const char *listenAddress) {
  if(mySocket.open(port, ArSocket::TCP, listenAddress?listenAddress:myListenAddress) == true || mySocket.getError() == ArSocket::NoErr)
  //if(mySocket.open(port, ArSocket::TCP) == true || mySocket.getError() == ArSocket::NoErr)
    return &mySocket;
  else  {
    fprintf(stderr, "Error %d opening socket for robot factory: %s\n", mySocket.getError(), mySocket.getErrorStr().c_str());
    return NULL;
  }
}

void RobotFactory::listen() {
  // Note, assumes that EmulatePioneer properly deletes client sockets
  // (otherwise leaks memory)
  ArSocket *clientSocket = new ArSocket;
  while(true)
  {
    ArTime timer;
    if(mySocket.accept(clientSocket))
    {
      RobotInterface *ri = createRobot(myModelName, clientSocket->getIPString());
      //log(("RobotFactory: created robot interface "));
      if(ri)
      {
        storeRobotInterface(ri);
        EmulatePioneer *ep = new EmulatePioneer(ri, myModelName, clientSocket, true, true, true); // three bools mean: end thread on disconnect; delete robot interface on disconnect; delete client socket on disconnect
      //log(("RobotFactory: created EmulatePioneer "));
        ep->setSimulatorIdentification("MobileSim", MOBILESIM_VERSION);
        ep->setCommandsToIgnore(myCommandsToIgnore);
        ep->setVerbose(myVerbose);
          //log("RobotFactory: starting EmulatePioneer thread");
        if(!ep->runThread(true, true))
        {
            log(("Error running EmulatePioneer thread!"));
            clientSocket->close();
            log("Sleeping for 5 seconds...");
            delete ep;
            ArUtil::sleep(5000);
            continue;
        }
        //ep->runThread(true, true);
        storeEmulator(ep);
          //log("RobotFactory: ran EmulatePioneer thread");
        clientSocket = new ArSocket;  
      } 
      else
      {
        log("Robot factory: Error creating new robot. Closing client socket.");
        clientSocket->close();
      }
    }
    else
    {
      log("RobotFactory: error accepting client.");
    }
    char msg[100];
    snprintf(msg, 100, "Robot factory: accept retuned after %ld ms. Loop it 10 ms...", timer.mSecSince());
    log(msg);
    ArUtil::sleep(10);
  }
}


bool RobotFactory::runListenThread() {
  int err = pthread_create(&myThread, NULL, &runRobotFactoryListenThread, this);
  if(err != 0) {
    fputs("*** Robot Factory: Critical error creating new thread: ", stderr);
    fflush(stderr);
    if(err == EAGAIN) { 
      fputs("system thread resources exhausted.\n", stderr);
    } else {
      fprintf(stderr, "unknown error: %d\n", err);
    }
    fflush(stderr);
    return false;
  }
  return true;
}

RobotFactory::~RobotFactory() {
  //removeStoredRobotFactory(this);
}
