/*

  Copyright (C) 2005, ActivMedia Robotics, LLC
  Copyright (C) 2006, 2007, 2008, 2009 MobileRobots, Inc.

     This program is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published by
     the Free Software Foundation; either version 2 of the License, or
     (at your option) any later version.

     This program is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with this program; if not, write to the Free Software
     Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

// to maybe get extensions like sincos():
#ifndef _GNU_SOURCE
#define _GNU_SOURCE 1
#endif
#ifndef __USE_GNU
#define __USE_GNU 1
#endif


/*
#define STG_DEBUG_MODEL_LOCK 1
#define STG_DEBUG_WORLD_LOCK_VERBOSE 1
#define STG_DEBUG_WORLD_LOCK_TIME 1
*/


#include "StageInterface.hh"
#include "EmulatePioneer.hh"

#include "ArScopedLock.h"
#include "ArScopedTimer.h"
#include "ArMutex.h"
#include "ArMap.h"
#include "ariaUtil.h"
#include <map>
#include <deque>
#include <vector>

#include <math.h>


#include "stage.h"


#if 0
class SISTimer : public virtual ArScopedTimer
{
public:
  SISTimer(StageInterface *iface, std::string msg, unsigned int warnThresh = 0) 
    : ArScopedTimer(msg, new ArFunctor1C<StageInterface, const char*>(iface, &StageInterface::warn_s), warnThresh, true)
  {
  }

  virtual ~SISTimer()
  {
  }
};
#else
class SISTimer
{
public:
    SISTimer(StageInterface *iface, std::string msg, unsigned int warnThresh = 0)
    {}
};
#endif


// Lock a model in the given world. May lock the world as well. (You can
// experiment with not locking the world by modifying these macros.)
#ifdef LOCK_BOTH_WORLD_AND_MODEL

#define LOCK_MODEL(model, world) { \
  { \
    SISTimer tlm(this, "  (time to lock world in LOCK_MODEL)", 10); \
    stg_world_lock(world); \
  } \
  { \
    SISTimer tlm(this, "  (time to lock model in LOCK_MODEL)", 10); \
    stg_model_lock(model);  \
  } \
}

#define UNLOCK_MODEL(model, world) { stg_model_unlock(model); stg_world_unlock(world); }

#else

#define LOCK_MODEL(model, world) { \
  { \
    SISTimer tlm(this, "  (time to lock model in LOCK_MODEL)", 10); \
    stg_model_lock(model);  \
  } \
}

#define UNLOCK_MODEL(model, world) { stg_model_unlock(model); }

#endif

#define PRINT_NUM_POINTS_CREATED 100000
#define PRINT_NUM_LINES_CREATED  10000

const double StageMapLoader::ReflectorThickness = 0.0200; // meters (=2cm).
bool StageMapLoader::threadCreated = false;
ArMutex StageMapLoader::activityMutex;
pthread_cond_t StageMapLoader::haveNewMapToLoadAsync; // = PTHREAD_COND_INITIALIZER; not in gcc 3.4.1! -- call staticInit()! 
std::deque<StageMapLoader::Request> StageMapLoader::requests;
std::set<stg_model_t*> StageMapLoader::mapModels;
pthread_t StageMapLoader::thread
#ifndef WIN32
	= 0
#endif
	;
StageMapLoader::Request StageMapLoader::currentRequest;
ArMutex StageMapLoader::requestsMutex;

void StageInterface::staticInit()
{
  pthread_cond_init(&StageMapLoader::haveNewMapToLoadAsync, NULL);
}

/** @todo Perhaps throw exceptions from accessors if lock fails?  that should
* only happen if a model is destroyed, and that should only happen when a
* StageInterface is destroyed and the deleteModels flag is set, and it's
* normally not.
*/

StageInterface::StageInterface(stg_world_t* _world, std::string _robotModel, std::string _robotName, bool _deleteModels) :
RobotInterface(_robotName),
stageWorld(_world), robotModel(_robotModel), robotName(_robotName),
positionModel(NULL), sonarModel(NULL), laserModel(NULL), messagesModel(NULL), 
mapLoader(_world, this),
subscribedToSonar(false), subscribedToLaser(false),
openedSonar(false), openedLaser(false),
motorsEnabled(true),

// these are set for real in connect(), or by commands:
laserStart(-90.0), laserEnd(90.0),

deleteModels(_deleteModels)

{
#ifdef STAGE_MUTEX_LOG
//stageMutex.setLogName("StageInterface");  
//stageMutex.setLog(true);
// (note that stageMutex is static though, so these will be called repeatedly on it)
#endif
}

StageInterface::StageInterface(stg_world_t* _world, stg_model_t* _model, std::string _robotModel, std::string _robotName, bool _deleteModels) :
  RobotInterface(_robotName),
  stageWorld(_world), robotModel(_robotModel), robotName(_robotName),
  positionModel(_model), sonarModel(NULL), laserModel(NULL), messagesModel(NULL), 
  mapLoader(_world, this),
  subscribedToSonar(false), subscribedToLaser(false),
  openedSonar(false), openedLaser(false),
  motorsEnabled(true),
  // these are set for real in connect(), or by commands:
  laserStart(-90.0), laserEnd(90.0),

  deleteModels(_deleteModels)

{
#ifdef STAGE_MUTEX_LOG
  //stageMutex.setLogName("StageInterface");  
  //stageMutex.setLog(true);
  // (note that stageMutex is static though, so these will be called repeatedly on it)
#endif
}

StageInterface::~StageInterface()
{
  mapLoader.interfaceDestroyed(this);
  if(deleteModels)
  {
    stg_world_lock(stageWorld);
    if(messagesModel)
    {
      stg_world_remove_model(stageWorld, messagesModel);
      stg_model_destroy(messagesModel);
    }
    if(laserModel)
    {
      stg_world_remove_model(stageWorld, laserModel);
      stg_model_destroy(laserModel);
    }
    if(sonarModel)
    {
      stg_world_remove_model(stageWorld, sonarModel);
      stg_model_destroy(sonarModel);
    }
    if(positionModel)
    {
      stg_world_remove_model(stageWorld, positionModel);
      stg_model_destroy(positionModel);
    }
    stg_world_unlock(stageWorld);
  }
}


void StageInterface::setup(RobotParams* params)
{
  SISTimer t0(this, " --> time to do StageInterface::setup()", 1);
  ArTime timer;
  timer.setToNow();
  /* Set initial robot subtype from name the user used to make it */
  {
    SISTimer t(this, "  time to copy robot identifier strings", 1);
    strncpy(params->RobotSubclass, robotModel.c_str(), ROBOT_IDENTIFIER_LEN);
  }

  // initialize command struct for position commands:
  {
  SISTimer t(this, "  time to memset positionCmd", 1);
  memset(&positionCmd, 0, sizeof(positionCmd));
  }

  /* find stage models and subscribe */

  // Get position model if neccesary
  if(!positionModel)
  {
    SISTimer t(this, "  time to lock world and look up position model by name", 1);
    stg_world_lock(stageWorld);
    positionModel = stg_world_model_name_lookup(stageWorld, robotName.c_str());
    stg_world_unlock(stageWorld);
  }
  if(!positionModel) 
  {
    //UNLOCK_MODEL(positionModel, stageWorld);
    error("could not find a base robot position model named \"%s\" in the world", robotName.c_str());
    shutdown(-10);
    return; 	// todo return an error code so emulatepioneer thread can stop
  }

  {
    SISTimer t(this, "  time to lock position model and world for param setup", 1);
    LOCK_MODEL(positionModel, stageWorld);
  }

  {
  SISTimer t(this, "  time to subscribe to position model", 1);
  stg_model_subscribe(positionModel);
  }

  // Get the robot subtype from the position model, that's important:
  {
  SISTimer t(this, "  time to get robot type string from stage and copy it.", 1);
  size_t len;
  const char* robot_subtype = (const char*) stg_model_get_property(positionModel, "pioneer_robot_subtype", &len);
  if(robot_subtype != NULL)
  {
    strncpy(params->RobotSubclass, robot_subtype, ROBOT_IDENTIFIER_LEN);
  }
  }
  {
  SISTimer t(this, "  time to unlock position model and world.", 1);
  UNLOCK_MODEL(positionModel, stageWorld);
  }

  // Get other models that are children of the position: 
  if(stg_world_gui_enabled(stageWorld)) {
    SISTimer t(this, "  time to find messages model", 1);
    messagesModel = stg_model_find_first_child_with_type(positionModel, (char*) "messages");
  }

  {
    SISTimer t(this, "  time to find sonar model", 1);
  sonarModel = stg_model_find_first_child_with_type(positionModel, (char*) "ranger");
}
  if(!sonarModel && !stg_world_get_quiet(stageWorld)) {
    SISTimer t(this, "  time to warn about missing sonar model", 1);
    warn_s("No sonar model defined for this robot");
  }

  /* Laser model */
  {
    SISTimer t(this, "  time to find laser model", 1);
    laserModel = stg_model_find_first_child_with_type(positionModel, (char*) "laser");
  }
  if(laserModel) {
    SISTimer t(this, "  time to lock laser model, get config property, and unlock.", 1);
    {
      SISTimer t2(this, "   time to lock laser model", 1);
      LOCK_MODEL(laserModel, stageWorld);
    }
    stg_laser_config_t* lasercfg;
    {
      SISTimer t2(this, "   time to get laser config struct", 1);
      lasercfg = (stg_laser_config_t*)stg_model_get_property_fixed(laserModel, "laser_cfg", sizeof(stg_laser_config_t));
    }
    //assert(lasercfg);
    if(lasercfg->reverse_scan)
    {
      laserEnd = (RTOD(lasercfg->fov) / 2.0); 
      laserStart = (RTOD(lasercfg->fov) / 2.0);
    } else {
      laserStart = (RTOD(lasercfg->fov) / 2.0); 
      laserEnd = (RTOD(lasercfg->fov) / 2.0);
    }
    UNLOCK_MODEL(laserModel, stageWorld)
  }
  else
  {
    if(!stg_world_get_quiet(stageWorld))
      inform_s("Note: No laser model defined for this robot");
  }



  /* Get our custom parameters from Stage's model definitions (these properties
  * were registered by MobileSim's main.cc): */

  /// @todo Need to get *all* robot parameters, including speed config etc. !!

  {
  SISTimer t(this, "  time to copy default params (hardParams)", 1);
  hardParams = params;  // make a copy to store the original defaults
  }
  //assert(positionModel);
{
SISTimer t(this, "  time to lock position model and get properties.", 1);
  {
    SISTimer t2(this, "   time to lock position model");
    LOCK_MODEL(positionModel, stageWorld);
  }

  float *f;

  {
    SISTimer t2(this, "   time to get pioneer_distconv property");
    f = (float*)stg_model_get_property_fixed(positionModel, "pioneer_distconv", sizeof(float));
    if(f) params->DistConvFactor = *f;
  }

  f = (float*)stg_model_get_property_fixed(positionModel, "pioneer_diffconv", sizeof(float));
  if(f) params->DiffConvFactor = *f;

  f = (float*)stg_model_get_property_fixed(positionModel, "pioneer_angleconv", sizeof(float));
  if(f) params->AngleConvFactor = *f;

  f = (float*)stg_model_get_property_fixed(positionModel, "pioneer_rangeconv", sizeof(float));
  if(f) params->RangeConvFactor = *f;

  f = (float*)stg_model_get_property_fixed(positionModel, "pioneer_vel2div", sizeof(float));
  if(f) params->Vel2DivFactor = *f;

  f = (float*)stg_model_get_property_fixed(positionModel, "pioneer_velconv", sizeof(float));
  if(f) params->VelConvFactor = *f;

  int *i = (int*)stg_model_get_property_fixed(positionModel, "pioneer_sip_cycle", sizeof(int));
  if(i) params->SIPFreq = *i;

  {
    SISTimer t2(this, "   time to get pioneer_watchdog property");
    i = (int*)stg_model_get_property_fixed(positionModel, "pioneer_watchdog", sizeof(int));
    if(i) params->WatchdogTime = *i;
  }

  UNLOCK_MODEL(positionModel, stageWorld)
}

  if(sonarModel)
  {
//SISTimer t(this, "  time to lock sonar model and get properties", 1);
  LOCK_MODEL(sonarModel, stageWorld);
int*  i = (int*) stg_model_get_property_fixed(sonarModel, "pioneer_max_readings_per_packet", sizeof(int));
  if(i) params->Sim_MaxSonarReadingsPerSIP = *i;
  UNLOCK_MODEL(sonarModel, stageWorld)
}

// For Debugging:
/*
puts("StageInterface: now set up and ready to use with the following stage model(s):");
stg_model_print_children(positionModel);
*/

  //log(" (end of setup(), time to run: %d)", timer.mSecSince());
 // assert(timer.mSecSince() <= 2000);
} 


void StageInterface::disconnect()
{
  stop();
  enableMotors(false);
  if(positionModel)
  {
    LOCK_MODEL(positionModel, stageWorld);
    stg_model_unsubscribe(positionModel);
    UNLOCK_MODEL(positionModel, stageWorld)
  }
  closeSonar();
  closeLaser();
  setOdom(0, 0, 0);
}




void StageInterface::enableMotors(bool e) 
{
  motorsEnabled = e;
  if(!e) stop();
}


void StageInterface::transVel(int v) 
{
  if(!motorsEnabled) return;
  // TODO get property pointer and modify rather than copying new struct in
  positionCmd.x = (v / 1000.0);  // mm to m
  positionCmd.transmode = STG_POSITION_CONTROL_VELOCITY;
  positionCmd.override_decel.x = 0.0;
  stg_model_set_property_locked(positionModel, "position_cmd", &positionCmd, sizeof(positionCmd));
}

void StageInterface::latVel(int v)
{
  if(!motorsEnabled) return;
  // TODO get property pointer and modify rather than copying new struct in
  positionCmd.y = (v / 1000.0);   // mm t m
  positionCmd.transmode = STG_POSITION_CONTROL_VELOCITY;
  positionCmd.override_decel.y = 0.0;
  stg_model_set_property_locked(positionModel, "position_cmd", &positionCmd, sizeof(positionCmd));
}

void StageInterface::rotVel(int v)
{
  if(!motorsEnabled) return;
  // TODO get property pointer and modify rather than copying new struct in
  positionCmd.a = DTOR((double)v); // degrees to radians
  positionCmd.rotmode = STG_POSITION_CONTROL_VELOCITY;
  positionCmd.override_decel.a = 0.0;
  stg_model_set_property_locked(positionModel, "position_cmd", &positionCmd, sizeof(positionCmd));
}

void StageInterface::move(int m)
{
  if(!motorsEnabled) return;
  // TODO get property pointer and modify rather than copying new struct in
  positionCmd.x = (m / 1000.0);
  positionCmd.transmode = STG_POSITION_CONTROL_RELATIVE;
  positionCmd.override_decel.x = 0.0;
  char active = 1;
  LOCK_MODEL(positionModel, stageWorld);
  stg_model_set_property(positionModel, "position_rel_ctrl_active_X", &active, sizeof(active));
  stg_model_set_property(positionModel, "position_cmd", &positionCmd, sizeof(positionCmd));
  stg_pose_t *progress = (stg_pose_t*)stg_model_get_property_fixed(positionModel, "position_rel_ctrl_progress", sizeof(stg_pose_t));
  progress->x = 0.0; // reset if a current move is currenly in progress.
  stg_model_property_changed(positionModel, "position_rel_ctrl_progress");
  UNLOCK_MODEL(positionModel, stageWorld)
}


void StageInterface::heading(int h)
{
  if(!motorsEnabled) return;
  // TODO get property pointer and modify rather than copying new struct in
  positionCmd.a = DTOR((double)h);
  positionCmd.rotmode = STG_POSITION_CONTROL_POSITION;
  positionCmd.override_decel.a = 0.0;
  stg_model_set_property_locked(positionModel, "position_cmd", &positionCmd, sizeof(positionCmd));
}

void StageInterface::deltaHeading(int h)
{
  if(!motorsEnabled) return;
  // TODO get property pointer and modify rather than copying new struct in
  // TODO: normalize?
  positionCmd.a = DTOR((double)h);
  positionCmd.rotmode = STG_POSITION_CONTROL_RELATIVE;
  positionCmd.override_decel.a = 0.0;
  char active = 1;
  LOCK_MODEL(positionModel, stageWorld);
  stg_model_set_property(positionModel, "position_cmd", &positionCmd, sizeof(positionCmd));
  stg_model_set_property(positionModel, "position_rel_ctrl_active_A", &active, sizeof(active));
  stg_pose_t *progress = (stg_pose_t*)stg_model_get_property_fixed(positionModel, "position_rel_ctrl_progress", sizeof(stg_pose_t));
  progress->a = 0.0; // reset if a current delta heading is currenly in progress.
  stg_model_property_changed(positionModel, "position_rel_ctrl_progress");
  UNLOCK_MODEL(positionModel, stageWorld)
}


void StageInterface::stop()
{
  // TODO get property pointer and modify rather than copying new struct in
  // clear the position command to all zero
  memset(&positionCmd, 0, sizeof(positionCmd));
  stg_model_set_property_locked(positionModel, "position_cmd", &positionCmd, sizeof(positionCmd));
}

void StageInterface::stop(int transDecel, int rotDecel)
{
  memset(&positionCmd, 0, sizeof(positionCmd));
  positionCmd.override_decel.x = positionCmd.override_decel.y = (transDecel / 1000.0);
  positionCmd.override_decel.a = DTOR(rotDecel);
  stg_model_set_property_locked(positionModel, "position_cmd", &positionCmd, sizeof(positionCmd));
}

void StageInterface::stall(bool stalled)
{
  LOCK_MODEL(positionModel, stageWorld);
  stg_position_stall_t* stall = (stg_position_stall_t*)stg_model_get_property_fixed(positionModel, "position_stall", sizeof(stg_position_stall_t));
  if(stalled) *stall = 1;
  else        *stall = 0;
  stg_model_property_changed(positionModel, "position_stall");
  UNLOCK_MODEL(positionModel, stageWorld)
}

void StageInterface::setAccel(int a)
{
  if(a < 0)
  {
    warn("Negative rot. acceleration value requested! Ignoring.");
    return;
  }
  LOCK_MODEL(positionModel, stageWorld);
  stg_position_speed_config_t* speed_cfg = (stg_position_speed_config_t*)stg_model_get_property_fixed(positionModel, "position_speed_config", sizeof(stg_position_speed_config_t));
  speed_cfg->current_accel.x = (a/1000.0);
  stg_model_property_changed(positionModel, "position_speed_config");
  UNLOCK_MODEL(positionModel, stageWorld)
}

void StageInterface::setDecel(int d)
{
  if(d < 0)
  {
    warn("Negative trans. deceleration value requested! Ignoring.");
    return;
  }
  LOCK_MODEL(positionModel, stageWorld);
  stg_position_speed_config_t* speed_cfg = (stg_position_speed_config_t*)stg_model_get_property_fixed(positionModel, "position_speed_config", sizeof(stg_position_speed_config_t));
  speed_cfg->current_decel.x = (d/1000.0);
  stg_model_property_changed(positionModel, "position_speed_config");
  UNLOCK_MODEL(positionModel, stageWorld)
}


void StageInterface::setRotAccel(int a)
{
  if(a < 0)
  {
    warn("Negative rot. acceleration value requested! Ignoring.");
    return;
  }
  LOCK_MODEL(positionModel, stageWorld);
  stg_position_speed_config_t* speed_cfg = (stg_position_speed_config_t*)stg_model_get_property_fixed(positionModel, "position_speed_config", sizeof(stg_position_speed_config_t));
  speed_cfg->current_accel.a = DTOR(a);
  stg_model_property_changed(positionModel, "position_speed_config");
  UNLOCK_MODEL(positionModel, stageWorld);
}

void StageInterface::setRotDecel(int d)
{
  if(d < 0)
  {
    warn("Negative rot. deceleration value requested! Ignoring.");
    return;
  }
  LOCK_MODEL(positionModel, stageWorld);
  stg_position_speed_config_t* speed_cfg = (stg_position_speed_config_t*)stg_model_get_property_fixed(positionModel, "position_speed_config", sizeof(stg_position_speed_config_t));
  speed_cfg->current_decel.a = DTOR(d);
  stg_model_property_changed(positionModel, "position_speed_config");
  UNLOCK_MODEL(positionModel, stageWorld)
}

void StageInterface::setLatAccel(int a)
{
  LOCK_MODEL(positionModel, stageWorld);
  stg_position_speed_config_t* speed_cfg = (stg_position_speed_config_t*)stg_model_get_property_fixed(positionModel, "position_speed_config", sizeof(stg_position_speed_config_t));
  speed_cfg->current_accel.y = (a/1000.0);
  stg_model_property_changed(positionModel, "position_speed_config");
  UNLOCK_MODEL(positionModel, stageWorld)
}

void StageInterface::setLatDecel(int d)
{
  LOCK_MODEL(positionModel, stageWorld);
  stg_position_speed_config_t* speed_cfg = (stg_position_speed_config_t*)stg_model_get_property_fixed(positionModel, "position_speed_config", sizeof(stg_position_speed_config_t));
  speed_cfg->current_decel.y = (d/1000.0);
  stg_model_property_changed(positionModel, "position_speed_config");
  UNLOCK_MODEL(positionModel, stageWorld)
}

void StageInterface::setMaxVel(int v) 
{
  LOCK_MODEL(positionModel, stageWorld);
  stg_position_speed_config_t* speed_cfg = (stg_position_speed_config_t*)stg_model_get_property_fixed(positionModel, "position_speed_config", sizeof(stg_position_speed_config_t));
  speed_cfg->max_speed.x = (v/1000.0);
  stg_model_property_changed(positionModel, "position_speed_config");
  UNLOCK_MODEL(positionModel, stageWorld)
}

void StageInterface::setMaxRotVel(int v)
{
  LOCK_MODEL(positionModel, stageWorld);
  stg_position_speed_config_t* speed_cfg = (stg_position_speed_config_t*)stg_model_get_property_fixed(positionModel, "position_speed_config", sizeof(stg_position_speed_config_t));
  speed_cfg->max_speed.a = DTOR(v);
  stg_model_property_changed(positionModel, "position_speed_config");
  UNLOCK_MODEL(positionModel, stageWorld)
}

void StageInterface::setDefaultVel(int v)
{
  LOCK_MODEL(positionModel, stageWorld);
  stg_position_speed_config_t* speed_cfg = (stg_position_speed_config_t*)stg_model_get_property_fixed(positionModel, "position_speed_config", sizeof(stg_position_speed_config_t));
  speed_cfg->default_speed.x = (v/1000.0);
  stg_model_property_changed(positionModel, "position_speed_config");
  UNLOCK_MODEL(positionModel, stageWorld)
}

void StageInterface::setDefaultRotVel(int v)
{
  LOCK_MODEL(positionModel, stageWorld);
  stg_position_speed_config_t* speed_cfg = (stg_position_speed_config_t*)stg_model_get_property_fixed(positionModel, "position_speed_config", sizeof(stg_position_speed_config_t));
  speed_cfg->default_speed.a = DTOR(v);
  stg_model_property_changed(positionModel, "position_speed_config");
  UNLOCK_MODEL(positionModel, stageWorld)
}

bool StageInterface::haveGripper() {
  return false;
}

bool StageInterface::haveSonar() {
  return sonarModel != NULL;
    // TODO front || rear
}

bool StageInterface::haveFrontSonar() {
  return haveSonar();
  // TODO check sonar model for # sonar and poses? or a user property in the
  // model?
}

bool StageInterface::haveRearSonar() {
  return haveSonar();
  // TODO check sonar model for # sonar and poses? or a user property in the
  // model?
}

void StageInterface::setOdom(int x, int y, int theta)
{
  LOCK_MODEL(positionModel, stageWorld);
  stg_position_data_t* odomData = (stg_position_data_t*) stg_model_get_property_fixed(positionModel, "position_data", sizeof(stg_position_data_t));
  if(odomData)
  {
    odomData->pose.x = (x / 1000.0);
    odomData->pose.y = (y / 1000.0);
    odomData->pose.a = DTOR((double)theta);
    stg_model_property_changed(positionModel, "position_data");
  }
  UNLOCK_MODEL(positionModel, stageWorld)
}

void StageInterface::setSimulatorPose(long int x, long int y, long int /*z*/, int theta)
{
  LOCK_MODEL(positionModel, stageWorld);
  stg_pose_t* p = (stg_pose_t*) stg_model_get_property_fixed(positionModel, "pose", sizeof(stg_pose_t));
  p->x = (x / 1000.0);
  p->y = (y / 1000.0);
  p->a = DTOR((double)theta);
  stg_model_property_changed(positionModel, "pose");
  UNLOCK_MODEL(positionModel, stageWorld)
}

void StageInterface::resetSimulatorPose()
{
  LOCK_MODEL(positionModel, stageWorld);
  stg_model_reset_pose(positionModel);
  UNLOCK_MODEL(positionModel, stageWorld)
}

bool StageInterface::havePositionData() {
  return positionModel != NULL;
}

bool StageInterface::haveLaser() {
  return laserModel != NULL;
}

void StageInterface::getPosition(int &x, int &y, int &theta) {
  LOCK_MODEL(positionModel, stageWorld);
  stg_position_data_t* posdata = stagePositionData();
  x = (int) ArMath::roundInt(posdata->pose.x * 1000.0);
  y = (int) ArMath::roundInt(posdata->pose.y * 1000.0);
  theta = (int) ArMath::roundInt(RTOD(posdata->pose.a));
  UNLOCK_MODEL(positionModel, stageWorld)
}

void StageInterface::getVelocity(int &x, int &y, int &theta) {
  LOCK_MODEL(positionModel, stageWorld);
  stg_velocity_t* veldata = stageVelocityData();
  x = (int) ArMath::roundInt(veldata->x * 1000.0);
  y = (int) ArMath::roundInt(veldata->y * 1000.0);
  theta = (int) ArMath::roundInt(RTOD(veldata->a));
  UNLOCK_MODEL(positionModel, stageWorld)
}


void StageInterface::getMotionState(int &x, int &y, int &theta, int &transVel, int &rotVel, bool &stalled, bool &enabled) {
  LOCK_MODEL(positionModel, stageWorld);
  stg_position_data_t* posdata = stagePositionData();
  x = (int) ArMath::roundInt(posdata->pose.x * 1000.0);
  y = (int) ArMath::roundInt(posdata->pose.y * 1000.0);
  theta = (int) ArMath::roundInt(RTOD(posdata->pose.a));
  stg_velocity_t* veldata = stageVelocityData();
  transVel = (int) ArMath::roundInt(veldata->x * 1000.0);
  rotVel = (int) ArMath::roundInt(RTOD(veldata->a));
  stg_position_stall_t* stall = (stg_position_stall_t*)stg_model_get_property_fixed(positionModel, "position_stall", sizeof(stg_position_stall_t));
  stalled = ((*stall) != 0);
  enabled = motorsEnabled;
  UNLOCK_MODEL(positionModel, stageWorld)
}

int StageInterface::xpos() {
  LOCK_MODEL(positionModel, stageWorld);
  int x;
  stg_position_data_t* data = stagePositionData();
  if(data)
    x = (int) ArMath::roundInt(data->pose.x * 1000.0);
  else
    x = 0;
  UNLOCK_MODEL(positionModel, stageWorld)
  return x;
}

int StageInterface::ypos() {
  LOCK_MODEL(positionModel, stageWorld);
  int y;
  stg_position_data_t* data = stagePositionData();
  if (data) 
    y = (int) ArMath::roundInt(data->pose.y * 1000.0);
  else
    y = 0;
  UNLOCK_MODEL(positionModel, stageWorld)
  return y;
}


int StageInterface::theta() {
  LOCK_MODEL(positionModel, stageWorld);
  int th;
  stg_position_data_t* data = stagePositionData();
  if(data)
    th = (int) ArMath::roundInt(RTOD(data->pose.a));
  else
    th = 0;
  UNLOCK_MODEL(positionModel, stageWorld)
  return th;
}

int StageInterface::xspeed() {
  LOCK_MODEL(positionModel, stageWorld);
  stg_velocity_t* data = stageVelocityData();
  int x;
  if(data)
    x = (int) ArMath::roundInt(data->x * 1000.0);
  else
    x = 0;
  UNLOCK_MODEL(positionModel, stageWorld)
  return x;
}

int StageInterface::yspeed() {
  LOCK_MODEL(positionModel, stageWorld);
  stg_velocity_t* data = stageVelocityData();
  int y;
  if(data)
    y = (int) ArMath::roundInt(data->y * 1000.0);
  else 
    y = 0;
  UNLOCK_MODEL(positionModel, stageWorld);
  return y;
}

int StageInterface::rotspeed() {
  LOCK_MODEL(positionModel, stageWorld);
  stg_velocity_t* data = stageVelocityData();
  int r;
  if(data) r = (int) ArMath::roundInt(RTOD(data->a));
  else r = 0;
  UNLOCK_MODEL(positionModel, stageWorld);
  return r;
}

bool StageInterface::stalled() {
  LOCK_MODEL(positionModel, stageWorld);
  stg_position_stall_t* stall = (stg_position_stall_t*)stg_model_get_property_fixed(positionModel, "position_stall", sizeof(stg_position_stall_t));
  bool s = (stall && (*stall) != 0);
  UNLOCK_MODEL(positionModel, stageWorld);
  return s;
}

size_t StageInterface::numSonarReadings() {
  if(!subscribedToSonar) {
    return 0;
  }
  size_t len = 0;
  LOCK_MODEL(sonarModel, stageWorld);
  stg_model_get_property(sonarModel, "ranger_data", &len);
  UNLOCK_MODEL(sonarModel, stageWorld);
  return len / sizeof(stg_ranger_sample_t);
}

size_t StageInterface::numLaserReadings() {
  if(!subscribedToLaser || !laserModel) {
    return 0;
  }
  size_t len = 0;
  LOCK_MODEL(laserModel, stageWorld);
  stg_model_get_property(laserModel, "laser_data", &len);
  UNLOCK_MODEL(laserModel, stageWorld);
  return len / sizeof(stg_laser_sample_t);
}

int StageInterface::getSonarReading(int i) {
  assert(subscribedToSonar);
  size_t len = 0;
  LOCK_MODEL(sonarModel, stageWorld);
  stg_ranger_sample_t* data = (stg_ranger_sample_t*)stg_model_get_property(sonarModel, "ranger_data", &len);
  int numSonarReadings = len / sizeof(stg_ranger_sample_t);
  assert(i <= numSonarReadings && i >= 0);
  int r =(int) ArMath::roundInt(data[i].range * 1000.0); 
  UNLOCK_MODEL(sonarModel, stageWorld);
  return  r;
}

size_t StageInterface::forEachSonarReading(SonarReadingFunc &func, const size_t &start) {
  LOCK_MODEL(sonarModel, stageWorld);
  size_t len = 0;
  stg_ranger_sample_t *data = (stg_ranger_sample_t*)stg_model_get_property(sonarModel, "ranger_data", &len);
  if(!data) {
    //puts(">> sonar data is null.");
    UNLOCK_MODEL(sonarModel, stageWorld);
    return 0;
  }
  size_t n = len / sizeof(stg_ranger_sample_t);
  size_t i;
  //printf(">> there are %d sonar readings.\n", n);
  for(i = start; i < n; ++i) {
    if(!func( (int) ArMath::roundInt(data[i].range * 1000.0 ))) break;
  }
  UNLOCK_MODEL(sonarModel, stageWorld);
  return i - start;
}

int StageInterface::getLaserReading(int i) {
  if(!subscribedToLaser || !laserModel)
  return 0;
  LOCK_MODEL(laserModel, stageWorld);
  size_t len = 0;
  stg_laser_sample_t* data = (stg_laser_sample_t*)stg_model_get_property(laserModel, "laser_data", &len);
  int numLaserReadings = len / sizeof(stg_laser_sample_t);
  assert(i <= numLaserReadings && i >= 0);
  int r = data[i].range;
  UNLOCK_MODEL(laserModel, stageWorld);
  return r;
}

int StageInterface::getLaserReflectance(int i) {
  if(!subscribedToLaser || !laserModel)
  return 0;
  size_t len = 0;
  LOCK_MODEL(laserModel, stageWorld);
  stg_laser_sample_t* data = (stg_laser_sample_t*)stg_model_get_property(laserModel, "laser_data", &len);
  int numLaserReadings = len / sizeof(stg_laser_sample_t);
  assert(i <= numLaserReadings && i >= 0);
  int r = data[i].reflectance;
  UNLOCK_MODEL(laserModel, stageWorld);
  return r;
}

size_t StageInterface::forEachLaserReading(LaserReadingFunc &func, const size_t &start) {
  LOCK_MODEL(laserModel, stageWorld);
  size_t len = 0;
  stg_laser_sample_t *data = (stg_laser_sample_t*)stg_model_get_property(laserModel, "laser_data", &len);
  if(!data)
  {
    UNLOCK_MODEL(laserModel, stageWorld);
    return 0;
  }
  size_t n = len / sizeof(stg_laser_sample_t);
  size_t i;
  for(i = start; i < n; ++i) {
    if(!func(data[i].range, data[i].reflectance)) break;
  }
  UNLOCK_MODEL(laserModel, stageWorld);
  return i - start;
}

void StageInterface::openSonar() {
  RobotInterface::openSonar();
  if(!sonarModel)
  {
    return;
  }
  if(openedSonar) {
    return;
  }
  openedSonar = true;
  if(!subscribedToSonar) {
    LOCK_MODEL(sonarModel, stageWorld);
    stg_model_subscribe(sonarModel);
    subscribedToSonar = true;
    UNLOCK_MODEL(sonarModel, stageWorld);
  }
  if(!stg_world_get_quiet(stageWorld))
    inform("Sonar on");
}

void StageInterface::closeSonar() {
  RobotInterface::closeSonar();
  if(!openedSonar) {
    return;
  }
  openedSonar = false;
  //memset(sonarData, 0, sizeof(sonarData));
  //numSonarSamples = 0;
  if(sonarModel && subscribedToSonar) {
    LOCK_MODEL(sonarModel, stageWorld);
    stg_model_unsubscribe(sonarModel);
    subscribedToSonar = false;
    UNLOCK_MODEL(sonarModel, stageWorld);
  }
  if(!stg_world_get_quiet(stageWorld))
    inform("Sonar off");
}


void StageInterface::openLaser() {
  if(!laserModel)
    return;
  if(openedLaser) {
    return;
  }
  openedLaser = true;
  if(!subscribedToLaser) {
    LOCK_MODEL(laserModel, stageWorld);
    stg_model_subscribe(laserModel);
    UNLOCK_MODEL(laserModel, stageWorld);
    subscribedToLaser = true;
  }
}

void StageInterface::closeLaser() {
  if(!openedLaser) return;
  if(laserModel && subscribedToLaser) {
    LOCK_MODEL(laserModel, stageWorld);
    stg_model_unsubscribe(laserModel);
    UNLOCK_MODEL(laserModel, stageWorld);
    subscribedToLaser = false;
  }
  openedLaser = false;
}

char StageInterface::gripperState() {
  // TODO 
  return 0;
}

void StageInterface::setLaserResolution(double inc) {
  if(!laserModel) {
    warn_s("this robot has no laser model, can't set resolution.");
    return;
  }
  LOCK_MODEL(laserModel, stageWorld);
  stg_laser_config_t* lasercfg = (stg_laser_config_t*)stg_model_get_property_fixed(laserModel, "laser_cfg", sizeof(stg_laser_config_t));
  assert(lasercfg);
  if(inc == 0)
    lasercfg->samples = 0;
  else
    lasercfg->samples = (int) ceil( ceil(RTOD(lasercfg->fov)) / inc) + 1;  // readings should fall at beginning of each degree as well as the end of the last degree
  stg_model_property_changed(laserModel, "laser_cfg");
  UNLOCK_MODEL(laserModel, stageWorld);
}

double StageInterface::getLaserResolution() {
  if(!laserModel) {
    warn_s("this robot has no laser model, can't get its resolution.");
    return 0.0;
  }
  LOCK_MODEL(laserModel, stageWorld);
  stg_laser_config_t* lasercfg = (stg_laser_config_t*)stg_model_get_property_fixed(laserModel, "laser_cfg", sizeof(stg_laser_config_t));
  assert(lasercfg);
  //fprintf(stderr, "StageInterface::getLaserResolution(): returning %f\n", RTOD(lasercfg->fov) / lasercfg->samples);
  double res;
  if(lasercfg->samples == 0)
    res = 0;
  else
    res = RTOD(lasercfg->fov) / lasercfg->samples;
  UNLOCK_MODEL(laserModel, stageWorld);
  return res;
}

void StageInterface::setLaserFOV(double deg) {
  if(!laserModel) return ;
  LOCK_MODEL(laserModel, stageWorld);
  stg_laser_config_t* lasercfg = (stg_laser_config_t*)stg_model_get_property_fixed(laserModel, "laser_cfg", sizeof(stg_laser_config_t));
  assert(lasercfg);
  lasercfg->fov = DTOR(deg);
  stg_model_property_changed(laserModel, "laser_cfg");
  UNLOCK_MODEL(laserModel, stageWorld);
}

double StageInterface::getLaserFOV() {
  LOCK_MODEL(laserModel, stageWorld);
  if(!laserModel) return 0.0;
  stg_laser_config_t* lasercfg = (stg_laser_config_t*)stg_model_get_property_fixed(laserModel, "laser_cfg", sizeof(stg_laser_config_t));
  assert(lasercfg);
  double fov = RTOD(lasercfg->fov);
  UNLOCK_MODEL(laserModel, stageWorld);
  return fov;
}

double StageInterface::getLaserStartAngle() {
  return laserStart;
}

double StageInterface::getLaserEndAngle() {
  return laserEnd;
}

void StageInterface::setLaserAngles(double start, double end) {
  if(!laserModel) return;
  LOCK_MODEL(laserModel, stageWorld);
  stg_laser_config_t* lasercfg = (stg_laser_config_t*)stg_model_get_property_fixed(laserModel, "laser_cfg", sizeof(stg_laser_config_t));
  assert(lasercfg);
  //fprintf(stderr, "StageInterface::setLaserAngles(%f,%f): setting fov to %f. reverse scan to %d.\n", start, end, DTOR( fabs( fabs(end+360.0) - fabs(start+360.0) ) ), (end < start));
  lasercfg->fov = DTOR( fabs( fabs(end+360.0) - fabs(start+360.0) ) );
  lasercfg->reverse_scan = (end < start);
  laserStart = start;
  laserEnd = end;
  stg_model_property_changed(laserModel, "laser_cfg");
  UNLOCK_MODEL(laserModel, stageWorld);
}

stg_position_data_t* StageInterface::stagePositionData() {
  stg_position_data_t* data = (stg_position_data_t*) stg_model_get_property_fixed(positionModel, "position_data", sizeof(stg_position_data_t));
  assert(data);
  return data;
}

stg_velocity_t* StageInterface::stageVelocityData() {
  stg_velocity_t* data = (stg_velocity_t*) stg_model_get_property_fixed(positionModel, "velocity", sizeof(stg_velocity_t));
  assert(data);
  return data;
}

void StageInterface::getSimulatorPose(long &x, long &y, long &z, int &theta)
{
  LOCK_MODEL(positionModel, stageWorld);
  stg_pose_t *pose = (stg_pose_t*)stg_model_get_property_fixed(positionModel, "pose", sizeof(stg_pose_t));
  x = (long)(pose->x * 1000.0);
  y = (long)(pose->y * 1000.0);
  z = 0;
  theta = (int)ArMath::roundInt(RTOD(pose->a));
  UNLOCK_MODEL(positionModel, stageWorld);
}

long StageInterface::getSimulatorPoseX() {
  LOCK_MODEL(positionModel, stageWorld);
  stg_pose_t* pose = (stg_pose_t*)stg_model_get_property_fixed(positionModel, "pose", sizeof(stg_pose_t));
  assert(pose);
  int x = (int)(pose->x * 1000.0);
  UNLOCK_MODEL(positionModel, stageWorld);
  return x;
}

long StageInterface::getSimulatorPoseY() {
  LOCK_MODEL(positionModel, stageWorld);
  stg_pose_t* pose = (stg_pose_t*)stg_model_get_property_fixed(positionModel, "pose", sizeof(stg_pose_t));
  assert(pose);
  int y = (int)(pose->y * 1000.0);
  UNLOCK_MODEL(positionModel, stageWorld);
  return y;
}

int StageInterface::getSimulatorPoseTheta() {
  LOCK_MODEL(positionModel, stageWorld);
  stg_pose_t* pose = (stg_pose_t*)stg_model_get_property_fixed(positionModel, "pose", sizeof(stg_pose_t));
  assert(pose);
  int th = (int)ArMath::roundInt(RTOD(pose->a));
  UNLOCK_MODEL(positionModel, stageWorld);
  return th;
}

int StageInterface::getLastInterval() { 
  lockStageWorld();
  int i = stg_world_get_last_interval(stageWorld);
  unlockStageWorld();
  return i;
}

int StageInterface::getSimInterval() {
  lockStageWorld();
  int i = stg_world_get_sim_interval(stageWorld);
  unlockStageWorld();
  return i;
}

int StageInterface::getRealInterval() {
  lockStageWorld();
  int i = stg_world_get_real_interval(stageWorld);
  unlockStageWorld();
  return i;
}

void StageInterface::error_s(const char* message)
{
  //LOCK_MODEL(messagesModel, stageWorld);
  if(messagesModel) stg_model_lock(messagesModel);
  /*stg_message_t* msg =*/ 
  stg_messages_send(messagesModel, (char*) robotName.c_str(), STG_MSG_CRITICAL, message);
  // can't call this from any old thread //   if(msg) while(!msg->displayed) stg_world_update(stageWorld, FALSE);
  if(messagesModel) stg_model_unlock(messagesModel);
  //UNLOCK_MODEL(messagesModel, stageWorld);
}

void StageInterface::warn_s(const char* message)
{
  //LOCK_MODEL(messagesModel, stageWorld);
  if(messagesModel) stg_model_lock(messagesModel);
  //stg_message_t* msg = 
  stg_messages_send(messagesModel, robotName.c_str(), STG_MSG_WARNING, message);
  // can't call this from any old thread // if(msg) while(!msg->displayed) stg_world_update(stageWorld, FALSE);
  //UNLOCK_MODEL(messagesModel, stageWorld);
  if(messagesModel) stg_model_unlock(messagesModel);
  //ArUtil::sleep(10); // hack to work around messages model habit of clobering old messages with new ones.
}

void StageInterface::inform_s(const char* message)
{
  //LOCK_MODEL(messagesModel, stageWorld);
  if(messagesModel) stg_model_lock(messagesModel);
  //stg_message_t* msg = 
  stg_messages_send(messagesModel, robotName.c_str(), STG_MSG_INFORMATION, message);
  // can't call this from any old thread //  if(msg) while(!msg->displayed) stg_world_update(stageWorld, FALSE);
  //UNLOCK_MODEL(messagesModel, stageWorld);
  if(messagesModel) stg_model_unlock(messagesModel);
}


void StageInterface::log_s(const char* message)
{
  stg_print_msg("%s: %s", robotName.c_str(), message);
}

void StageInterface::log_error_s(const char* message)
{
  stg_print_error("%s: %s", robotName.c_str(), message);
  stg_flush_log_file();
}

void StageInterface::shutdown(int errorcode)
{
  lockStageWorld();
  stg_quit_request_code(errorcode + 1); // 1 means to quit normally, >1 for an error
  unlockStageWorld();
}

/*
void StageInterface::loadNewMap(char* mapfile)
{
  inform("Loading new map file: \"%s\".", mapfile);
  lockStageWorld();
  loadNewMap_NoLock(mapfile);
  unlockStageWorld();
}
*/


//void StageInterface::loadNewMap_NoLock(char* mapfile)
void StageInterface::loadNewMap(const char* mapfile)
{
  mapLoader.loadNewMapSync(mapfile);
  /*
  //StageMapLoader loader(stageWorld, messagesModel);
  if(!mapLoader.loadNewMapIfChanged(mapfile))
  {
  stg_messages_send(messagesModel, "MobileSim", ::strinSTG_MSG_WARNING, "Not reloading map file \"%s\", it has not changed since last load.", mapfile);
  }
  */
}

void StageInterface::loadNewMapAsync(const char *mapfile)
{
  mapLoader.loadNewMapAsync(mapfile);
}

/* Map Loader/Converter: */


/// Synchronous (blocks if another thread is doing something)
void StageMapLoader::loadNewMapSync(const char *mapfile)
{
  activityMutex.lock();
  if(shouldReloadMap(mapfile))
    loadNewMap(mapfile);
  else
    inst_interface->warn("Not reloading map file \"%s\', it has not changed since last load.", mapfile);
  activityMutex.unlock();
}


/// @note locks @a world
bool StageMapLoader::loadNewMap(bool loadLines, bool loadPoints)
{
  bool loadedData = false;
  //  ArTime timer;

  stg_world_t *world = inst_world;

  /// @todo Get points and lines change timestamps from ArMap and check against saved times. 
  ///  (Though note that this will create a bug, that modifying reflectors or
  //obstacle cairns will be ignored)

  if(!inst_map) return false;
  ArMap *map = inst_map;
  const char* mapfile = map->getFileName();

  // Load map data (this is slightly time consuming, so don't lock world yet)
  stg_polygon_t *polys = NULL;
  size_t npolys = 0;
  size_t nlines = 0;

  //timer.setToNow();

  // load lines from map
  if(loadLines)
  {
    nlines = map->getLines()->size();
    if(nlines > 0)
    {
      polys = stg_polygons_create(nlines);
      for(std::vector<ArLineSegment>::const_iterator i = map->getLines()->begin();
          i != map->getLines()->end(); ++i)
      {
        // make an infintessimally thin rectangle for this line. convert mm to m
        stg_point_t v1 = {(*i).getX1() / 1000.0, (*i).getY1() / 1000.0};
        stg_point_t v2 = {(*i).getX2() / 1000.0, (*i).getY2() / 1000.0};
        stg_polygon_append_points(&polys[npolys], &v1, 1);
        stg_polygon_append_points(&polys[npolys], &v1, 1);
        stg_polygon_append_points(&polys[npolys], &v2, 1);
        stg_polygon_append_points(&polys[npolys], &v2, 1);
        ++npolys;
        #ifdef PRINT_NUM_LINES_CREATED
        if(npolys > 0 && npolys % PRINT_NUM_LINES_CREATED == 0) printf("MobileSim map loader: At %u lines...\n", npolys);
        #endif
      }
      loadedData = true;
    }
  }


  stg_point_t *points = NULL;
  size_t npoints = 0;

  // load points from map
  if(loadPoints)
  {
    npoints = map->getPoints()->size();
    if(npoints > 0)
    {
      points = stg_points_create(npoints);
      stg_point_t* p = points;
      size_t count = 0;
      for(std::vector<ArPose>::const_iterator i = map->getPoints()->begin();
        i != map->getPoints()->end(); ++i)
      {
        p->x = (double) (*i).getX() / 1000.0;     // convert mm to m
        p->y = (double) (*i).getY() / 1000.0;
        ++p;
        #ifdef PRINT_NUM_POINTS_CREATED
        if(count > 0 && count % PRINT_NUM_POINTS_CREATED == 0) printf("MobileSim map loader: At %u points...\n", count);
        ++count;
        #endif
      }

      loadedData = true;
    }
  }

  //printf("XXX Took %d msec to load data from ArMap (world unlocked).\n", timer.mSecSince());

  if(!loadedData)
  {
    stg_print_warning("No obstacle data loaded from map file \"%s\"!", mapfile);
    if(!loadPoints) stg_print_warning("Requested not to load point data, try enabling.");
    if(!loadLines) stg_print_warning("Requested not to load line data, try enabling.");
  }

  // Lock world and remove old models, add new models to it
  stg_world_lock(world);


  // Clear any existing map models
  for(std::set<stg_model_t*>::const_iterator i = mapModels.begin(); i != mapModels.end(); i++)
  {
    assert(*i);
    stg_world_remove_model(world, *i);
    //printf("clearMap: destroying model \"%s\"...\n", stg_model_get_token(*i));
    stg_model_destroy(*i);
  }
  mapModels.clear();

  // find a unique id number
  stg_id_t id = 0;
  for(id = 0; stg_world_get_model(world, id) != NULL && id <= STG_ID_T_MAX; ++id)
    ;
  if(id == STG_ID_T_MAX)
  {
    stg_print_error("!!! too many models in the world, can't create a new one.");
    return false;
  }

    // create model TODO use stg_world_new_model instead!
  // set as a background figure, so it isn't redrawn every time the robot moves
  // or whatever.
  stg_model_t* map_model = stg_model_create(world, NULL, id, mapfile, "model", "model", 0, NULL, TRUE);


  // store the map file name
  stg_model_set_property(map_model, "source", (void*)mapfile, strlen(mapfile)+1);

  // set color
  stg_color_t mapcolor = stg_lookup_color("dark gray");
  if(mapcolor == 0xFF0000) mapcolor = 0; // black if not found
  stg_model_set_property(map_model, "color", &mapcolor, sizeof(mapcolor));

  // set size
  double maxX_mm = 0;
  double minX_mm = 0;
  double maxY_mm = 0;
  double minY_mm = 0;
  if(npoints > 0 && nlines > 0)
  {
    maxX_mm = fmax(map->getLineMaxPose().getX(), map->getMaxPose().getX());
    minX_mm = fmin(map->getLineMinPose().getX(), map->getMinPose().getX());
    maxY_mm = fmax(map->getLineMaxPose().getY(), map->getMaxPose().getY());
    minY_mm = fmin(map->getLineMinPose().getY(), map->getMinPose().getY());
  }
  else if(npoints > 0)
  {
    maxX_mm = map->getMaxPose().getX();
    minX_mm = map->getMinPose().getX();
    maxY_mm = map->getMaxPose().getY();
    minY_mm = map->getMinPose().getY();
  }
  else if(nlines > 0)
  {
    maxX_mm = map->getLineMaxPose().getX();
    minX_mm = map->getLineMinPose().getX();
    maxY_mm = map->getLineMaxPose().getY();
    minY_mm = map->getLineMinPose().getY();
  }
  stg_size_t size;
  size.x = maxX_mm/1000.0 - minX_mm/1000.0; //mm to m
  size.y = maxY_mm/1000.0 - minY_mm/1000.0; //mm to m
  //printf("Setting map size to %f, %f\n", size.x, size.y);
  stg_model_set_size(map_model, size);

  // but don't scale it to that size, lines and points are already at the right
  // places for correct scale
  stg_model_set_scaling(map_model, FALSE);

  // set origin offset
/*
  stg_pose_t offset;
  offset.x = (size.x / 2.0) + (minX_mm / 1000.0);
  offset.y = (size.y / 2.0) + (minY_mm / 1000.0);
  offset.a = 0;
  stg_model_set_origin(map_model, offset);
KXXX RH */

  // Turn off grid
  int grid = 0;
  stg_model_set_property(map_model, "grid", &grid, sizeof(int));

  // Make it unmovable
  int movemask = 0; 
  stg_model_set_property(map_model, "mask", &movemask, sizeof(int));

  // store creation time and file source
  stg_model_set_property(map_model, "source", (void*)mapfile, strlen(mapfile)+1);

  time_t tm = time(NULL);
  stg_model_set_property(map_model, "creation_time", &tm, sizeof(tm));


  // store map data. 
  //timer.setToNow();
  if(polys)
  {
    //puts("XXX Storing polygons...");
    stg_model_init_polygons(map_model, polys, npolys);
    free(polys);  // it was copied by stg_model_init_polygons
  }

  if(points)
  {
    //puts("XXX Storing points...");
    stg_model_init_points(map_model, points, npoints);
    stg_points_destroy(points); // it was copied by stg_model_init_points
  }

  //printf("XXX Took %d msec to initialize model points and polygons (world locked).\n", timer.mSecSince());


  // Remember this model
  mapModels.insert(map_model);

  // Add model to world. Will be mapped in.
  //timer.setToNow();
  stg_world_add_model(world, map_model);
  //printf("XXX Took %d msec to add model to world.\n", timer.mSecSince());
  //printf("loadMap: added map model \"%s\" to world.\n", stg_model_get_token(map_model));


  // Check special simulator attributes of custom map object type definitions,
  // and create models for objects as neccesary.
  // TODO check Color0 and Color1, SonarReflect.
  std::map<std::string, int> laser_return_vals;
  std::set<std::string> obstacles;
  std::set<std::string> boxshapes;
  std::set<std::string> lineshapes;

  // Built in reflector type always has a high laser retun value by default
  laser_return_vals["Reflector"] = 2;

  for(std::list<ArArgumentBuilder*>::const_iterator i = map->getMapInfo()->begin(); i != map->getMapInfo()->end(); ++i)
  {
    const char *type = (*i)->getArg(1);
    if( strncmp(type, "Name=", 5) != 0 )
    {
      //stg_print_warning("First MapInfo attribute is not \"Name\", skipping.");
      continue;
    }
    type += strlen("Name=");

    const char *shape = (*i)->getArg(0);
    if(strcmp(shape, "SectorType") == 0)
      boxshapes.insert(type);
    else if(strcmp(shape, "BoundaryType") == 0)
      lineshapes.insert(type);


    // Reflectors can be built-in reflectors, or have a name that
    // previous versions of MobileSim interpreted as automatically being
    // reflectors.
    int val = 1;
    char* endTag = strrchr(type, '.');
    if(   strcmp( type, "Sim.Reflector") == 0 
      || strcmp(  type, "Reflector") == 0
      || (endTag && strcmp(endTag, ".Reflect") == 0)
    )
    {
      val = 2;
      //laser_return_vals[type] = val;
    }

    // Check remaining attributes for special simulator things
    for(size_t a = 2; a < (*i)->getArgc(); ++a)
    {
      char buf[256];
      ArUtil::stripQuotes(buf, (*i)->getArg(a), 256);

      // Reflective to laser?
      if(strncmp(buf, "Sim.LaserReflect=", strlen("Sim.LaserReflect=")) == 0)
      {
        val = atoi( buf + strlen("Sim.LaserReflect=") ) + 1;   // Need to add one because Stage starts highly reflective objects at 2, but SICK/Aria at 1
        stg_print_msg("Will use reflector value %d for objects with type %s (from Sim.LaserReflect attribute in MapInfo)", val, type);
      }

      // Obstacle to robot?
      if(strcasecmp(buf, "Sim.Obstacle=yes") == 0 || strcasecmp(buf, "Sim.Obstacle=true") == 0)
      {
        stg_print_msg("Objects of type %s will be represented as obstacles (from Sim.Shape attribute in MapInfo)", type);
        obstacles.insert(type);
      }
    }

    laser_return_vals[type] = val;
  }

  // Create special objects for certain Cairn objects.
  // Reflector: make a line in the map with bright reflectance. 
  // Sim.BoxObstacle: Make a box that the user can move. 
  for(std::list<ArMapObject*>::const_iterator i = map->getMapObjects()->begin(); i != map->getMapObjects()->end(); ++i)
  {
    ArMapObject* obj = (*i);
    if(obj == NULL) continue;

    stg_model_t* model = NULL;

    int laser_return = 1;
    if(laser_return_vals.find(obj->getType()) != laser_return_vals.end())
      laser_return = laser_return_vals[obj->getType()];
    bool builtinReflector = false;

    // XXX this needs to be refactored a bit, probably eliminate the seperate
    // loadReflector and LoadBoxObstacle functions, just set unique properties
    // seperately.

    // Built-in Reflector objects have special properties (color, shape, etc.) 
    if(strcmp(obj->getType(), "Reflector") == 0) 
    {
      builtinReflector = true;
      model = loadReflector(obj, map_model, laser_return);
    }

    // Is it a box-shaped obstacle?
    else if(   strncmp(obj->getType(), "Sim.BoxObstacle", strlen("Sim.BoxObstacle")) == 0
      || (boxshapes.find(obj->getType()) != boxshapes.end() && obstacles.find(obj->getType()) != obstacles.end())  )
    {
      model = loadBoxObstacle(obj, map_model);
    }

    // TODO support line-shaped obstacles and reflective thngs that aren't
    // obstacles

    if(model == NULL) continue;

    if(laser_return > 1 && !builtinReflector)
    {
      stg_color_t color = stg_lookup_color("magenta");
      stg_model_set_property(model, "color", &color, sizeof(color));
      stg_model_set_property(model, "laser_return", &laser_return, sizeof(laser_return));
    }

    // store file it was loaded from and current time
    stg_model_set_property(model, "source", (void*)mapfile, strlen(mapfile)+1);
    time_t t = time(NULL);
    stg_model_set_property(model, "creation_time", &t, sizeof(t));

    mapModels.insert(model);
  }


  // resize world 
  /// @todo Only resize if it got bigger. Also should optimize this, it takes forever.
  //timer.setToNow();
  //puts("XXX Resizing...");
  stg_world_resize_to_contents(world, 10);
  //printf("XXX Took %d msec to resize world to contents (world locked).\n", timer.mSecSince());

  stg_world_unlock(world);


  return true;
}


/** Does not lock world */
stg_model_t* StageMapLoader::loadReflector(ArMapObject* obj, stg_model_t* /*map_model*/, stg_laser_return_t laser_return)
{
  stg_world_t *world = inst_world;
  if(!obj->hasFromTo()) 
  {
    stg_print_warning("Found a Reflector Cairn in the map, but the line has no 'from' and 'to' position; skipping.");
    return NULL;
  }


  /*
  stg_print_msg("Found a Reflector in the map file at pose (%fmm,%fmm); a line from (%fmm,%fmm) to (%fmm,%fmm)",
  obj->getPose().getX(), obj->getPose().getY(), 
  obj->getFromPose().getX(), obj->getFromPose().getY(), 
  obj->getToPose().getX(), obj->getToPose().getY());
  */

  // Make a very thin magenta box aligned with the reflector line

  stg_pose_t reflector_pose;




  double line_x1 = (obj->getFromPose().getX() / 1000.0);
  double line_x2 = (obj->getToPose().getX() / 1000.0);
  double line_y1 = (obj->getFromPose().getY() / 1000.0);
  double line_y2 = (obj->getToPose().getY() / 1000.0);
  double line_dx = line_x2 - line_x1;
  double line_dy = line_y2 - line_y1;


  // Turn it into a box with some thickness
  double line_angle = atan( fabs(line_dy) / fabs(line_dx) );
  double reflector_theta = fabs(line_angle + reflector_pose.a);
  double disp_x = fabs(ReflectorThickness * sin(reflector_theta));
  double disp_y = fabs(ReflectorThickness * cos(reflector_theta));

  // If an object in stage has size (x or y) == 0, then it become
  // infinitely large!
  if(disp_x <= 0.000001) disp_x = 0.000001;
  if(disp_y <= 0.000001) disp_y = 0.000001;

  // Use cairn pose:
  //reflector_pose.x = obj->getPose().getX() / 1000.0;
  //reflector_pose.y = obj->getPose().getY() / 1000.0;
  //reflector_pose.a = obj->getPose().getTh(); // Specifically *don't* use rotation, ARIA doesn't.
  //reflector_pose.a = 0;

  // Alternatively, figure pose from line endpoints:
  reflector_pose.x = line_x1 + (line_dx / 2.0);
  reflector_pose.y = line_y1 + (line_dy / 2.0);
  reflector_pose.a = 0;

  // Find a unique name
  char token[32];
  memset(token, 0, 32);
  strcat(token, "Reflector:0");
  int i = 0;
  for(; stg_world_model_name_lookup(world, token) != NULL; i++)
  {
    snprintf(token, 32, "Reflector:%i", i);
  }

  // find a unique id number
  stg_id_t id = 0;
  for(id = 0; stg_world_get_model(world, id) != NULL; id++)
    ;

  // create reflector model.  Note, not making map a parent, this causes
  // problems currently. TODO use stg_world_new_model instead.
  stg_model_t* reflector_model = stg_model_create(world, NULL, id, token, "model", "model", i, NULL, FALSE);
  assert(reflector_model);


  // set size and position
  stg_size_t reflector_size = { fabs(line_dx) + disp_x, fabs(line_dy) + disp_y };
  stg_model_set_size(reflector_model, reflector_size);
  stg_model_set_pose(reflector_model, reflector_pose);

  // bright laser return
  stg_model_set_property(reflector_model, "laser_return", &laser_return, sizeof(laser_return));

  // color
  stg_color_t color = stg_lookup_color("magenta");
  stg_model_set_property(reflector_model, "color", &color, sizeof(color));

  // shape
  stg_polygon_t* reflector_polys = stg_polygons_create(1);
  stg_point_t v1 = { line_x1, line_y1 };
  stg_point_t v2 = { line_x1 + disp_x, line_y1 - disp_y };
  stg_point_t v3 = { line_x2 + disp_x, line_y2 - disp_y };
  stg_point_t v4 = { line_x2, line_y2 };
  stg_polygon_append_points(&reflector_polys[0], &v1, 1);
  stg_polygon_append_points(&reflector_polys[0], &v2, 1);
  stg_polygon_append_points(&reflector_polys[0], &v3, 1);
  stg_polygon_append_points(&reflector_polys[0], &v4, 1);
  stg_model_init_polygons(reflector_model, reflector_polys, 1);

  int reflector_movemask = 1|2; // movable|rotatable
  int reflector_outline = 0;    // no outline
  stg_model_set_property(reflector_model, "mask", &reflector_movemask, sizeof(int));
  stg_model_set_property(reflector_model, "outline", &reflector_outline, sizeof(int));

  stg_world_add_model(world, reflector_model);
  //printf("loadMap: added reflector model \"%s\" to world.\n", stg_model_get_token(reflector_model));
  return reflector_model;
}

/** Does not lock world */
stg_model_t* StageMapLoader::loadBoxObstacle(ArMapObject* obj, stg_model_t* /*map_model*/)
{

  stg_world_t *world = inst_world;

  // Todo, look for a MapInfo declaration that defines properties like color for this
  // type of obstacle. (e.g. Sim.SonarOnly, Sim.Invisible, etc.)
  if(!obj->hasFromTo())
  {
    stg_print_warning("Found a Sim.BoxObstacle Cairn in the map, but the line has no 'from' and 'to' position. Skipping.");
    return NULL;
  }

  stg_print_msg("Found a BoxObstacle in the map file at pose (%.0fmm,%.0fmm,%.0fdeg); a box from (%.0fmm,%.0fmm) to (%.0fmm,%.0fmm)",
  obj->getPose().getX(), obj->getPose().getY(), obj->getPose().getTh(), 
  obj->getFromPose().getX(), obj->getFromPose().getY(), 
  obj->getToPose().getX(), obj->getToPose().getY());

  // Find a unique name and id
  char token[32];
  memset(token, 0, 32);
  strcat(token, "BoxObstacle:0");
  int i = 0;
  for(; stg_world_model_name_lookup(world, token) != NULL; i++)
  {
    snprintf(token, 32, "BoxObstacle:%i", i);
  }

  // find a unique name
  stg_id_t id = 0;
  for(id = 0; stg_world_get_model(world, id) != NULL; id++)
    ;

  // create model. note, using null parent. making the map a parent causes
  // problems right now. TODO use stg_world_new_model instead!
  stg_model_t* box_model = stg_model_create(world, NULL, id, token, "model", "model", i, NULL, FALSE);


  // size
  stg_size_t box_size;
  box_size.x = (obj->getToPose().getX() - obj->getFromPose().getX()) / 1000.0;
  box_size.y = (obj->getToPose().getY() - obj->getFromPose().getY()) / 1000.0;
  stg_model_set_size(box_model, box_size);

  // pose. Map object pose is not used for some reason, it needs to be
  // positioned according to the "from" and "to" poses, in this slightly
  // complex way.
  stg_pose_t box_pose;
  double sinObjTh, cosObjTh;
  STG_SINCOS(DTOR(obj->getPose().getTh()), sinObjTh, cosObjTh);
  double x = (obj->getFromPose().getX() / 1000.0) + (box_size.x / 2.0);
  double y = (obj->getFromPose().getY() / 1000.0) + (box_size.y / 2.0);
  box_pose.x = (x * cosObjTh) - (y * sinObjTh);
  box_pose.y = (x * sinObjTh) + (y * cosObjTh);
  box_pose.a = DTOR(obj->getPose().getTh());
  stg_model_set_pose(box_model, box_pose);

  // color
  stg_color_t color = stg_lookup_color("light green");
  stg_model_set_property(box_model, "color", &color, sizeof(color));

  // shape
  stg_polygon_t* box_polys = stg_polygons_create(1);
  //stg_point_t v1 = { obj->getFromPose().getX()/1000.0, obj->getFromPose().getY()/1000.0 };
  //stg_point_t v2 = { obj->getToPose().getX()/1000.0, obj->getFromPose().getY()/1000.0 };
  //stg_point_t v3 = { obj->getToPose().getX()/1000.0, obj->getToPose().getY() / 1000.0 };
  //stg_point_t v4 = { obj->getFromPose().getX()/1000.0, obj->getToPose().getY()/1000.0 };
  stg_point_t v1 = { -box_size.x/2.0, -box_size.y/2.0 };   // actually equivalent... ?
  stg_point_t v2 = { box_size.x/2.0, -box_size.y/2.0 };
  stg_point_t v3 = { box_size.x/2.0, box_size.y/2.0 };
  stg_point_t v4 = { -box_size.x/2.0, box_size.y/2.0 };
  stg_polygon_append_points(&box_polys[0], &v1, 1);
  stg_polygon_append_points(&box_polys[0], &v2, 1);
  stg_polygon_append_points(&box_polys[0], &v3, 1);
  stg_polygon_append_points(&box_polys[0], &v4, 1);
  stg_model_init_polygons(box_model, box_polys, 1);

  int box_movemask = 1|2; // movable and rotatable
  int box_outline = 1;  // has a black outline
  stg_model_set_property(box_model, "mask", &box_movemask, sizeof(int));
  stg_model_set_property(box_model, "outline", &box_outline, sizeof(int));

  stg_world_add_model(world, box_model);
  //printf("loadMap: added box model \"%s\" to world.\n", stg_model_get_token(box_model));
  return box_model;
}

bool StageMapLoader::shouldReloadMap(const char *mapfile) 
{
  struct stat filestat;
  int s = stat(mapfile, &filestat);
  if(s != 0)
  return true;  // no way to know modification time, force reload
  bool hadModelsFromThatMap = false;
  stg_world_t *world = inst_world;
  stg_world_lock(world);
  for(std::set<stg_model_t*>::const_iterator i = mapModels.begin(); i != mapModels.end(); i++)
  {
    size_t len;
    char *source = (char*)stg_model_get_property(*i, "source", &len);
    if( source && len > 0 && strcmp(mapfile, source) == 0)
    {
      // model came from this map file
      hadModelsFromThatMap = true;
      time_t *modelTime = (time_t*)stg_model_get_property_fixed(*i, "creation_time", sizeof(time_t));
      if(!modelTime) {
        stg_world_unlock(world);
        return true;  // no creation_time, no way to know if it's newer, always reload map file
      }
      if( difftime( filestat.st_mtime, *modelTime ) > 0.0) {
        //printf("map file has modification time %d (%s), newer than model %s's creation time %d (%s). must reload.\n", filestat.st_mtime, ctime(&filestat.st_mtime), stg_model_get_token(*i), *modelTime, ctime(modelTime));
        stg_world_unlock(world);
        return true; 
      }
    }
  }
  stg_world_unlock(world);
  if(hadModelsFromThatMap) {
    return false;
  } else {
    return true;   // force reload if no models came from that map (it's a completely new map)
  }
}


static void* runStageMapLoaderThread(void *ptr)
{
  ((StageMapLoader*)ptr)->loadNewMapThread();
  return 0;
}

void StageMapLoader::loadNewMapAsync(const char *mapfile, MapLoadedCallback callback)
{
  requestsMutex.lock();
  requests.push_back(StageMapLoader::Request(inst_interface, inst_world, mapfile, callback));
  if(!threadCreated)
  {
    int err = pthread_create(&thread, NULL, &runStageMapLoaderThread, this);
    if(err != 0)
    {
      if(err == EAGAIN)
      {
        if(inst_interface)
        {
          inst_interface->error("error creating now thread in StageMapLoader: system thread resources exhausted.");
        }
        else
        {
          puts("error creating now thread in StageMapLoader: system thread resources exhausted.");
        }
      }
      if(inst_interface)
        inst_interface->error("map Loader: critical error creating new thread!");
      else
        puts("map Loader: critical error creating new thread!");
      return;
    }
    threadCreated = true;
  }
  requestsMutex.unlock();
  pthread_cond_signal(&haveNewMapToLoadAsync);
}

void StageMapLoader::loadNewMapThread()
{
  // todo just pass in callback function to loadNewMap?
  //double minx, miny, maxx, maxy, homex, homey, hometh;
  pthread_mutex_t condmutex; // = PTHREAD_MUTEX_INITIALIZER;
  pthread_mutex_init(&condmutex, NULL);
  while(true)
  {
    activityMutex.lock();

    requestsMutex.lock();
    // todo make this not be a queue, it doesn't need to be, can just be a
    // single "map file to load next" string.
    while(requests.size() > 0)
    {
      currentRequest = requests.front();
      requests.pop_front();
      requestsMutex.unlock();

      currentRequest.doIt();

      requestsMutex.lock();
    }
    requestsMutex.unlock();

    activityMutex.unlock();

    pthread_mutex_lock(&condmutex);
    pthread_cond_wait(&haveNewMapToLoadAsync, &condmutex);
    pthread_mutex_unlock(&condmutex);
  }
}

bool StageMapLoader::loadNewMap(const char* mapfile, MapLoadedCallback callback,
//  double *min_x, double *min_y, double *max_x, double *max_y, 
//  double *home_x, double *home_y, double *home_th, 
std::string *errorMsg)
{
  if(!inst_map) {
    inst_map = new ArMap;
    inst_created_map = true;
  }
  ArMap *map = inst_map;
  char errbuf[128];
  if(!map->readFile(mapfile, errbuf, 127))
  {
    if(errorMsg) *errorMsg = errbuf;
    return false;
  }
  if(!loadNewMap(true, true))
  {
    return false;
  }
  if(callback != NULL)
  {
    MapLoadedInfo info;
    ArMapObject *home = map->findFirstMapObject(NULL, "RobotHome");
    if(!home)
    home = map->findFirstMapObject(NULL, "Dock");
    if(home)
    {
      info.have_home = true;
      info.home_x = home->getPose().getX();
      info.home_y = home->getPose().getY();
      info.home_th = home->getPose().getTh();
    }

    info.min_x = map->getLineMinPose().getX();
    info.min_y = map->getLineMinPose().getY();
    info.max_x = map->getLineMaxPose().getX();
    info.max_y = map->getLineMaxPose().getY();
    (*callback)(info);
  }
  return true;
}

std::vector< RobotInterface::DeviceInfo > StageInterface::getDeviceInfo()
{
  std::vector<RobotInterface::DeviceInfo> devs;
  GPtrArray* models = stg_model_get_children_ptr(positionModel);
  for(size_t i = 0; i < models->len; i++)
  {
    stg_model_t* m = (stg_model_t*)g_ptr_array_index(models, i);
    if(m)
    {
      DeviceInfo inf;
      inf.name = stg_model_get_token(m);
      inf.type = stg_model_get_base_type_name(m);
      inf.which = stg_model_get_instance_index(m);
      inf.status = 0;
      devs.push_back(inf);
    }
  }
  return devs;
}

void StageMapLoader::Request::error(const char *fmt, ...)  {
  mutex.lock();
  va_list args;
  va_start(args, fmt);
  if(sinterface)
    sinterface->error_v(fmt, args);
  else
    stg_print_error_v(fmt, args);
  va_end(args);
  mutex.unlock();
}

void StageMapLoader::Request::warn(const char *fmt, ...) {
  mutex.lock();
  va_list args;
  va_start(args, fmt);
  if(sinterface)
    sinterface->warn_v(fmt, args);
  else
    stg_print_warning_v(fmt, args);
  va_end(args);
  mutex.unlock();
}

void StageMapLoader::Request::inform(const char *fmt, ...) {
  mutex.lock();
  va_list args;
  va_start(args, fmt);
  if(sinterface)
    sinterface->inform_v(fmt, args);
  else
    stg_print_message_v(fmt, args);
  va_end(args);
  mutex.unlock();
}

void StageMapLoader::interfaceDestroyed(StageInterface *iface)
{
  requestsMutex.lock();
  if(currentRequest.sinterface == iface)
  {
    currentRequest.invalidateInterface();
  }
  for(std::deque<Request>::iterator i = requests.begin(); i != requests.end(); ++i)
  {
    if((*i).sinterface == iface)
    {
      (*i).invalidateInterface();
    }
  }
  requestsMutex.unlock();
}

void StageInterface::logState()
{
  stg_world_lock(stageWorld);
  log("Desired sim interval=%lu, desired real interval=%lu, last real interval=%lu, current ratio=%2.3f, average ratio=%2.3f, total elapsed=%lu", 
    (unsigned long) stg_world_get_sim_interval(stageWorld), 
    (unsigned long) stg_world_get_real_interval(stageWorld), 
    stg_world_get_last_interval(stageWorld), 
    (double)stg_world_get_sim_interval(stageWorld) / (double)stg_world_get_last_interval(stageWorld),
    stg_world_get_avg_interval_ratio(stageWorld),
    (unsigned long) stg_timenow()
  );
  log("%lu zero-interval warnings so far, %lu 10%%-too-long warnings.", 
    (unsigned long) stg_world_num_zero_interval_warnings(stageWorld),
    (unsigned long) stg_world_num_interval_too_long_warnings(stageWorld)
  );
  stg_world_unlock(stageWorld);

  stg_model_lock(positionModel);
  stg_position_cmd_t *cmd = (stg_position_cmd_t*) stg_model_get_property(positionModel, "position_cmd", NULL);
  log("Current command: x=%dmm/s y=%dmm/s th=%ddeg/s, transmode=%s, rotmode=%s", (int)(cmd->x / 1000.0), (int)(cmd->y / 1000.0), (int)RTOD(cmd->a ),
    stg_position_control_mode_name(cmd->transmode),
    stg_position_control_mode_name(cmd->rotmode)
  );
  char *rel_active_x  = (char*) stg_model_get_property(positionModel, "position_rel_ctrl_active_X", NULL);
  char *rel_active_a  = (char*) stg_model_get_property(positionModel, "position_rel_ctrl_active_A", NULL);
  stg_pose_t *prog = (stg_pose_t*) stg_model_get_property(positionModel, "position_rel_ctrl_progress", NULL);
  log(" relctrl_active_x=%d, relctrl_active_a=%d, progress_x=%d mm, progress_a=%d deg.",
    *rel_active_x, *rel_active_a, (int)(prog->x / 1000.0), (int)RTOD(prog->a) );
  stg_flush_log_file();
  stg_model_unlock(positionModel);
}

void StageMapLoader::Request::doIt() //StageMapLoader *loader)
{
  StageMapLoader loader(world, sinterface);
  if(loader.shouldReloadMap(mapfile.c_str()))
  {
    std::string errormsg;
    inform("Loading map from file \"%s\"...", mapfile.c_str());
    if(loader.loadNewMap(mapfile.c_str(), callback, &errormsg))
    {
      inform("Finished loading map from file \"%s\"...", mapfile.c_str());
    }
    else
    {
      warn("Error loading map from file \"%s\" (%s)", mapfile.c_str(), errormsg.c_str());
    }
  } 
  else
  {
    inform("Not reloading map file \"%s\", since it has not changed since last loaded.", mapfile.c_str());
  }
}


