/*  
    (C) Copyright 2005, ActivMedia Robotics LLC <http://www.activmedia.com>
    (C) Copyright 2006, 2007, 2008, 2009 MobileRobots, Inc. <http://www.mobilerobots.com>

    This is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This software is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this software; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef _EP_STAGE_INTERFACE_HH_
#define _EP_STAGE_INTERFACE_HH_

#include "RobotInterface.hh"
#include "ArMutex.h"
#include "ArMap.h"
#include <string>
#include <set>
#include <deque>
#include <map>

#include <stage.h>

#define MAX_SONAR_SAMPLES 128
#define MAX_LASER_SAMPLES (361*2)

// If defined, log about locking stageWorldMutex (for debugging, especially on windows where it's hard to interrupt GDB at a deadlock):
//#define STAGE_MUTEX_LOG

class ArMap;
class ArMapObject;

class StageInterface;

/** Load data from an ActivMedia map file into
 * a Stage world, and keep track of models created. 
 *
 * @todo make this not keep everything static, and have main() or StageInterface
 * keep an instance.
 */
class StageMapLoader
{
  public:
    class MapLoadedInfo {
    public:
      double min_x;
      double min_y;
      double max_x;
      double max_y;
      double home_x;
      double home_y;
      double home_th;
      bool have_home;
      MapLoadedInfo() : min_x(0), min_y(0), max_x(0), max_y(0), home_x(0), home_y(0), home_th(0), have_home(false)
      {}
    };
    typedef void (*MapLoadedCallback)(MapLoadedInfo);
      ///< minx, miny, maxx, maxy, homex, homey, hometh

    class Request {
      private:
        ArMutex mutex;
      public:
        StageInterface *sinterface;
        stg_world_t *world;
        std::string mapfile;
        MapLoadedCallback callback;
        Request(StageInterface *_if = NULL, stg_world_t *_w = NULL, const std::string& _m = "", MapLoadedCallback _cb = NULL) :
          sinterface(_if), world(_w), mapfile(_m), callback(_cb)
        {}

        /// copy everything but mutex
        Request(const Request& other) :
          sinterface(other.sinterface),
          world(other.world),
          mapfile(other.mapfile),
          callback(other.callback)
        {}

        void error(const char *fmt, ...);
        void warn(const char *fmt, ...);
        void inform(const char *fmt, ...);

        /// Clear 'sinterface' in case it becomes invalid (e.g. destroyed)
        void invalidateInterface() { 
          mutex.lock();
          sinterface = NULL;
          mutex.unlock();
        }

        void doIt(); //StageMapLoader *loader);
  };

  static const double ReflectorThickness; ///< meters
  static std::set<stg_model_t*> mapModels; ///< all models created by loading maps
  static ArMutex mapModelsMutex;
  static pthread_t thread;
  static bool threadCreated;
  static std::deque<Request> requests;
  static Request currentRequest;
  static ArMutex requestsMutex;
  static ArMutex activityMutex; // locked while async thread or sync function is working on loading a map, unlocked when it's done
  static pthread_cond_t haveNewMapToLoadAsync;

  ArMap *inst_map;
  bool inst_created_map;
  stg_world_t* inst_world;
  StageInterface* inst_interface;

public:

  StageMapLoader(stg_world_t *_world, StageInterface *_if = NULL) :
    inst_map(NULL), inst_created_map(false), inst_world(_world), inst_interface(_if)
  { }

  StageMapLoader(ArMap *_map, stg_world_t *_world, StageInterface *_if = NULL)
  :
    inst_map(_map), inst_created_map(false), inst_world(_world), inst_interface(_if)
  {
  }

  ~StageMapLoader() {
    if(inst_map && inst_created_map)
      delete inst_map;
  }
    
public:
  /// Load the given ActivMedia map and create models in the world from its contents.  Clears out the old map.
  bool loadNewMap(bool loadLines = true, bool loadPoints = true);

/*
  bool loadMap(const char* mapfile, bool lines = true, bool points = true, std::string *errorMsg = NULL) {
    if(!map) {
      map = new ArMap;
      createdMap = true;
    }
    char errbuf[128];
    //map.setCompareForChanges(); // XXX TODO @todo
    if(!map->readFile(mapfile, errbuf, 127))
    {
      //char cwd[512];
      //getcwd(cwd, 512);
      //warn("Error reading map file \"%s\": %s (Current directory is \"%s\")", mapfile, errbuf, cwd);
      if(errorMsg) (*errorMsg) = "Could not read file";
      return false;
    }
    return loadMap(lines, points);
  }
*/

  /// Load an ActivMedia map from the given file.
  /// Clears out the old map's models from
  /// stage, and replace it with the data read from the ActivMedia map.
  /// If not null, the new map's bounds and home point are set.
  bool loadNewMap(const char *mapfile, MapLoadedCallback callback = NULL, std::string *errormsg = NULL);
//    double *min_x = NULL, double *min_y = NULL, double *max_x = NULL, double *max_y = NULL, 
//    double *home_x = NULL, double *home_y = NULL, double *home_th = NULL, std::string *errormsg = NULL);

  /// @internal
  static void loadNewMapThread();

  /// Run a new thread which loads the given map file asynchronously (in a new background thread)
  /// optional callback's args are this StageMapLoader object, then map info
  /// (min x, min y, max x, max y, home x, home y, home th)
  /// checks if should reload because new map file or file changed
  void loadNewMapAsync(const char *mapfile, MapLoadedCallback callback = NULL);

  /// synchronous
  void loadNewMapSync(const char *mapfile);

  // dissasociate any request that has this robot interface object from this
  // robot interface object. this is neccesary if interface is destroyed
  static void interfaceDestroyed(StageInterface *iface);

  /// return if should reload because new map file or file changed.
  bool shouldReloadMap(const char *mapfile);

private:
  stg_model_t* loadReflector(ArMapObject* cairnObj, stg_model_t* map_model, stg_laser_return_t laser_return_val = 2) ;
  stg_model_t* loadBoxObstacle(ArMapObject* cairnObj, stg_model_t* map_model) ;


};

/** Provides access to Stage commands and data to the EmulatePioneer class.
 *  Note, must call staticInit() at some point before using this (due to an
 *  ommision from GCC 3.4.1 standard library!)
 */
class StageInterface : public virtual RobotInterface
{
  public:
    static void staticInit();
    StageInterface(stg_world_t* _world, std::string _robotModel, std::string _robotName, bool deleteModels = false);
    StageInterface(stg_world_t* _world, stg_model_t* _model, std::string _robotModel, std::string _robotName, bool deleteModels = false);
    virtual void setup(RobotParams* params);
    virtual void disconnect();
    void lockStageWorld() {
#ifdef STAGE_MUTEX_LOG
      printf("StageInterface::lockStage()... ");
#endif
      stg_world_lock(stageWorld);
      //stageWorldMutex.lock();
    }
    void unlockStageWorld() {
#ifdef STAGE_MUTEX_LOG
      printf("StageInterface::unlockStage()... ");
#endif
      stg_world_unlock(stageWorld);
      //stageWorldMutex.unlock();

    }

    virtual ~StageInterface();

    virtual void logState();

  protected:

    RobotParams* hardParams;
    stg_world_t* stageWorld;
    std::string robotModel;
    std::string robotName;

    stg_model_t* positionModel;
    stg_model_t* sonarModel;
    stg_model_t* laserModel;
    stg_model_t* messagesModel;

    stg_position_cmd_t positionCmd;

    StageMapLoader mapLoader;

    /** @groupname Utilities for accessing Stage */
    //@{
    
  private:
    //static ArMutex stageWorldMutex; ///< Mutex for calls to Stage affecting the world (models can be individually locked)

  protected:
    /// easier accossor than stage's property interface. does NOT lock stage mutex.
    stg_position_data_t* stagePositionData();
    /// easier accossor than stage's property interface. does NOT lock stage mutex.
    stg_velocity_t* stageVelocityData();
    //@}


    /* Implement the RobotInterface for stage: */

  private:
    bool subscribedToSonar, subscribedToLaser, openedSonar, openedLaser;
    bool motorsEnabled;
    double originalLaserPoseAngle; ///< Angle of laser device pose on robot.
    double laserStart; ///< Start angle requested by client
    double laserEnd;   ///< End angle requested by client

    bool deleteModels;

  public:
    virtual void openSonar();
    virtual void closeSonar();
    virtual bool sonarOpen() { return subscribedToSonar && openedSonar; }
    virtual void openLaser();
    virtual void closeLaser();
    virtual bool laserOpen() { return subscribedToLaser && openedLaser; }
    virtual void enableMotors(bool e) ;
    virtual void transVel(int v);
    virtual void latVel(int v);
    virtual void rotVel(int v);
    virtual void move(int m);
    virtual void heading(int h);
    virtual void deltaHeading(int h);
    virtual void stop();
    virtual void stop(int transdecel, int rotdecel);
    virtual void setAccel(int a);
    virtual void setDecel(int d);
    virtual void setRotAccel(int a);
    virtual void setRotDecel(int d);
    virtual void setLatAccel(int a);
    virtual void setLatDecel(int d);
    virtual void setMaxVel(int v) ;
    virtual void setMaxRotVel(int v);
    virtual void setDefaultVel(int v);
    virtual void setDefaultRotVel(int v);
    virtual void setOdom(int x, int y, int theta);
    virtual void setSimulatorPose(long int x, long int y, long int z, int theta);
    virtual void resetSimulatorPose();
    virtual void stall(bool stalled);

    // data:
    virtual bool havePositionData();
    virtual bool haveGripper();
    virtual bool haveFrontSonar();
    virtual bool haveRearSonar();
    virtual bool haveSonar();
    virtual bool haveLaser();
    virtual int xpos();
    virtual int ypos();
    virtual int theta();
    virtual int xspeed();
    virtual int yspeed();
    virtual int rotspeed();
    virtual void getMotionState(int &x, int &y, int &theta, int &transVel, int &rotVel, bool &stallflag, bool &enabled);
    virtual void getPosition(int &x, int &y, int &theta);
    virtual void getVelocity(int &x, int &y, int &theta);
    virtual long getSimulatorPoseX();
    virtual long getSimulatorPoseY();
    virtual int getSimulatorPoseTheta();
    virtual void getSimulatorPose(long &x, long &y, long &z, int &theta);
    virtual bool stalled();
    virtual size_t numSonarReadings();
    virtual int getSonarReading(int i);
    virtual size_t forEachSonarReading(SonarReadingFunc &func, const size_t &start = 0);
    virtual char gripperState();
    virtual size_t numLaserReadings();
    virtual int getLaserReading(int i);
    virtual size_t forEachLaserReading(LaserReadingFunc &func, const size_t &start = 0);
    virtual int getLaserReflectance(int i);
    virtual double getLaserResolution();  ///< Degrees between readings
    virtual void setLaserResolution(double deg);  ///< Degrees between readings
    virtual double getLaserFOV(); ///< Total laser FOV
    virtual void setLaserFOV(double deg); ///< Total laser FOV
    virtual double getLaserStartAngle();  ///< Angle of first reading relative to robot
    virtual double getLaserEndAngle();  ///< Angle of first reading relative to robot
    virtual void setLaserAngles(double first, double last); ///< Angle of first and last readings relative to robot
    virtual void error_s(const char* message); ///< critical error! locks stage mutex
    virtual void warn_s(const char* message);  ///< locks stage mutex
    virtual void inform_s(const char* message); ///< locks stage mutex
    virtual void log_s(const char* message); ///< locks stage mutex
    virtual void log_error_s(const char *message);
    virtual void shutdown(int errorcode = 0);
    virtual void loadNewMap(const char* mapfile);
    virtual void loadNewMapAsync(const char *mapfile);
    //virtual void loadNewMap_NoLock(char* mapfile);
    virtual int getLastInterval();
    virtual int getSimInterval();
    virtual int getRealInterval();

    virtual std::vector< RobotInterface::DeviceInfo > getDeviceInfo();

};

 

#endif
