/*  
 *  Copyright (C) 2005 ActivMedia Robotics
 *  Copyright (C) 2006, 2007, 2008, 2009 MobileRobots Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#define _BSD_SOURCE 1	// to get getcwd() from Linux's <unistd.h> 

#define STG_WORLD_DEBUG_LOCK

#include <stage.h>
#include <assert.h>
#include <gtk/gtk.h>
#include <glib.h>
#include <map>
#include <set>
#include <locale.h>
#include <string.h>
#include <errno.h>
#include <dirent.h>
#include <limits.h>
#include <unistd.h>
#include <pthread.h>
#include <signal.h>


#include "ariaUtil.h"
#include "ArRobotParams.h"
#include "ArMap.h"

#include "StageInterface.hh"
#include "EmulatePioneer.hh"

#include "RobotFactory.hh"
#include "StageRobotFactory.hh"

#include "MobileSim.hh"
#include "Config.hh"
#include "util.h"

#include "ArSocket.h"

// if defined, measure time it takes to do world updates in main loop
//#define DEBUG_LOOP_TIME 1

// if defined, stop all emulator threads and delete the objects on program
// exit. usually not really neccesary since the program is exiting anyway, and
// is not fully debugged (it can crash)
//#define DELETE_EMULATOR_THREADS 1

// if defined, delete all robot interface objects on program exit.
// usually not really neccesary since the program is exiting anyway, 
// and is not fully debugged (it can crash)
//#define DELETE_ROBOT_INTERFACES 1

// if defined, destroy stage world and call uninit at program
// exit. unually not really neccesary since the program is exiting
// anyway, and causes crashes on windows.
//#define STAGE_UNINIT 1



/* Arbitrary limit for number of robots allowed in the initial
 * configuration dialog.  Theoretically, we're only limited by
 * the number of TCP ports possible (1024 through 65545), but
 * the OS probably also has a limit on number of threads. 
 * MobileSim will probably become more and more unusable, however, 
 * the closer you approach this limit.
 */
#define NUM_ROBOTS_LIMIT 200

/* If undefined, temp files won't be deleted. Useful 
   for debugging them. 
   */
#define CLEANUP_TEMP_FILES 1


/* Remember the world file we created so we can delete it */
char TempWorldFile[MAX_PATH_LEN];


/* Global variables for some options */
bool NonInteractive = false;
bool RestartedAfterCrash = false;
bool Daemon = false;

/* Some predefined ActivMedia robot models for use in the GUI */
const char* CommonRobotModels[] = {"p3dx", "p3at", "amigo-sh", "amigo", "powerbot", "peoplebot-sh", "patrolbot-sh", "p2de", "p2at", "p3atiw-sh", "seekur", 0};



/* Type used to store what simulated robot instances to create. Map from unique robot
 * name to model or .p file.
 */
typedef std::map<std::string, std::string> RobotModels;

/* Type to store what factories to create.*/
typedef  std::list<std::string> RobotFactoryRequests;


/* Create a temporary world file and load it. Return NULL on error */
stg_world_t* create_stage_world(const char* mapfile, std::map<std::string, std::string>& robotInstanceRequests, std::list<std::string>& robotFactoryRequests, const char* libdir, 
    mobilesim_start_place_t start, double start_override_x, double start_override_y, double start_override_th, double world_res = 0.0);

/* Return a name for a robot defined in a pfile that is usable in stage etc.
 * This will not be the same as the robot subtype, it's based on the pfile name.
 */
const char *pfile_model_name(const char *pfile)
{
    /* Skip past any directories */
    const char *modelname = strrchr(pfile, '/');
    if(modelname) 
    {
      if(modelname[0] == '/') ++modelname;
    }
    else
    {
      modelname = pfile;
    }
    return modelname;
}


/* Figure out what directory to use for our external resources */
const char* find_libdir();

/* Delete temporary files: */
void cleanup_temp_files();

/* Utility to get data out of a GHashTable */
void add_model_to_string_map_if_position(stg_model_t* model, char* model_name, void* strmap_p);

/* Write a model definition to the given file based on parameters loaded from
 * the given ARIA parameter file.
 * Returns the name the model type was defined as, which will be a substring in
 * pfile, or NULL on error.
 */
const char* worldfile_define_model_from_pfile(FILE* fp, const char* pfile);





#if GTK_CHECK_VERSION(2, 4, 0)
# define GTKFILEDIALOG GtkFileChooserDialog
# define GTKFILECHOOSER GtkFileChooser
# define GTKCOMBOWIDGET GtkComboBox
# define USE_GTKFILECHOOSER 1
# define USE_GTKCOMBOBOX 1
# define USE_GTKEXPANDER 1
#else
# define GTKFILEDIALOG GtkFileSelection
# define GTKFILECHOOSER GtkFileSelection
# define GTKCOMBOWIDGET GtkCombo
#endif

/* Show the map/options dialog displayed at startup if no map is
 * given on the command line. Exit the program if the dialog is closed.
 * If robotInstanceRequests is NULL, do not show any robot model or num. robots selection
 * controls. If not NULL, then robot model selection and number controls are
 * shown, and entries are added to the map accordingly.
 */
void map_options_dialog(char** map, RobotModels* robotInstanceRequests, RobotFactoryRequests *robotFactoryRequests, int *startport);

void map_options_dialog(char** map, int *startport) {
  map_options_dialog(map, NULL, NULL, startport);
}


/* Print help info */

void usage()
{
  puts(
/*                                                                   column 80--v */
"MobileSim " MOBILESIM_VERSION "\n"\
"Usage: MobileSim [-m <map>] [-r <robot model> ...] [...options...]\n\n"\
"  --map <map>      : Load map file (e.g. created with Mapper3)\n"\
"  -m <map>         : Same as -map <map>.\n"\
"  --nomap          : Create an \"empty\" map to start with.\n"\
"                     Same as -map \"\". Ignored if -m is also given.\n"\
"  --robot <model>[:<name>] :\n"\
"                     Create a simulated robot of the given model.\n"\
"                     If an ARIA robot parameter file is given for <model>,\n"\
"                     then MobileSim attempts to load that file from the\n"\
"                     current working directory, and create an equivalent\n"\
"                     model definition. May be repeated with different names\n"\
"                     and models to create multiple simulated robots. Name \n"\
"                     portion is optional.\n"\
"                     Example: -robot p3dx -robot amigo:bot1 -robot custom.p:bot2\n"\
"                     See PioneerRobotModels.world.inc for model definitions.\n"\
"  -r <model>[:name]: Same as --robot <model>[:<name>]\n"\
"  --robot-factory <model> :\n"\
"                     Instead of creating one robot of the given model, accept\n"\
"                     any number of client programs on its port, creating a new\n"\
"                     instance of the model for each client, and destroying it\n"\
"                     when the client disconnects.\n"\
"  -R <model>       : Same as --robot-factory <model>\n"\
"  -p <port>        : Emulate Pioneer connections starting with TCP port <port>.\n"\
"                     (Default: 8101)\n"\
"  -W <worldfile>   : Use Stage worldfile <worldfile> instead of map and all\n"\
"                     other standard world configuration files.\n"\
"  --stageworld     : Same as -W.\n"\
"  --fullscreen-gui : Display window in fullscreen mode.\n"\
"  --maximize-gui   : Display window maximized.\n"\
"  --minimize-gui   : Display window minimized (iconified)\n"\
"  --noninteractive : Don't display any interactive dialog boxes that might\n"\
"                     block program execution, enable automatic crash-restart,\n"\
"                     and enable log rotation after default maximum log file\n"\
"                     size of 10MB\n"\
"  --daemonize      : Run as a daemon (background process) after initialization\n"\
"                     (still creates GUI). Forces noninteractive mode.  \n"\
"                     Not available on Windows.\n"\
"  --lite-graphics  : Disable some graphics for slightly better performance\n"\
"  --no-graphics    : Disable all graphics drawing for slightly better\n"\
"                     performance\n"\
"  --no-gui         : Disable GUI entirely. Automatically enables noninteractive\n"\
"                     mode as well.\n"\
"  --html           : Print log messages and other output in HTML rather than\n"\
"                     plain text\n"\
"  --cwd <dir>      : Change directory to <dir> at startup.  Client programs\n"\
"                     can then load maps relative to this directory.\n"\
"  --log-file <file>: Print log messages to <file> instead of standard error.\n"\
"  -l <file>        : Same as --log-file <file>.\n"\
"  --log-file-max-size <size> :\n"\
"                     If the amount of data (bytes) written to the log file\n"\
"                     exceeds <size>, then rotate the files (up to 5) and open\n"\
"                     a new file. This keeps the total size of the log files\n"\
"                     under <size>*5 bytes.  If --noninteractive is given,\n"\
"                     default is 5 MB (5*1024^2 bytes).\n"\
"                     If --noninteractive is not given, or <size> is 0, no\n"\
"                     limit is used and logs won't automatically rotate.\n"\
"  --update-interval <ms> : \n"\
"                     Time between each simulation update step.  Default is 100\n"\
"                     ms. Less may improve simulation accuracy but impact client\n"
"                     responsiveness and data update; more may reduce CPU load\n"\
"                     but impact accuracy (especially movement resolution).\n"\
"  --update-sim-time <ms> : \n"\
"                     How much simulated time each simulation update takes.\n"\
"                     Default is equal to --update-interval.\n"\
"  --start <x>,<y>,<th> OR --start outside OR --start random :\n"\
"                     Use <x>, <y>, <th> (mm and degrees) as robot starting\n"\
"                     point (even if the map has Home objects). Or, use the\n"\
"                     keyword \"outside\" to cause robots to start 2m outside\n"\
"                     the map bounds -- it later must be moved within the map\n"\
"                     bounds to be used -- or, use the keyword \"random\" to\n"\
"                     randomly choose a starting place within the map bounds.\n"\
"  --resolution <r> : Use resolution <r> (milimeters) for collisions and\n"\
"                     sensors. Default is 20mm (2cm)\n"\
"  --ignore-command <num> :\n"\
"                     Ignore the command whose number is given. Refer to robot\n"\
"                     manual and MobileSim documentation for command numbers.\n"\
"                     May be repeated. Warning, MobileSim and/or your program\n"\
"                     may not function correctly if some critical commands are\n"\
"                     ignored.\n"\
"  --less-verbose   : Be a bit less verbose about logging, especially things \n"\
"                     that might be logged frequently (e.g. ignored unsupported\n"\
"                     commands, some internal/debugging information, etc.)\n"\
"  --log-timing-stats :\n"\
"                     Log timing stats every 30 seconds.\n"\
"  --bind-to-address <address> :\n"\
"                     Only listen on network interface with IP address \n"\
"                     <address> for new client connections (default is to\n"\
"                     listen on all addresses).\n"\
"  --no-crash-handler :\n"\
"                     Disable GDB crash handler (just abort program on fatal\n"\
"                     signals).\n"\
"\n"\
"MobileSim is based on the Stage simulator library: see <http://playerstage.sf.net>\n\n"\
"Stage is (C) Copyright 2007  Richard Vaughan, Brian Gerkey, Andrew Howard, and others.\n"\
"MobileSim is (C) Copyright 2005, ActivMedia Robotics LLC and (C) Copyright 2006, 2007, 2008, 2009 MobileRobots Inc.\n"\
"This software is released under the GNU General Public License.\n"\
"This software comes with NO WARRANTY.  This is free software, and you are\n"\
"welcome to redistribute it under certain conditions; see the LICENSE.txt file\n"\
"for details.\n\n");
}


/* Get home directory from environment */
const char* home_dir()
{
#ifdef WIN32
  return getenv("HOMEPATH");
#else
  return getenv("HOME");
#endif
}

/* Get temp. dir. from environment or use default path */
const char* temp_dir()
{
  const char* tempdir = getenv("TEMP");
  if(!tempdir)
    tempdir = getenv("TMP");
  if(!tempdir)
#ifdef WIN32
    tempdir = "\\TEMP";
#else
  tempdir = "/tmp";
#endif
  return tempdir;
}

/* Return path to home directory if Linux, or My Documents directory if Windows. */
#ifdef WIN32

char MyDocs[MAX_PATH_LEN];
bool HaveMyDocs = false;
const char* user_docs_dir()
{
  if(!HaveMyDocs)
  {
    snprintf(MyDocs, MAX_PATH_LEN-1, "%s\\My Documents", home_dir());
    HaveMyDocs = true;
  }
  return MyDocs;
}

#else

const char* user_docs_dir() 
{
  return home_dir();
}

#endif


/* Remember some info about the map we loaded so that other objects can use it. */
static mobilesim_start_place_t mobilesim_startplace = mobilesim_start_home;
static double map_min_x = 0;
static double map_min_y = 0;
static double map_max_x = 0;
static double map_max_y = 0;
static double map_home_x = 0;
static double map_home_y = 0;
static double map_home_th = 0;
pthread_mutex_t global_map_data_mutex; // = PTHREAD_MUTEX_INITIALIZER;

void stage_load_map_done_cb(StageMapLoader::MapLoadedInfo info)
//StageMapLoader::Request *loadReq, double min_x, double min_y, double max_x, double max_y, double home_x, double home_y, double home_th)
{
  pthread_mutex_lock(&global_map_data_mutex);
    // this callback is called asynchronously, theoretically could get called
    // twice at once.
  map_min_x = info.min_x;
  map_min_y = info.min_y;
  map_max_x = info.max_x;
  map_max_y = info.max_y;
  map_home_x = info.have_home ? info.home_x : 0;
  map_home_y = info.have_home ? info.home_y : 0;
  map_home_th = info.have_home ? info.home_th : 0;
  pthread_mutex_unlock(&global_map_data_mutex);
}

int stage_load_file_cb(stg_world_t* world, char* filename, void*)
{
  printf("MobileSim load file callback for filename %s!\n", filename);
  StageMapLoader(world, NULL).loadNewMapAsync(filename, &stage_load_map_done_cb);
  return 0;
}

int mobilesim_argc;
char **mobilesim_argv;
char *mobilesim_log_file;

std::set<RobotInterface*> robotInterfaces;
std::set<EmulatePioneer*> emulators;
std::set<RobotFactory*> factories;

ArMutex emulatorsMutex;

void storeEmulator(EmulatePioneer *em)
{
  emulatorsMutex.lock();
  emulators.insert(em);
  emulatorsMutex.unlock();
}

void removeStoredEmulator(EmulatePioneer *em)
{
  emulatorsMutex.lock();
  emulators.erase(em);
  emulatorsMutex.unlock();
}

ArMutex interfacesMutex;

void storeRobotInterface(RobotInterface *em)
{
  interfacesMutex.lock();
  robotInterfaces.insert(em);
  interfacesMutex.unlock();
}

void removeStoredRobotInterface(RobotInterface *em)
{
  interfacesMutex.lock();
  robotInterfaces.erase(em);
  interfacesMutex.unlock();
}

void log_file_full_cb(FILE* fp, size_t sz, size_t max)
{
    stg_print_warning("MobileSim: This log file is full (maximum size %d bytes). Rotating logs and starting a new one...", max);
    mobilesim_rotate_log_files(NULL);
}

/* Main: */
int main(int argc, char** argv) 
{ 

//  global_map_data_mutex = PTHREAD_MUTEX_INITIALIZER;
  pthread_mutex_init(&global_map_data_mutex, NULL);
  StageInterface::staticInit();

  // save command-line arguments 
  mobilesim_argc = argc;
  mobilesim_argv = argv;

  // Name of map file
  char* map = 0;

  // Map of robot name -> model string.
  RobotModels robotInstanceRequests;

  // List or set of model type strings.
  RobotFactoryRequests robotFactoryRequests;

  // Stored config
  MobileSimConfig config;


  // Command line arguments
  char* userstageworld = 0;
  int port = 8101;
  enum {NORMAL_WINDOW, MAXIMIZE_WINDOW, MINIMIZE_WINDOW, FULLSCREEN_WINDOW} windowmode = NORMAL_WINDOW;
  enum {NORMAL_GRAPHICS, LITE_GRAPHICS, NO_GRAPHICS, NO_GUI} graphicsmode = NORMAL_GRAPHICS;
  NonInteractive = false;
  Daemon = false;
  bool enable_crash_handler = true;
  bool html = false;
  char* change_to_directory = 0;
  char before_change_to_directory[MAX_PATH_LEN];
  before_change_to_directory[0] = 0;
  stg_msec_t interval_real = (stg_msec_t) STG_DEFAULT_INTERVAL_REAL;
  stg_msec_t interval_sim = 0;
  double world_res = 0;
  long int start_pos_override_pose_x;
  long int start_pos_override_pose_y;
  int start_pos_override_pose_th;
  std::set<int> ignore_commands;
  bool verbose = true;
  mobilesim_log_file = NULL;
  const char *listen_address = NULL;
  size_t log_file_max_size = 0;

  for(int i = 1; i < argc; ++i) {
    if(command_argument_match(argv[i], "h", "help")) {
      usage();
      exit(0);
    }
    else if(command_argument_match(argv[i], "", "version")) {
      puts("MobileSim " MOBILESIM_VERSION "\nUse --help for more information.\n");
      exit(0);
    }
    else if(command_argument_match(argv[i], "p", "port")) {
      if(++i < argc) {
        port = atoi(argv[i]);
      } else {
        usage();
        exit(ERR_USAGE);
      }
    }
    else if(command_argument_match(argv[i], "m", "map")) {
      if(++i < argc) {
        map = argv[i];
      } else {
        usage();
        exit(ERR_USAGE);
      }
    }
    else if(command_argument_match(argv[i], "", "nomap")) {
      // if -m was already given, don't change it. otherwise set to empty string
      if(map == NULL)
        map = (char*)"";
    }
    else if(command_argument_match(argv[i], "r", "robot")) {
      if(++i < argc) {
        std::string arg(argv[i]);
        size_t sep = arg.find(":");
        std::string model;
        std::string name;
        if(sep == arg.npos) {
          // No name given, invent one
          model = arg.substr(0, sep);
          name = model;
          char buf[4];
          for(int i = 2; robotInstanceRequests.find(name) != robotInstanceRequests.end(); i++) {
            snprintf(buf, 4, "%03d", i);
            name = model + "_" + buf;
          }
        } else {
          model = arg.substr(0, sep);
          name = arg.substr(sep+1);
        }
        stg_print_msg("MobileSim: Robot named \"%s\" will be a \"%s\" model.", name.c_str(), model.c_str());
        robotInstanceRequests[name] = model;
      } else {
        usage();
        exit(ERR_USAGE);
      }
    } 
    else if(command_argument_match(argv[i], "R", "robot-factory")) {
      if(++i < argc) {
        stg_print_msg("MobileSim: Will create robot factory for model \"%s\".", argv[i]);
        robotFactoryRequests.push_back(argv[i]);
      } else {
        usage();
        exit(ERR_USAGE);
      }
    }
    else if(command_argument_match(argv[i], "W", "stageworld")) {
      if(++i < argc) {
        userstageworld = argv[i];
      } else {
        usage();
        exit(ERR_USAGE);
      }
    }
    else if(command_argument_match(argv[i], "", "maximize") || command_argument_match(argv[i], "", "maximize-gui"))
    {
      windowmode = MAXIMIZE_WINDOW;
    }
    else if(command_argument_match(argv[i], "", "minimize") || command_argument_match(argv[i], "", "minimize-gui"))
    {
      windowmode = MINIMIZE_WINDOW;
    }
    else if(command_argument_match(argv[i], "", "fullscreen") || command_argument_match(argv[i], "", "fullscreen-gui"))
    {
      windowmode = FULLSCREEN_WINDOW;
    }
    else if(command_argument_match(argv[i], "", "noninteractive") || command_argument_match(argv[i], "", "non-interactive"))
    {
      NonInteractive = true;
    }
#ifndef WIN32
    else if(command_argument_match(argv[i], "", "daemonize"))
    {
      Daemon = true;
    }
#endif
    else if(command_argument_match(argv[i], "", "no-graphics") || command_argument_match(argv[i], "", "nographics"))
    {
      graphicsmode = NO_GRAPHICS;
    }
    else if(command_argument_match(argv[i], "", "no-gui") || command_argument_match(argv[i], "", "nogui"))
    {
      graphicsmode = NO_GUI;
      NonInteractive = true;
    }
    else if(command_argument_match(argv[i], "", "lite-graphics") || command_argument_match(argv[i], "", "litegraphics"))
    {
      graphicsmode = LITE_GRAPHICS;
    }
    else if(command_argument_match(argv[i], "", "html"))
    {
      html = true;
      stg_set_print_format(STG_PRINT_HTML);
    }
    else if(!strcasecmp(argv[i], "--cwd") || !strcasecmp(argv[i], "-cwd") || !strcasecmp(argv[i], "--cd") || !strcasecmp(argv[i], "-cd") ) 
    {
      if(++i < argc)
      {
        change_to_directory = argv[i];
      }
      else
      {
        usage();
        exit(ERR_USAGE);
      }
    }
    else if(command_argument_match(argv[i], "l", "log-file"))
    {
      if(++i < argc)
      {

        mobilesim_log_file = argv[i];
      }
      else
      {
        usage();
        exit(ERR_USAGE);
      }
    }
    else if(command_argument_match(argv[i], "", "log-file-max-size"))
    {
        if(++i < argc)
            log_file_max_size = atoi(argv[i]);
        else
        {
            usage();
            exit(ERR_USAGE);
        }
    }
    else if(command_argument_match(argv[i], "", "update-interval"))
    {
      if(++i < argc)
      {
        interval_real = (stg_msec_t) atoi(argv[i]);
      }
      else
      {
        usage();
        exit(ERR_USAGE);
      }
    }
    else if(command_argument_match(argv[i], "", "update-sim-time"))
    {
      if(++i < argc)
      {
        interval_sim = (stg_msec_t) atoi(argv[i]);
      }
      else
      {
        usage();
        exit(ERR_USAGE);
      }
    }
    else if(command_argument_match(argv[i], "", "resolution"))
    {
      if(++i < argc)
      {
        world_res = atof(argv[i]);
      }
      else
      {
        usage();
        exit(ERR_USAGE);
      }
    }
    else if(command_argument_match(argv[i], "", "start"))
    {
      if(++i < argc)
      {
        if(strcmp(argv[i], "outside") == 0)
        {
          mobilesim_startplace = mobilesim_start_outside;
        }
        else if(strcmp(argv[i], "random") == 0)
        {
          mobilesim_startplace = mobilesim_start_random;
        }
        else
        {
          if(sscanf(argv[i], "%ld,%ld,%d", &start_pos_override_pose_x, &start_pos_override_pose_y, &start_pos_override_pose_th) != 3)
          {
            fputs("Error: --start must be used either with the form <x>,<y>,<th> (e.g. 200,4650,90) (mm and deg), or the keyword \"random\" or \"outside\".", stderr);
            exit(ERR_USAGE);
          }
          mobilesim_startplace = mobilesim_start_fixedpos;
        }
      }
      else
      {
        usage();
        exit(ERR_USAGE);
      }
    }
    else if(command_argument_match(argv[i], "", "ignore-command"))
    {
      int c = atoi(argv[++i]);
      printf("Will ignore command #%d.\n", c);
      ignore_commands.insert(c);
    }
    else if(command_argument_match(argv[i], "", "less-verbose"))
    {
      verbose = false;
    }
    else if(command_argument_match(argv[i], "", "log-timing-stats"))
    {
      stg_log_stats_freq = 30000; // msec (30 sec)
    }
    else if(command_argument_match(argv[i], "", "bind-to-address"))
    {
      listen_address = argv[++i];
    }
    else if(command_argument_match(argv[i], "", "no-crash-handler"))
    {
      enable_crash_handler = false;
    }
    else if(command_argument_match(argv[i], "", "restart-after-crash"))
    {
      RestartedAfterCrash = true;
    }
    else if(i == (argc-1))
    {
      // last argument on the end of the command line. if it's not a flag,
      // assume it's a map file.
      if(argv[i][0] == '-')
      {
        usage();
        exit(ERR_USAGE);
      }
      map= argv[i];
    }
    else {
      stg_print_warning("MobileSim: Ignoring unrecognized command-line argument \"%s\".", argv[i]);
    }
  }

  if(mobilesim_log_file)
  {
    // force noncolor text if not html
    if(!html)
      stg_set_print_format(STG_PRINT_PLAIN_TEXT);

    stg_print_msg("MobileSim: Rotating log files (%s) if present...", mobilesim_log_file);
    mobilesim_rotate_log_files(NULL);
    FILE* fp = ArUtil::fopen(mobilesim_log_file, "w");
    if(!fp)
    {
      stg_print_error("MobileSim: Could not open log file \"%s\" for writing.", mobilesim_log_file);
      exit(ERR_USAGE);
    }
    stg_set_log_file(fp);
    if(log_file_max_size == 0 && NonInteractive)
      log_file_max_size = 5 * 1024 * 1024;  // default of 5 MB
    if(log_file_max_size > 0)
        stg_set_log_file_max_size(log_file_max_size, &log_file_full_cb);
  }


  /* Directory where supporting files (robot model defs, icons)
     can be found.
     */
  const char* libdir = find_libdir();


  /* Initialize Stage and GTK */
  stg_about_info_appname = (char*)"MobileSim";
  stg_about_info_description = (char*)"Simulator for MobileRobots/ActivMedia robots, based on the Stage robot simulator library (with modifications by MobileRobots Inc).";
  stg_about_info_url = (char*)"http://robots.mobilerobots.com";
  stg_about_info_copyright = (char*)"Copyright Richard Vaughan, Andrew Howard, Brian Gerkey,\n"\
                              "ActivMedia Robotics, MobileRobots Inc, and other contributors 2000-2009.\n\n"\
                              "Released under the GNU General Public License (see COPYING for details).";
  stg_about_info_appversion = (char*)MOBILESIM_VERSION;


#ifndef WIN32
  if(Daemon)
  {
    NonInteractive = true;
    puts("Forking into background (--daemonize option was given)...");
    pid_t pid = fork();
    if(pid == -1)
    {
      perror("Error forking new background process! ");
      exit(ERR_FORK);
    }
    if(pid != 0)
    {
      printf("Background process has PID %d.\n", pid);
      exit(0);
    }
  }
#endif

  if(graphicsmode == NO_GUI)
  {
    NonInteractive = true;
  }

  if(NonInteractive)
  {
    stg_use_error_dialog = FALSE;
  }

  stg_init(argc, argv);

  if(RestartedAfterCrash)
    stg_print_warning("MobileSim crashed and was automatically restarted.  See log files (-crash) for diagnostic information.");

#ifndef WIN32
  // ignore this signal, instead let write() return an error 
  signal(SIGPIPE, SIG_IGN);

  if(enable_crash_handler)
  {
      // catch crashes, save debugging information, and restart if
      // noninteractive
      signal(SIGSEGV, mobilesim_crash_handler);
      signal(SIGILL, mobilesim_crash_handler);
      signal(SIGFPE, mobilesim_crash_handler);
      signal(SIGABRT, mobilesim_crash_handler);
      signal(SIGBUS, mobilesim_crash_handler);
  }
#endif


#if GTK_CHECK_VERSION(2,2,0)
  /* Set GTK's default icon so Stage uses it */
  char iconfile[MAX_PATH_LEN];
  snprintf(iconfile, MAX_PATH_LEN-1, "%s%cicon.png", libdir, PATHSEPCH);
  GError* err = NULL;
  if(!gtk_window_set_default_icon_from_file(iconfile, &err))
  {
    stg_print_warning("MobileSim: Could not load program icon \"%s\": %s", iconfile, err->message);
    g_error_free(err);
  }
#endif

  /* Open config file */
  //XXX TODO


  /* Check for user's init.map */
  char initmapfile[MAX_PATH_LEN];
  if(map == NULL) // neither -m or -nomap options given
  {
    strncpy(initmapfile, home_dir(), MAX_PATH_LEN-1);
    strncat(initmapfile, "/.MobileSim/init.map", MAX_PATH_LEN-1);
    struct stat s;
    if( stat(initmapfile, &s) == 0 )
    {
      stg_print_msg("MobileSim: Found init. map file \"%s\".", initmapfile);
      map = initmapfile;
    }
  }

  /* Change to startup directory before dialog box, so that it shows the startup
   * dir. 
   */
  if(change_to_directory)
  {
    stg_print_msg("MobileSim: Changing to directory \"%s\"...", change_to_directory);
    if(getcwd(before_change_to_directory, MAX_PATH_LEN) == NULL)
	    	before_change_to_directory[0] = '\0';
    chdir(change_to_directory);
  }

  /* If no map given, display a dialog box. */
  if(map)
  {
    if(strlen(map) == 0)
      stg_print_msg("MobileSim: Will start with no map.");
    else
      stg_print_msg("MobileSim: Will use map \"%s\" for simulation environment.", map);
  }
  else if(userstageworld)
  {
    stg_print_msg("MobileSim: Will use Stage world \"%s\".", userstageworld);
  }
  else
  {
    if(NonInteractive)
    {
      stg_print_error("MobileSim: No map given in noninteractive mode. Use --map to specify a map file, or use --nomap for an empty map.");
      exit(ERR_USAGE);
    }

    /* Show the dialog box. Only show the robot models selection if instances or
     * factories are already requested on the command line.
     */
    if(robotInstanceRequests.empty() && robotFactoryRequests.empty())
    {
      map_options_dialog(&map, &robotInstanceRequests, &robotFactoryRequests, &port );
    }
    else
    {
      map_options_dialog(&map, &port);
    }

    if(map == NULL || strlen(map) == 0)
      stg_print_msg("MobileSim: Will start with no map.");
    else
      stg_print_msg("MobileSim: Will use map \"%s\" for simulation environment.", map);
  }


  /* Setup our special pioneer-specific properties for the world file. These
   * will be read from the model definitions by
   * emulatePioneer/stageInterface.cc. */
  stg_model_user_property("position", "pioneer_robot_subtype", STG_STRING);
  stg_model_user_property("position", "pioneer_diffconv", STG_FLOAT);
  stg_model_user_property("position", "pioneer_angleconv", STG_FLOAT);
  stg_model_user_property("position", "pioneer_distconv", STG_FLOAT);
  stg_model_user_property("position", "pioneer_vel2div", STG_FLOAT);
  stg_model_user_property("position", "pioneer_velconv", STG_FLOAT);
  stg_model_user_property("position", "pioneer_rangeconv", STG_FLOAT);
  stg_model_user_property("ranger",   "pioneer_hasfrontarray", STG_INT);
  stg_model_user_property("ranger",   "pioneer_hasreararray", STG_INT);
  stg_model_user_property("ranger",   "pioneer_max_readings_per_packet", STG_INT);
  stg_model_user_property("position", "pioneer_sip_cycle", STG_INT);
  stg_model_user_property("position", "pioneer_watchdog", STG_INT);

  /* create Stage world with our map and robot(s) or load the user's world file */
  stg_world_t* world;
  if(userstageworld) {
    /* The user specified his own Stage world file. */

    /* Load stage worldfile: */
    world = stg_world_create_from_file(userstageworld);
    map = userstageworld; // for display in titlebar etc.

    /* Now find position models that are defined in the world and add to our list: */
    stg_world_foreach_model_by_name(world, add_model_to_string_map_if_position, &robotInstanceRequests);

  }
  else
  {
    /* The user did not give a Stage world file, so we create one from the map */

    /* Create one p3dx model by default, if no instances or factories given: */
    if(robotInstanceRequests.size() == 0 && robotFactoryRequests.size() == 0)
      robotInstanceRequests["p3dx"] = "p3dx";

    /* You need to specify a map or nomap if you specify any arguments: */
    if(map == NULL)
    {
      stg_print_error("MobileSim: No map or stage world file given. Use --map to specify a map file, or --nomap.");
      exit(ERR_USAGE);
    }

    /* Convert Aria map to Stage world, and load it. TODO don't create robot
     * modelste there.  */
    world = create_stage_world(map, robotInstanceRequests, robotFactoryRequests, libdir, mobilesim_startplace, start_pos_override_pose_x, start_pos_override_pose_y, start_pos_override_pose_th, world_res/1000.0);
  }

  if(!world) {
    exit(ERR_STAGELOAD);
  }

  // save list of map files for use in startup GUI.
  config.updateRecentMaps(map);

  // add callback for loading a new map file
  stg_world_add_file_loader(world, &stage_load_file_cb, "*.map", "MobileRobots/ActivMedia Map Files", NULL);

  // Set locking policy for stg_world_update and enable internal
  // world locking in stage
  stg_world_set_model_update_locking_policy(world, STG_WORLD_LOCK);
  stg_world_enable_world_lock(world, TRUE);

  stg_world_set_quiet(world, !verbose);

  /* Start the world now (needs to be running on order to both create robot
   * models and show the gui without deadlocks or race conditions) */
  stg_world_start(world);
  stg_world_set_cursor_busy(world);


  /* Initialize Winsock */
  ArSocket::init();



  /* Create any robot factories specified */
  for(std::list<std::string>::iterator i = robotFactoryRequests.begin(); i != robotFactoryRequests.end(); i++)
  {
    const char *modelname = pfile_model_name((*i).c_str());
    stg_print_msg("MobileSim: Creating new robot factory for \"%s\"...", modelname);
    RobotFactory *stagefac;
    if(mobilesim_startplace == mobilesim_start_fixedpos)
      stagefac = new StageRobotFactory(world, modelname, start_pos_override_pose_x, start_pos_override_pose_y, start_pos_override_pose_th);
    else
      stagefac = new StageRobotFactory(world, modelname, &mobilesim_get_map_home, &mobilesim_get_map_bounds, (mobilesim_startplace==mobilesim_start_outside) );
    stagefac->setCommandsToIgnore(ignore_commands);
    stagefac->setVerbose(verbose);
    ArSocket *facsock = stagefac->open(port, listen_address);
    if(listen_address != NULL)
      stg_print_msg("MobileSim: Robot factory for \"%s\" ready on %s port %d", modelname, listen_address, port);
    else
      stg_print_msg("MobileSim: Robot factory for \"%s\" ready on port %d", modelname, port);
    if(!facsock) {
      stg_print_error("MobileSim: Could not open port %d for robot factory %s. Try another with -p.\n", port, modelname);
      exit(ERR_OPEN_PORT);
    }
    factories.insert(stagefac);
    stagefac->runListenThread();
  }


  /* Create Pioneer emulators for each position model and run in background threads. 
   * TODO create robot models in stage here too instead of in the world file in create_stage_world. */
  for(std::map<std::string, std::string>::const_iterator i = robotInstanceRequests.begin();
      i != robotInstanceRequests.end(); i++)
  {
    const std::string& name = (*i).first;
    const std::string& model = (*i).second;
    //    stg_print_msg("MobileSim: Creating emulated Pioneer connection for robot named \"%s\" (\"%s\") on TCP port %d.", (*i).first.c_str(), (*i).second.c_str(), port);
    //    stg_world_display_message(world, 0, "MobileSim", STG_MSG_INFORMATION, "Creating emulated Pioneer connection for \"%s\" (\"%s\") on TCP port %d.", (*i).second.c_str(), (*i).first.c_str(), port);
    StageInterface* stageint = new StageInterface(world, model, name); 
    EmulatePioneer* emulator = new EmulatePioneer(stageint, model, port++);
    robotInterfaces.insert(stageint);
    emulators.insert(emulator);  
    emulator->setSimulatorIdentification("MobileSim", MOBILESIM_VERSION);
    emulator->setCommandsToIgnore(ignore_commands);
    emulator->setVerbose(verbose);
    if(listen_address)
      emulator->setListenAddress(listen_address);
    if(!emulator->runThread())
    {
      stageint->error("Unrecoverable error creating thread for robot \"%s\".", name.c_str());
      exit(ERR_ROBOT_INIT);
    }
  }


  stg_world_lock(world);

  /* Change Stage's window title. TODO: include map file name? */
  stg_world_set_window_title(world, "MobileSim");

  /* Set window mode as requested */
  switch(windowmode)
  {
    case MAXIMIZE_WINDOW:
      stg_world_window_request_maximize(world);
      break;
    case MINIMIZE_WINDOW:
      stg_world_window_request_minimize(world);
      break;
    case FULLSCREEN_WINDOW:
      stg_world_window_request_fullscreen(world);
      break;
    default:
      /* do nothing. */
      ;
  }

  /* Set graphics mode as requested */
  switch(graphicsmode)
  {
    case NO_GUI:
      stg_world_disable_gui(world);
      break;
    case NO_GRAPHICS:
      stg_world_hide_all_graphics(world);
      break;
    case LITE_GRAPHICS:
      stg_world_set_fill_polygons(world, FALSE);
      break;
    default:
      /* do nothing */
      ;
  }

  /* Set time intervals as requested */
  if(interval_sim == 0)
    interval_sim = interval_real;
  stg_world_set_interval_real(world, interval_real);
  stg_world_set_interval_sim(world, interval_sim);


  /* run until exit. */
  stg_world_set_cursor_normal(world);
  stg_world_unlock(world);

#ifdef DEBUG_LOOP_TIME
  ArTime stageUpdateTime;  // debugging
  ArTime loopTime;  // debugging
#endif

  int stageStat = 0;

  stageStat = stg_world_run(world);

  /*
     while(stageStat == 0)
     {

#ifdef DEBUG_LOOP_TIME
stageUpdateTime.setToNow();
stageStat = stg_world_update(world, TRUE);
stg_print_msg("stage update %d ms.", stageUpdateTime.mSecSince());
#else
stageStat = stg_world_update(world, TRUE);
#endif

#ifdef DEBUG_LOOP_TIME
stg_print_msg("loop %d ms.", loopTime.mSecSince());
loopTime.setToNow();
#endif

}
*/



  /* Cleanup and return */


#ifdef DELETE_EMULATOR_THREADS
  puts("Stage exited. Stopping and deleting emulators...");
  for(std::set<EmulatePioneer*>::const_iterator i = emulators.begin(); i != emulators.end(); i++)
  {
    (*i)->stopThread();
    delete(*i);
  }
#endif


#ifdef DELETE_ROBOT_INTERFACES
  puts("Stage exited. Deleting robot interfaces...");
  for(std::set<RobotInterface*>::const_iterator i = robotInterfaces.begin(); i != robotInterfaces.end(); i++)
  {
    delete(*i);
  }
#endif

  //stg_world_destroy( world );// crashes
  cleanup_temp_files();

#ifdef STAGE_UNINIT
  stg_world_destroy(world);
  stg_uninit();
#endif

  ArSocket::shutdown();

  if(before_change_to_directory[0] != '\0')
	chdir(before_change_to_directory);	// this is needed to profiling info can be written if built with -pg

  return stageStat - 1; // stageStat == 1 should cause normal exit (0)
}




void mobilesim_get_map_bounds(double *min_x, double *min_y, double *max_x, double *max_y)
{
  //printf(">>> Mobilesim_get_map_bounds: min_x = %f, min_y = %f, max_x = %f, max_y = %f.\n", map_min_x, map_min_y, map_max_x, map_max_y);
  if(min_x) *min_x = map_min_x;
  if(min_y) *min_y = map_min_y;
  if(max_x) *max_x = map_max_x;
  if(max_y) *max_y = map_max_y;
}

void mobilesim_get_map_home(double *home_x, double *home_y, double *home_th)
{
  if(mobilesim_startplace == mobilesim_start_random)
  {

    if(home_x) {
      if(map_min_x == 0 && map_max_x == 0) 
        *home_x = map_home_x = STG_RANDOM_RANGE(-25000, 25000);
      else
        *home_x = map_home_x = STG_RANDOM_RANGE(map_min_x, map_max_x); 
    }
    if(home_y) {
      if(map_min_y == 0 && map_max_y == 0)
        *home_y = map_home_y = STG_RANDOM_RANGE(-25000, 25000);
      else
        *home_y = map_home_y = STG_RANDOM_RANGE(map_min_y, map_max_y);
    }
    if(home_th) {
      *home_th = map_home_th = STG_RANDOM_RANGE(0, 359);
    }
  }
  else
  {
    if(home_x) *home_x = map_home_x;
    if(home_y) *home_y = map_home_y;
    if(home_th) *home_th = map_home_th;
  }
}

void mobilesim_set_map_bounds(double min_x, double min_y, double max_x, double max_y)
{
  map_min_x = min_x;
  map_min_y = min_y;
  map_max_x = max_x;
  map_max_y = max_y;
}

void mobilesim_set_map_home(double home_x, double home_y, double home_th)
{
  map_home_x = home_x;
  map_home_y = home_y;
  map_home_th = home_th;
}

/* Create a temporary world file and load it: 
   If map is null or empty, will make an empty world of some size with
   the robot in the middle, else will convert an aria map by that
   name and include it in the world.
   You may supply a pointer to a set of stg_model_t*. If non-null,
   each model created while loading the map file is added
   to this list.
   */
stg_world_t* create_stage_world(const char* mapfile, std::map<std::string, std::string>& robotInstanceRequests, std::list<std::string>& robotFactoryRequests, const char* libdir, 
    mobilesim_start_place_t startplace, double start_override_x, double start_override_y, double start_override_th, double world_res)
{

  char worldfile[MAX_PATH_LEN];

  /* Filenames */

  const char* tempdir = temp_dir();


#ifdef WIN32
  // Don't include the PID on Windows. Temp files aren't properly deleted on
  // Windows (file locking issue?) so we just always replace the old one.
  snprintf(worldfile, MAX_PATH_LEN-1, "%s%cMobileSim-stage_world.world", tempdir, PATHSEPCH);
#else
  snprintf(worldfile, MAX_PATH_LEN-1, "%s%cMobileSim-stage_world-%d.world", tempdir, PATHSEPCH, getpid());
#endif

  /* Create stage world file. */
  FILE* world_fp = ArUtil::fopen(worldfile, "w");
  if(!world_fp)
  {
    if(NonInteractive)
    {
      stg_print_error("MobileSim: could not create temporary file \"%s\" (%s). Is the system temporary directory \"%s\" accessible?", worldfile, strerror(errno), tempdir);
      exit(ERR_TEMPFILE);
    }
    else
    {
      char buf[256];
      snprintf(buf, 255, "Could not write Stage world file \"%s\" (%s).\nIs the system temporary directory \"%s\" accessible?", worldfile, strerror(errno), tempdir);
      stg_gui_fatal_error_dialog("MobileSim: Error creating temprorary file.", buf, ERR_TEMPFILE, FALSE); 
      exit(ERR_TEMPFILE);
    }
  }

  // Change locale to good old C to get decimal points rather than commas 
  // in floating point numbers (stage expects the C locale, and AM map files
  // use C locale when written).  stg_init is  supposed to do this but I 
  // guess something else changed it back.
  if(!setlocale(LC_ALL, "C"))
    stg_print_warning("MobileSim: failed to set locale to \"C\", Stage world files may not parse correctly!");

  // Write comment, include model definitions, and GUI spec.
  fprintf(world_fp, "# World file for Stage\n# Automatically generated by MobileSim, the MobileRobots/ActivMedia mobile robot simulator.\n\n");
  fprintf(world_fp, "include \"%s%cPioneerRobotModels.world.inc\"\n", libdir, PATHSEPCH);
  fprintf(world_fp, "window\n(\n"\
      "\tscale 0.1\n"\
      "\tmouse_button_pan 3\n"\
      "\tmouse_button_zoom 2\n"\
      "\tlaser_data 1\n"\
      "\tranger_data 1\n"\
      "\tposition_data 0\n"\
      ")\n\n");
  if(world_res > 0) {
    //fprintf(world_fp, "world\n(\n\tresolution %f\n)\n\n", world_res);
    fprintf(world_fp, "resolution %f\n\n", world_res);
  }

  /* Include user's model definition files if they have them. */
#ifdef WIN32
  char* homedir = getenv("HOMEPATH");
#else
  char* homedir = getenv("HOME");
#endif
  char includedir[MAX_PATH_LEN];
  strncpy(includedir, homedir, MAX_PATH_LEN-1);
  strncat(includedir, "/.MobileSim/include", MAX_PATH_LEN-1);
  DIR* dir = opendir(includedir);
  if(dir)
  {
    struct dirent* e;
    while((e = readdir(dir)))
    {
#ifndef WIN32
      // Check that the entry is a regular file (or a symlink), except on Windows where we don't have d_type.
      if(e->d_type == DT_REG || e->d_type == DT_LNK)
#endif
      {
        // Don't include a name that starts with a '.'
        if(e->d_name[0] != '.')
        {
          stg_print_msg("MobileSim: including \"%s\" into world configuration file...", e->d_name);
          fprintf(world_fp, "include \"%s/%s\"\n", includedir, e->d_name);
        }
      }
    }
    closedir(dir);
  }


  /* Load the ActivMedia map file to get the size, dock/home points, etc. */
  ArMap map;
  if(mapfile && strlen(mapfile) > 0)
  {
    if(!map.readFile(mapfile))
    {
      if(NonInteractive) {
        stg_print_error("MobileSim: Could not load map \"%s\".", mapfile);
        exit(ERR_MAPCONV);
      }
      else {
        char buf[256];
        snprintf(buf, 256, "Error loading map \"%s\".\n.", mapfile);
        stg_gui_fatal_error_dialog("MobileSim: Error loading map", buf, ERR_MAPCONV, FALSE); 
      }
      return NULL;
    }
  }
  else
  {
    /* If mapfile is the empty string, make a 500x500m  empty space. */
    fprintf(world_fp, "# Requested that no map be used:\nsize [500.0 500.0]\n\n");
  }

  /* Remember map bounds for future use */
  map_min_x = map.getLineMinPose().getX();
  map_min_y = map.getLineMinPose().getY();
  map_max_x = map.getLineMaxPose().getX();
  map_max_y = map.getLineMaxPose().getY();

  /* Find place to start robot at */
  double startPoseX = 0.0; //m
  double startPoseY = 0.0; //m
  double startPoseTheta = 0.0; //deg
  switch(startplace)
  {
    case mobilesim_start_fixedpos:
      {
        stg_print_msg("MobileSim: Starting robots at position given by --start option: %f, %f, %f.", start_override_x/1000.0, start_override_y/1000.0, start_override_th);
      map_home_x = start_override_x;
      map_home_y = start_override_y;
      map_home_th = start_override_th;
    }
    break;

    case mobilesim_start_outside:
    {
      map_home_x = map_min_x - 2000.0;
      map_home_y = map_max_y;
      map_home_th = 0;
      stg_print_msg("MobileSim: Starting robots 2m outside edge of map, at: %f, %f, %f.", map_home_x/1000.0, map_home_y/1000.0, map_home_th);
    }
    break;

    case mobilesim_start_random:
    {
      /* Nothing, will be randomly generated below */
      stg_print_msg("MobileSim: Starting robots at random positions in map.");
      map_home_x = map_home_y = map_home_th = 0.0;
    }
    break;


    default: /* home */
    {
      ArMapObject* home = map.findFirstMapObject(NULL, "RobotHome");
      if(!home)
        home = map.findFirstMapObject(NULL, "Dock");
      if(home)
      {
        map_home_x = home->getPose().getX();
        map_home_y = home->getPose().getY();
        map_home_th = home->getPose().getTh();
      }
      stg_print_msg("MobileSim: Starting robots at map's default starting place of: %f, %f, %f", map_home_x/1000.0, map_home_y/1000.0, map_home_th);
    }
  }

  startPoseX = map_home_x / 1000.0;
  startPoseY = map_home_y / 1000.0;
  startPoseTheta = map_home_th;


  /* If a member of robotInstanceRequests or robotFactoryRequests is really a .p file, then create a model
   * definition for that robot type: */
  std::set<std::string> modelNames;
  for(RobotModels::const_iterator i = robotInstanceRequests.begin(); i != robotInstanceRequests.end(); i++)
    modelNames.insert( (*i).second );
  for(std::list<std::string>::const_iterator i = robotFactoryRequests.begin(); i != robotFactoryRequests.end(); i++)
    modelNames.insert( (*i) );
  for(std::set<std::string>::const_iterator i = modelNames.begin(); i != modelNames.end(); i++)
  {
    const char* model = (*i).c_str();
    struct stat filestat;

    if(stat( model, &filestat) == -1)
    {
      // file does not exist or other error accessing it
      continue;
    }
    stg_print_msg("MobileSim: File \"%s\" exists, loading parameters to define the robot model.", model);
    const char* pfile = model;
    model = worldfile_define_model_from_pfile(world_fp, pfile);
    if(!model)
    {
      stg_print_error("MobileSim: Could not read or parse parameter file \"%s\" to define robot model.", pfile);
      exit(ERR_PFILE);
    }
    stg_print_msg("MobileSim: Loaded parameter file \"%s\" to define robot model \"%s\".", pfile, model);
  }
  
  /* Create robot models in the world */
  for(RobotModels::const_iterator i = robotInstanceRequests.begin();
        i != robotInstanceRequests.end(); i++) 
  {
    // if the modelname has /, it will cause a syntax error, so find the last
    // one. if the modelname has no /, this procedure should result in an
    // identical string.
    std::string modelname = (*i).second.c_str();
    std::string::size_type lastslash = modelname.rfind('/');
    if(lastslash != 0) lastslash++; // don't include the / we found
    modelname = modelname.substr(lastslash);

    // same thing with robot name, it's annoying for it to be a file path
    std::string robotname = (*i).first.c_str();
    lastslash = modelname.rfind('/');
    if(lastslash != 0) lastslash++; // don't include the / we found
    robotname = robotname.substr(lastslash);
    // TODO: remove the old entry in the robotInstanceRequests map and add this new one.
    // Can't do it while we're iterating it though...
    
    if(startplace == mobilesim_start_random)
    {
      startPoseX = STG_RANDOM_RANGE(map_min_x, map_max_x) / 1000.0;
      startPoseY = STG_RANDOM_RANGE(map_min_y, map_max_y) / 1000.0;
      startPoseTheta = rand() % 360;
    }

    fprintf(world_fp, "%s\n(\n"\
                      "\tname \"%s\"\n"\
                      "\tpose [%g %g %g]\n",
                      modelname.c_str(), robotname.c_str(), 
                      startPoseX, startPoseY, startPoseTheta
    );
    fprintf(world_fp, "\tmessages ( )\n");
    fprintf(world_fp, ")\n\n");
    startPoseY -= 1.0;
  }

  
  /* Close the worldfile, then have stage load it */
  fclose(world_fp);

  // Store filename in global variables so we can delete it on exit:
  strncpy(TempWorldFile, worldfile, 256);

  stg_print_msg("MobileSim: Loading stage world file \"%s\"...", worldfile);
  stg_world_t* world = stg_world_create_from_file(worldfile); 
  if(!world)
  {
    if(NonInteractive)
    {
      stg_print_error("MobileSim: Error creating and initializing the world environment. %s", stg_last_error_message);
      cleanup_temp_files();
      exit(ERR_STAGEINIT);
    }
    else
    {
      cleanup_temp_files();
      stg_gui_fatal_error_dialog("MobileSim: Error creating and initializing world environment", stg_last_error_message, ERR_STAGEINIT, FALSE); 
    }
    return NULL;
  }




  /* Create models for the map and map objects */
  if(mapfile && (strlen(mapfile) > 0))
  {
    stg_print_msg("MobileSim: Loading map \"%s\"...", mapfile);
    StageMapLoader loader(&map, world, NULL);
    if(!loader.loadNewMap())
    {
      stg_print_error("MobileSim: Error loading map \"%s\"!", mapfile);
      exit(ERR_MAPCONV);
    }
    stg_print_msg("MobileSim: Finished loading map \"%s\".", mapfile);
  }

  return world;
}

void cleanup_temp_files()
{

#ifdef CLEANUP_TEMP_FILES

  if(TempWorldFile == NULL || strlen(TempWorldFile) == 0)
    return;

#ifdef WIN32
  // On Windows we must change '/' to '\\':
  for(char* c = TempWorldFile; c && *c; c++)
    if(*c == '/') *c = '\\';
#endif


  if(unlink(TempWorldFile) != 0)
    stg_print_warning("MobileSim: Failed to delete temporary file \"%s\".", TempWorldFile);
#else
  stg_print_warning("MobileSim: Temporary world file has been left behind: \"%s\"; .", TempWorldFile);
#endif

}



const char* find_libdir()
{
  const char* libdir = getenv("MOBILESIM");
  //unused? char buf[256];
  if(libdir == NULL)
  {

#ifdef WIN32
    char buf[1024];
    if (ArUtil::getStringFromRegistry(ArUtil::REGKEY_LOCAL_MACHINE,
        "SOFTWARE\\MobileRobots\\MobileSim", "Install Directory", buf, 255))
    {
      libdir = strdup(buf);
      stg_print_msg("MobileSim: Expecting supporting resources to be installed in \"%s\" (according to registry key).", libdir);
    }
    else if (ArUtil::getStringFromRegistry(ArUtil::REGKEY_LOCAL_MACHINE,
        "SOFTWARE\\ActivMedia Robotics\\MobileSim", "Install Directory", buf, 255))
    {
      libdir = strdup(buf);
      stg_print_msg("MobileSim: Expecting supporting resources to be installed in \"%s\" (according to ActivMedia Robotics registry key).", libdir);
    }
    else  
    {
      libdir = "\\Program Files\\MobileRobots\\MobileSim";
      //libdir = "\\PROGRA~1\\ACTIVM~\\MobileSim";
      //libdir = "\\src\\MobileSim\\config";
      stg_print_msg("MobileSim: Expecting supporting resources to be installed in the default location: \"%s\".", libdir);
    }

#else 

    libdir = "/usr/local/MobileSim";
    stg_print_msg("MobileSim: Expecting supporting resources to be installed in the default location: \"%s\".", libdir);
#endif

  } 
  else 
  {
    stg_print_msg("MobileSim: Expecting supporting resources to be installed in \"%s\" (according to MOBILESIM environment variable).", libdir);
  }
  return libdir;
}


void add_model_to_string_map_if_position(stg_model_t* model, char* name, void* strmap_p)
{
  std::map<std::string, std::string>* strmap = (std::map<std::string, std::string>*)strmap_p;
  if(strcmp(stg_model_get_base_type_name(model), "position") == 0)
  {
    std::string type;
    // property is not null terminated:
    size_t l;
    char* t = (char*)stg_model_get_property(model, "pioneer_robot_subtype", &l);
    if(t == NULL)
    {
      type = stg_model_get_token(model);
      stg_print_msg("Warning: World position model \"%s\" does not have pioneer_robot_subtype property. Using \"%s\" for robot type, this may be invalid.", name, type.c_str());
    } else {
      type.assign(t, l);
    }
    strmap->insert(std::pair<std::string, std::string>(name, type));
  }
}



const char* worldfile_define_model_from_pfile(FILE* fp, const char* pfile)
{
  ArRobotParams p;
  char err[256];
  if(!p.parseFile(pfile, false /*continueOnErrors*/, false /*noFileNotFoundError*/, err, 255))
  {
    stg_print_error("MobileSim: Error reading or parsing parameter file \"%s\": %s", pfile, err);
    return NULL;
  }

  /* Start definition. */
  const char* modelname = pfile_model_name(pfile);
  /*
  strrchr(pfile, '/');
  if(modelname)
  {
    if(modelname != pfile)
      modelname++;  // avoid '/'
  }
  else
  {
    modelname = (char*)pfile;
  }
  */
  stg_print_msg("MobileSim: Defining model \"%s\" in world file based on parameter file \"%s\"...", modelname, pfile);
  fprintf(fp, "define %s pioneer (\n", modelname);
  fprintf(fp, "\tpioneer_robot_subtype \"%s\"\n", p.getSubClassName());
  
  /* Speed profile: */
  if(p.getAbsoluteMaxVelocity() > 0 || p.getAbsoluteMaxRotVelocity() > 0)
    fprintf(fp, "\tmax_speed [%f 0 %f]\n", p.getAbsoluteMaxVelocity()/1000.0, ArMath::degToRad(p.getAbsoluteMaxRotVelocity()));
  if(p.getTransAccel() > 0 || p.getRotAccel() > 0) 
    fprintf(fp, "\taccel [%f 0 %f]\n", p.getTransAccel()/1000.0, ArMath::degToRad(p.getRotAccel()));
  if(p.getTransDecel() > 0 || p.getRotDecel() >0)
    fprintf(fp, "\tdecel [%f 0 %f]\n", p.getTransDecel()/1000.0, ArMath::degToRad(p.getRotDecel()));

  /* Body shape: */
  const double d = p.getRobotDiagonal()/1000.0;
  const double w = p.getRobotWidth()/1000.0;
  const double l = p.getRobotLength()/1000.0;
  fprintf(fp, "\tsize [%f %f]\n", l, w);
  fprintf(fp, "\tpolygons 1\n\tpolygon[0].points 8\n\tpolygon[0].filled 1\n");
  fprintf(fp, "\tpolygon[0].point[0] [%f %f]\n", l/2.0, -d);
  fprintf(fp, "\tpolygon[0].point[1] [%f %f]\n", l/2.0, d);
  fprintf(fp, "\tpolygon[0].point[2] [%f %f]\n", d, w/2.0);
  fprintf(fp, "\tpolygon[0].point[3] [%f %f]\n", -d, w/2.0);
  fprintf(fp, "\tpolygon[0].point[4] [%f %f]\n", -l/2.0, d);
  fprintf(fp, "\tpolygon[0].point[5] [%f %f]\n", -l/2.0, -d);
  fprintf(fp, "\tpolygon[0].point[6] [%f %f]\n", -d, -w/2.0);
  fprintf(fp, "\tpolygon[0].point[7] [%f %f]\n", d, -w/2.0);


  /* Client conversion factors: */
  fprintf(fp, "\tpioneer_diffconv %f\n", p.getDiffConvFactor());
  fprintf(fp, "\tpioneer_distconv %f\n", p.getDistConvFactor());
  fprintf(fp, "\tpioneer_rangeconv %f\n", p.getRangeConvFactor());
  fprintf(fp, "\tpioneer_angleconv %f\n", p.getAngleConvFactor());

  /* Sonar positions: */
  if(p.getNumSonar() > 0)
  {
    fprintf(fp, "\tpioneerSonar (\n\t\tscount %d\n", p.getNumSonar());
    for(int i = 0; i < p.getNumSonar(); i++)
    {
      fprintf(fp, "\t\tspose[%d] [%f %f %d]\n", i, p.getSonarX(i)/1000.0, p.getSonarY(i)/1000.0, p.getSonarTh(i));
      if(!p.haveSonar(i))
        fprintf(fp, "\t\tsview[%d] [5 5 0]\n", i);  // "disabled"
    }
    fprintf(fp, "\t)\n");
  }

  /* Laser?: */
  if(p.getLaserPossessed())
  {
    fprintf(fp, "\tsicklms200 (\n\t\tpose [%f %f %f]\n", p.getLaserX()/1000.0, p.getLaserY()/1000.0, p.getLaserTh());
    if(p.getLaserFlipped())
      fprintf(fp, "\t\treverse_scan 1\n");
    fprintf(fp, "\t)\n");
  }


  
  /* Done. */
  fprintf(fp, ")\n\n");
  return modelname;
}



void map_options_dialog(char** map, RobotModels *robotInstanceRequests, RobotFactoryRequests *robotFactoryRequests, int *port)
{
  GtkDialog* dialog;
  GTKFILECHOOSER* filechooser;
  GtkEntry* portentry = NULL;
  GtkEntry* numentry = NULL;
  GtkRadioButton *factory_radio = NULL;
  GtkRadioButton *multirobot_radio = NULL;
  GTKCOMBOWIDGET* modelmenu = NULL;
  gboolean include_modelmenu = (robotInstanceRequests != NULL && robotFactoryRequests != NULL);


#ifdef USE_GTKFILECHOOSER

  dialog = GTK_DIALOG(
    gtk_file_chooser_dialog_new("MobileSim: Load Map File...", NULL, 
      GTK_FILE_CHOOSER_ACTION_OPEN, 
      "No Map", GTK_RESPONSE_CANCEL, 
      "Load Map", GTK_RESPONSE_ACCEPT, 
    NULL)
  );
  filechooser = GTK_FILE_CHOOSER(dialog);

  GtkFileFilter* filter_maps = gtk_file_filter_new();
  GtkFileFilter* filter_all = gtk_file_filter_new();
  // TODO: need to ref the filters and destroy in the destructor?
  gtk_file_filter_set_name(filter_maps, "MobileRobots/ActivMedia Map Files");
  gtk_file_filter_add_pattern(filter_maps, "*.map");
  gtk_file_filter_set_name(filter_all, "All Files");
  gtk_file_filter_add_pattern(filter_all, "*");
  gtk_file_chooser_add_filter(filechooser, filter_maps);
  gtk_file_chooser_add_filter(filechooser, filter_all);
  gtk_file_chooser_set_current_folder(filechooser, user_docs_dir());

#else
  char prev_dir[MAX_PATH_LEN];
  getcwd(prev_dir, MAX_PATH_LEN);
  chdir(user_docs_dir());
  filechooser = GTK_FILE_SELECTION(gtk_file_selection_new("MobileSim: Load Map File..."));
  dialog = GTK_DIALOG(filechooser);
  gtk_file_selection_hide_fileop_buttons(filechooser);
  gtk_button_set_label(GTK_BUTTON(filechooser->ok_button), "Load Map");
  gtk_button_set_label(GTK_BUTTON(filechooser->cancel_button), "No Map");

#endif

  gtk_dialog_set_default_response(GTK_DIALOG(dialog), GTK_RESPONSE_ACCEPT);

  // Options: select model, advanced, etc.:
  {   
      /* Options Boxes
       *   0              1              2
       * 0 +--------------+--------------+ 0
       *   | Robot Model: | [Model Menu] |
       * 1 +--------------+--------------+ 1
       *   | > More Options              |
       * 2 +-----------------------------+
       *
       * More Options Expandable panel:
       *
       *   Multiple Robots: 
       *    (o) Create [Num Entry] robot(s), starting at port [Port Entry]
       *    ( ) Create a new robot for each connection to port [Port Entry]
       */


    GtkFrame* optionsframe = GTK_FRAME(gtk_frame_new("Options"));
    gtk_widget_show(GTK_WIDGET(optionsframe));

    /*
    GtkTable* optionstable = GTK_TABLE(gtk_table_new(2, 2, FALSE));
    gtk_container_add(GTK_CONTAINER(optionsframe), GTK_WIDGET(optionstable));
    gtk_widget_show(GTK_WIDGET(optionstable));
    */

    GtkBox* optionsbox = GTK_BOX(gtk_vbox_new(FALSE, 10));
    gtk_container_set_border_width(GTK_CONTAINER(optionsbox), 10);
    gtk_container_add(GTK_CONTAINER(optionsframe), GTK_WIDGET(optionsbox));
    gtk_widget_show(GTK_WIDGET(optionsbox));
    
    //const guint pad = 5;

    // Menu to select a robot model to create, unless -r was given:
    if(include_modelmenu)
    {
      GtkBox* modelbox = GTK_BOX(gtk_hbox_new(FALSE, 10));
      gtk_widget_show(GTK_WIDGET(modelbox));
      gtk_box_pack_start(optionsbox, GTK_WIDGET(modelbox), FALSE, FALSE, 0);

      GtkLabel* modellabel = GTK_LABEL(gtk_label_new("Robot Model:"));
      gtk_widget_show(GTK_WIDGET(modellabel));

      /*
      gtk_misc_set_alignment(GTK_MISC(modellabel), 1, 0.5);
      gtk_table_attach(optionstable, GTK_WIDGET(modellabel), 0, 1, 0, 1, GTK_FILL, GTK_FILL, pad, pad); 
        // L, R, T, B, xopt, yopt, xpad, ypad
      */

      gtk_box_pack_start(modelbox, GTK_WIDGET(modellabel), FALSE, FALSE, 0);

# ifdef USE_GTKCOMBOBOX
# if GTK_CHECK_VERSION(2, 6, 0)
      modelmenu = GTK_COMBO_BOX(gtk_combo_box_entry_new_text());
# else
      modelmenu = GTK_COMBO_BOX(gtk_combo_box_new_text());
# endif

      for(const char** p = CommonRobotModels; *p; p++) {
        gtk_combo_box_append_text(modelmenu, *p);
      }
      gtk_combo_box_set_active(modelmenu, 0);

#else   /* (not using newer combobox) */ 

      modelmenu = GTK_COMBO(gtk_combo_new());
      GList* items = NULL;
      for(const char** m = CommonRobotModels; *m != 0; m++ ) 
      {
        items = g_list_append(items, (gchar*)*m);
      }
      gtk_combo_set_popdown_strings(modelmenu, items);
      gtk_entry_set_text(GTK_ENTRY(modelmenu->entry), CommonRobotModels[0]);

#endif

      gtk_widget_show(GTK_WIDGET(modelmenu));
      /*
      gtk_table_attach(optionstable, GTK_WIDGET(modelmenu), 1, 2, 0, 1, GTK_FILL, GTK_FILL, pad, pad); 
        // L, R, T, B, xopt, yopt, xpad, ypad
      */
      gtk_box_pack_start(modelbox, GTK_WIDGET(modelmenu), FALSE, FALSE, 0);
    }


    // - Advanced options for multirobots:
    //
    
#ifdef USE_GTKEXPANDER
    GtkContainer *more = GTK_CONTAINER(gtk_expander_new("More Options"));
#else
    GtkContainer *more = GTK_CONTAINER(gtk_frame_new(NULL));
#endif
    gtk_box_pack_start(optionsbox, GTK_WIDGET(more), FALSE, FALSE, 0);

    GtkBox *more_box = GTK_BOX(gtk_vbox_new(FALSE, 0));
    gtk_container_add(more, GTK_WIDGET(more_box));
    gtk_widget_show(GTK_WIDGET(more_box));
    

    // Only include num. robot options if neither -r or -R given
    if(include_modelmenu)
    {

      /*    (o) Create [Num Entry] robot(s), starting at port [Port Entry] */
      GtkBox *multirobot_row = GTK_BOX(gtk_hbox_new(FALSE, 0));
      gtk_widget_show(GTK_WIDGET(multirobot_row));
      gtk_box_pack_start(more_box, GTK_WIDGET(multirobot_row), FALSE, FALSE, 3);
      multirobot_radio = GTK_RADIO_BUTTON(gtk_radio_button_new_with_label(NULL, "Create"));
      gtk_widget_show(GTK_WIDGET(multirobot_radio));
      gtk_box_pack_start(multirobot_row, GTK_WIDGET(multirobot_radio), FALSE, FALSE, 5);
      numentry = GTK_ENTRY(gtk_spin_button_new_with_range(1, NUM_ROBOTS_LIMIT, 1));
      gtk_widget_show(GTK_WIDGET(numentry));
      gtk_box_pack_start(multirobot_row, GTK_WIDGET(numentry), FALSE, FALSE, 5);
      GtkWidget *label = gtk_label_new(" robot(s) on seperate TCP ports.");
      gtk_widget_show(label);
      gtk_box_pack_start(multirobot_row, label, FALSE, FALSE, 0);

      /*    ( ) Create a new robot for each connection to port [Port Entry] */
      GtkBox *factory_row = GTK_BOX(gtk_hbox_new(FALSE, 0));
      gtk_widget_show(GTK_WIDGET(factory_row));
      gtk_box_pack_start(more_box, GTK_WIDGET(factory_row), FALSE, FALSE, 3);
      factory_radio = GTK_RADIO_BUTTON(gtk_radio_button_new_with_label(gtk_radio_button_get_group(GTK_RADIO_BUTTON(multirobot_radio)), "Create a new robot for each connection."));
      gtk_widget_show(GTK_WIDGET(factory_radio));
      gtk_box_pack_start(factory_row, GTK_WIDGET(factory_radio), FALSE, FALSE, 5);
    }

    /*    Port: [8101] */
    GtkBox *port_row = GTK_BOX(gtk_hbox_new(FALSE, 0));
    gtk_widget_show(GTK_WIDGET(port_row));
    gtk_box_pack_start(more_box, GTK_WIDGET(port_row), FALSE, FALSE, 15);
    GtkWidget *label = gtk_label_new("TCP Port: ");
    gtk_widget_show(label);
    gtk_box_pack_start(port_row, label, FALSE, FALSE, 10);
    portentry = GTK_ENTRY(gtk_spin_button_new_with_range(1024, 65535, 1)); // 1024..65535 is the allowed range for non-privilaged TCP ports
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(portentry), *port);
    gtk_widget_show(GTK_WIDGET(portentry));
    gtk_box_pack_start(port_row, GTK_WIDGET(portentry), FALSE, FALSE, 1);


    gtk_widget_show(GTK_WIDGET(more));

/****** Old table: 
 * 
 *   0              1              2   3              4              5
 * 0 +--------------+--------------+---+--------------+--------------+ 0
 *   | Num. Robots: | [Num Entry ] |   | Start Port:  | [Port Entry] |
 * 1 +--------------+--------------+---+--------------+--------------+ 1
 */
#if 0

    {
#ifdef USE_GTKEXPANDER
      GtkExpander* more = GTK_EXPANDER(gtk_expander_new("More Options"));
#else
      GtkFrame* more = GTK_FRAME(gtk_frame_new(NULL));   // It's obvious that they are "more" :)
#endif
      gtk_widget_show(GTK_WIDGET(more));
      /*
      gtk_table_attach(optionstable, GTK_WIDGET(more), 0, 2, 1, 2, GTK_FILL, GTK_FILL, pad, pad); 
        // L, R, T, B, xopt, yopt, xpad, ypad
      */
      gtk_box_pack_start(optionsbox, GTK_WIDGET(more), FALSE, FALSE, 0);

      GtkTable* moretable = GTK_TABLE(gtk_table_new(1, 5, FALSE));
      gtk_container_add(GTK_CONTAINER(more), GTK_WIDGET(moretable));
      gtk_widget_show(GTK_WIDGET(moretable));

      // Num. Robots, unless -r was given above:
      if(include_modelmenu)
      {
        GtkLabel* numlabel = GTK_LABEL(gtk_label_new("Num. Robots:"));
        gtk_widget_show(GTK_WIDGET(numlabel));
        gtk_misc_set_alignment(GTK_MISC(numlabel), 1, 0.5);
        gtk_table_attach(moretable, GTK_WIDGET(numlabel), 0, 1, 0, 1, GTK_FILL, GTK_FILL, pad, pad); 
          // L, R, T, B, xopt, yopt, xpad, ypad
        numentry = GTK_ENTRY(gtk_spin_button_new_with_range(1, NUM_ROBOTS_LIMIT, 1));
        gtk_widget_show(GTK_WIDGET(numentry));
        gtk_table_attach(moretable, GTK_WIDGET(numentry), 1, 2, 0, 1, GTK_FILL, GTK_FILL, pad, pad); 
          // L, R, T, B, xopt, yopt, xpad, ypad
      }

      // Spacer in column 2,3
      gtk_table_set_col_spacing(moretable, 2, 40);

      // Starting port, defaulting to 8101 or value given with -p:
      {
        GtkLabel* portlabel = GTK_LABEL(gtk_label_new("First TCP Port:"));
        gtk_widget_show(GTK_WIDGET(portlabel));
        gtk_misc_set_alignment(GTK_MISC(portlabel), 1, 0.5);
        gtk_table_attach(moretable, GTK_WIDGET(portlabel), 3, 4, 0, 1, GTK_FILL, GTK_FILL, pad, pad);
          // L, R, T, B, xopt, yopt, xpad, ypad
        portentry = GTK_ENTRY(gtk_spin_button_new_with_range(1024, 65535, 1)); // 1024..65535 is the allowed range for non-privilaged TCP ports
        gtk_widget_show(GTK_WIDGET(portentry));
        gtk_table_attach(moretable, GTK_WIDGET(portentry), 4, 5, 0, 1, GTK_FILL, GTK_FILL, pad, pad); 
          // L, R, T, B, xopt, yopt, xpad, ypad
        gtk_spin_button_set_value(GTK_SPIN_BUTTON(portentry), *port);
      }
    }
#endif

#ifdef USE_GTKFILECHOOSER
    gtk_file_chooser_set_extra_widget(filechooser, GTK_WIDGET(optionsframe));

    // Make the dialog a bit taller, to help deal with long lists of maps,
    // and also when you expand the more options expander.
    GtkRequisition szreq;
    gtk_widget_size_request(GTK_WIDGET(dialog), &szreq);
    gtk_widget_set_size_request(GTK_WIDGET(dialog), szreq.width, szreq.height * 2);
#else
    gtk_box_pack_end(GTK_BOX(dialog->vbox), GTK_WIDGET(optionsframe), TRUE, TRUE, 0);
#endif
  }




  // Show dialog and check responses
  int r = gtk_dialog_run(dialog);
  if(r == GTK_RESPONSE_ACCEPT || r == GTK_RESPONSE_OK) {
    // selected a map.
#ifdef USE_GTKFILECHOOSER
    *map = (char*)gtk_file_chooser_get_filename(filechooser);
#else
    *map = (char*)gtk_file_selection_get_filename(filechooser);
#endif
  } else if( r == GTK_RESPONSE_DELETE_EVENT ) {
    // closed dialog box. exit program.
    exit(0);
  } else {
    // clicked "No Map"
    *map = "";
  }

  // Check selected robot model
  if(modelmenu)
  {
    std::string selected_robot;
#ifdef USE_GTKCOMBOBOX
# if GTK_CHECK_VERSION(2,6,0)
    selected_robot = gtk_combo_box_get_active_text(modelmenu);
# else
    gint i = gtk_combo_box_get_active(modelmenu);
    if(i != -1)
      selected_robot = CommonRobotModels[i];
# endif
#else
    selected_robot = (char*)gtk_entry_get_text(GTK_ENTRY(modelmenu->entry));
#endif
    if(selected_robot == "")
      selected_robot = "p3dx";

    if(multirobot_radio && gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(multirobot_radio)))
    {
      int n = 1;
      if(numentry)
        n = (int) gtk_spin_button_get_value(GTK_SPIN_BUTTON(numentry));

      std::string name = selected_robot;
      for(int i = 0; i < n; i++)
      {
        (*robotInstanceRequests)[name] = selected_robot;
        stg_print_msg("MobileSim: Robot named \"%s\" will be a \"%s\"", name.c_str(), selected_robot.c_str());
        char numstr[8];
        snprintf(numstr, 8, "_%03d", i+2);
        name = selected_robot + numstr;
      }
    }

    else if(factory_radio && gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(factory_radio)))
    {
      stg_print_msg("MobileSim: Will create a robot factory for model \"%s\".", selected_robot.c_str());
      robotFactoryRequests->push_back(selected_robot);
    }

  }

  *port = (int) gtk_spin_button_get_value(GTK_SPIN_BUTTON(portentry));

  // hide file dialog
  gtk_widget_hide(GTK_WIDGET(dialog));
  // TODO: destroy widgets
 
#ifndef USE_GTKFILECHOOSER
  // go back to previous working directory if we changed for the file dialog
  chdir(prev_dir);
#endif

}


