/*
 *  drivers/mtd/nand/ts7250.c
 *
 *  Copyright (C) 2004 Technologic Systems (support@embeddedARM.com)
 *
 *  Derived from drivers/mtd/nand/edb7312.c
 *       Copyright (C) 2004 Marius Grger (mag@sysgo.de)
 *
 *  Derived from drivers/mtd/nand/autcpu12.c
 *       Copyright (c) 2001 Thomas Gleixner (gleixner@autronix.de)
 *
 * $Id: ts7250.c,v 1.4 2004/12/30 22:02:07 joff Exp $
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 *  Overview:
 *   This is a device driver for the NAND flash device found on the
 *   TS-7250 board which utilizes a Samsung 32 Mbyte part.
 */

#include <linux/slab.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/nand.h>
#include <linux/mtd/partitions.h>
#include <asm/io.h>
#include <asm/arch/hardware.h>
#include <asm/sizes.h>

/*
 * MTD structure for TS7250 board
 */
static struct mtd_info *ts7250_mtd = NULL;


/*
 * Module stuff
 */

static unsigned int ts7250_nandctrl = IO_ADDRESS(TS7XXX_NAND_FLASH_CTRL);
static unsigned int ts7250_nandbusy = IO_ADDRESS(TS7XXX_NAND_FLASH_BUSY);

#ifdef CONFIG_MTD_PARTITIONS
/*
 * Define static partitions for flash device
 */
static struct mtd_partition partition_info32[] = {
	{ .name= "TS-BOOTROM",
		  .offset= 0,
		  .size= 0x4000 },
	{ .name= "Linux",
		  .offset= 0x4000,
		  .size= 0x1d00000 },
	{ .name= "RedBoot",
		  .offset= 0x1d04000,
		  .size= 0x2fc000 },
};
/*
 * Define static partitions for flash device
 */
static struct mtd_partition partition_info128[] = {
	{ .name= "TS-BOOTROM",
		  .offset= 0,
		  .size= 0x4000 },
	{ .name= "Linux",
		  .offset= 0x4000,
		  .size= 0x7d00000 },
	{ .name= "RedBoot",
		  .offset= 0x7d04000,
		  .size= 0x2fc000 },
};
#define NUM_PARTITIONS 3

extern int parse_cmdline_partitions(struct mtd_info *master, 
				    struct mtd_partition **pparts,
				    const char *mtd_id);
#endif


/* 
 *	hardware specific access to control-lines
 */
static void ts7250_hwcontrol(struct mtd_info *mtd, int cmd) 
{
	switch(cmd) {
		
	case NAND_CTL_SETCLE: 
		writeb(readb(ts7250_nandctrl) | 0x2, ts7250_nandctrl);
		break;
	case NAND_CTL_CLRCLE: 
		writeb(readb(ts7250_nandctrl) & ~0x2, ts7250_nandctrl);
		break;
		
	case NAND_CTL_SETALE:
		writeb(readb(ts7250_nandctrl) | 0x1, ts7250_nandctrl);
		break;
	case NAND_CTL_CLRALE:
		writeb(readb(ts7250_nandctrl) & ~0x1, ts7250_nandctrl);
		break;
		
	case NAND_CTL_SETNCE:
		writeb(readb(ts7250_nandctrl) | 0x4, ts7250_nandctrl);
		break;
	case NAND_CTL_CLRNCE:
		writeb(readb(ts7250_nandctrl) & ~0x4, ts7250_nandctrl);
		break;
	}
}

/*
 *	read device ready pin
 */
static int ts7250_device_ready(struct mtd_info *mtd)
{
	return readb(ts7250_nandbusy) & 0x20;
}

/*
 * Main initialization routine
 */
static int __init ts7250_init (void)
{
	struct nand_chip *this;
	const char *part_type = 0;
	int mtd_parts_nb = 0;
	struct mtd_partition *mtd_parts = 0;

	if (readb(IO_ADDRESS(TS7XXX_MODEL)) == 0x0)  /* is TS-7200 */
		return -ENXIO;
	
	/* Allocate memory for MTD device structure and private data */
	ts7250_mtd = kmalloc(sizeof(struct mtd_info) + 
			     sizeof(struct nand_chip),
			     GFP_KERNEL);
	if (!ts7250_mtd) {
		printk("Unable to allocate TS7250 NAND MTD device structure.\n");
		return -ENOMEM;
	}
	
	/* Get pointer to private data */
	this = (struct nand_chip *) (&ts7250_mtd[1]);
	
	/* Initialize structures */
	memset((char *) ts7250_mtd, 0, sizeof(struct mtd_info));
	memset((char *) this, 0, sizeof(struct nand_chip));
	
	/* Link the private data with the MTD structure */
	ts7250_mtd->priv = this;
	
	
	/* insert callbacks */
	if (readb(IO_ADDRESS(TS7XXX_STATUS2)) & 0x4) { 
		this->IO_ADDR_R = IO_ADDRESS(TS7XXX_NAND_FLASH2_DATA);
		this->IO_ADDR_W = IO_ADDRESS(TS7XXX_NAND_FLASH2_DATA);
		ts7250_nandbusy = IO_ADDRESS(TS7XXX_NAND_FLASH2_BUSY);
		ts7250_nandctrl = IO_ADDRESS(TS7XXX_NAND_FLASH2_CTRL);
	} else {
		this->IO_ADDR_R = IO_ADDRESS(TS7XXX_NAND_FLASH_DATA);
		this->IO_ADDR_W = IO_ADDRESS(TS7XXX_NAND_FLASH_DATA);
		ts7250_nandbusy = IO_ADDRESS(TS7XXX_NAND_FLASH_BUSY);
		ts7250_nandctrl = IO_ADDRESS(TS7XXX_NAND_FLASH_CTRL);
	}
	this->hwcontrol = ts7250_hwcontrol;
	this->dev_ready = ts7250_device_ready;
	this->chip_delay = 15;
	this->eccmode = NAND_ECC_SOFT;
	
	printk("Searching for NAND flash...\n");
	/* Scan to find existence of the device */
	if (nand_scan (ts7250_mtd, 1)) {
		kfree (ts7250_mtd);
		return -ENXIO;
	}
	
#ifdef CONFIG_MTD_CMDLINE_PARTS
	mtd_parts_nb = parse_cmdline_partitions(ts7250_mtd, &mtd_parts, 
						"ts7250-nand");
	if (mtd_parts_nb > 0)
	  part_type = "command line";
	else
	  mtd_parts_nb = 0;
#endif
	if (mtd_parts_nb == 0)
	{
		mtd_parts = partition_info32;
		if (ts7250_mtd->size >= (128 * 0x100000))
			mtd_parts = partition_info128;
		mtd_parts_nb = NUM_PARTITIONS;
		part_type = "static";
	}
	
	/* Register the partitions */
	printk(KERN_NOTICE "Using %s partition definition\n", part_type);
	add_mtd_partitions(ts7250_mtd, mtd_parts, mtd_parts_nb);
	
	/* Return happy */
	return 0;
}
module_init(ts7250_init);

/*
 * Clean up routine
 */
static void __exit ts7250_cleanup (void)
{
	struct nand_chip *this = (struct nand_chip *) &ts7250_mtd[1];
	
	/* Unregister the device */
	del_mtd_device (ts7250_mtd);
	
	/* Free the MTD device structure */
	kfree (ts7250_mtd);
}
module_exit(ts7250_cleanup);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Jesse Off <joff@embeddedARM.com>");
MODULE_DESCRIPTION("MTD map driver for Technologic Systems TS-7250 board");
