/*
 * ep93xx-audio.h
 *
 * Common audio handling for the Cirrus EP93xx processor.
 *
 * Copyright (c) 2003 Cirrus Logic Corp.
 * Copyright (C) 2000, 2001 Nicolas Pitre <nico@cam.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License.
 *
 * Taken from sa1100-audio.c
 *
 * This module handles the generic buffering/DMA/mmap audio interface for
 * codecs connected to the EP93xx chip.  All features depending on specific
 * hardware implementations like supported audio formats or samplerates are
 * relegated to separate specific modules.
 *
 *
 * History:
 *
 * 2000-05-21	Nicolas Pitre	Initial release.
 *
 * 2000-06-10	Erik Bunce	Add initial poll support.
 *
 * 2000-08-22	Nicolas Pitre	Removed all DMA stuff. Now using the
 * 				generic SA1100 DMA interface.
 *
 * 2000-11-30	Nicolas Pitre	- Validation of opened instances;
 * 				- Power handling at open/release time instead
 * 				  of driver load/unload;
 *
 * 2001-06-03	Nicolas Pitre	Made this file a separate module, based on
 * 				the former sa1100-uda1341.c driver.
 *
 * 2001-07-22	Nicolas Pitre	- added mmap() and realtime support
 * 				- corrected many details to better comply
 * 				  with the OSS API
 *
 * 2001-10-19	Nicolas Pitre	- brought DMA registration processing
 * 				  into this module for better ressource
 * 				  management.  This also fixes a bug
 * 				  with the suspend/resume logic.
 *
 * 2003-04-04   Adapted for EP93xx I2S/Ac97 audio.
 *
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/types.h>
#include <linux/fs.h>
#include <linux/mm.h>
#include <linux/slab.h>
#include <linux/sched.h>
#include <linux/poll.h>
#include <linux/pm.h>
#include <linux/errno.h>
#include <linux/sound.h>
#include <linux/soundcard.h>
#include <linux/sysrq.h>
#include <linux/delay.h>

#include <asm/uaccess.h>
#include <asm/io.h>
#include <asm/hardware.h>
#include <asm/semaphore.h>
#include <asm/arch/dma.h>

#include "ep93xx-audio.h"

#undef DEBUG 
// #define DEBUG 1
#ifdef DEBUG
#define DPRINTK( x... )  printk( ##x )
#else
#define DPRINTK( x... )
#endif

/* Mostly just prints what ioctls got called */
/* #define DEBUG 1 */
#ifdef DEBUG
#define DPRINTK_IOCTL( x... )  printk( ##x )
#else
#define DPRINTK_IOCTL( x... )
#endif


#define AUDIO_NAME		"ep93xx-audio"

#define AUDIO_NBFRAGS_DEFAULT	8
#define AUDIO_FRAGSIZE_DEFAULT	16384

/*
 * Translates to:
 * 		s->buf_idx++;
 * 		s->buf_idx %= s->nbfrags;
 *		s->buf = s->buffers + s->buf_idx;
 *
 * So s->buf always points to the s->buf_idx-nth element of s->buffers.
 */
#define NEXT_BUF(_s_,_b_) { \
	(_s_)->_b_##_idx++; \
	(_s_)->_b_##_idx %= (_s_)->nbfrags; \
	(_s_)->_b_ = (_s_)->buffers + (_s_)->_b_##_idx; }

#define AUDIO_ACTIVE(state)	((state)->rd_ref || (state)->wr_ref)

/* Function prototypes */
static int copy_n_convert
(
	audio_state_t *state,
	const char *to, 
	const char *from, 
	int toCount 
);
static int scale_to_expanded_samples( audio_state_t *state, int input_bufsize );
static void audio_dma_start( audio_state_t * state, audio_stream_t * stream );
static void audio_dma_pause( audio_state_t * state, audio_stream_t * stream );
static void audio_prime_dma( audio_state_t * state, audio_stream_t * s );

/*
 * We samples of arbitrary format to be 32 bit for our hardware.
 *	Input format is arbitrary.
 *	Output format is either CM (compact mode) where two 16 bit samples
 *  together in a 32 bit word become a stereo sample or non-CM where
 *  each channel gets a 32 bit word.
 *
 *  Input format       Input sample     Output sample size    ratio (out:in)
 *  bits   channels    size (bytes)       CM   non-CM          CM   non-CM
 *   8       mono          1               4      8            4:1   8:1
 *   8      stereo         2			   4      8            2:1   4:1
 *   16      mono          2			   4      8            2:1   4:1
 *   16     stereo         4			   4      8            1:1   2:1
 *
 *   24      mono          3			   4      8             X    8:3
 *   24     stereo         6			   4      8             X    8:6
 *   32      mono          4			   4      8             X    2:1
 *   32     stereo         8			   4      8             X    1:1
 */
static int scale_to_expanded_samples( audio_state_t *state, int input_bufsize )
{
	int output_bufsize = input_bufsize;

#ifdef I2S_SCALE_BUFFERS
	audio_stream_t *os = state->output_stream;
	int output_sample_size, input_sample_size;
	
	if( state->bCompactMode )
	{
		output_sample_size = 4;
	}
	else
	{
		output_sample_size = 8;
	}
	
	input_sample_size = os->audio_stream_bitwidth / 8;
	
	if( os->audio_num_channels != 1 )
	{
		input_sample_size <<= 1;
	}

	output_bufsize = input_bufsize * output_sample_size / input_sample_size;

#endif
	
	return output_bufsize;
}

/*
 * audio_deallocate_buffers
 * 
 * This function frees all buffers
 */
static void audio_deallocate_buffers( audio_state_t * state, audio_stream_t * s )
{
	int frag, i;
	
	DPRINTK("EP93xx - audio_deallocate_buffers\n");

	/* ensure DMA won't run anymore */
	audio_dma_pause( state, s );
	s->active = 0;
	s->stopped = 0;

	for( i=0 ; i<s->NumDmaChannels ; i++ )
	{
		ep93xx_dma_flush( s->dmahandles[0] );
	}
	
	if (s->buffers) 
	{
		for (frag = 0; frag < s->nbfrags; frag++) 
		{
			if (!s->buffers[frag].master)
			{
				continue;
			}

			consistent_free(s->buffers[frag].start,
					s->buffers[frag].master,
					s->buffers[frag].dma_addr);
		}

		/*
		 * Free the space allocated to the array of buf structs.
		 */
		kfree(s->buffers);
		s->buffers = NULL;
	}

	s->buf_idx = 0;
	s->buf = NULL;

	DPRINTK("EP93xx - audio_deallocate_buffers - EXIT\n");
}


/*
 * audio_allocate_buffers
 *
 * This function allocates the buffer structure array and buffer data space
 * according to the current number of fragments and fragment size.
 * Note that the output_stream and input_stream structs are allocated
 * in ep93xx-ac97.c or ep93xx-i2s.c.
 */
static int audio_allocate_buffers( audio_state_t * state, audio_stream_t * s)
{
	int frag;
	int dmasize = 0;
	char *dmabuf = NULL;
	dma_addr_t dmaphys = 0;

	if (s->buffers)
		return -EBUSY;

	DPRINTK("EP93xx audio_allocate_buffers\n" );
	
	/*
	 * Allocate space for the array of audio_buf_t structs.
	 */
	s->buffers = (audio_buf_t *)
		kmalloc(sizeof(audio_buf_t) * s->nbfrags, GFP_KERNEL);

	if (!s->buffers)
	{
		DPRINTK(AUDIO_NAME ": unable to allocate audio memory\n ");
		audio_deallocate_buffers(state,s);
		return -ENOMEM;
	}

	memset( s->buffers, 0, sizeof(audio_buf_t) * s->nbfrags );

	/*
	 * Let's allocate non-cached memory for DMA buffers.
	 * We try to allocate all memory at once.
	 * If this fails (a common reason is memory fragmentation),
	 * then we allocate more smaller buffers.
	 */
	for (frag = 0; frag < s->nbfrags; frag++) 
	{
		audio_buf_t *buf = &s->buffers[frag];

		if (!dmasize) 
		{
			/*
			 * First try to allocate enough for all the frags that
			 * don't yet have memory allocated.
			 */
			dmasize = (s->nbfrags - frag) * s->fragsize;
			do {
			  	dmabuf = consistent_alloc(GFP_KERNEL|GFP_DMA,
							  dmasize, &dmaphys);
				
				/* 
				 * If that fails, try to allocate a chunk of memory
				 * that is one less fragment is size.
				 */
				if (!dmabuf)
				{
					dmasize -= s->fragsize;
				}

			  /* 
			   * Keep trying but the minimum we'll attempt is one 
			   * fragment.  If we can't even get that, give up.
			   */
			} while (!dmabuf && dmasize);
		
			/*
			 * If we do fail to allocate enough for all the frags, 
			 * deallocate whatever we did get and quit.
			 */
			if (!dmabuf)
			{
				DPRINTK(AUDIO_NAME ": unable to allocate audio memory\n ");
				audio_deallocate_buffers(state, s);
				return -ENOMEM;
			}

			DPRINTK("EP93xx allocated %d bytes:  dmabuf=0x%08x  dmaphys=0x%08x\n",
					dmasize, (int)dmabuf, (int)dmaphys );
			
			/*
			 * Success!	 Save the size of this chunk to use when we deallocate it.
			 */
			buf->master = dmasize;
			memzero(dmabuf, dmasize);
		}

		/*
		 * Save the start address of the buf fragment.
		 * We know the size of the fragment is fragsize.
		 */
		buf->start = dmabuf;
		buf->dma_addr = dmaphys;
		buf->stream = s;
		buf->size = 0;
		buf->sent = 0;

		sema_init(&buf->sem, 1);

		/*
		 * Now if we only allocated the minimal one frag of space, the
		 * dmasize will be ==0 after this subtraction so it will allocate more
		 * for the next frag.  Otherwise, the next time(s) thru this for loop
		 * will dole out frag sized pieces of this big master chunk.
		 */
		dmabuf += s->fragsize;
		dmaphys += s->fragsize;
		dmasize -= s->fragsize;
	}

	/*
	 * Initialize the stream.
	 */
	s->buf_idx = 0;				/* Init the current buffer index. 			*/
	s->buf = &s->buffers[0];	/* Point buf to the current buffer struct. 	*/
	s->bytecount = 0;
	s->getptrCount = 0;
	s->fragcount = 0;

	DPRINTK("EP93xx audio_allocate_buffers -- exit SUCCESS\n" );
	return 0;

}


/*
 * audio_reset_buffers
 *
 * This function stops and flushes the dma, gets all buffers back
 * from the DMA driver and resets them ready to be used again.
 */
static void audio_reset_buffers( audio_state_t * state, audio_stream_t * s )
{
	int frag, i;

	audio_dma_pause( state, s );
	s->active = 0;
	s->stopped = 0;

	for( i=0 ; i < s->NumDmaChannels ; i++ )
	{
		ep93xx_dma_flush( s->dmahandles[0] );
	}

	if (s->buffers) 
	{
		for (frag = 0; frag < s->nbfrags; frag++) 
		{
			audio_buf_t *buf = &s->buffers[frag];
			buf->size = 0;
			buf->sent = 0;
			sema_init(&buf->sem, 1);
		}
	}

	s->bytecount = 0;
	s->getptrCount = 0;
	s->fragcount = 0;
}


/*
 * DMA callback functions
 */
static void audio_dma_tx_callback
( 
	ep93xx_dma_int_t DMAInt,
	ep93xx_dma_dev_t device, 
	unsigned int user_data 
)
{
	unsigned int buf_id;
	int handle;
	
	audio_state_t *state = (audio_state_t *)user_data;
	audio_stream_t *s = state->output_stream;
	
 /*	DPRINTK( "audio_dma_tx_callback  - enter\n"); */
	
	switch( device )
	{
    	case DMATx_I2S3:
			handle = s->dmahandles[2];
			break;

    	case DMATx_I2S2:
			handle = s->dmahandles[1];
			break;

		case DMATx_I2S1:
		default:
			handle = s->dmahandles[0];
			break;
	}

  	if (s->mapped)
	{
		/*
		 * If we are mapped, get one buffer back and recycle it.
		 */
		if( ep93xx_dma_remove_buffer( handle, &buf_id ) >= 0 )
		{
		   	audio_buf_t *buf = (audio_buf_t *)buf_id;

		 /*	DPRINTK( "audio_dma_tx_callback  - got buffer index=%d\n", buf_id); */
		
			/* Accounting */
			s->bytecount += buf->size;
			s->fragcount++;
			buf->size = 0;
		
			/* Recycle buffer */
		 	buf->size = s->fragsize;
			
			ep93xx_dma_add_buffer( s->dmahandles[0], 	/* dma instance 		*/
								   (unsigned int)buf->dma_addr, /* source 		*/
								   0, 					/* dest 				*/
								   s->fragsize,			/* size					*/
								   0,					/* is the last chunk? 	*/
								   (unsigned int) buf ); /* buf id			 	*/	
	  		
			buf->sent = 1;
		}
	}
	else
	{
		/*
		 * Get all buffers that are free'ed back and clear their semephores.
		 */
		while( ep93xx_dma_remove_buffer( handle, &buf_id ) >= 0 )
		{
		   	audio_buf_t *buf = (audio_buf_t *)buf_id;

		 /*	DPRINTK( "audio_dma_tx_callback  - got buffer index=%d\n", buf_id); */
		
			/* Accounting */
			s->bytecount += buf->size;
			s->fragcount++;
			buf->size = 0;
			buf->sent = 0;
			
			/*
			 * Release the semaphore on this buf.
			 * If write is waiting on this buf then it can go ahead and fill
			 * it and send it to the dma.
			 */
			up(&buf->sem);
		}
	}

	/* And any process polling on write. */
	wake_up(&s->wq);
}

static void audio_dma_rx_callback
( 
	ep93xx_dma_int_t DMAInt,
	ep93xx_dma_dev_t device, 
	unsigned int user_data 
)
{
	unsigned int buf_id;
	int handle;
	
	audio_state_t *state = (audio_state_t *)user_data;
	audio_stream_t *s = state->input_stream;

	switch( device )
	{
    	case DMARx_I2S3:
			handle = s->dmahandles[2];
			break;

    	case DMARx_I2S2:
			handle = s->dmahandles[1];
			break;

		case DMARx_I2S1:
		default:
			handle = s->dmahandles[0];
			break;
	}
	
	/*
	 * Keep removing and recycling buffers as long as there are buffers
	 * to remove.
	 */
	while( !ep93xx_dma_remove_buffer( handle, &buf_id ) )
	{
	   	audio_buf_t *buf = (audio_buf_t *) buf_id;

		/* Accounting */
		s->bytecount += buf->size;
		s->fragcount++;

		/* Recycle buffer */
		if( s->mapped ) 
		{
		 	ep93xx_dma_add_buffer( s->dmahandles[0], 	/* dma instance 		*/
								   (unsigned int)buf->dma_addr, /* source 		*/
								   0, 					/* dest 				*/
								   s->fragsize,			/* size					*/
								   0,					/* is the last chunk? 	*/
								   (unsigned int) buf ); /* buf id			 	*/	
		} 
		else 
		{
			buf->size = s->fragsize;
			up(&buf->sem);
		}

		/* And any process polling on write. */
		wake_up(&s->wq);
	}
}

/*
 * audio_sync
 *
 * Wait until the last byte written to this device has been played.
 */
static int audio_sync(struct file *file)
{
	audio_state_t *state = (audio_state_t *)file->private_data;
	audio_stream_t *s = state->output_stream;
	audio_buf_t *buf;
/*	int frag; */
	int buf_wait_index;
  /*	audio_buf_t *buftemp; */

	DPRINTK( "EP93xx - audio_sync - enter\n" );

	if (!(file->f_mode & FMODE_WRITE) || !s->buffers || s->mapped)
	{
		DPRINTK( "EP93xx - audio_sync - exit immediately.\n" );
		return 0;
	}
	
	/*
	 * Send current buffer if it contains data and hasn't been sent. 
	 */
#if 0	 
	for (frag = 0; frag < s->nbfrags; frag++) 
	{
		buftemp = &s->buffers[frag];
		DPRINTK("audio_sync - buf# %d: size=%d  sent=%d\n",
			frag, buftemp->size, buftemp->sent );
	}
#endif

	buf = s->buf;
	if( buf->size && !buf->sent ) 
	{
		DPRINTK("audio_sync -- SENDING BUFFER index=%d size=%d\n",
				s->buf_idx, buf->size );
		
		down(&buf->sem);
	 	ep93xx_dma_add_buffer( s->dmahandles[0], 	/* dma instance 		*/
							   (unsigned int)buf->dma_addr, /* source 		*/
							   0, 					/* dest 				*/
							   buf->size,			/* size					*/
							   0,					/* is the last chunk? 	*/
							   (unsigned int) buf ); /* buf id			 	*/	
	 	buf->sent = 1;
		NEXT_BUF(s, buf);
	}

	/*
	 * Let's wait for the last buffer we sent i.e. the one before the
	 * current buf_idx.  When we acquire the semaphore, this means either:
	 * - DMA on the buffer completed or
	 * - the buffer was already free thus nothing else to sync.
	 */
	buf_wait_index = ((s->nbfrags + s->buf_idx - 1) % s->nbfrags);
	buf = s->buffers + buf_wait_index;

#if 0
	while( buf->sent )
	{
		mdelay( 1000 );
		for (frag = 0; frag < s->nbfrags; frag++) 
		{
			buftemp = &s->buffers[frag];
			DPRINTK("audio_sync - buf# %d: size=%d  sent=%d\n",
				frag, buftemp->size, buftemp->sent );
		}
		DPRINTK("\n");
	}
	DPRINTK("audio_sync -- current buf index is %d.  Waiting on buffer index=%d\n",
			s->buf_idx, buf_wait_index );
#endif

	if (down_interruptible(&buf->sem))
	{
		return -EINTR;
	}
	
	up(&buf->sem);
	DPRINTK("EP93xx - audio_sync - EXIT\n");
	return 0;
}

//static int iWriteCount = 0;

/*
 * Need to convert to 32 bit stereo format:	  
 * 16 bit signed
 * 16 bit unsigned
 *  8 bit signed
 *  8 bit unsigned
 */
static int audio_write
(
	struct file 	*file, 
	const char 		*buffer,
	size_t 			src_count, 
	loff_t 			*ppos
)
{
	const char *buffer0 = buffer;
	audio_state_t *state = (audio_state_t *)file->private_data;
	audio_stream_t *s = state->output_stream;
	int dma_xfer_count, src_xfer_count, expanded_count, ret = 0;
//	int i,j;
	
 	/* DPRINTK_IOCTL( "EP93xx - audio_write: count=%d\n", src_count ); */

	if (ppos != &file->f_pos)
		return -ESPIPE;
		
	if (s->mapped)
		return -ENXIO;
		
	if( !access_ok( VERIFY_READ, buffer, src_count ) )
	{
		DPRINTK( "access_ok failed!!!!\n" );
		return -EFAULT;
	}

	/*
	 * Allocate dma buffers if we haven't already done so.
	 */
	if( !s->buffers && audio_allocate_buffers( state, s) )
	{
		return -ENOMEM;
	}

//	if( iWriteCount > 40 )
//	{
//		for( i=0; i<4 ; i++ )
//		{
//			for( j=0; j<8 ; j++ )
//			{
//				printk("%02x ", buffer[(i*8)+j] );
//			}
//			printk("\n");
//		}
//		iWriteCount = 0;
//	}
//	iWriteCount++;
	
	/*
	 * Stay in this loop until we have copied all of the file
	 * into user memory.
	 */
	while( src_count > 0 ) 
	{
		audio_buf_t *buf = s->buf;

		/* Wait for a buffer to become free */
		if (file->f_flags & O_NONBLOCK)
		{
			ret = -EAGAIN;
			if (down_trylock(&buf->sem))
				break;
		} 
		else 
		{
			ret = -ERESTARTSYS;
			if (down_interruptible(&buf->sem))
				break;
		}

		/* 
		 * Feed the current buffer (s->buf) 
		 * This involves expanding sample size from the source buffer
		 * to be 32 bit stereo for our dma.
		 */
		
		/*
		 * How much space is left in the current buf?
		 *
		 * dma_xfer_count is # of bytes placed into the dma buffer
		 * where each sample is 4 bytes (4 bytes left, 4 bytes right)
		 */
		dma_xfer_count = s->fragsize - buf->size;
		
		/*
		 * Input buffer is src_count (bytes) of whatever format.
		 * How big will src_count be when sample size is expanded
		 * to 32 bit samples for our hardware?
		 */
		expanded_count = scale_to_expanded_samples( state, src_count );
		
		/*
		 * See if we can fit all the remaining input buffer in
		 * the current buffer...
		 */
		if (dma_xfer_count > expanded_count)
		{
			dma_xfer_count = expanded_count;
		}
		
		//DPRINTK( "EP93xx - audio_write: %d to buf # %d\n", dma_xfer_count, s->buf_idx );
		
		src_xfer_count = copy_n_convert( 	state, 
											buf->start + buf->size, 
											buffer, 
											dma_xfer_count);
		if( src_xfer_count <= 0 ) 
		{
			up(&buf->sem);
			return -EFAULT;
		}

		/*
		 * Increment dma buffer pointer.
		 */
		buf->size += dma_xfer_count;

		/* 
		 * Increment source buffer pointer.
		 * Decrement the input buffer size count.
		 */
		buffer += src_xfer_count;
		src_count  -= src_xfer_count;

		/*
		 * If the chunk is less than a whole fragment, don't
		 * send it, wait for more ??? I say, go ahead and send it.
		 */
#if 0
		if (buf->size < s->fragsize) 
		{
			up(&buf->sem);
			break;
		}	 
#endif

		/*
		 * If we haven't already started the DMA start it.
		 * But don't start it if we are waiting on a trigger.
		 */
		if( !s->active && !s->stopped )
		{
			s->active = 1;
			audio_dma_start( state, s );
		}
		
		/* Send current buffer to dma */
	 	ep93xx_dma_add_buffer( s->dmahandles[0], 	/* dma instance 		*/
							   (unsigned int)buf->dma_addr, /* source 		*/
							   0, 					/* dest 				*/
							   buf->size,			/* size					*/
							   0,					/* is the last chunk? 	*/
							   (unsigned int) buf ); /* buf id			 	*/	
		
		/* 
		 * Indicate that the buffer has been sent
	 	 * used to set buf->size = 0; but we can't use that anymore
		 */
		buf->sent = 1;
		
		/*
		 *
		 */
		NEXT_BUF(s, buf);
	}
	
	/*
	 * Return the number of bytes transferred.
	 */
	if( buffer - buffer0 )
	{
		ret = buffer - buffer0;
	}
	
/*	DPRINTK( "EP93xx - audio_write: return=%d\n", ret );*/
	return ret;
}


/*
 * audio_dma_start
 *
 * Our Ac97 has a specific start order that it likes.  Enable the 
 * Ac97 channel AFTER enabling DMA.  Our I2S is not so picky.
 */
void audio_dma_start( audio_state_t * state, audio_stream_t * stream )
{
	ep93xx_dma_start( stream->dmahandles[0],
					  stream->NumDmaChannels, 
					  stream->dmahandles );
	
	if (state->hw_enable)
	{
		state->hw_enable( state->output_stream == stream );
	}
}

/*
 * audio_dma_pause
 */
void audio_dma_pause( audio_state_t * state, audio_stream_t * stream )
{
	if (state->hw_disable)
	{
		state->hw_disable( state->output_stream == stream );
	}
	
	ep93xx_dma_pause( stream->dmahandles[0],
					  stream->NumDmaChannels, 
					  stream->dmahandles );
}

static void audio_prime_dma( audio_state_t * state, audio_stream_t * s )
{
	int i;

	/*
	 * If we haven't already started the DMA start it.
	 * But don't start it if we are waiting on a trigger.
	 */
	if( !s->active && !s->stopped )
	{
		s->active = 1;
		audio_dma_start( state, s );
	}

	for (i = 0; i < s->nbfrags; i++) 
	{
		audio_buf_t *buf = s->buf;
		down(&buf->sem);
	 	ep93xx_dma_add_buffer( s->dmahandles[0], 	/* dma instance 		*/
							   (unsigned int)buf->dma_addr, /* source 		*/
							   0, 					/* dest 				*/
							   s->fragsize,			/* size					*/
							   0,					/* is the last chunk? 	*/
							   (unsigned int) buf ); /* buf id			 	*/	
		NEXT_BUF(s, buf);
	}
}


static int audio_read(struct file *file, char *buffer,
		      size_t count, loff_t * ppos)
{
	char *buffer0 = buffer;
	audio_state_t *state = (audio_state_t *)file->private_data;
	audio_stream_t *s = state->input_stream;
	int dest_xfer_count, ret = 0;

	DPRINTK("EP93xx - audio_read: count=%d\n", count);

	if (ppos != &file->f_pos)
		return -ESPIPE;
	if (s->mapped)
		return -ENXIO;

	if (!s->active) 
	{
		if (!s->buffers && audio_allocate_buffers( state, s))
		{
			return -ENOMEM;
		}
		audio_prime_dma( state, s);
	}

	while (count > 0) 
	{
		audio_buf_t *buf = s->buf;

		/* Wait for a buffer to become full */
		if (file->f_flags & O_NONBLOCK) 
		{
			ret = -EAGAIN;
			if (down_trylock(&buf->sem))
				break;
		} 
		else 
		{
			ret = -ERESTARTSYS;
			if (down_interruptible(&buf->sem))
				break;
		}

		/* Grab data from the current buffer */
		dest_xfer_count = buf->size;
		if (dest_xfer_count > count)
		{
			dest_xfer_count = count;
		}
			
		DPRINTK("EP93xx - read %d from %d\n", dest_xfer_count, s->buf_idx);
		
		if (copy_to_user(buffer,
				 buf->start + s->fragsize - buf->size,
				 dest_xfer_count)) 
		{
			up(&buf->sem);
			return -EFAULT;
		}
		
		buf->size -= dest_xfer_count;
		buffer += dest_xfer_count;
		count -= dest_xfer_count;
		
		if (buf->size > 0) 
		{
			up(&buf->sem);
			break;
		}

		/* Make current buffer available for DMA again */
	 	ep93xx_dma_add_buffer( s->dmahandles[0], 	/* dma instance 		*/
							   (unsigned int)buf->dma_addr, /* source 		*/
							   0, 					/* dest 				*/
							   s->fragsize,			/* size					*/
							   0,					/* is the last chunk? 	*/
							   (unsigned int) buf ); /* buf id			 	*/	

		NEXT_BUF(s, buf);
	}

	if( buffer - buffer0 )
	{
		ret = buffer - buffer0;
	}
	
	DPRINTK("EP93xx - audio_read: return=%d\n", ret);
	return ret;
}


static int audio_mmap(struct file *file, struct vm_area_struct *vma)
{
	audio_state_t *state = (audio_state_t *)file->private_data;
	audio_stream_t *s;
	unsigned long size, vma_addr;
	int i, ret;

	if (vma->vm_pgoff != 0)
	{
		return -EINVAL;
	}

	if (vma->vm_flags & VM_WRITE) 
	{
		if (!state->wr_ref)
		{
			return -EINVAL;
		}
		s = state->output_stream;
	}
	else if (vma->vm_flags & VM_READ) 
	{
		if (!state->rd_ref)
		{
			return -EINVAL;
		}
		s = state->input_stream;
	} 
	else 
	{
		return -EINVAL;
	}

	if (s->mapped)
	{
		return -EINVAL;
	}

	size = vma->vm_end - vma->vm_start;
	
	if (size != s->fragsize * s->nbfrags)
	{
		return -EINVAL;
	}
	
	if (!s->buffers && audio_allocate_buffers( state, s))
	{
		return -ENOMEM;
	}

	vma_addr = vma->vm_start;

	for (i = 0; i < s->nbfrags; i++) 
	{
		audio_buf_t *buf = &s->buffers[i];
		
		if (!buf->master)
		{
			continue;
		}

		ret = remap_page_range(vma_addr, buf->dma_addr, buf->master, vma->vm_page_prot);

		if (ret)
		{
			return ret;
		}

		vma_addr += buf->master;
	}

	s->mapped = 1;

	return 0;
}


static unsigned int audio_poll(struct file *file,
			       struct poll_table_struct *wait)
{
	audio_state_t *state = (audio_state_t *)file->private_data;
	audio_stream_t *is = state->input_stream;
	audio_stream_t *os = state->output_stream;
	unsigned int mask = 0;
	int i;

	DPRINTK("EP93xx - audio_poll(): mode=%s%s\n",
		(file->f_mode & FMODE_READ) ? "r" : "",
		(file->f_mode & FMODE_WRITE) ? "w" : "");

	if (file->f_mode & FMODE_READ) 
	{
		/* Start audio input if not already active */
		if (!is->active) 
		{
			if (!is->buffers && audio_allocate_buffers( state, is))
			{
				return -ENOMEM;
			}

			audio_prime_dma( state, is);
		}

		poll_wait(file, &is->wq, wait);
	}

	if (file->f_mode & FMODE_WRITE) 
	{
		if (!os->buffers && audio_allocate_buffers( state, os))
		{
			return -ENOMEM;
		}

		poll_wait(file, &os->wq, wait);
	}

	if (file->f_mode & FMODE_READ) 
	{
		if (is->mapped) 
		{
			/* 
			 * if the buffer is mapped assume we care that there are 
			 * more bytes available than when we last asked using 
			 * SNDCTL_DSP_GETxPTR 
			 */
			if (is->bytecount != is->getptrCount)
			{
				mask |= POLLIN | POLLRDNORM;
			}
		} 
		else 
		{
			for (i = 0; i < is->nbfrags; i++) 
			{
				if (atomic_read(&is->buffers[i].sem.count) > 0) 
				{
					mask |= POLLIN | POLLRDNORM;
					break;
				}
			}
		}
	}
	
	if (file->f_mode & FMODE_WRITE) 
	{
		if (os->mapped) 
		{
			if (os->bytecount != os->getptrCount)
				mask |= POLLOUT | POLLWRNORM;
		} 
		else 
		{
			for (i = 0; i < os->nbfrags; i++) 
			{
				if (atomic_read(&os->buffers[i].sem.count) > 0) 
				{
					mask |= POLLOUT | POLLWRNORM;
					break;
				}
			}
		}
	}

	DPRINTK("EP93xx - audio_poll() returned mask of %s%s\n",
		(mask & POLLIN) ? "r" : "",
		(mask & POLLOUT) ? "w" : "");

	return mask;
}

/*
 * audio_set_fragments
 *
 * Used to process SNDCTL_DSP_SETFRAGMENT.
 *
 * Argument is 0xMMMMSSSS where:
 * 		SSSS sets dma fragment (buf) size.  size = 2^SSSS bytes
 *		MMMM sets number of fragments.
 */
static int audio_set_fragments( audio_state_t *state, audio_stream_t *s, int val )
{
	if (s->active)
	{
		return -EBUSY;
	}
	
	/*
	 * We're gonna change the buffers, so free the ones we currently have
	 */
	if (s->buffers)
	{
		audio_deallocate_buffers(state, s);
	}
	
	/*
	 * Get number of fragments from upper 16 or so bits of argument.
	 */
	s->nbfrags = (val >> 16) & 0x7FFF;

	/*
	 * Get size of fragments from lower 16 bits of argument.
	 */
	val &= 0xffff;

	DPRINTK_IOCTL( "audio_set_fragments: requested num = %d ; size = 2^%d\n",
				s->nbfrags, val );

	if (val < 4)
	{
		val = 4;
	}
	
	if (val > 15)
	{
		val = 15;
	}
	
	s->fragsize = 1 << val;

	if (s->nbfrags < 2)
	{
		s->nbfrags = 2;
	}
	
	/*
	 * Limit total dma buffer size to 128K
	 */
	if (s->nbfrags * s->fragsize > 128 * 1024)
	{
		s->nbfrags = 128 * 1024 / s->fragsize;
	}
	
	/*
	 * Keep track of how many frags we are telling the app we allocated.
	 */
	s->reportedfrags = s->nbfrags;

	s->nbfrags = scale_to_expanded_samples( state, s->nbfrags );

	if (audio_allocate_buffers( state, s))
	{
		return -ENOMEM;
	}
	
	DPRINTK_IOCTL( "audio_set_fragments: allocated num = %d ; size = %d bytes\n",
				s->nbfrags, s->fragsize );

	return val|(s->reportedfrags << 16);
}

	
static void audio_set_num_channels( audio_stream_t * s, long val)
{
	DPRINTK( "audio_set_num_channels = %d\n", (int)val );
	s->audio_num_channels = val;
}

static void print_audio_format( long format )
{
	switch( format )
	{
		case AFMT_U8:		   
			DPRINTK_IOCTL( "AFMT_U8\n" );		   
			break;

		case AFMT_S8:
			DPRINTK_IOCTL( "AFMT_S8\n" );		   
			break;

		case AFMT_S32_BLOCKED:
			DPRINTK_IOCTL( "AFMT_S32_BLOCKED\n" );		   
			break;

		case AFMT_S16_LE:
			DPRINTK_IOCTL( "AFMT_S16_LE\n" );		   
			break;

		case AFMT_S16_BE:
			DPRINTK_IOCTL( "AFMT_S16_BE\n" );		   
			break;

		case AFMT_U16_LE:
			DPRINTK_IOCTL( "AFMT_U16_LE\n" );		   
			break;

		case AFMT_U16_BE:
		default:
			DPRINTK_IOCTL( "AFMT_U16_BE\n" );		   
			break;
	}
}

/*
 * We convert to 24 bit samples that occupy 32 bits each.
 * Formats we support:
 *
 * AFMT_U8		   
 * AFMT_S16_LE	   	Little endian signed 16
 * AFMT_S8		   
 * AFMT_U16_LE	   	Little endian U16
 * AFMT_S32_BLOCKED	32 bit little endian format, taken from the rme96xx driver.
 *
 */
static void audio_set_format( audio_stream_t * s, long val )
{
	DPRINTK_IOCTL( "audio_set_format enter.  Format requested (%d) ", 
				(int)val );
	print_audio_format( val );
	
	switch( val )
	{
		case AFMT_U8:		   
			s->audio_format = val;
			s->audio_stream_bitwidth = 8;
			break;

		case AFMT_S8:
			s->audio_format = val;
			s->audio_stream_bitwidth = 8;
			break;

		case AFMT_S32_BLOCKED:
			s->audio_format = val;
			s->audio_stream_bitwidth = 32;
			break;

		case AFMT_S16_LE:
		case AFMT_S16_BE:
			s->audio_format = AFMT_S16_LE;
			s->audio_stream_bitwidth = 16;
			break;

		case AFMT_U16_LE:
		case AFMT_U16_BE:
		default:
			s->audio_format = AFMT_U16_LE;
			s->audio_stream_bitwidth = 16;
			break;
	}

	DPRINTK_IOCTL( "audio_set_format EXIT format set to be (%d) ", 
				(int)s->audio_format );
	print_audio_format( (long)s->audio_format );
}

static int copy_from_user_U16_LE_CM
(
	audio_state_t *state,
	const char *to, 
	const char *from, 
	int toCount 
)
{
	unsigned short data;
	unsigned int val;
	int toCount0 = toCount;
	unsigned int * p = (unsigned int *)to;
	unsigned short * userPtr = (unsigned short *)from;
	int mono = (state->output_stream->audio_num_channels == 1);

	/*
	 * Compact mode - slot 3 (left) is the lower 16 bits.  slot 4 (right)
	 * is the upper 16 bits.
	 */
	while (toCount > 0) 
	{
		if (__get_user(data, userPtr++))
		{
			return -EFAULT;
		}
		val = (unsigned int)data & 0x0000ffff;
		
		/*
		 * If mono, use this mono sample for both channels, else we just got
		 * the left channel sample.  So get the right sample and use it for 
		 * the right channel.
		 */
		if( !mono )
		{
			if (__get_user(data, userPtr++))
			{
				return -EFAULT;
			}
		}

		*p++ = (((unsigned int)data << 16) | val) ^ 0x80008000;

		toCount -= 4;
	}

	/*
	 * Each mono or stereo 16 bit sample going in results
	 * in one stereo 32 bit sample going out.  So
	 * mono   = (2 bytes in)/(4 bytes out)
	 * stereo = (4 bytes in)/(4 bytes out)
	 */
	if( mono )
	{
		return toCount0 / 2;
	}
	
	return toCount0;
}

static int copy_from_user_U16_LE
(
	audio_state_t *state,
	const char *to, 
	const char *from, 
	int toCount 
)
{
	unsigned short data;
	int val;
	int toCount0 = toCount;
	int * p = (int *)to;
	unsigned short * userPtr = (unsigned short *)from;
	int mono = (state->output_stream->audio_num_channels == 1);
	int shift = state->DAC_bit_width - 16;

	while (toCount > 0) 
	{
		if (__get_user(data, userPtr++))
		{
			return -EFAULT;
		}
		val = (data ^ 0x8000) << shift;
		*p++ = val;
		toCount -= 4;

		if (!mono) 
		{
			if (__get_user(data, userPtr++))
			{
				return -EFAULT;
			}
			val = (data ^ 0x8000) << shift;
		}

		*p++ = val;
		toCount -= 4;
	}

	/*
	 * Each mono or stereo 16 bit sample going in results
	 * in a two 32 bit (left & right) samples going out.  So
	 * mono   = (2 bytes in)/(8 bytes out)
	 * stereo = (4 bytes in)/(8 bytes out)
	 */
	if( mono )
	{
		return toCount0 / 4;
	}
	
	return toCount0 / 2;
}

static int copy_from_user_S16_LE_CM
(
	audio_state_t *state,
	const char *to, 
	const char *from, 
	int toCount 
)
{
	short data;
	unsigned int val;
	unsigned int * p = (int *)to;
	int toCount0 = toCount;
	unsigned short * userPtr = (unsigned short *)from;
	int mono = (state->output_stream->audio_num_channels == 1);

	while (toCount > 0) 
	{
		if (__get_user(data, userPtr++))
		{
			return -EFAULT;
		}
		val = (unsigned int)data & 0x0000ffff;

		if( !mono )
		{
			if (__get_user(data, userPtr++))
			{
				return -EFAULT;
			}
		}

		*p++ = ((unsigned int)data << 16) | val;
		toCount -= 4;
	}

	/*
	 * Each mono or stereo 16 bit sample going in results
	 * in one stereo 32 bit sample going out.  So
	 * mono   = (2 bytes in)/(4 bytes out)
	 * stereo = (4 bytes in)/(4 bytes out)
	 */
	if( mono )
	{
		return toCount0 / 2;
	}
	
	return toCount0;
}


static int copy_from_user_S16_LE
(
	audio_state_t *state,
	const char *to, 
	const char *from, 
	int toCount 
)
{
	short data;
	int val;
	int toCount0 = toCount;
	int * p = (int *)to;
	unsigned short * userPtr = (unsigned short *)from;
	int mono = (state->output_stream->audio_num_channels == 1);
	int shift = state->DAC_bit_width - 16;

	while (toCount > 0) 
	{
		if (__get_user(data, userPtr++))
		{
			return -EFAULT;
		}
		val = data << shift;
		*p++ = val;
		toCount -= 4;

		if (!mono) 
		{
			if (__get_user(data, userPtr++))
			{
				return -EFAULT;
			}
			val = data << shift;
		}

		*p++ = val;
		toCount -= 4;
	}

	/*
	 * Each mono or stereo 16 bit sample going in results
	 * in a two 32 bit samples (left & right) going out.  So
	 * mono   = (2 bytes in)/(8 bytes out)
	 * stereo = (4 bytes in)/(8 bytes out)
	 */
	if( mono )
	{
		return toCount0 / 4;
	}
	
	return toCount0 / 2;
}

static int copy_from_user_S8_CM
(
	audio_state_t *state,
	const char *to, 
	const char *from, 
	int toCount 
)
{
	char data;
	int val;
	int toCount0 = toCount;
	int * p = (int *)to;
	unsigned char * userPtr = (unsigned char *)from;
	int mono = (state->output_stream->audio_num_channels == 1);

	while (toCount > 0) 
	{
		if (__get_user(data, userPtr++))
		{
			return -EFAULT;
		}
		val = (data << 8) & 0x0000ffff;

		if (!mono) 
		{
			if (__get_user(data, userPtr++))
			{
				return -EFAULT;
			}
		}

		*p++ = (data << 24) | val;
		toCount -= 4;
	}

	/*
	 * Each mono or stereo 8 bit sample going in results
	 * in one stereo 32 bit sample going out.  So
	 * mono   = (1 byte  in)/(4 bytes out)
	 * stereo = (2 bytes in)/(4 bytes out)
	 */
	if( mono )
	{
		return toCount0 / 4;
	}
	
	return toCount0 / 2;
}

static int copy_from_user_S8
(
	audio_state_t *state,
	const char *to, 
	const char *from, 
	int toCount 
)
{
	char data;
	int val;
	int toCount0 = toCount;
	int * p = (int *)to;
	unsigned char * userPtr = (unsigned char *)from;
	int mono = (state->output_stream->audio_num_channels == 1);
	int shift = state->DAC_bit_width - 8;

	while (toCount > 0) 
	{
		if (__get_user(data, userPtr++))
		{
			return -EFAULT;
		}
		val = data << shift;
		*p++ = val;
		toCount -= 4;

		if (!mono) 
		{
			if (__get_user(data, userPtr++))
			{
				return -EFAULT;
			}
			val = data << shift;
		}

		*p++ = val;
		toCount -= 4;
	}

	/*
	 * Each mono or stereo 8 bit sample going in results
	 * in a two 32 bit samples (left & right) going out.  So
	 * mono   = (1 byte  in)/(8 bytes out)
	 * stereo = (2 bytes in)/(8 bytes out)
	 */
	if( mono )
	{
		return toCount0 / 8;
	}
	
	return toCount0 / 4;
}

static int copy_from_user_U8_CM
(
	audio_state_t *state,
	const char *to, 
	const char *from, 
	int toCount 
)
{
	int * p = (int *)to;
	unsigned char * userPtr = (unsigned char *)from;
	unsigned char data;
	int val;
	int toCount0 = toCount;

	int mono = (state->output_stream->audio_num_channels == 1);

	while (toCount > 0) 
	{
		if (__get_user(data, userPtr++))
		{
			return -EFAULT;
		}
		val = ((data ^ 0x80) << 8) & 0x0000ffff;

		if (!mono) 
		{
			if (__get_user(data, userPtr++))
			{
				return -EFAULT;
			}
		}

		*p++ = ((data ^ 0x80) << 24) | val;
		toCount -= 4;
	}

	/*
	 * Each mono or stereo 8 bit sample going in results
	 * in one stereo 32 bit sample going out.  So
	 * mono   = (1 byte  in)/(4 bytes out)
	 * stereo = (2 bytes in)/(4 bytes out)
	 */
	if( mono )
	{
		return toCount0 / 4;
	}
	
	return toCount0 / 2;
}

static int copy_from_user_U8
(
	audio_state_t *state,
	const char *to, 
	const char *from, 
	int toCount 
)
{
	int * p = (int *)to;
	unsigned char * userPtr = (unsigned char *)from;

	unsigned char data;
	int val;
	int toCount0 = toCount;
	int shift = state->DAC_bit_width - 8;

	int mono = (state->output_stream->audio_num_channels == 1);

	while (toCount > 0) 
	{
		if (__get_user(data, userPtr++))
		{
			return -EFAULT;
		}
		val = (data ^ 0x80) << shift;
		*p++ = val;
		toCount -= 4;

		if (!mono) 
		{
			if (__get_user(data, userPtr++))
			{
				return -EFAULT;
			}
			val = (data ^ 0x80) << shift;
		}

		*p++ = val;
		toCount -= 4;
	}

	/*
	 * Each mono or stereo 8 bit sample going in results
	 * in a two 32 bit samples (left & right) going out.  So
	 * mono   = (1 byte  in)/(8 bytes out)
	 * stereo = (2 bytes in)/(8 bytes out)
	 */
	if( mono )
	{
		return toCount0 / 8;
	}
	
	return toCount0 / 4;
}

static int copy_from_user_U32
(
	audio_state_t *state,
	const char *to, 
	const char *from, 
	int toCount 
)
{
	if (copy_from_user( (char *)to, from, toCount)) 
	{
		return -EFAULT;
	}

	return toCount;
}

/*
 * Returns negative for error
 * Returns # of bytes transferred out of the from buffer
 * for success.
 */
static int copy_n_convert
(
	audio_state_t *state,
	const char *to, 
	const char *from, 
	int toCount 
)
{
	int ret=0;
	
	if( toCount == 0 )
	{
		DPRINTK("copy_n_convert - nothing to copy!\n");
	}

	/*
	 * Compact Mode means that each 32 bit word has both the
	 * left and right sample in it.
	 */
	if( state->bCompactMode )
	{
		switch( state->output_stream->audio_format )
		{
			case AFMT_U8:
				ret = copy_from_user_U8_CM( state, to, from, toCount );
				break;

			case AFMT_S16_LE:
				ret = copy_from_user_S16_LE_CM( state, to, from, toCount );
				break;
		
			case AFMT_S8:
				ret = copy_from_user_S8_CM( state, to, from, toCount );
				break;
				
			case AFMT_U16_LE:
				ret = copy_from_user_U16_LE_CM( state, to, from, toCount );
				break;
				
			case AFMT_S32_BLOCKED:
			default:
				break;
		}
	}
	else
	{
		switch( state->output_stream->audio_format )
		{
			case AFMT_U8:
				ret = copy_from_user_U8( state, to, from, toCount );
				break;

			case AFMT_S16_LE:
				ret = copy_from_user_S16_LE( state, to, from, toCount );
				break;
		
			case AFMT_S8:
				ret = copy_from_user_S8( state, to, from, toCount );
				break;
				
			case AFMT_U16_LE:
				ret = copy_from_user_U16_LE( state, to, from, toCount );
				break;
				
			case AFMT_S32_BLOCKED:
			default:
				ret = copy_from_user_U32( state, to, from, toCount );
				break;
		}
	}
	
	if( ret == -EFAULT )
	{
		DPRINTK( "copy_n_convert: get_user failed!!!!\n" );
	}
	return ret;
}

static int audio_ioctl(struct inode *inode, struct file *file,
		       uint cmd, ulong arg)
{
	audio_state_t *state = (audio_state_t *)file->private_data;
	audio_stream_t *os = state->output_stream;
	audio_stream_t *is = state->input_stream;
	long val;

	/* dispatch based on command */
	switch (cmd) 
	{
		case SNDCTL_DSP_STEREO:
			DPRINTK_IOCTL("audio_ioctl - SNDCTL_DSP_STEREO\n");
			if (get_user(val, (int *)arg))
			{
				return -EFAULT;
			}
			if (val > 1 || val < 0)
			{
				return -EINVAL;
			}
			audio_set_num_channels( os, val + 1 );
			return put_user( (os->audio_num_channels-1), (long *) arg);

		case SNDCTL_DSP_CHANNELS:
			DPRINTK_IOCTL("audio_ioctl - SNDCTL_DSP_CHANNELS\n");
			if (get_user(val, (int *)arg))
			{
				return -EFAULT;
			}
			if (val > 2 || val < 0)
			{
				return -EINVAL;
			}
			audio_set_num_channels( os, val);
			return put_user( os->audio_num_channels, (long *) arg);

		case SOUND_PCM_READ_CHANNELS:
			DPRINTK_IOCTL("audio_ioctl - SOUND_PCM_READ_CHANNELS\n");
			return put_user( os->audio_num_channels, (long *) arg);

		case SNDCTL_DSP_SETFMT:
			DPRINTK_IOCTL("audio_ioctl - SNDCTL_DSP_SETFMT\n");
			if (get_user(val, (int *)arg))
			{
				return -EFAULT;
			}
			
			/*
			 * Do the set format for audio streaming.
			 */
			audio_set_format( os, val);
			 
			/*
			 * If the audio hardware needs some setup, do it.
			 */
			if( state->set_hw_serial_format )
			{
				state->set_hw_serial_format( os->audio_format );
			}

			return put_user( os->audio_format, (long *) arg);

		case SNDCTL_DSP_GETFMTS:
			DPRINTK_IOCTL("audio_ioctl - SNDCTL_DSP_GETFMTS\n");
			return put_user( (AFMT_U8 | AFMT_S8 | AFMT_S16_LE | AFMT_U16_LE), 
							 (long *) arg);

		case OSS_GETVERSION:
			DPRINTK_IOCTL("audio_ioctl - OSS_GETVERSION\n");
			return put_user(SOUND_VERSION, (int *)arg);

		case SNDCTL_DSP_GETBLKSIZE:
			DPRINTK_IOCTL("audio_ioctl - SNDCTL_DSP_GETBLKSIZE\n");
			if (file->f_mode & FMODE_WRITE)
			{
				return put_user(os->fragsize, (int *)arg);
			}
			return put_user(is->fragsize, (int *)arg);
			
		case SNDCTL_DSP_GETCAPS:
			DPRINTK_IOCTL("audio_ioctl - SNDCTL_DSP_GETCAPS\n");
			/*
			 * Note that we are not capable of precisely reporting
			 * dma position so we can't claim DSP_CAP_REALTIME.
			 */
		 /*	val = DSP_CAP_REALTIME|DSP_CAP_TRIGGER|DSP_CAP_MMAP; */
			val = DSP_CAP_TRIGGER|DSP_CAP_MMAP;
			if (is && os)
			{
				val |= DSP_CAP_DUPLEX;
			}
			return put_user(val, (int *)arg);

		case SNDCTL_DSP_SETFRAGMENT:
			DPRINTK_IOCTL("audio_ioctl - SNDCTL_DSP_SETFRAGMENT\n");
			if (get_user(val, (long *) arg))
			{
				return -EFAULT;
			}
			
			if (file->f_mode & FMODE_READ) 
			{
				int ret = audio_set_fragments(state, is, val);
				if (ret < 0)
				{
					return ret;
				}
				
				ret = put_user(ret, (int *)arg);
				if (ret)
				{
					return ret;
				}
			}
			
			if (file->f_mode & FMODE_WRITE) 
			{
				int ret = audio_set_fragments(state, os, val);
				if (ret < 0)
				{
					return ret;
				}
				
				ret = put_user(ret, (int *)arg);
				if (ret)
				{
					return ret;
				}
			}
			return 0;

		case SNDCTL_DSP_SYNC:
			DPRINTK_IOCTL("audio_ioctl - SNDCTL_DSP_SYNC\n");
			return audio_sync(file);

		case SNDCTL_DSP_SETDUPLEX:
			DPRINTK_IOCTL("audio_ioctl - SNDCTL_DSP_SETDUPLEX\n");
			return 0;

		case SNDCTL_DSP_POST:
			DPRINTK_IOCTL("audio_ioctl - SNDCTL_DSP_POST\n");
			return 0;

		case SNDCTL_DSP_GETTRIGGER:
			DPRINTK_IOCTL("audio_ioctl - SNDCTL_DSP_GETTRIGGER\n");
			val = 0;
			
			if (file->f_mode & FMODE_READ && is->active && !is->stopped)
				val |= PCM_ENABLE_INPUT;
				
			if (file->f_mode & FMODE_WRITE && os->active && !os->stopped)
				val |= PCM_ENABLE_OUTPUT;
				
			return put_user(val, (int *)arg);

		case SNDCTL_DSP_SETTRIGGER:
			DPRINTK_IOCTL("audio_ioctl - SNDCTL_DSP_SETTRIGGER\n");
			if (get_user(val, (int *)arg))
				return -EFAULT;
				
			if (file->f_mode & FMODE_READ) 
			{
				if (val & PCM_ENABLE_INPUT) 
				{
					if (!is->active) 
					{
						if (!is->buffers && audio_allocate_buffers( state, is))
							return -ENOMEM;
						audio_prime_dma( state, is);
					}
					
					if (is->stopped) 
					{
						is->stopped = 0;
						os->active = 1;
						audio_dma_start( state, is );
					}
				} 
				else 
				{
					audio_dma_pause( state, is );
					is->stopped = 1;
					is->active = 0;
				}
			}
			
			if (file->f_mode & FMODE_WRITE) 
			{
				if (val & PCM_ENABLE_OUTPUT) 
				{
					if (!os->active) 
					{
						if (!os->buffers && audio_allocate_buffers( state, os))
							return -ENOMEM;
							
						if (os->mapped)
							audio_prime_dma( state, os );
					}
					
					if (os->stopped) 
					{
						os->stopped = 0;
						os->active = 1;
						audio_dma_start( state, os );
					}
				} 
				else 
				{
					audio_dma_pause( state, os );
					os->stopped = 1;
					os->active = 0;
				}
			}
			return 0;

		case SNDCTL_DSP_GETOPTR:
		case SNDCTL_DSP_GETIPTR:
		{
			count_info inf = { 0, };
			audio_stream_t *s = (cmd == SNDCTL_DSP_GETOPTR) ? os : is;
			int bytecount, flags;
			unsigned int buf_id, total;

			DPRINTK_IOCTL("audio_ioctl - called SNDCTL_DSP_GETOPTR.\n");

			if ((s == is && !(file->f_mode & FMODE_READ)) ||
			    (s == os && !(file->f_mode & FMODE_WRITE)))
			{
				return -EINVAL;
			}	
				
			if( !s->active ) 
			{
				return copy_to_user((void *)arg, &inf, sizeof(inf));
			}
			
		   	save_flags_cli(flags);
			
			/*
			 * Our DMA block doesn't update its buffer position regs unless
			 * it is stopped.  So we don't calculate an offset.  All we know
			 * is what buf is currently being played.
			 */
			if( (ep93xx_dma_get_position(s->dmahandles[0], &buf_id, &total) == 0) &&
				(buf_id != -1) )
			{
				inf.ptr = buf_id * s->fragsize;
			} 

			bytecount = s->bytecount;
			s->getptrCount = s->bytecount;	/* so poll can tell if it changes */

			inf.blocks = s->fragcount;
			s->fragcount = 0;
			
			restore_flags(flags);
			
			if (bytecount < 0)
			{
				bytecount = 0;
			}
			
			inf.bytes = bytecount;

#ifdef I2S_SCALE_BUFFERS
			inf.blocks = inf.blocks * os->reportedfrags / os->nbfrags;
			inf.bytes = inf.bytes * os->reportedfrags / os->nbfrags;
#endif				

			return copy_to_user((void *)arg, &inf, sizeof(inf));
		}
			
		case SNDCTL_DSP_GETOSPACE:
		    {
				/*
				 * It looks like this code is assuming that nobody
				 * is writing the device while calling this ioctl
				 * which may be a reasonable assumption.  I hope.
				 * Maybe.
				 */
				audio_buf_info inf = { 0, };
				int i;

				DPRINTK_IOCTL("audio_ioctl - SNDCTL_DSP_GETOSPACE\n");

				if (!(file->f_mode & FMODE_WRITE))
					return -EINVAL;
				if (!os->buffers && audio_allocate_buffers( state, os))
					return -ENOMEM;
					
				for (i = 0; i < os->nbfrags; i++) 
				{
					/*
					 * If the semaphore is 1, the buffer is available.
					 */
					if (atomic_read(&os->buffers[i].sem.count) > 0) 
					{
						if (os->buffers[i].size == 0)
						{
							inf.fragments++;
						}
						
						inf.bytes += os->fragsize - os->buffers[i].size;
					}
				}
				
				inf.fragsize = os->fragsize;
				inf.fragstotal = os->reportedfrags;

#ifdef I2S_SCALE_BUFFERS
				inf.fragments = inf.fragments * os->reportedfrags / os->nbfrags;
				inf.bytes = inf.bytes * os->reportedfrags / os->nbfrags;
#endif				
			
				return copy_to_user((void *)arg, &inf, sizeof(inf));
		    }

		case SNDCTL_DSP_GETISPACE:
		    {
				audio_buf_info inf = { 0, };
				int i;

			 	DPRINTK_IOCTL("audio_ioctl - SNDCTL_DSP_GETISPACE\n");

				if (!(file->f_mode & FMODE_READ))
					return -EINVAL;
				if (!is->buffers && audio_allocate_buffers( state, is))
					return -ENOMEM;
					
				for (i = 0; i < is->nbfrags; i++) 
				{
					if (atomic_read(&is->buffers[i].sem.count) > 0) 
					{
						if (is->buffers[i].size == is->fragsize)
							inf.fragments++;
						
						inf.bytes += is->buffers[i].size;
					}
				}
				
				inf.fragsize = is->fragsize;
				inf.fragstotal = is->reportedfrags;

#ifdef I2S_SCALE_BUFFERS
				inf.fragments = inf.fragments * is->reportedfrags / is->nbfrags;
				inf.bytes = inf.bytes * is->reportedfrags / is->nbfrags;
#endif				

				return copy_to_user((void *)arg, &inf, sizeof(inf));
		    }

		case SNDCTL_DSP_NONBLOCK:
			DPRINTK_IOCTL("audio_ioctl - SNDCTL_DSP_NONBLOCK\n");
			file->f_flags |= O_NONBLOCK;
			return 0;

		case SNDCTL_DSP_RESET:
			DPRINTK_IOCTL("audio_ioctl - SNDCTL_DSP_RESET\n");
			if (file->f_mode & FMODE_READ) 
			{
				audio_reset_buffers(state,is);
			}
			
			if (file->f_mode & FMODE_WRITE) 
			{
				audio_reset_buffers(state,os);
			}
			return 0;

		default:
			/*
			 * Let the client of this module handle the
			 * non generic ioctls
			 */
			return state->client_ioctl(inode, file, cmd, arg);
	}

	return 0;
}


static int audio_release(struct inode *inode, struct file *file)
{
	int i;
	audio_state_t *state = (audio_state_t *)file->private_data;

	DPRINTK("EP93xx - audio_release -- enter.  Write=%d  Read=%d\n",
			((file->f_mode & FMODE_WRITE) != 0),
			((file->f_mode & FMODE_READ) != 0)	);

	down(&state->sem);

	if (file->f_mode & FMODE_READ) 
	{
		audio_deallocate_buffers(state, state->input_stream);
		
		for( i=0 ; i < state->input_stream->NumDmaChannels ; i++ )
		{
			ep93xx_dma_free( state->input_stream->dmahandles[i] );
		}
		state->rd_ref = 0;
	}

	if (file->f_mode & FMODE_WRITE) 
	{
		/*
		 * Wait for all our buffers to play.
		 */
		audio_sync(file);
		
		/*
		 * audio_deallocate_buffers will pause and flush the dma
		 * then deallocate the buffers.
		 */
		audio_deallocate_buffers(state, state->output_stream);
		for( i=0 ; i < state->output_stream->NumDmaChannels ; i++ )
		{
			ep93xx_dma_free( state->output_stream->dmahandles[i] );
		}
		state->wr_ref = 0;
	}

	up(&state->sem);

	DPRINTK("EP93xx - audio_release -- EXIT\n");

	return 0;
}

/*
 *	ep93xx_audio_attach
 *
 *  Initialize the state structure for read or write.  Allocate dma channels.
 *  Default is for 2 channel (stereo) operation.
 *
 */
int ep93xx_audio_attach
(
	struct inode *inode, 
	struct file *file,
	audio_state_t *state
)
{
	int err;
	audio_stream_t *os = state->output_stream;
	audio_stream_t *is = state->input_stream;

	DPRINTK("ep93xx_audio_attach -- enter.  Write=%d  Read=%d\n",
			((file->f_mode & FMODE_WRITE) != 0),
			((file->f_mode & FMODE_READ) != 0)	);

	down(&state->sem);

	/* access control */
	if( ((file->f_mode & FMODE_WRITE) && !os) ||
	    ((file->f_mode & FMODE_READ) && !is) )
	{
		up(&state->sem);
		return -ENODEV;
	}

	if( ((file->f_mode & FMODE_WRITE) && state->wr_ref) ||
	    ((file->f_mode & FMODE_READ) && state->rd_ref) )
	{
		up(&state->sem);
		return -EBUSY;
	}

	/* 
	 * Request DMA channels 
	 */
	if( file->f_mode & FMODE_WRITE ) 
	{
		DPRINTK("ep93xx_audio_attach -- FMODE_WRITE\n");
		strncpy( os->devicename, "I2S_Tx1", MAX_DEVICE_NAME );
	
		err = ep93xx_dma_request( 	&os->dmahandles[0], 
									os->devicename,
                       				state->output_dma );
		if (err)
		{
			up(&state->sem);
			DPRINTK("ep93xx_audio_attach -- EXIT ERROR dma request failed\n");
			return err;
		}

		err = ep93xx_dma_config( 	os->dmahandles[0], 
									IGNORE_CHANNEL_ERROR, 
									0,
									audio_dma_tx_callback, 
									(unsigned int)state );
		if (err) 
		{
			ep93xx_dma_free( os->dmahandles[0] );
			up(&state->sem);
			DPRINTK("ep93xx_audio_attach -- EXIT ERROR dma config failed\n");
			return err;
		}

		os->NumDmaChannels = 1;
		state->wr_ref = 1;
		audio_deallocate_buffers(state,os);

		os->fragsize 	= AUDIO_FRAGSIZE_DEFAULT;
		os->nbfrags  	= AUDIO_NBFRAGS_DEFAULT;

		os->reportedfrags = scale_to_expanded_samples( state, AUDIO_NBFRAGS_DEFAULT );

		os->mapped 	= 0;

		init_waitqueue_head(&os->wq);
	}
	
	if( file->f_mode & FMODE_READ ) 
	{
		strncpy( is->devicename, "I2S_Rx1", MAX_DEVICE_NAME );
	
		err = ep93xx_dma_request( 	&is->dmahandles[0], 
									is->devicename,
                       				state->input_dma);
		if (err) 
		{
			up(&state->sem);
			DPRINTK("ep93xx_audio_attach -- EXIT ERROR dma request failed\n");
			return err;
		}

		err = ep93xx_dma_config( 	is->dmahandles[0], 
									IGNORE_CHANNEL_ERROR, 
									0,
									audio_dma_rx_callback, 
									(unsigned int)state );
		if (err) 
		{
			ep93xx_dma_free( is->dmahandles[0] );
			up(&state->sem);
			DPRINTK("ep93xx_audio_attach -- EXIT ERROR dma config failed\n");
			return err;
		}

		is->NumDmaChannels = 1;
		state->rd_ref = 1;
		audio_deallocate_buffers(state,is);
		is->fragsize 	= AUDIO_FRAGSIZE_DEFAULT;
		is->nbfrags 	= AUDIO_NBFRAGS_DEFAULT;

		os->reportedfrags = scale_to_expanded_samples( state, AUDIO_NBFRAGS_DEFAULT );

		is->mapped 	= 0;

		init_waitqueue_head(&is->wq);
	}

	/*
	 * Fill out the rest of the file_operations struct.
	 */
	file->private_data	= state;
	file->f_op->release	= audio_release;
	file->f_op->write	= audio_write;
	file->f_op->read	= audio_read;
	file->f_op->mmap	= audio_mmap;
	file->f_op->poll	= audio_poll;
	file->f_op->ioctl	= audio_ioctl;
	file->f_op->llseek	= no_llseek;

	up(&state->sem);

	DPRINTK("ep93xx_audio_attach -- EXIT\n");

	return 0;
}

EXPORT_SYMBOL(ep93xx_audio_attach);

MODULE_DESCRIPTION("Common audio handling for the Cirrus EP93xx processor");
MODULE_LICENSE("GPL");
