/*
 * YAFFS: Yet another FFS. A NAND-flash specific file system.
 * yaffs_fs.c
 *
 * Copyright (C) 2002 Aleph One Ltd.
 *   for Toby Churchill Ltd and Brightstar Engineering
 *
 * Created by Charles Manning <charles@aleph1.co.uk>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * This is the file system front-end to YAFFS that hooks it up to
 * the VFS.
 *
 * Special notes: 
 * >> sb->u.generic_sbp points to the yaffs_Device associated with this superblock
 * >> inode->u.generic_ip points to the associated yaffs_Object.
 *
 *
 * Acknowledgements:
 * * Luc van OostenRyck for numerous patches.
 * * Nick Bane for numerous patches.
 * * Nick Bane for 2.5/2.6 integration.
 * * Andras Toth for mknod rdev issue.
 * * Michael Fischer for finding the problem with inode inconsistency.
 * * Some code bodily lifted from JFFS2.
 */


const char *yaffs_fs_c_version = "$Id: yaffs_fs.c,v 1.3 2005/01/25 00:38:25 eddie Exp $";
extern const char *yaffs_guts_c_version;


#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/slab.h>
#include <linux/init.h>
#include <linux/list.h>
#include <linux/fs.h>
#include <linux/proc_fs.h>
#include <linux/pagemap.h>
#include <linux/mtd/mtd.h>
#include <linux/interrupt.h>
#include <linux/string.h>


#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,5,0))

#include <linux/statfs.h>	/* Added NCB 15-8-2003 */
#include <asm/statfs.h>
#define UnlockPage(p) unlock_page(p)
#define Page_Uptodate(page)	test_bit(PG_uptodate, &(page)->flags)
//#define kdevname(x) cdevname(to_kdev_t(x))
#define kdevname(x) "(unavailable)"	// temporary fix

#else

#include <linux/locks.h>

#endif


#include <asm/uaccess.h>

#include "yportenv.h"
#include "yaffs_guts.h"




unsigned yaffs_traceMask = YAFFS_TRACE_ALWAYS | YAFFS_TRACE_BAD_BLOCKS;

#ifdef CONFIG_YAFFS_RAM_ENABLED
#include "yaffs_nandemul.h" 
// 2 MB of RAM for emulation
#define YAFFS_RAM_EMULATION_SIZE  0x200000
#endif //CONFIG_YAFFS_RAM_ENABLED

#ifdef CONFIG_YAFFS_MTD_ENABLED
#include <linux/mtd/mtd.h>
#include "yaffs_mtdif.h"
#endif //CONFIG_YAFFS_MTD_ENABLED

//#define T(x) printk x



#define yaffs_InodeToObject(iptr) ((yaffs_Object *)((iptr)->u.generic_ip))
#define yaffs_DentryToObject(dptr) yaffs_InodeToObject((dptr)->d_inode)
//NCB #define yaffs_SuperToDevice(sb)	((yaffs_Device *)sb->u.generic_sbp)

//#if defined(CONFIG_KERNEL_2_5)
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,5,0))
#define yaffs_SuperToDevice(sb)	((yaffs_Device *)sb->s_fs_info)
#else
#define yaffs_SuperToDevice(sb)	((yaffs_Device *)sb->u.generic_sbp)
#endif


static void yaffs_put_super(struct super_block *sb);

static ssize_t yaffs_file_read(struct file *f, char *buf, size_t n, loff_t *pos);
static ssize_t yaffs_file_write(struct file *f, const char *buf, size_t n, loff_t *pos);
static int yaffs_file_flush(struct file* file);

static int yaffs_sync_object(struct file * file, struct dentry *dentry, int datasync);

static int yaffs_readdir(struct file *f, void *dirent, filldir_t filldir);

//#if defined(CONFIG_KERNEL_2_5)	/* Added NCB 185-8-2003 */
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,5,0))
static int yaffs_create(struct inode *dir, struct dentry *dentry, int mode, struct nameidata *n);
static struct dentry * yaffs_lookup(struct inode *dir, struct dentry *dentry, struct nameidata *n);
#else
static int yaffs_create(struct inode *dir, struct dentry *dentry, int mode);
static struct dentry * yaffs_lookup(struct inode *dir, struct dentry *dentry);
#endif
static int yaffs_link(struct dentry *old_dentry, struct inode * dir, struct dentry * dentry);
static int yaffs_unlink(struct inode * dir, struct dentry *dentry);
static int yaffs_symlink(struct inode * dir, struct dentry *dentry, const char * symname);
static int yaffs_mkdir(struct inode * dir, struct dentry * dentry, int mode);

//#if defined(CONFIG_KERNEL_2_5)
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,5,0))
static int yaffs_mknod(struct inode *dir, struct dentry *dentry, int mode, dev_t dev);
#else
static int yaffs_mknod(struct inode *dir, struct dentry *dentry, int mode, int dev);
#endif
static int yaffs_rename(struct inode * old_dir, struct dentry *old_dentry, struct inode * new_dir,struct dentry *new_dentry);
static int yaffs_setattr(struct dentry *dentry, struct iattr *attr);

//#if defined(CONFIG_KERNEL_2_5)	/* Added NCB 185-8-2003 */
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,5,0))
static int yaffs_statfs(struct super_block *sb, struct kstatfs *buf);
#else
static int yaffs_statfs(struct super_block *sb, struct statfs *buf);
#endif
static void yaffs_read_inode (struct inode *inode);

//#if defined(CONFIG_KERNEL_2_5)
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,5,0))
static struct super_block *yaffs_read_super(struct file_system_type * fs, int flags, const char *dev_name, void *data);
#else
static struct super_block *yaffs_read_super(struct super_block * sb, void * data, int silent);
#endif

static void yaffs_put_inode (struct inode *inode);
static void yaffs_delete_inode(struct inode *);
static void yaffs_clear_inode(struct inode *);

static int yaffs_readpage(struct file *file, struct page * page);
static int yaffs_writepage(struct page *page);
static int yaffs_prepare_write(struct file *f, struct page *pg, unsigned offset, unsigned to);
static int yaffs_commit_write(struct file *f, struct page *pg, unsigned offset, unsigned to);

static int yaffs_readlink(struct dentry *dentry, char *buffer, int buflen);
static int yaffs_follow_link(struct dentry *dentry, struct nameidata *nd);




static struct address_space_operations yaffs_file_address_operations = {
	readpage:		yaffs_readpage,
	writepage:		yaffs_writepage,
	prepare_write:	yaffs_prepare_write,
	commit_write:	yaffs_commit_write
};


static struct file_operations yaffs_file_operations = {
#ifdef CONFIG_YAFFS_USE_GENERIC_RW
	read:		generic_file_read,
	write:		generic_file_write,
#else
	read:		yaffs_file_read,
	write:		yaffs_file_write,
#endif
	mmap:		generic_file_mmap,
	flush:		yaffs_file_flush,
	fsync:		yaffs_sync_object,
	
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,5,0))
	sendfile:	generic_file_sendfile,
#endif
};


static struct inode_operations yaffs_file_inode_operations = {
	setattr:	yaffs_setattr,
};


struct inode_operations yaffs_symlink_inode_operations =
{	
	readlink:	yaffs_readlink,
	follow_link:	yaffs_follow_link,
	setattr:	yaffs_setattr
};

static struct inode_operations yaffs_dir_inode_operations = {
	create:		yaffs_create,
	lookup:		yaffs_lookup,
	link:		yaffs_link,
	unlink:		yaffs_unlink,	
	symlink:	yaffs_symlink,
	mkdir:		yaffs_mkdir,
	rmdir:		yaffs_unlink,
	mknod:		yaffs_mknod,
	rename:		yaffs_rename,
	setattr:	yaffs_setattr,
};

static struct file_operations yaffs_dir_operations = {
	read:		generic_read_dir,
	readdir:	yaffs_readdir,
	fsync:		yaffs_sync_object,
};


static struct super_operations yaffs_super_ops = {
	statfs:			yaffs_statfs,
	read_inode:		yaffs_read_inode,
	put_inode:		yaffs_put_inode,
	put_super:		yaffs_put_super,
//	remount_fs:
	delete_inode:		yaffs_delete_inode,
	clear_inode:		yaffs_clear_inode,
};



static void yaffs_GrossLock(yaffs_Device *dev)
{
	T(YAFFS_TRACE_OS,("yaffs locking\n"));

	down(&dev->grossLock);
}

static void yaffs_GrossUnlock(yaffs_Device *dev)
{
	T(YAFFS_TRACE_OS,("yaffs unlocking\n"));
	up(&dev->grossLock);

}

static int yaffs_readlink(struct dentry *dentry, char *buffer, int buflen)
{
	unsigned char *alias;
	int ret;

	yaffs_Device *dev = yaffs_DentryToObject(dentry)->myDev;


	yaffs_GrossLock(dev);
	
	alias = yaffs_GetSymlinkAlias(yaffs_DentryToObject(dentry));
	
	yaffs_GrossUnlock(dev);
	
	if(!alias)
		return -ENOMEM;

	ret = vfs_readlink(dentry, buffer, buflen, alias);
	kfree(alias);
	return ret;
}

static int yaffs_follow_link(struct dentry *dentry, struct nameidata *nd)
{
	unsigned char *alias;
	int ret;
	yaffs_Device *dev = yaffs_DentryToObject(dentry)->myDev;


	yaffs_GrossLock(dev);

	alias = yaffs_GetSymlinkAlias(yaffs_DentryToObject(dentry));
	
	yaffs_GrossUnlock(dev);
	
	if(!alias)
		return -ENOMEM;

	ret = vfs_follow_link(nd,alias);
	kfree(alias);
	return ret;
}


struct inode *yaffs_get_inode(struct super_block *sb, int mode, int dev,yaffs_Object *obj);

/*
 * Lookup is used to find objects in the fs
 */
//#if defined(CONFIG_KERNEL_2_5)	/* Added NCB 185-8-2003 */
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,5,0))

static struct dentry * yaffs_lookup(struct inode *dir, struct dentry *dentry, struct nameidata *n)
#else
static struct dentry * yaffs_lookup(struct inode *dir, struct dentry *dentry)
#endif
{
	yaffs_Object *obj;
	struct inode *inode = NULL; // NCB 2.5/2.6 needs NULL here
	
	yaffs_Device *dev = yaffs_InodeToObject(dir)->myDev;


	yaffs_GrossLock(dev);

	
	T(YAFFS_TRACE_OS,("yaffs_lookup for %d:%s\n",yaffs_InodeToObject(dir)->objectId,dentry->d_name.name));
	
	obj = yaffs_FindObjectByName(yaffs_InodeToObject(dir),dentry->d_name.name);
	
	obj = yaffs_GetEquivalentObject(obj); // in case it was a hardlink
	

	
	if(obj)
	{
		T(YAFFS_TRACE_OS,("yaffs_lookup found %d\n",obj->objectId));
		
		inode = yaffs_get_inode(dir->i_sb, obj->st_mode,0,obj);
		
		if(inode)
		{
			T(YAFFS_TRACE_OS,("yaffs_loookup dentry \n"));
/* #if 0 asserted by NCB for 2.5/6 compatability - falls through to d_add even if NULL inode */	
#if 0
			//dget(dentry); // try to solve directory bug
			d_add(dentry,inode);
			
			yaffs_GrossUnlock(dev);

			// return dentry;
			return NULL;
#endif
		}

	}
	else
	{
		T(YAFFS_TRACE_OS,("yaffs_lookup not found\n"));
		
	}
	yaffs_GrossUnlock(dev);

/* added NCB for 2.5/6 compatability - forces add even if inode is NULL which creates dentry hash*/	
	d_add(dentry,inode);
	
	return NULL;
	//	return (ERR_PTR(-EIO));
	
}

// For now put inode is just for debugging
// Put inode is called when the inode **structure** is put.
static void yaffs_put_inode(struct inode *inode)
{
	T(YAFFS_TRACE_OS,("yaffs_put_inode: ino %d, count %d nlink %d\n",
	                       (int)inode->i_ino, atomic_read(&inode->i_count), inode->i_nlink));
	
}

// clear is called to tell the fs to release any per-inode data it holds
static void yaffs_clear_inode(struct inode *inode)
{
	yaffs_Object *obj;
	yaffs_Device *dev;
	
	obj = yaffs_InodeToObject(inode);
	
	T(YAFFS_TRACE_OS,("yaffs_clear_inode: ino %d, count %d %s\n",(int)inode->i_ino, atomic_read(&inode->i_count),
		obj ? "object exists" : "null object"));	

	if(obj)
	{
		dev = obj->myDev;
		yaffs_GrossLock(dev);
		
		// Clear the association between the inode ant the yaffs_Object.
		obj->myInode = NULL;
		inode->u.generic_ip = NULL;
		
		// If the object freeing was deferred, then the real free happens now.
		// This should fix the inode inconsistency problem.
		
		yaffs_HandleDeferedFree(obj);
		
		yaffs_GrossUnlock(dev);
	}
	
	
}

// delete is called when the link count is zero and the inode
// is put (ie. nobody wants to know about it anymore, time to
// delete the file).
// NB Must call clear_inode()
static void yaffs_delete_inode(struct inode *inode)
{
	yaffs_Object *obj = yaffs_InodeToObject(inode);
	yaffs_Device *dev;

	T(YAFFS_TRACE_OS,("yaffs_delete_inode: ino %d, count %d %s\n",(int)inode->i_ino, atomic_read(&inode->i_count),
		obj ? "object exists" : "null object"));
	
	if(obj)
	{
	 	dev = obj->myDev;
		yaffs_GrossLock(dev);
		yaffs_DeleteFile(obj);
		yaffs_GrossUnlock(dev);
	}
	clear_inode(inode);
}


static int yaffs_file_flush(struct file* file)
{
	yaffs_Object *obj = yaffs_DentryToObject(file->f_dentry);
	
	yaffs_Device *dev = obj->myDev;
	
	T(YAFFS_TRACE_OS,("yaffs_file_flush object %d (%s)\n",obj->objectId,
				obj->dirty ? "dirty" : "clean"));

	yaffs_GrossLock(dev);
	
    yaffs_FlushFile(obj,1);

	yaffs_GrossUnlock(dev);

    return 0;
}



static int yaffs_readpage_nolock(struct file *f, struct page * pg)
{
	// Lifted from jffs2
	
	yaffs_Object *obj;
	unsigned char *pg_buf;
	int ret;

	yaffs_Device *dev;

	T(YAFFS_TRACE_OS,("yaffs_readpage at %08x, size %08x\n",
	              (unsigned)(pg->index << PAGE_CACHE_SHIFT), (unsigned)PAGE_CACHE_SIZE));

	obj  = yaffs_DentryToObject(f->f_dentry);

	dev = obj->myDev;
	
	
	if (!PageLocked(pg))
                PAGE_BUG(pg);

	pg_buf = kmap(pg);
	/* FIXME: Can kmap fail? */

	yaffs_GrossLock(dev);
	
	ret = yaffs_ReadDataFromFile(obj, pg_buf, pg->index << PAGE_CACHE_SHIFT, PAGE_CACHE_SIZE);

	yaffs_GrossUnlock(dev);
	
	if(ret >= 0) ret = 0;

	if (ret) {
		ClearPageUptodate(pg);
		SetPageError(pg);
	} else {
		SetPageUptodate(pg);
		ClearPageError(pg);
	}

	flush_dcache_page(pg);
	kunmap(pg);


	T(YAFFS_TRACE_OS,("yaffs_readpage done\n"));
	return ret;
}

static int yaffs_readpage_unlock(struct file *f, struct page *pg)
{
	int ret = yaffs_readpage_nolock(f,pg);
	UnlockPage(pg);
	return ret;
}

static int yaffs_readpage(struct file *f, struct page * pg)
{
	return yaffs_readpage_unlock(f,pg);
}

// writepage inspired by/stolen from smbfs
//

static int yaffs_writepage(struct page *page)
{
	struct address_space *mapping = page->mapping;
	struct inode *inode;
	unsigned long end_index;
	char *buffer;
	yaffs_Object *obj;
	int nWritten = 0;
	unsigned nBytes;

	if (!mapping)
		BUG();
	inode = mapping->host;
	if (!inode)
		BUG();

	end_index = inode->i_size >> PAGE_CACHE_SHIFT;

	/* easy case */
	if (page->index < end_index)
	{
		nBytes = PAGE_CACHE_SIZE;
	}
	else
	{
		nBytes = inode->i_size & (PAGE_CACHE_SIZE-1);
	}
	//  What's happening here?
	///* OK, are we completely out? */
	//if (page->index >= end_index+1 || !offset)
	//	return -EIO;

	get_page(page);


	buffer = kmap(page);

	obj = yaffs_InodeToObject(inode);
	yaffs_GrossLock(obj->myDev);


	nWritten = yaffs_WriteDataToFile(obj,buffer,page->index << PAGE_CACHE_SHIFT,nBytes);

	yaffs_GrossUnlock(obj->myDev);
	
	kunmap(page);
	SetPageUptodate(page);
	UnlockPage(page);
	put_page(page);

	return (nWritten == nBytes) ? 0  : -ENOSPC;
}



static int yaffs_prepare_write(struct file *f, struct page *pg, unsigned offset, unsigned to)
{

	T(YAFFS_TRACE_OS,("yaffs_prepair_write\n"));
	if(!Page_Uptodate(pg) && (offset || to < PAGE_CACHE_SIZE))
		return  yaffs_readpage_nolock(f,pg);    

	return 0;
	
}

static int yaffs_commit_write(struct file *f, struct page *pg, unsigned offset, unsigned to)
{

	void *addr = page_address(pg) + offset;
	loff_t pos = (((loff_t)pg->index) << PAGE_CACHE_SHIFT) + offset;
	int nBytes = to - offset;
	int nWritten;
	
	unsigned spos = pos;
	unsigned saddr = (unsigned)addr;

	T(YAFFS_TRACE_OS,("yaffs_commit_write addr %x pos %x nBytes %d\n",saddr,spos,nBytes));
	
	nWritten = yaffs_file_write(f,addr, nBytes, &pos);
	
	if(nWritten != nBytes)
	{
		T(YAFFS_TRACE_OS,("yaffs_commit_write not same size nWritten %d  nBytes %d\n",nWritten,nBytes));
		SetPageError(pg);
		ClearPageUptodate(pg);
	}
	else
	{
		SetPageUptodate(pg);
	}

	T(YAFFS_TRACE_OS,("yaffs_commit_write returning %d\n",nWritten));
	
	return nWritten;

}



static void yaffs_FillInodeFromObject(struct inode *inode, yaffs_Object *obj)
{
	if (inode && obj) 
	{
		inode->i_ino = obj->objectId;
		inode->i_mode = obj->st_mode;
		inode->i_uid = obj->st_uid;
		inode->i_gid = obj->st_gid;
		inode->i_blksize = inode->i_sb->s_blocksize;
//#if defined(CONFIG_KERNEL_2_5)
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,5,0))

		inode->i_rdev = old_decode_dev(obj->st_rdev);
		inode->i_atime.tv_sec = (time_t)(obj->st_atime);
		inode->i_atime.tv_nsec = 0;
		inode->i_mtime.tv_sec = (time_t)obj->st_mtime;
		inode->i_mtime.tv_nsec =0;
		inode->i_ctime.tv_sec = (time_t)obj->st_ctime;
		inode->i_ctime.tv_nsec = 0;
#else
		inode->i_rdev = obj->st_rdev;
		inode->i_atime = obj->st_atime;
		inode->i_mtime = obj->st_mtime;
		inode->i_ctime = obj->st_ctime;
#endif
		inode->i_size = yaffs_GetObjectFileLength(obj);
		inode->i_blocks = (inode->i_size + 511) >> 9;

		inode->i_nlink = yaffs_GetObjectLinkCount(obj);
		
		T(YAFFS_TRACE_OS,("yaffs_FillInode mode %x uid %d gid %d size %d count %d\n",
				inode->i_mode, inode->i_uid, inode->i_gid, (int)inode->i_size, atomic_read(&inode->i_count)));
		
		switch (obj->st_mode & S_IFMT) 
		{
			default: // fifo, device or socket
				init_special_inode(inode, obj->st_mode,(dev_t)(obj->st_rdev));
				break;
			case S_IFREG:	// file		
				inode->i_op = &yaffs_file_inode_operations;
				inode->i_fop = &yaffs_file_operations;
				inode->i_mapping->a_ops = &yaffs_file_address_operations;
				break;
			case S_IFDIR:	// directory
				inode->i_op = &yaffs_dir_inode_operations;
				inode->i_fop = &yaffs_dir_operations;
				break;
			case S_IFLNK:	// symlink
				inode->i_op = &yaffs_symlink_inode_operations;
				break;
		}
		
		
		inode->u.generic_ip = obj;
		obj->myInode = inode;
		
	}
	else
	{
		T(YAFFS_TRACE_OS,("yaffs_FileInode invalid parameters\n"));
	}

}

struct inode *yaffs_get_inode(struct super_block *sb, int mode, int dev,yaffs_Object *obj)
{
	struct inode * inode;
	
	T(YAFFS_TRACE_OS,("yaffs_get_inode for object %d\n",obj->objectId));

	inode = iget(sb,obj->objectId);

	// NB Side effect: iget calls back to yaffs_read_inode().
	// iget also increments the inode's i_count
	
	return inode;
}

static ssize_t yaffs_file_read(struct file *f, char *buf, size_t n, loff_t *pos)
{
	yaffs_Object *obj;
	int nRead,ipos;
	struct inode *inode;
	yaffs_Device *dev;
	
	T(YAFFS_TRACE_OS,("yaffs_file_read\n"));

	obj  = yaffs_DentryToObject(f->f_dentry);
	
	dev = obj->myDev;
	
	yaffs_GrossLock(dev);
	
	inode = f->f_dentry->d_inode;
	
	if (*pos < inode->i_size) 
	{
			if (*pos + n > inode->i_size)
			{
				n = inode->i_size - *pos;
			}
	}
	else
	{
		n = 0;
	}
	
	nRead = yaffs_ReadDataFromFile(obj,buf,*pos,n);
	if(nRead > 0)
	{
		f->f_pos += nRead;
	}
	
	yaffs_GrossUnlock(dev);
	
	ipos = *pos;
	
	T(YAFFS_TRACE_OS,("yaffs_file_read read %d bytes, %d read at %d\n",n,nRead,ipos));
	return nRead;
	
}


static ssize_t yaffs_file_write(struct file *f, const char *buf, size_t n, loff_t *pos)
{
	yaffs_Object *obj;
	int nWritten,ipos;
	struct inode *inode;
	yaffs_Device *dev;
	
	
	obj  = yaffs_DentryToObject(f->f_dentry);
	
	dev = obj->myDev;
	
	yaffs_GrossLock(dev);

	inode = f->f_dentry->d_inode;

	if(!S_ISBLK(inode->i_mode) && f->f_flags & O_APPEND)
	{
		ipos = inode->i_size;
	}
	else
	{
		ipos = *pos;
	}
	
	
	if(!obj)
	{
		T(YAFFS_TRACE_OS,("yaffs_file_write: hey obj is null!\n"));
	}
	else
	{
		T(YAFFS_TRACE_OS,("yaffs_file_write about to write writing %d bytes to object %d at %d\n",n,obj->objectId,ipos));
	}

	nWritten = yaffs_WriteDataToFile(obj,buf,ipos,n);

	T(YAFFS_TRACE_OS,("yaffs_file_write writing %d bytes, %d written at %d\n",n,nWritten,ipos));
	if(nWritten > 0)
	{
		ipos += nWritten;
		*pos = ipos;
		if(ipos > inode->i_size)
		{
			inode->i_size = ipos;
			inode->i_blocks = (ipos + 511)>>9;
			
			T(YAFFS_TRACE_OS,("yaffs_file_write size updated to %d bytes, %d blocks\n",ipos,(int)(inode->i_blocks)));
		}
		
	}
	yaffs_GrossUnlock(dev);
	
	return nWritten != n ? -ENOSPC : nWritten;
}


static int yaffs_readdir(struct file *f, void *dirent, filldir_t filldir)
{
	yaffs_Object *obj;
	yaffs_Device *dev;
	struct inode *inode = f->f_dentry->d_inode;
	unsigned long offset, curoffs;
	struct list_head *i;	
	yaffs_Object *l;
	
	char name[YAFFS_MAX_NAME_LENGTH	+1];
		
	obj =  yaffs_DentryToObject(f->f_dentry);
	dev = obj->myDev;
	
	yaffs_GrossLock(dev);
	
	offset = f->f_pos;
	
	T(YAFFS_TRACE_OS,("yaffs_readdir: starting at %d\n",(int)offset));
	
	if(offset == 0)
	{
        T(YAFFS_TRACE_OS,("yaffs_readdir: entry . ino %d \n",(int)inode->i_ino));
		if(filldir(dirent,".",1,offset,inode->i_ino,DT_DIR) < 0)
		{
			goto out;
		}
		offset++;
		f->f_pos++;
	}
	if(offset == 1)
	{
		T(YAFFS_TRACE_OS,("yaffs_readdir: entry .. ino %d \n",(int)f->f_dentry->d_parent->d_inode->i_ino));
		if(filldir(dirent,"..",2,offset,f->f_dentry->d_parent->d_inode->i_ino,DT_DIR) < 0)
		{
			goto out;
		}
		offset++;
		f->f_pos++;
	}
	
	curoffs = 1;
	
	list_for_each(i,&obj->variant.directoryVariant.children)
	{
		curoffs++;
		if(curoffs >= offset)
		{		
			l = list_entry(i, yaffs_Object,siblings);
			
			yaffs_GetObjectName(l,name,YAFFS_MAX_NAME_LENGTH+1); 
			T(YAFFS_TRACE_OS,("yaffs_readdir: %s inode %d\n",name,yaffs_GetObjectInode(l)));
			
			if(filldir(dirent,
					   name,
					   strlen(name),
					   offset,
					   yaffs_GetObjectInode(l),
					   yaffs_GetObjectType(l))
					   < 0)
			{
				goto up_and_out;
			}
			
			offset++;
			f->f_pos++;	   
		}
	}

  up_and_out:
  out:
  
    yaffs_GrossUnlock(dev);
    
	return 0;
}


/*
 * File creation. Allocate an inode, and we're done..
 */
//#if defined(CONFIG_KERNEL_2_5)
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,5,0))
static int yaffs_mknod(struct inode *dir, struct dentry *dentry, int mode, dev_t rdev)
#else
static int yaffs_mknod(struct inode *dir, struct dentry *dentry, int mode, int rdev)
#endif
{
	struct inode *inode;
	
	yaffs_Object *obj = NULL;
	yaffs_Device *dev;
	
	yaffs_Object *parent = yaffs_InodeToObject(dir);
	
	int error = -ENOSPC;

	if(parent)
	{
		T(YAFFS_TRACE_OS,("yaffs_mknod: parent object %d type %d\n",
					 parent->objectId,parent->variantType));
	}
	else
	{
		T(YAFFS_TRACE_OS,("yaffs_mknod: could not get parent object\n"));
		return -EPERM;
	}
	
	T(YAFFS_TRACE_OS,("yaffs_mknod: making oject for %s, mode %x dev %x\n",
					dentry->d_name.name, mode,rdev));

	dev = parent->myDev;
	
	yaffs_GrossLock(dev);

	switch (mode & S_IFMT) 
	{
		default:
			// Special (socket, fifo, device...)
			T(YAFFS_TRACE_OS,("yaffs_mknod: making special\n"));
			obj = yaffs_MknodSpecial(parent,dentry->d_name.name,mode,current->uid, current->gid,rdev);
			break;
		case S_IFREG:	// file		
			T(YAFFS_TRACE_OS,("yaffs_mknod: making file\n"));
			obj = yaffs_MknodFile(parent,dentry->d_name.name,mode,current->uid, current->gid);
			break;
		case S_IFDIR:	// directory
			T(YAFFS_TRACE_OS,("yaffs_mknod: making directory\n"));
			obj = yaffs_MknodDirectory(parent,dentry->d_name.name,mode,current->uid, current->gid);
			break;
		case S_IFLNK:	// symlink
			T(YAFFS_TRACE_OS,("yaffs_mknod: making file\n"));
			obj = NULL; // Do we ever get here?
			break;
	}
	
	if(obj)
	{
		inode = yaffs_get_inode(dir->i_sb, mode, rdev, obj);
		d_instantiate(dentry, inode);
		T(YAFFS_TRACE_OS,("yaffs_mknod created object %d count = %d\n",obj->objectId,atomic_read(&inode->i_count)));
		error = 0;
	}
	else
	{
		T(YAFFS_TRACE_OS,("yaffs_mknod failed making object\n"));
		error = -ENOMEM;
	}

	yaffs_GrossUnlock(dev);

	return error;
}

static int yaffs_mkdir(struct inode * dir, struct dentry * dentry, int mode)
{
	int retVal;
	T(YAFFS_TRACE_OS,("yaffs_mkdir\n"));
	retVal =  yaffs_mknod(dir, dentry, mode | S_IFDIR, 0);
#if 0
 // attempt to fix dir bug - didn't work
	if(!retVal)
	{
		dget(dentry);
	}
#endif
	return retVal;
}

//#if defined(CONFIG_KERNEL_2_5)	/* Added NCB 185-8-2003 */
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,5,0))
static int yaffs_create(struct inode *dir, struct dentry *dentry, int mode, struct nameidata *n)
#else
static int yaffs_create(struct inode *dir, struct dentry *dentry, int mode)
#endif
{
	T(YAFFS_TRACE_OS,("yaffs_create\n"));
	return yaffs_mknod(dir, dentry, mode | S_IFREG, 0);
}


static int yaffs_unlink(struct inode * dir, struct dentry *dentry)
{
	int retVal;
	
	yaffs_Device *dev;
	
	
	T(YAFFS_TRACE_OS,("yaffs_unlink %d:%s\n",(int)(dir->i_ino),dentry->d_name.name));
	
	dev = yaffs_InodeToObject(dir)->myDev;
	
	yaffs_GrossLock(dev);
	
	
	retVal = yaffs_Unlink(yaffs_InodeToObject(dir),dentry->d_name.name);
	
	
	yaffs_GrossUnlock(dev);
	
	if( retVal == YAFFS_OK)
	{
		dentry->d_inode->i_nlink--;
		mark_inode_dirty(dentry->d_inode);
		return 0;
	}
	else
	{
		return -ENOTEMPTY;
	}
}


/*
 * Create a link...
 */
static int yaffs_link(struct dentry *old_dentry, struct inode * dir, struct dentry * dentry)
{
	struct inode *inode = old_dentry->d_inode;
	yaffs_Object *obj = NULL;
	yaffs_Object *link=NULL;
	yaffs_Device *dev;
	
	T(YAFFS_TRACE_OS,("yaffs_link\n"));
	
	obj = yaffs_InodeToObject(inode);
	dev = obj->myDev;
	
	yaffs_GrossLock(dev);

	if (!S_ISDIR(inode->i_mode))	// Don't link directories
	{
		link = yaffs_Link(yaffs_InodeToObject(dir),dentry->d_name.name,obj);
	}
	

	if(link)
	{
		old_dentry->d_inode->i_nlink =  yaffs_GetObjectLinkCount(obj);
		d_instantiate(dentry, old_dentry->d_inode);
		atomic_inc(&old_dentry->d_inode->i_count);
		T(YAFFS_TRACE_OS,("yaffs_link link count %d i_count %d\n",	
			old_dentry->d_inode->i_nlink,atomic_read(&old_dentry->d_inode->i_count)));
	
	}
	
	yaffs_GrossUnlock(dev);
	

	if(link)
	{
	
		return 0;
	}
	
	
	return -EPERM; 
}


static int yaffs_symlink(struct inode * dir, struct dentry *dentry, const char * symname)
{
	yaffs_Object *obj;
	yaffs_Device *dev;
	
	T(YAFFS_TRACE_OS,("yaffs_symlink\n"));
	
	dev = yaffs_InodeToObject(dir)->myDev;
	yaffs_GrossLock(dev);
	obj = yaffs_MknodSymLink(yaffs_InodeToObject(dir), dentry->d_name.name, 
							 S_IFLNK | S_IRWXUGO, current->uid, current->gid,
							 symname);
	yaffs_GrossUnlock(dev);

	if(obj)
	{

		struct inode* inode;
	
		inode = yaffs_get_inode(dir->i_sb, obj->st_mode, 0, obj);
		d_instantiate(dentry, inode);
		T(YAFFS_TRACE_OS,("symlink created OK\n"));
		return 0;
	}
	else
	{
		T(YAFFS_TRACE_OS,("symlink not created\n"));

	}
	
	return -ENOMEM;
}

static int yaffs_sync_object(struct file * file, struct dentry *dentry, int datasync)
{

	yaffs_Object *obj;
	yaffs_Device *dev;
	
	obj = yaffs_DentryToObject(dentry);

	dev = obj->myDev;
	
	T(YAFFS_TRACE_OS,("yaffs_sync_object\n"));
	yaffs_GrossLock(dev);
	yaffs_FlushFile(obj,1);
	yaffs_GrossUnlock(dev);
	return 0;
}

/*
 * The VFS layer already does all the dentry stuff for rename.
 *
 * NB: POSIX says you can rename an object over an old object of the same name
 */
static int yaffs_rename(struct inode * old_dir, struct dentry *old_dentry, struct inode * new_dir,struct dentry *new_dentry)
{
	yaffs_Device *dev;
	int retVal = YAFFS_FAIL;
	int removed = 0;
	yaffs_Object *target;
	
	dev = yaffs_InodeToObject(old_dir)->myDev;

	yaffs_GrossLock(dev);
	
	// Check if the target is an existing directory that is not empty.
	target = yaffs_FindObjectByName(yaffs_InodeToObject(new_dir),new_dentry->d_name.name);
	
	if(target &&
	   target->variantType == YAFFS_OBJECT_TYPE_DIRECTORY &&
	   !list_empty(&target->variant.directoryVariant.children))
	{
		retVal = YAFFS_FAIL;
	}
	else
	{
	   
		// Unlink the target if it exists
		removed = yaffs_Unlink(yaffs_InodeToObject(new_dir),new_dentry->d_name.name);

	
		retVal =  yaffs_RenameObject(yaffs_InodeToObject(old_dir),old_dentry->d_name.name,
									yaffs_InodeToObject(new_dir),new_dentry->d_name.name);
									
	}
	yaffs_GrossUnlock(dev);
	
	if(retVal == YAFFS_OK)
	{
		if(removed == YAFFS_OK)
		{
			new_dentry->d_inode->i_nlink--;
			mark_inode_dirty(new_dentry->d_inode);
		}
		
		return 0;
	}
	else
	{
		return -ENOTEMPTY;
	}
	

}

static int yaffs_setattr(struct dentry *dentry, struct iattr *attr)
{
	struct inode *inode = dentry->d_inode;
	int error;
	yaffs_Device *dev;
	
	T(YAFFS_TRACE_OS,("yaffs_setattr of object %d\n",yaffs_InodeToObject(inode)->objectId));
	
	if((error = inode_change_ok(inode,attr)) == 0)
	{
	
		dev = yaffs_InodeToObject(inode)->myDev;
		yaffs_GrossLock(dev);
		if(yaffs_SetAttributes(yaffs_InodeToObject(inode),attr) == YAFFS_OK)
		{
			error = 0;
		}
		else
		{
			error = -EPERM;
		}
		yaffs_GrossUnlock(dev);
		inode_setattr(inode,attr);
	}
	return error;
}

//#if defined(CONFIG_KERNEL_2_5)	/* Added NCB 185-8-2003 */
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,5,0))
static int yaffs_statfs(struct super_block *sb, struct kstatfs *buf)
#else
static int yaffs_statfs(struct super_block *sb, struct statfs *buf)
#endif
{
	yaffs_Device *dev = yaffs_SuperToDevice(sb);
	T(YAFFS_TRACE_OS,("yaffs_statfs\n"));

	yaffs_GrossLock(dev);
	
	buf->f_type = YAFFS_MAGIC;
	buf->f_bsize = sb->s_blocksize;
	buf->f_namelen = 255;
	buf->f_blocks = (dev->endBlock - dev->startBlock + 1) * YAFFS_CHUNKS_PER_BLOCK/
						(sb->s_blocksize/YAFFS_BYTES_PER_CHUNK);
	buf->f_files = 0;
	buf->f_ffree = 0;
	buf->f_bfree = yaffs_GetNumberOfFreeChunks(dev)/
						(sb->s_blocksize/YAFFS_BYTES_PER_CHUNK);
	buf->f_bavail =  buf->f_bfree;
	
	yaffs_GrossUnlock(dev);
	return 0;
}

static void yaffs_read_inode (struct inode *inode)
{
	// NB This is called as a side effect of other functions and
	// thus gross locking should always be in place already.
	
	yaffs_Object *obj ; 
	yaffs_Device *dev = yaffs_SuperToDevice(inode->i_sb);
	
	T(YAFFS_TRACE_OS,("yaffs_read_inode for %d\n",(int)inode->i_ino));

	obj  = yaffs_FindObjectByNumber(dev,inode->i_ino);
	
	yaffs_FillInodeFromObject(inode,obj);

}



static yaffs_Device *yaffs_dev;
static yaffs_Device *yaffsram_dev;



static void yaffs_put_super(struct super_block *sb)
{
	yaffs_Device *dev = yaffs_SuperToDevice(sb);
	
	yaffs_GrossLock(dev);
	if(dev->putSuperFunc)
	{
		 dev->putSuperFunc(sb);
	}
	yaffs_Deinitialise(dev);
	yaffs_GrossUnlock(dev);

	if(dev == yaffs_dev) yaffs_dev = NULL;
	if(dev == yaffsram_dev) yaffsram_dev = NULL;
		
	kfree(dev);
}


#ifdef CONFIG_YAFFS_MTD_ENABLED

static void  yaffs_MTDPutSuper(struct super_block *sb)
{
	
	struct mtd_info *mtd = yaffs_SuperToDevice(sb)->genericDevice;
	
	if(mtd->sync)
	{
		mtd->sync(mtd);
	}
	
	put_mtd_device(mtd);
}

#endif


static struct super_block *yaffs_internal_read_super(int useRam, struct super_block * sb, void * data, int silent)
{
	int nBlocks;
	struct inode * inode;
	struct dentry * root;
	yaffs_Device *dev;
	
	sb->s_magic = YAFFS_MAGIC;
	sb->s_op = &yaffs_super_ops;
	
	if(!sb)
		T(YAFFS_TRACE_OS,("yaffs: sb is NULL\n"));
	else if(!sb->s_dev)
		T(YAFFS_TRACE_OS,("yaffs: sb->s_dev is NULL\n"));
	else if(! kdevname(sb->s_dev))
		T(YAFFS_TRACE_OS,("yaffs: kdevname is NULL\n"));
	else
		T(YAFFS_TRACE_OS,("yaffs: dev is %d name is \"%s\"\n", sb->s_dev, kdevname(sb->s_dev)));

	

#ifdef CONFIG_YAFFS_USE_CHUNK_SIZE
	sb->s_blocksize = YAFFS_BYTES_PER_CHUNK;
	sb->s_blocksize_bits = YAFFS_CHUNK_SIZE_SHIFT;
#else
	sb->s_blocksize = PAGE_CACHE_SIZE;
	sb->s_blocksize_bits = PAGE_CACHE_SHIFT;
#endif
	T(YAFFS_TRACE_OS,("yaffs_read_super: %s block size %d\n", useRam ? "RAM" : "MTD",(int)(sb->s_blocksize)));

#ifdef CONFIG_YAFFS_DISABLE_WRITE_VERIFY
	T(YAFFS_TRACE_OS,("yaffs: Write verification disabled. All guarantees null and void\n"));
#endif


	
	if(useRam)
	{

#ifdef CONFIG_YAFFS_RAM_ENABLED
		// Set the yaffs_Device up for ram emulation

		sb->u.generic_sbp =	dev = kmalloc(sizeof(yaffs_Device),GFP_KERNEL);
		if(!dev)
		{
			// Deep shit could not allocate device structure
			T(YAFFS_TRACE_OS,("yaffs_read_super: Failed trying to allocate yaffs_Device. Terminating debug.\n"));
			return NULL;
		}

		memset(dev,0,sizeof(yaffs_Device));
		dev->genericDevice = NULL; // Not used for RAM emulation.

		nBlocks = YAFFS_RAM_EMULATION_SIZE / (YAFFS_CHUNKS_PER_BLOCK * YAFFS_BYTES_PER_CHUNK);
		dev->startBlock = 1;  // Don't use block 0
		dev->endBlock = nBlocks - 1;
		dev->nChunksPerBlock = YAFFS_CHUNKS_PER_BLOCK;
		dev->nBytesPerChunk = YAFFS_BYTES_PER_CHUNK;
		dev->nReservedBlocks = 5;

		dev->writeChunkToNAND = nandemul_WriteChunkToNAND;
		dev->readChunkFromNAND = nandemul_ReadChunkFromNAND;
		dev->eraseBlockInNAND = nandemul_EraseBlockInNAND;
		dev->initialiseNAND = nandemul_InitialiseNAND;

		yaffsram_dev = dev;
		
#endif

	}
	else
	{	
#ifdef CONFIG_YAFFS_MTD_ENABLED
		struct mtd_info *mtd;
		
		T(YAFFS_TRACE_ALWAYS,( "yaffs: Attempting MTD mount on %u.%u, \"%s\"\n",
		 MAJOR(sb->s_dev),MINOR(sb->s_dev),kdevname(sb->s_dev)));
			
		// Check it's an mtd device.....
		if(MAJOR(sb->s_dev) != MTD_BLOCK_MAJOR)
		{
			return NULL; // This isn't an mtd device
		} 
		
		// Get the device
		mtd = get_mtd_device(NULL, MINOR(sb->s_dev));
		if (!mtd) 
		{
			T(YAFFS_TRACE_ALWAYS,( "yaffs: MTD device #%u doesn't appear to exist\n", MINOR(sb->s_dev)));
			return NULL;
		}
		
		// Check it's NAND
		if(mtd->type != MTD_NANDFLASH)
		{
			T(YAFFS_TRACE_ALWAYS,( "yaffs: MTD device is not NAND it's type %d\n", mtd->type));
			return NULL;
		}

		//T(YAFFS_TRACE_OS,(KERN_DEBUG" erase %x\n",mtd->erase);
		//T(YAFFS_TRACE_OS,(KERN_DEBUG" read %x\n",mtd->read);
		//T(YAFFS_TRACE_OS,(KERN_DEBUG" write %x\n",mtd->write);
		//T(YAFFS_TRACE_OS,(KERN_DEBUG" readoob %x\n",mtd->read_oob);
		//T(YAFFS_TRACE_OS,(KERN_DEBUG" writeoob %x\n",mtd->write_oob);
		//T(YAFFS_TRACE_OS,(KERN_DEBUG" oobblock %x\n",mtd->oobblock);
		//T(YAFFS_TRACE_OS,(KERN_DEBUG" oobsize %x\n",mtd->oobsize);


		if(!mtd->erase ||
		   !mtd->read  ||
		   !mtd->write ||
#ifndef CONFIG_YAFFS_USE_OLD_MTD
		   !mtd->write_ecc ||
		   !mtd->read_ecc ||
#endif
		   !mtd->read_oob ||
		   !mtd->write_oob )
		{
			T(YAFFS_TRACE_ALWAYS,( "yaffs: MTD device does not support required functions\n"));
			return NULL;
		}
		
		if(mtd->oobblock != YAFFS_BYTES_PER_CHUNK ||
		   mtd->oobsize != YAFFS_BYTES_PER_SPARE)
		{
			T(YAFFS_TRACE_ALWAYS,( "yaffs: MTD device does not support have the right page sizes\n"));
			return NULL;
		}
		   

		// OK, so if we got here, we have an MTD that's NAND and looks 
		// like it has the right capabilities
		// Set the yaffs_Device up for ram emulation

//#if defined(CONFIG_KERNEL_2_5)
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,5,0))
		sb->s_fs_info =	dev = kmalloc(sizeof(yaffs_Device),GFP_KERNEL);
#else
		sb->u.generic_sbp =	dev = kmalloc(sizeof(yaffs_Device),GFP_KERNEL);
#endif
		if(!dev)
		{
			// Deep shit could not allocate device structure
			T(YAFFS_TRACE_ALWAYS,("yaffs_read_super: Failed trying to allocate yaffs_Device. Terminating debug.\n"));
			return NULL;
		}

		memset(dev,0,sizeof(yaffs_Device));
		dev->genericDevice = mtd; 

		// Set up the memory size parameters....
		
		nBlocks = mtd->size / (YAFFS_CHUNKS_PER_BLOCK * YAFFS_BYTES_PER_CHUNK);
		dev->startBlock = 1;  // Don't use block 0
		dev->endBlock = nBlocks - 1;
		dev->nChunksPerBlock = YAFFS_CHUNKS_PER_BLOCK;
		dev->nBytesPerChunk = YAFFS_BYTES_PER_CHUNK;
		dev->nReservedBlocks = 5;
		dev->nShortOpCaches = 0; // Enable short op caching
		

		// ... and the functions.
		dev->writeChunkToNAND = nandmtd_WriteChunkToNAND;
		dev->readChunkFromNAND = nandmtd_ReadChunkFromNAND;
		dev->eraseBlockInNAND = nandmtd_EraseBlockInNAND;
		dev->initialiseNAND = nandmtd_InitialiseNAND;
				
		dev->putSuperFunc = yaffs_MTDPutSuper;
		
#ifdef CONFIG_YAFFS_USE_NANDECC
		dev->useNANDECC = 1;
#endif

		yaffs_dev = dev;
		
#endif
	}

	init_MUTEX(&dev->grossLock);
	
	
	yaffs_GrossLock(dev);
	yaffs_GutsInitialise(yaffs_SuperToDevice(sb));

	T(YAFFS_TRACE_OS,("yaffs_read_super: guts initialised\n"));

	// Create root inode
	inode = yaffs_get_inode(sb, S_IFDIR | 0755, 0,yaffs_Root(yaffs_SuperToDevice(sb)));

	yaffs_GrossUnlock(dev);

	if (!inode)
		return NULL;
		
// added NCB
	inode->i_op = & yaffs_dir_inode_operations;
	inode->i_fop = & yaffs_dir_operations;

	T(YAFFS_TRACE_OS,("yaffs_read_super: got root inode\n"));
		

	root = d_alloc_root(inode);

	T(YAFFS_TRACE_OS,("yaffs_read_super: d_alloc_root done\n"));

	if (!root) {
		iput(inode);
		return NULL;
	}
	sb->s_root = root;

	T(YAFFS_TRACE_OS,("yaffs_read_super: done\n"));
	return sb;
}

static int yaffs_internal_read_super_ram(struct super_block * sb, void * data, int silent)
{
	 return yaffs_internal_read_super(1,sb,data,silent) ? 0 : -1;
}
static int yaffs_internal_read_super_mtd(struct super_block * sb, void * data, int silent)
{
	 return yaffs_internal_read_super(0,sb,data,silent) ? 0 : -1;
}


#ifdef CONFIG_YAFFS_MTD_ENABLED
//#if defined(CONFIG_KERNEL_2_5)
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,5,0))
static struct super_block *yaffs_read_super(struct file_system_type * fs, int flags, const char *dev_name, void *data)
{

    return get_sb_bdev(fs, flags, dev_name, data, yaffs_internal_read_super_mtd);
}

/* changes NCB 2.5.70 */
//static DECLARE_FSTYPE(yaffs_fs_type, "yaffs", yaffs_read_super, FS_REQUIRES_DEV);
static struct file_system_type yaffs_fs_type = {
	.owner		= THIS_MODULE,
	.name		= "yaffs",
	.get_sb		= yaffs_read_super,
	.kill_sb	= kill_block_super,
//	.kill_sb	= kill_litter_super,
	.fs_flags	= FS_REQUIRES_DEV,
};
#else
static struct super_block *yaffs_read_super(struct super_block * sb, void * data, int silent)
{
	return yaffs_internal_read_super(0,sb,data,silent);
}

static DECLARE_FSTYPE(yaffs_fs_type, "yaffs", yaffs_read_super, FS_REQUIRES_DEV);
#endif
#endif

#ifdef CONFIG_YAFFS_RAM_ENABLED

//#if defined(CONFIG_KERNEL_2_5)
#if (LINUX_VERSION_CODE > KERNEL_VERSION(2,5,0))
static struct super_block *yaffs_ram_read_super(struct file_system_type * fs, int flags, const char *dev_name, void *data)
{

    return get_sb_bdev(fs, flags, dev_name, data, yaffs_internal_read_super_ram);
}

/* changes NCB 2.5.70 */
//static DECLARE_FSTYPE(yaffs_fs_type, "yaffs", yaffs_read_super, FS_REQUIRES_DEV);
static struct file_system_type yaffs_fs_type = {
	.owner		= THIS_MODULE,
	.name		= "yaffsram",
	.get_sb		= yaffs_ram_read_super,
	.kill_sb	= kill_block_super,
//	.kill_sb	= kill_litter_super,
	.fs_flags	= FS_SINGLE,
};
#else
static struct super_block *yaffs_ram_read_super(struct super_block * sb, void * data, int silent)
{
	return yaffs_internal_read_super(1,sb,data,silent);
}

static DECLARE_FSTYPE(yaffs_ram_fs_type, "yaffsram", yaffs_ram_read_super, FS_SINGLE);
#endif
#endif // CONFIG_YAFFS_RAM_ENABLED


static struct proc_dir_entry *my_proc_entry;
static struct proc_dir_entry *my_proc_ram_write_entry;

static char * yaffs_dump_dev(char *buf,yaffs_Device *dev,char *name)
{
	buf +=sprintf(buf,"\nDevice %s\n",name);
	buf +=sprintf(buf,"startBlock......... %d\n",dev->startBlock);
	buf +=sprintf(buf,"endBlock........... %d\n",dev->endBlock);
	buf +=sprintf(buf,"chunkGroupBits..... %d\n",dev->chunkGroupBits);
	buf +=sprintf(buf,"chunkGroupSize..... %d\n",dev->chunkGroupSize);
	buf +=sprintf(buf,"nErasedBlocks...... %d\n",dev->nErasedBlocks);
	buf +=sprintf(buf,"nTnodesCreated..... %d\n",dev->nTnodesCreated);
	buf +=sprintf(buf,"nFreeTnodes........ %d\n",dev->nFreeTnodes);
	buf +=sprintf(buf,"nObjectsCreated.... %d\n",dev->nObjectsCreated);
	buf +=sprintf(buf,"nFreeObjects....... %d\n",dev->nFreeObjects);
	buf +=sprintf(buf,"nFreeChunks........ %d\n",dev->nFreeChunks);
	buf +=sprintf(buf,"nPageWrites........ %d\n",dev->nPageWrites);
	buf +=sprintf(buf,"nPageReads......... %d\n",dev->nPageReads);
	buf +=sprintf(buf,"nBlockErasures..... %d\n",dev->nBlockErasures);
	buf +=sprintf(buf,"nGCCopies.......... %d\n",dev->nGCCopies);
	buf +=sprintf(buf,"garbageCollections. %d\n",dev->garbageCollections);
	buf +=sprintf(buf,"passiveGCs......... %d\n",dev->passiveGarbageCollections);
	buf +=sprintf(buf,"nRetriedWrites..... %d\n",dev->nRetriedWrites);
	buf +=sprintf(buf,"nRetireBlocks...... %d\n",dev->nRetiredBlocks);
	buf +=sprintf(buf,"eccFixed........... %d\n",dev->eccFixed);
	buf +=sprintf(buf,"eccUnfixed......... %d\n",dev->eccUnfixed);
	buf +=sprintf(buf,"tagsEccFixed....... %d\n",dev->tagsEccFixed);
	buf +=sprintf(buf,"tagsEccUnfixed..... %d\n",dev->tagsEccUnfixed);
	buf +=sprintf(buf,"cacheHits.......... %d\n",dev->cacheHits);
	buf +=sprintf(buf,"nDeletedFiles...... %d\n",dev->nDeletedFiles);
	buf +=sprintf(buf,"nUnlinkedFiles..... %d\n",dev->nUnlinkedFiles);
	buf +=sprintf(buf,"nBackgroudDeletions %d\n",dev->nBackgroundDeletions);
	buf +=sprintf(buf,"useNANDECC......... %d\n",dev->useNANDECC);

	
	return buf;	
}

static int  yaffs_proc_read(
        char *page,
	char **start,
	off_t offset,
	int count,
	int *eof,
	void *data
	)
{

	char my_buffer[3000];
	char *buf;
	buf = my_buffer;

	if (offset > 0) return 0;

	/* Fill the buffer and get its length */
	buf +=sprintf(buf,"YAFFS built:"__DATE__ " "__TIME__"\n%s\n%s\n", yaffs_fs_c_version,yaffs_guts_c_version);
	
	if(yaffs_dev) buf = yaffs_dump_dev(buf,yaffs_dev,"yaffs");
	if(yaffsram_dev) buf = yaffs_dump_dev(buf,yaffsram_dev,"yaffsram");
	

	strcpy(page,my_buffer);
	return strlen(my_buffer);
}


static int  yaffs_proc_ram_write(
        char *page,
	char **start,
	off_t offset,
	int count,
	int *eof,
	void *data
	)
{

	T(YAFFS_TRACE_OS,( "yaffs write size %d\n",count));
	return count;
}

static int __init init_yaffs_fs(void)
{
	int error = 0;
	
	yaffs_dev = yaffsram_dev = NULL;
	
	T(YAFFS_TRACE_OS,( "yaffs " __DATE__ " " __TIME__ " Initialisation\n"));
#ifdef CONFIG_YAFFS_USE_GENERIC_RW
	T(YAFFS_TRACE_OS,( "yaffs is using generic read/write (caching)\n"));
#else
	T(YAFFS_TRACE_OS,( "yaffs is using direct read/write (uncached)\n"));
#endif


    /* Install the proc_fs entry */
    my_proc_entry = create_proc_read_entry("yaffs",
                                           S_IRUGO | S_IFREG,
					   &proc_root,
					   yaffs_proc_read,
					   NULL);
    if(!my_proc_entry)
    {
       return -ENOMEM;
    }

#ifdef CONFIG_YAFFS_RAM_ENABLED
#if 0
    my_proc_ram_write_entry = create_proc_entry("yaffs_ram",
                                           S_IRUGO | S_IFREG,
					   &proc_root);
 
    if(!my_proc_ram_write_entry)
    {
       return -ENOMEM;
    }
    else
    {
    	my_proc_ram_write_entry->write_proc = yaffs_proc_ram_write;
    }
#endif

    error = register_filesystem(&yaffs_ram_fs_type);
    if(error)
    {
    	return error;
    }
#endif //CONFIG_YAFFS_RAM_ENABLED

#ifdef CONFIG_YAFFS_MTD_ENABLED
	error = register_filesystem(&yaffs_fs_type);
	if(error)
	{
#ifdef CONFIG_YAFFS_RAM_ENABLED
		unregister_filesystem(&yaffs_ram_fs_type);
#endif //CONFIG_YAFFS_RAM_ENABLED
	}
#endif // CONFIG_YAFFS_MTD_ENABLED

	return error;
}

static void __exit exit_yaffs_fs(void)
{
	T(YAFFS_TRACE_OS,( "yaffs " __DATE__ " " __TIME__ " Clean up\n"));

    remove_proc_entry("yaffs",&proc_root);
    
#ifdef CONFIG_YAFFS_RAM_ENABLED
	unregister_filesystem(&yaffs_ram_fs_type);
#endif
#ifdef CONFIG_YAFFS_MTD_ENABLED
	unregister_filesystem(&yaffs_fs_type);
#endif

}

module_init(init_yaffs_fs)
module_exit(exit_yaffs_fs)

MODULE_DESCRIPTION("YAFFS - a NAND specific flash file system");
MODULE_AUTHOR("Charles Manning, Aleph One Ltd., 2002");
MODULE_LICENSE("GPL");

