/*
 * linux/mm/debug.c
 *
 *  Copyright (C) 2001 Russell King.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 *  Dump out page information on SysRQ-G, and provide show_page_info()
 *
 *  You are advised to use a serial console with this patch - it
 *  saturates a 38400baud link for 1 minute on a 32MB machine.
 */
#include <linux/kernel.h>
#include <linux/mm.h>
#include <linux/pagemap.h>
#include <linux/sysrq.h>
#include <linux/init.h>

/*
 * We print out the following information for each page in the system:
 *	address: use count, age, mapping, [RSsr] rD [acd]
 *
 * The flags are:
 *	R - reserved
 *	S - in swapcache
 *	s - slab page
 *
 *	r - referenced
 *	D - dirty
 *
 *	a - active page
 */
static void page_detail(struct page *page)
{
	if (!page)
		return;

	printk("%p: %2d %p [%c%c%c] %c%c [%c]\n",
			page_address(page),
			page_count(page),
			page->mapping,

			PageReserved(page) ? 'R' : '-',
			PageSwapCache(page) ? 'S' : '-',
			PageSlab(page) ? 's' : '-',

			PageReferenced(page) ? 'r' : '-',
			PageDirty(page) ? 'D' : '-',

			PageActive(page) ? 'a' : '-');
}

/*
 * This version collects statistics
 */
static int anon_pages, slab_pages, resvd_pages, unused_pages;

static void page_statistics(struct page *page)
{
	if (page) {
		if (PageReserved(page))
			resvd_pages++;
		else if (PageSlab(page))
			slab_pages++;
		else if (!page_count(page))
			unused_pages ++;
		else if (!page->mapping)
			anon_pages ++;
		return;
	}

	printk("  anon: %d, slab: %d, reserved: %d, free: %d\n",
		anon_pages, slab_pages, resvd_pages, unused_pages);
}

static void show_zone_info(zone_t *zone, void (*fn)(struct page *))
{
	int i;

	printk("  total %ld, free %ld\n",
		zone->size, zone->free_pages);

	anon_pages    = 0;
	slab_pages    = 0;
	resvd_pages   = 0;
	unused_pages  = 0;

	for (i = 0; i < zone->size; i++) {
		struct page *page = zone->zone_mem_map + i;

		fn(page);
	}

	fn(NULL);
}

static void show_node_info(pg_data_t *pg, void (*fn)(struct page *))
{
	int type;

	for (type = 0; type < MAX_NR_ZONES; type++) {
		zone_t *zone = pg->node_zones + type;

		if (zone->size == 0)
			continue;

		printk("----- Zone %d ------\n", type);

		show_zone_info(zone, fn);
	}
}

static void __show_page_info(void (*fn)(struct page *))
{
	pg_data_t *pg;
	int pgdat = 0;

	for (pg = pgdat_list; pg; pg = pg->node_next) {

		printk("===== Node %d =====\n", pgdat++);

		show_node_info(pg, fn);
	}	
}

void show_page_info(void)
{
	__show_page_info(page_detail);
}

static void
show_pg_info(int key, struct pt_regs *regs, struct kbd_struct *kd,
	     struct tty_struct *tty)
{
	void (*fn)(struct page *);
	show_mem();
	if (key == 'g')
		fn = page_detail;
	else
		fn = page_statistics;
	__show_page_info(fn);
}

static struct sysrq_key_op page_info_op = {
	handler:	show_pg_info,
	help_msg:	"paGeinfo",
	action_msg:	"Page Info",
};

static int __init debug_mm_init(void)
{
	register_sysrq_key('g', &page_info_op);
	register_sysrq_key('h', &page_info_op);
	return 0;
}

__initcall(debug_mm_init);


#ifdef CONFIG_MMU_TABLE_DUMP
/*
 * This routine has been changed to consolidate the info for blocks
 * of memory whose properties are the same.  The same info can be 
 * communicated in 20 lines instead of 4000 (no exaggeration). 9/10/02
 *
 *
 * We print out the following information for each page in the system:
 *  address: use count, age, mapping, [RSsr] rD [acd]
 *
 * The flags are:
 *  R - reserved
 *  S - in swapcache
 *  s - slab page
 *  r - ramdisk
 *
 *  r - referenced
 *  D - dirty
 *
 *  a - active page
 *  c - on inactive clean list
 *  d - on inactive dirty list
 */
static char szCurrentPageInfo[100], szPreviousPageInfo[100], szBuff[200];
static unsigned long ulStartAddr, ulEndAddr;

void page_detail_condensed( struct page *page )
{
	if( page != 0 )
	{
		/* 
		 * Construct a string that gives useful info about this page 
		 */
		sprintf(    szCurrentPageInfo, 
					"%2d %p [%c%c%c] %c%c [%c]",
					page_count(page),
					page->mapping,

					PageReserved(page) ? 'R' : '-',
					PageSwapCache(page) ? 'S' : '-',
					PageSlab(page) ? 's' : '-',

					PageReferenced(page) ? 'r' : '-',
					PageDirty(page) ? 'D' : '-',

					PageActive(page) ? 'a' : '-'
				);

		/*
		 * If this is the first time thru the loop, or is the first
		 * page of a different section, initialize the szPreviousPageInfo
		 * to be the same as the current page info and initialize
		 * the addresses.
		 */
		if( szPreviousPageInfo[0] == 0 )
		{
			strncpy( szPreviousPageInfo, szCurrentPageInfo, 100 );
		}

		/*
		 * We will print out the info for this set of zones if this page's 
		 * info is different from that for the last zone or if this is
		 * the last zone.
		 */
		if( strcmp( szCurrentPageInfo, szPreviousPageInfo ) != 0 )
		{
			/*
			 * Print the info for the region of memory that includes the
			 * page previous to 'page'.
			 */
			unsigned int uiSizeInK = ((int)(ulEndAddr - ulStartAddr + 4096)) >> 10;
		
			sprintf( szBuff, "%08x - %08x: %s =%6dK\n", 
				(unsigned int)ulStartAddr, (unsigned int)ulEndAddr, 
				szPreviousPageInfo, uiSizeInK );
			printk( szBuff );

			/*
			 *  Done with that section of memory.  The current zone
			 *  starts a new section...
			 */
			ulStartAddr = (unsigned long)page_address( page );
			ulEndAddr = ulStartAddr;
			strncpy( szPreviousPageInfo, szCurrentPageInfo, 100 );
		}
		else
		{
			/*
			 * This page's properties are still the same as the page
			 * that corresponds to start_addr.  So include this page
			 * in on the end address and loop back to check the next page.
			 */
			ulEndAddr = (unsigned long)page_address( page );
		}
	}
	else
	{
		/*
		 * This is the last page in this zone.
		 * Print the info for the region of memory that includes the
		 * page previous to 'page'.
		 */
		unsigned int uiSizeInK = ((int)(ulEndAddr - ulStartAddr + 4096)) >> 10;
		
		sprintf( szBuff, "%08x - %08x: %s =%6dK\n", 
			(unsigned int)ulStartAddr, (unsigned int)ulEndAddr, 
			szPreviousPageInfo, uiSizeInK );
		printk( szBuff );

	}
}

static void show_zone_info_condensed( zone_t *zone )
{
	int i;

	printk("  total %ld, free %ld\n",
		zone->size,
		zone->free_pages);

	printk("  min %ld, low %ld, high %ld\n",
		zone->pages_min,
		zone->pages_low,
		zone->pages_high);

	szPreviousPageInfo[0] = 0;
	ulStartAddr = (unsigned long)page_address( zone->zone_mem_map );
	ulEndAddr = ulStartAddr;


	for (i = 0; i < zone->size ; i++) 
	{
		struct page *page = zone->zone_mem_map + i;

		if( page )
		{
			page_detail_condensed( page );
		}
	}

	page_detail_condensed( 0 );
}

static void show_node_info_condensed( pg_data_t *pg )
{
	int type;

	for( type = 0 ; type < MAX_NR_ZONES ; type++ ) 
	{
		zone_t *zone = pg->node_zones + type;

		if( zone->size != 0 )
		{
			printk("----- Zone %d ------\n", type);
			show_zone_info_condensed(zone );
		}
	}
}

void show_page_info_condensed(void)
{
	pg_data_t *pg;
	int pgdat = 0;

	show_mem();

	for (pg = pgdat_list; pg; pg = pg->node_next) 
	{
		printk("===== Node %d =====\n", pgdat++);
		show_node_info_condensed( pg );
	}   

	printk("=====================\n\n");
}

void dump_mmu_level1_table(void)
{
	pgd_t *pgd;
	unsigned long pg, ulLevel2table;
	unsigned int i=0, j=0, uiLevel1Descr1, uiLevel1Descr2;
	unsigned int uiLevel2Descr1;
	int iSectionOffset1 = 0, iSectionOffset2 = 0;
	unsigned int uiCoursePTs = 0, uiFinePTs = 0;
	unsigned int *puiPT;
	unsigned int iL1EntryNum, uiEntries, uiShift;
	

	/*
	 * I borrowed some code from cpu_get_pgd().
	 * Get the physical address of the MMU's Level 1 table.
	 */
	__asm__("mrc p15, 0, %0, c2, c0, 0"	: "=r" (pg));

	pg &= ~0x3fff;

	printk("===================================\n");
	printk("MMU Table located at physical addr %08x - %08x.\n", 
		(unsigned int)pg, (unsigned int)pg + (4096*4) - 1 );

	/*
	 * Funny thing is that MMU is on, so we need to go thru the MMU
	 * to look at how the MMU is mapping things.  This is really no
	 * problem, it's just kinda trippy.  Get the virtual address
	 * of the Level one table.
	 */
	pgd = (pgd_t *)phys_to_virt(pg);

	/*
	 * Get the first descriptor in the L1 table.
	 */
	uiLevel1Descr1 = pgd_val( *pgd++ );
	
	/*
	 * Go thru the whole L1 table looking for sections that are
	 * contiguous and have the rights, domain, and AP bits set the
	 * same.
	 */
	for( j = 1 ; j <= 4096 ; j++ )
	{
		/*
		 * If the descriptor is a Section descriptor, calculate the
		 * offset that it sets up between virtual and physical address.
		 */
		if( (uiLevel1Descr1 & 3) == 2 )
		{
			iSectionOffset1 = (i<<20) - (uiLevel1Descr1 & 0xfff00000);
		}

		/*
		 * Get the j'th descriptor (or 0 if we are going thru the loop 
		 * just one last time to print out the last chunk of data).
		 */
		if( j < 4096 )
		{
			uiLevel1Descr2 = pgd_val(*pgd++);
		}
		else
		{
			uiLevel1Descr2 = 0;
		}
		
		/*
		 * If the j'th descriptor is different from uiLevel1Descr1, we
		 * may be in a new section of the mapping.  Else, skip
		 * all this and look at the next descriptor.
		 */
		if( uiLevel1Descr2 != uiLevel1Descr1 )
		{
			/*
			 * We're here because the j'th descriptor may be starting
			 * a new section of memory.  So first print out info on
			 * the last section of memory.
			 *
			 * Switch is conditioned on what type of L1 descriptor
			 * starts out this section of memory.
			 */
			switch( uiLevel1Descr1 & 3 )
			{
				/*
				 * Fault
				 */
				case 0:
					printk( "%08x - %08x: %08x            (%4d MB) ", 
						(i<<20), ((j-1)<<20), uiLevel1Descr1, (j-i) );
					printk("- Fault\n");
					i = j;
					uiLevel1Descr1 = uiLevel1Descr2;
					break;

				/*
				 * Course page table
				 */
				case 1:
					printk( "%08x - %08x: %08x            (%4d MB) ", 
						(i<<20), ((j-1)<<20), uiLevel1Descr1, (j-i) );
					printk("- CoarsePT       Domain:%2d\n", (uiLevel1Descr1>>5) & 0xf );
					i = j;
					uiLevel1Descr1 = uiLevel1Descr2;
					uiCoursePTs++;
					break;

				/*
				 * Section
				 * The descriptor points contains a section base address.
				 */
				case 2:
					/*
					 * If the j'th descriptor is also a section descriptor,
					 * get its offset.
					 */
					if( (uiLevel1Descr2 & 3) == 2 )
					{
						iSectionOffset2 = (j<<20) - (uiLevel1Descr2 & 0xfff00000);
					}
					else
					{
						iSectionOffset2 = -1;    
					}
					
					/*
					 * Check to see if we already are in a section of memory with
					 * the same offset.  If so, don't print data yet until we have
					 * walked through all descriptors in this section.
					 */
					if( ( (uiLevel1Descr1 & 0x000fffff) != (uiLevel1Descr2 & 0x000fffff) ) ||
						(iSectionOffset1 != iSectionOffset2) )
					{
						printk( "%08x - %08x: %08x - %08x (%4d MB) ", 
							(i<<20), 
							((j<<20) - 1),
							(uiLevel1Descr1 & 0xfff00000),
							(uiLevel1Descr1 & 0xfff00000) + ((j-i)<<20) - 1,
							(j-i) );
						
						printk("- Section  AP:%d, Domain:%2d ", 
							((uiLevel1Descr1>>10) & 0x3), ((uiLevel1Descr1>>5) & 0xf) );

						if( uiLevel1Descr1 & 0x8 )
						{
							printk("C");
						}
						else
						{
							printk("-");
						}

						if( uiLevel1Descr1 & 0x4 )
						{
							printk("B");
						}
						else
						{
							printk("-");
						}
					
						printk("\n");
						
						i = j;
						uiLevel1Descr1 = uiLevel1Descr2;
					}
					
					break;

				/*
				 * Fine page table
				 */
				case 3:
					printk( "%08x - %08x: %08x            (%4d MB) ", 
						(i<<20), ((j-1)<<20), uiLevel1Descr1, (j-i) );
					printk("- FinePT         Domain:%2d\n", (uiLevel1Descr1>>5) & 0xf );

					i = j;
					uiLevel1Descr1 = uiLevel1Descr2;
					uiFinePTs++;
					break;
			}
		}
	} /* for */

	printk("===================================\n");
	
#if 0
	/*
	 * If we saw any pointers to course or fine page tables the first 
	 * time through the PT, go through again and print them out now.
	 */
	if( (uiCoursePTs == 0) && (uiFinePTs == 0) )
	{
		return;
	}

	/*
	 * Again, get the virtual address of the L1 page table.
	 */
	pgd = (pgd_t *)phys_to_virt(pg);

	/*
	 * Go thru the L1 page table, expanding each Fine or Course page table
	 * as we get to it.
	 */
	for( iL1EntryNum = 0 ; iL1EntryNum < 4096 ; iL1EntryNum++ )
	{
		uiLevel1Descr1 = pgd_val( *pgd++ );
		
		/* Course Page Table */
		if( (uiLevel1Descr1 & 3) == 1 )
		{
			ulLevel2table = uiLevel1Descr1 & 0xfffffc00;
			uiEntries = 256;
			uiShift = 12;
			printk("===================================\n");
			printk("Course Page Table at physical addr %08x - %08x.\n", 
				(unsigned int)ulLevel2table, 
				(unsigned int)(ulLevel2table + (uiEntries * 4) - 1)   );
		}
		/* Fine Page Table */
		else if( (uiLevel1Descr1 & 3) == 3)
		{
			ulLevel2table = uiLevel1Descr1 & 0xfffff000;
			uiEntries = 1024;
			uiShift = 10;
			printk("===================================\n");
			printk("Fine Page Table at physical addr %08x - %08x.\n", 
				(unsigned int)ulLevel2table, 
				(unsigned int)(ulLevel2table + (uiEntries * 4) - 1)   );
		}
		else
		{
			continue;
		}

		/*
		 * get a pointer to the L2 table.
		 */
		puiPT = (unsigned int *)phys_to_virt(ulLevel2table);
		
		/* THis looks wrong to me.  */
		i = 0;
		for( j = 0 ; j < uiEntries ; j++ )
		{
			uiLevel2Descr1 = puiPT[i];
			printk( "%08x: %08x\n", 
				((iL1EntryNum<<20) + (j<<uiShift)), 
				uiLevel2Descr1 ); 
		}
		
	printk("===================================\n");
	}
#endif

}
#endif /* CONFIG_MMU_TABLE_DUMP */


