/*****************************************************************************
 *
 * drivers/ide/ide-ep93xx.c
 *
 *  Copyright (c) 2003 Cirrus Logic, Inc.
 *
 * Some code based in whole or in part on:
 *
 *      linux/drivers/ide/ide-dma.c		Version 4.10	June 9, 2000
 *
 *      Copyright (c) 1999-2000	Andre Hedrick <andre@linux-ide.org>
 *      May be copied or modified under the terms of the GNU General Public
 *      License
 *
 *      Special Thanks to Mark for his Six years of work.
 *
 *      Copyright (c) 1995-1998  Mark Lord
 *      May be copied or modified under the terms of the GNU General Public
 *      License
 *  
 * Some code taken from the PowerMac implentation
 *
 *      Copyright (C) 1998-2001 Paul Mackerras & Ben. Herrenschmidt
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * ***************************************************************************
 *
 * Cirrus Logic's EP93XX architecture specific ide driver. This driver
 * supports the following ide transfer modes:
 *  PIO modes 0 - 4
 *  MDMA modes 0 - 2
 *  UDMA modes 0 - 4
 *
 ****************************************************************************/
#include <linux/ide.h>
#include <linux/delay.h>
#include <asm/io.h>
#include <asm/ide.h>
#include <asm/io.h>

#include <asm/arch/ide.h>

#include <asm/arch/dma.h>
#include <asm/hardware.h>

/*****************************************************************************
 *
 * Debugging macros
 *
 ****************************************************************************/
//#undef DEBUG
//#define DEBUG
#ifdef DEBUG
#define DPRINTK( x... )  printk( ##x )
#else
#define DPRINTK( x... )
#endif

/*
 *  IDE control register mask used when setting the CS0n, CS1n and DA bits 
 *  in this register.
 */
#define IDECR_REG_MASK          (IDECtrl_CS0n | IDECtrl_CS1n | IDECtrl_DA_MASK)

/*****************************************************************************
 *
 * Function and macros for handling dma transfers.
 *
 ****************************************************************************/
#ifdef CONFIG_BLK_DEV_IDEDMA_EP93XX

/*****************************************************************************
 *
 * Our Physical Region Descriptor (PRD) table should be large enough
 * to handle the biggest I/O request we are likely to see.  Since requests
 * can have no more than 256 sectors, and since the typical blocksize is
 * two or more sectors, we could get by with a limit of 128 entries here for
 * the usual worst case.  Most requests seem to include some contiguous blocks,
 * further reducing the number of table entries required.
 *
 * The driver reverts to PIO mode for individual requests that exceed
 * this limit (possible with 512 byte blocksizes, eg. MSDOS f/s), so handling
 * 100% of all crazy scenarios here is not necessary.
 *
 * As it turns out though, we must allocate a full 4KB page for this,
 * so the two PRD tables (ide0 & ide1) will each get half of that,
 * allowing each to have about 256 entries (8 bytes each) from this.
 *
 ****************************************************************************/
#define PRD_BYTES	8
#define PRD_ENTRIES	(PAGE_SIZE / (2 * PRD_BYTES))

/*****************************************************************************
 *
 * Global to keep track of the number of entries in the dma buffer
 * table for each transfer.
 *
 ****************************************************************************/
static unsigned int g_prd_count;

/*****************************************************************************
 *                                        
 * Global to set during the dma callback function to indicate that from
 * the dma perspective, the transfer is complete.
 *
 ****************************************************************************/
static unsigned int g_done;

/*****************************************************************************
 *
 * dma_intr() is the handler for disk read/write DMA interrupts
 *
 ****************************************************************************/
#if 0
ide_startstop_t ide_dma_intr (ide_drive_t *drive)
{
	int i;
	byte stat, dma_stat;

	dma_stat = HWIF(drive)->dmaproc(ide_dma_end, drive);
	stat = GET_STAT();			/* get drive status */
	if (OK_STAT(stat,DRIVE_READY,drive->bad_wstat|DRQ_STAT)) {
		if (!dma_stat) {
			struct request *rq = HWGROUP(drive)->rq;
			rq = HWGROUP(drive)->rq;
			for (i = rq->nr_sectors; i > 0;) {
				i -= rq->current_nr_sectors;
				ide_end_request(1, HWGROUP(drive));
			}
			return ide_stopped;
		}
		printk("%s: dma_intr: bad DMA status (dma_stat=%x)\n", 
		       drive->name, dma_stat);
	}
	return ide_error(drive, "dma_intr", stat);
}


/*****************************************************************************
 *
 * ide_build_sglist()
 *
 * Builds a table of buffers to be used by the dma.  Each buffer consists 
 * of a region of memory which is contiguous in virtual memory.  
 *
 ****************************************************************************/
static int
ide_build_sglist(ide_hwif_t *hwif, struct request *rq)
{
	struct buffer_head *buf_head;
	struct scatterlist *sg = hwif->sg_table;
	int nents = 0;

	if (hwif->sg_dma_active)
		BUG();
		
	/*
     *  Set up the direction of the command
     */
    if (rq->cmd == READ)
    {
		hwif->sg_dma_direction = EP93XX_DMA_FROMDEVICE;
    }
	else
    {
		hwif->sg_dma_direction = EP93XX_DMA_TODEVICE;
    }
    
    /*
     *  Get a pointer to the buffer head.
     */
	buf_head = rq->bh;
	
    do
    {
		unsigned char *virt_addr = buf_head->b_data;
		unsigned int size = buf_head->b_size;

        if(nents >= PRD_ENTRIES)
        {
			return 0;
        }

		while((buf_head = buf_head->b_reqnext) != NULL)
        {
			if((virt_addr + size) != (unsigned char *)buf_head->b_data)
			{
            	break;
            }
			
            size += buf_head->b_size;
		}
		memset(&sg[nents], 0, sizeof(*sg));
		sg[nents].address = virt_addr;
		sg[nents].length = size;
		nents++;
	} while (buf_head != NULL);

	/*
     *  This call to map_sg will return the number of entries
     *  for which DMAable memory could be mapped.
     */
    return(ep93xx_map_sg(sg, nents, hwif->sg_dma_direction)); 
}

/****************************************************************************
 *
 * report_drive_dmaing()
 *
 ****************************************************************************/
int report_drive_dmaing (ide_drive_t * drive)
{
	struct hd_driveid * id = drive->id;

	if((id->field_valid & 4) && (eighty_ninty_three(drive)) &&
	    (id->dma_ultra & (id->dma_ultra >> 14) & 3))
    {
		if ((id->dma_ultra >> 15) & 1)
        {
			/*
             *  UDMA enabled
             */
            printk(", UDMA(mode 7)");
		}
        else
        {
			/*
             *  UDMA enabled.
             */
            printk(", UDMA(133)");
		}
	}
    else if((id->field_valid & 4) && (eighty_ninty_three(drive)) &&
	  	  (id->dma_ultra & (id->dma_ultra >> 11) & 7))
    {
        if((id->dma_ultra >> 13) & 1)
        {
			/*
             *  UDMA enabled!
             */
            printk(", UDMA(100)");
		}
        else if((id->dma_ultra >> 12) & 1)
        {
			/*
             *  UDMA enabled.
             */
            printk(", UDMA(66)");
		}
        else
        {
			/*
             *  UDMA enabled.
             */
            printk(", UDMA(44)");
		}
	}
    else if((id->field_valid & 4) &&
		   (id->dma_ultra & (id->dma_ultra >> 8) & 7))
    {
		if ((id->dma_ultra >> 10) & 1)
        {
			/*
             * UDMA enabled.
             */
            printk(", UDMA(33)");
		}
        else if ((id->dma_ultra >> 9) & 1)
        {
			/*
             *  UDMA enabled.
             */
            printk(", UDMA(25)");
		}
        else
        {
			/*
             *  UDMA enabled.
             */
            printk(", UDMA(16)");
		}
	}
    else if(id->field_valid & 4)
    {
		/*
         *  Can be enabled.
         */
        printk(", (U)DMA capable, but not enabled");
	}
    else
    {
		/*
         *  maybe MDMA?
         */
        printk(", DMA");
	}
	
    return(1);
}


/*****************************************************************************
 *
 * ide_build_dmatable() prepares a dma request.
 * Returns 0 if all went okay, returns 1 otherwise.
 * May also be invoked from trm290.c
 *
 ****************************************************************************/
int
ide_build_dmatable(ide_drive_t *drive, ide_dma_action_t func)
{
	unsigned int *table = HWIF(drive)->dmatable_cpu;
	unsigned int count = 0;
	int i;
	struct scatterlist *sg;
    unsigned int is_trm290_chipset = 0;

	HWIF(drive)->sg_nents = i = ide_build_sglist(HWIF(drive), HWGROUP(drive)->rq);

	if (!i)
		return 0;

	sg = HWIF(drive)->sg_table;
	while (i && sg_dma_len(sg))
    {
		u32 cur_addr;
		u32 cur_len;

		cur_addr = sg_dma_address(sg);
		cur_len = sg_dma_len(sg);

		/*                                                                                      1
		 * Fill in the dma table, without crossing any 64kB boundaries.
		 * Most hardware requires 16-bit alignment of all blocks,
		 * but the trm290 requires 32-bit alignment.
		 */
		while (cur_len) {
			if (count++ >= PRD_ENTRIES) {
				printk("%s: DMA table too small\n", drive->name);
				goto use_pio_instead;
			} else {
				u32 xcount, bcount = 0x10000 - (cur_addr & 0xffff);

				if (bcount > cur_len)
					bcount = cur_len;
				*table++ = cpu_to_le32(cur_addr);
				xcount = bcount & 0xffff;
				if (is_trm290_chipset)
					xcount = ((xcount >> 2) - 1) << 16;
				if (xcount == 0x0000) {
					/* 
					 * Most chipsets correctly interpret a length of 0x0000 as 64KB,
					 * but at least one (e.g. CS5530) misinterprets it as zero (!).
					 * So here we break the 64KB entry into two 32KB entries instead.
					 */
					if (count++ >= PRD_ENTRIES) {
						printk("%s: DMA table too small\n", drive->name);
						goto use_pio_instead;
					}
					*table++ = cpu_to_le32(0x8000);
					*table++ = cpu_to_le32(cur_addr + 0x8000);
					xcount = 0x8000;
				}
				*table++ = cpu_to_le32(xcount);
				cur_addr += bcount;
				cur_len -= bcount;
			}
		}

		sg++;
		i--;
	}

	if (count) {
		if (!is_trm290_chipset)
			*--table |= cpu_to_le32(0x80000000);
		return count;
	}
	printk("%s: empty DMA table?\n", drive->name);

use_pio_instead:
	ep93xx_unmap_sg(HWIF(drive)->sg_table,
		     HWIF(drive)->sg_nents,
		     HWIF(drive)->sg_dma_direction);
	HWIF(drive)->sg_dma_active = 0;
	return 0; /* revert to PIO for this request */
}

/* Teardown mappings after DMA has completed.  */
void ide_destroy_dmatable (ide_drive_t *drive)
{
	struct scatterlist *sg = HWIF(drive)->sg_table;
	int nents = HWIF(drive)->sg_nents;

	ep93xx_unmap_sg(sg, nents, HWIF(drive)->sg_dma_direction);
	HWIF(drive)->sg_dma_active = 0;
}

/*****************************************************************************
 *
 * ide_release_dma()
 * 
 * This function releases the memory allocated for the scatter gather list
 *  and for the dma table.
 *
 ****************************************************************************/
int
ep93xx_ide_release_dma(ide_hwif_t * hwif)
{
    /*
     *  Check if we have a valid dma handle.
     */
    if(hwif->hw.dma != 0)
    {
        /*
         * Close and free the dma instance.
         */
        ep93xx_dma_free(hwif->hw.dma);
        
        /*
         *  Reset the dma handle value to null so there's no confusion later.
         */
        hwif->hw.dma = 0;
    }
    
    /*
     *  Free the memory allocated for the scatter gather list and
     *  the dma table of buffer entries.
     */
    consistent_free(hwif->dmatable_cpu, PRD_ENTRIES * PRD_BYTES,
		            hwif->dmatable_dma);
    kfree(hwif->sg_table);
    
    /*
     *  Success.
     */
	return(1);
}


/*****************************************************************************
 *
 * ep93xx_ide_setfeature()
 * 
 * This function send the set features command to the device with the 
 * specified xfer subcommand and subcommand parameters.
 *
 * This function is based mostly on the PowerMac implementation.
 *
 ****************************************************************************/
static int
ep93xx_ide_setfeature(ide_drive_t * drive, byte transfer_mode)
{
	int result = 1;
	unsigned int flags;
    unsigned long timeout = 2000;
	ide_hwif_t *hwif = HWIF(drive);
    ide_startstop_t state;
	
    
    printk("CD: ep93xx_ide_setfeature - 0x%x", (unsigned char) drive->select.all);
    /*
     *  Mask the ide interrupt.
     */
    disable_irq(hwif->irq);
    udelay(1);
	
    /*
     *  Select the specified drive.
     */ 
    SELECT_DRIVE(HWIF(drive), drive);
	SELECT_MASK(HWIF(drive), drive, 0);
	udelay(1);
	
    /*
     *  Get rid of pending error stats.
     */
    (void)GET_STAT();
	
    /*
     *  Check if the device is ready.
     */
    if(!ide_wait_stat(&state, drive, DRIVE_READY, BUSY_STAT, timeout))
    {
	    udelay(10);
        
        /*
         *  Set up the set features command.
         */
	    OUT_BYTE(drive->ctl | 2, IDE_CONTROL_REG);
	    OUT_BYTE(transfer_mode, IDE_NSECTOR_REG);
	    OUT_BYTE(SETFEATURES_XFER, IDE_FEATURE_REG);
        /*
         * TODO: should I add this?
         */
        OUT_BYTE(drive->select.all, IDE_SELECT_REG);
        
	    OUT_BYTE(WIN_SETFEATURES, IDE_COMMAND_REG);
	    udelay(1);
        
	    /*
         *  local CPU only.
         */
        __save_flags(flags);
        
        /*
         *  local CPU only -- for jiffies.
         */
        ide__sti();
        result = ide_wait_stat(&state, drive, DRIVE_READY, BUSY_STAT, timeout);
	    
        /*
         *  local CPU only
         */
        __restore_flags(flags);
        
	    OUT_BYTE(drive->ctl, IDE_CONTROL_REG);
	    
        /*
         *  Check if the device is ready.
         */
        if(result)
        {
            printk(KERN_ERR
                "ep93xx Set Feature - disk not ready after SET_FEATURE!\n");
        }
	}
    /*
     *  Device is not ready.
     */
    else
    {
		printk(KERN_ERR
            "ep93xx Set Features - disk not ready before SET_FEATURE!\n");
    }
    
	SELECT_MASK(HWIF(drive), drive, 0);
	
    /*
     *  If we successfully set a new transfer mode then those values are
     *  reflected in the id structure.
     */
    if(result == 0)
    {
		DPRINTK("IDE: set feature- we successfully set feature on the device \n");
        /*
         *  deselect all udma and mdma modes.
         */
        drive->id->dma_ultra &= ~0xFF00;
		drive->id->dma_mword &= ~0x0F00;
        
        /*
         *  Set the appropriate fields in the id structure to reflect
         *  the transfer mode set.  Also set up the dma_extra field in the
         *  hwif structure to reflect the xfer mode.
         */
        switch(transfer_mode)
        {
            case XFER_UDMA_4:
            {
                drive->id->dma_ultra |= 0x1010;
                hwif->dma_extra = 7;
                break;
            }
            case XFER_UDMA_3:
            {
                drive->id->dma_ultra |= 0x0808;
                hwif->dma_extra = 6;
                break;
            }
            case XFER_UDMA_2:
            {
                drive->id->dma_ultra |= 0x0404;
                hwif->dma_extra = 5;
                break;
            }
            case XFER_UDMA_1:
            {
                drive->id->dma_ultra |= 0x0202;
                hwif->dma_extra = 4;
                break;
            }
            case XFER_UDMA_0:
            {
                drive->id->dma_ultra |= 0x0101;
                hwif->dma_extra = 3;
                break;
            }
            case XFER_MW_DMA_2:
            {
                drive->id->dma_mword |= 0x0404;
                hwif->dma_extra = 2;
                break;
            }
            case XFER_MW_DMA_1:
            {
                drive->id->dma_mword |= 0x0202;
                hwif->dma_extra = 1;
                break;
            }
			case XFER_MW_DMA_0:
            {
                drive->id->dma_mword |= 0x0101;
                hwif->dma_extra = 0;
                break;
            }
			default:
            {
                break;
            }
		}
	}
    
	/*
     *  Restore the ide interrupt.
     */
    enable_irq(hwif->irq);
    
    DPRINTK("IDE: set feature: result= %d \n", result);
	return(result);
}
#endif // 0

/*****************************************************************************
 *
 * ep93xx_ide_callback()
 *
 * Registered with the ep93xx dma driver and called at the end of the dma
 * interrupt handler, this function should process the dma buffers.  
 *
 ****************************************************************************/
#if 0
static void
ep93xx_ide_callback(ep93xx_dma_int_t dma_int, ep93xx_dma_dev_t device,
                    unsigned int user_data)
{
    ide_drive_t * drive = (ide_drive_t *)user_data;
    ide_hwif_t * hwif = HWIF(drive);
    unsigned int temp;
    
    /*
     *  Retrieve from the dma interface as many used buffers as are
     *  available.
     */
    while(ep93xx_dma_remove_buffer(hwif->hw.dma, &temp) == 0)
    {
    }
    
    /*
     *  Add new buffers if we have any available.
     */
    while(g_prd_count)
    {
        /*
         *  Check if this is a read or write operation.
         */ 
        if(hwif->sg_dma_direction == 2)
        {
            /*
             *  Set up buffers for a read op.
             */
            temp = ep93xx_dma_add_buffer(hwif->hw.dma, hwif->dma_base,
                        *hwif->dmatable_cpu++, *hwif->dmatable_cpu++, 0,
                        g_prd_count);
        }
        else
        {
            /*
             *  Set up buffers for a write op.
             */
            temp = ep93xx_dma_add_buffer(hwif->hw.dma, *hwif->dmatable_cpu++,
                        hwif->dma_base, *hwif->dmatable_cpu++, 0,
                        g_prd_count);
        }
        
        /*
         *  Add a buffer to the dma interface.
         */
        if(temp != 0)
        {
            /*
             *  This buffer didn't make it into the dma 
             *  interface, so move the dma table pointer
             *  to the start of the current entry.
             */
            hwif->dmatable_cpu -= 4;
            break;
        }
        
        /*
         *  Decrement the count of dmatable entries
         */
        g_prd_count--;
    }
    
    /*
     *  Check if the transfer is complete.
     */
    if(dma_int == DONE)
    {
        g_done = 1;
    }
}
    
/*****************************************************************************
 *
 *  ep93xx_dma_timer_expiry()
 * 
 *  This function returns the following:
 *      1 if the ide is dmaing
 *      2 if the ide has an error
 *      4 if the ide has an interrupt
 *
 ****************************************************************************/
static int
ep93xx_idedma_timer_expiry(ide_drive_t *drive)
{
    /*
     *  Read the device error register.
     */
    byte status = IN_BYTE(HWIF(drive)->io_ports[IDE_ALTSTATUS_REG]);
    
    /*
     * Check if the ide is DMAing.
     */
    if(status & ATABSY)
    {
		/*
         *  The device is still busy DMAing.
         */
        return(WAIT_CMD);
    }
    
    /*
     * Check if an error condition exists.
     */
    if((status & ATAERR) && !(status & ATADRDY))
    {
        /*
         *  Read the device error register.
         */
        status = IN_BYTE(HWIF(drive)->io_ports[IDE_ERROR_REG]);
        
	    /*
         *  Check if the command was aborted.
         */
        if(status && ATAABRT)
        {
		    /*
             *  the command was aborted.  Get the status register again
             *  and return it.
             */
            status = IN_BYTE(HWIF(drive)->io_ports[IDE_STATUS_REG]);
		    return(ide_error(drive, "dma_timer_expiry", status));
	    }
    }

	DPRINTK("%s: ep93xx_idedma_timer_expiry: status == 0x%x\n", drive->name,
            status);
	
	return(0);
}

/*****************************************************************************
 *
 *	ep93xx_ide_rwdma()
 *
 *  This function performs the read and write dma setup.
 *
 ****************************************************************************/
static int
ep93xx_ide_rwdma(ide_drive_t * drive, ide_dma_action_t action,
                 unsigned int flags)
{
    ide_hwif_t * hwif = HWIF(drive);
    int result;
    
    /*
     *  Determine if we need the the MDMA or UDMA data register.
     */
    if(action == ide_dma_read)
    {
        if(hwif->dma_extra <= 2)
        {
            flags |= WS_IDE_MDMA_READ_WRITE;
            
            /*
             *  MDMA data register address.
             */
            hwif->dma_base = (unsigned int)&ep93xx_ide_regs->IDEMDMADATAIN;
        }
        else
        {
            flags |= WS_IDE_UDMA_READ;
            
            
            /*
             *  UDMA data register address.
             */
            hwif->dma_base = (unsigned int)&ep93xx_ide_regs->IDEUDMADATAIN;
        }
    }
    else
    {
        if(hwif->dma_extra <= 2)
        {
            flags |= WS_IDE_MDMA_READ_WRITE;
            
            
            /*
             *  MDMA data register address.
             */
            hwif->dma_base = (unsigned int)&ep93xx_ide_regs->IDEMDMADATAOUT;
        }
        else
        {
            flags |= WS_IDE_UDMA_WRITE;
            
            /*
             *  UDMA data register address.
             */
            hwif->dma_base = (unsigned int)&ep93xx_ide_regs->IDEUDMADATAOUT;
        }
    }
    
    /*
     *  Configure the dma interface for this IDE operation.
     */
    if(ep93xx_dma_config(hwif->hw.dma, 0, flags, ep93xx_ide_callback,
        (unsigned int)drive) != 0)
    {
        return(1);
    }
    
    /*
     *  Build the table of dma-able buffers.
     */
    if (!(g_prd_count = ide_build_dmatable(drive, action)))
    {
        /*
         *  Fail, try PIO instead of DMA
         */
        return(1);
    }
    
    /*
     *  Indicate that we're waiting for dma.
     */
	drive->waiting_for_dma = 1;    
    
    /*
     *  Prepare the dma interface with some buffers from the
     *  dma_table.
     */
    do
    {
        /*
         *  Add a buffer to the dma interface.
         */
        if(action == ide_dma_read)
        {
            /*
             *  Set up buffers for a read op.
             */
            result = ep93xx_dma_add_buffer(hwif->hw.dma, hwif->dma_base,
                        *hwif->dmatable_cpu++, *hwif->dmatable_cpu++, 0,
                        g_prd_count);
        }
        else
        {
            /*
             *  Set up buffers for a write op.
             */
            result = ep93xx_dma_add_buffer(hwif->hw.dma, *hwif->dmatable_cpu++,
                        hwif->dma_base, *hwif->dmatable_cpu++, 0,
                        g_prd_count);
        }
        
        
        if(result != 0)
        {
            /*
             *  This buffer didn't make it into the dma 
             *  interface, so move the dma table pointer
             *  to the start of the current entry.
             */
            hwif->dmatable_cpu -= 4;
            break;
        }
        
        /*
         *  Decrement the count of dmatable entries
         */
        g_prd_count--;
         
    } while(g_prd_count);
    
    /*
     * Check if this is a disk or cd drive.
     */
    if(drive->media == ide_disk)
    {
    /*
     *  Set up the device interface with a handler to handle the end of the
     *  transfer.  This interrupt handler is called when the IDE interrupt
     *  is triggered, not the DMA interrupt.
     */
#ifdef CONFIG_BLK_DEV_IDEDMA_TIMEOUT

    ide_set_handler(drive, &ide_dma_intr, WAIT_CMD, NULL);
    
#else /* !CONFIG_BLK_DEV_IDEDMA_TIMEOUT */

    ide_set_handler(drive, &ide_dma_intr, WAIT_CMD,
                    ep93xx_idedma_timer_expiry);
            
#endif /* CONFIG_BLK_DEV_IDEDMA_TIMEOUT */

    /*
     *  Issue the command to the device.
     */
    if(action == ide_dma_read)
    {
        OUT_BYTE(WIN_READDMA, IDE_COMMAND_REG);
    }
    else
    {
        OUT_BYTE(WIN_WRITEDMA, IDE_COMMAND_REG);
        }
    }
    
    /*
     *  Configure the ep93xx ide controller for a dma read operation.
     */
    SELECT_READ_WRITE(hwif, drive, action);
    
    /*
     *  Success.
     */
    return(0);
}                    

/*****************************************************************************
 *
 *	ep93xx_config_ide_dma()
 *
 *  This function allocates dma-able memory and configures the ep93xx dma
 *  interface for use with the ide interface.
 *
 ****************************************************************************/
static void
ep93xx_config_ide_dma(ide_hwif_t *hwif)
{
    DPRINTK("ep93xx_setup_dma start \n");
    
    /*
     *  Allocate dma-able memory space, in one consistent chunk.  The call
     *  to alloc_consistent will return a virtual address and fill in a dma
     *  physical address.  (arch/arm/mm/consistent.c)
     *
     */
    hwif->dmatable_cpu = consistent_alloc(GFP_KERNEL|GFP_DMA,
							  PRD_ENTRIES * PRD_BYTES, &hwif->dmatable_dma);
	
    /*
     *  Check if we allocated memory for dma
     */
    if(hwif->dmatable_cpu == NULL)
    {
	    printk(" -- ERROR, UNABLE TO ALLOCATE DMA TABLES\n");
        return;
    }
    
    printk("    %s: EP93XX-DMA at 0x%x - 0x%x \n", hwif->name,
           hwif->dmatable_dma, (unsigned int)(hwif->dmatable_dma + (PRD_ENTRIES * PRD_BYTES)));
    
	/*
     *  Allocate memory for the scatterlist structures.
     */
    hwif->sg_table = kmalloc(sizeof(struct scatterlist) * PRD_ENTRIES,
				 GFP_KERNEL);
	
    /*
     *  Check if we allocated the memory we expected to.
     */ 
    if(hwif->sg_table == NULL)
    {
		consistent_free(hwif->dmatable_cpu, PRD_ENTRIES * PRD_BYTES,
				        hwif->dmatable_dma);
	    printk(" -- ERROR, UNABLE TO ALLOCATE DMA TABLES\n");
        return;
	}

	/*
     *  Set the dma handle value to NULL to start with.
     */
    hwif->hw.dma = 0;
     
    /*
     *  Open an instance of the ep93xx dma interface.
     */
    if(ep93xx_dma_request(&hwif->hw.dma, hwif->name, DMA_IDE) != 0)
    {
		consistent_free(hwif->dmatable_cpu, PRD_ENTRIES * PRD_BYTES,
				        hwif->dmatable_dma);
        kfree(hwif->sg_table);
	    printk(" -- ERROR, Unable to open a dma channel.\n");
        return;
    }
         
	DPRINTK("ep93xx_setup_dma end \n");
}

/*****************************************************************************
 *
 * ep93xx_config_ide_device()
 *
 * This function sets up the ep93xx ide device for a dma transfer by first
 * probing to find the best dma mode supported by the device.
 *
 ****************************************************************************/
static unsigned int
ep93xx_config_ide_device(ide_drive_t * drive)
{
    struct hd_driveid * id  = drive->id;
    ide_hwif_t  * hwif = HWIF(drive);
    unsigned int flags;
    byte transfer = 0;
	
    DPRINTK("%s: ep93xx_config_ide_device - dma on\n", hwif->name);
  
    /*
     *  mask off interrupts
     */
    local_irq_save(flags);
        
    /*
     * check if the device supports a udma mode.
     */
    if(transfer = (id->dma_ultra & 0x001F))
    {
        DPRINTK("IDE: udma xfer support, transfer = %d \n", transfer);
        
        /*
         *  UDMA mode 4 support.
         */
        if(transfer & 0x10)
        {
            transfer = XFER_UDMA_4;
        }
        /*
         *  UDMA mode 3 support.
         */
        else if(transfer & 0x08)
        {
            transfer = XFER_UDMA_3;
        }
        /*
         *  UDMA mode 2.
         */
        else if(transfer & 0x04)
        {
            transfer = XFER_UDMA_2;
        }
        /*
         *  UDMA mode 1.
         */
        else if(transfer & 0x02)
        {
            transfer = XFER_UDMA_1;
        }
        /*
         *  UDMA mode 0.
         */
        else if(transfer & 0x01)
        {
            transfer = XFER_UDMA_0;
        }
    }
    /*
     *  The device doesn't support a udma mode, so check if it
     *  supports an mdma mode.
     */
    else if(transfer = (drive->id->dma_mword & 0x0007))
    {
        DPRINTK("IDE: mdma support, transfer = %d \n", transfer);
            
        /*
         *  MDMA mode 2.
         */
        if(transfer & 0x04)
        {
            transfer = XFER_MW_DMA_2;
        }
        /*
         *  MDMA mode 1.
         */
        else if(transfer & 0x02)
        {
            transfer = XFER_MW_DMA_1;
        }
        /*
         *  MDMA mode 0.
         */
        else if(transfer & 0x01)
        {
            transfer = XFER_MW_DMA_0;
        }
    }
    
    if(transfer == 0)
    {
        DPRINTK("IDE: config device failed \n");
        
        /*
         *  Fail.
         */
        return(1);
    }
    
    /*
     *  Use the set features command to set up the device for a dma
     *  transaction.
     */
    DPRINTK("IDE: calling set feature \n");
    
    if(ep93xx_ide_setfeature(drive, transfer) == 0)
    {
        /*
         *  If there was no error setting up the command,
         *  then turn on dma.
         */
        DPRINTK("IDE: calling dmaproc w/ ide_dma_on \n");
        return(hwif->dmaproc(ide_dma_on, drive));
    }
    
    DPRINTK("IDE: config device failure \n");
    /*
     *  Fail
     */
    return(1);
}

/*****************************************************************************
 *
 *  ep93xx_set_pio()                                                
 *                                                      
 *  Configures the ep93xx controller for a PIO mode transfer.
 *              
 ****************************************************************************/
static void
ep93xx_set_pio(void)
{    
    /*
     *  Disable the interface.
     */
    ep93xx_ide_regs->IDECFG.Field.IDEEN = 0;
    
    /*
     *  Enable PIO mode of operation.
     */
    ep93xx_ide_regs->IDECFG.Field.PIO = 1;
    ep93xx_ide_regs->IDECFG.Field.UDMA = 0;
    ep93xx_ide_regs->IDECFG.Field.MDMA = 0;
    
    /*
     *  Enable the IDE interface.
     */
    ep93xx_ide_regs->IDECFG.Field.IDEEN = 1;
}

/*****************************************************************************
 *
 *  ep93xx_rwproc()                                                
 *                                                      
 *  Initializes the ep93xx IDE controller interface with the transfer type, 
 *  transfer mode, and transfer direction.
 *              
 ****************************************************************************/
static void
ep93xx_rwproc(ide_drive_t * drive, ide_dma_action_t action)
{
    ide_hwif_t * hwif = HWIF(drive);
    int write = 0;
    
    /*
     *  If this is a read action set reading to 1.
     */
    if(action == ide_dma_write)
    {
        write = 1;
    }
    
    
    /*
     *  Configure the IDE controller for the specified transfer mode.
     */
    switch(hwif->dma_extra)
    {        
        /*
         *  Confiure for an MDMA operation.
         */
        case 0:
        case 1:
        case 2:
        {
            ep93xx_ide_regs->IDECFG.Field.PIO = 0;
            ep93xx_ide_regs->IDECFG.Field.UDMA = 0;
            ep93xx_ide_regs->IDECFG.Field.MDMA = 1;
            ep93xx_ide_regs->IDECFG.Field.MODE = hwif->dma_extra;
            ep93xx_ide_regs->IDECFG.Field.IDEEN = 1;
            ep93xx_ide_regs->IDEMDMAOP.Field.RWOP = write;
            ep93xx_ide_regs->IDEMDMAOP.Field.MEN = 1;
            break;
        }

        /*
         *  Configure for a UDMA operation.
         */
        case 3:
        case 4:
        case 5:
        case 6:
        case 7:
        {
            ep93xx_ide_regs->IDECFG.Field.PIO = 0;
            ep93xx_ide_regs->IDECFG.Field.MDMA = 0;
            ep93xx_ide_regs->IDECFG.Field.UDMA = 0;
            ep93xx_ide_regs->IDEUDMAOP.Field.UEN = 0;
            ep93xx_ide_regs->IDECFG.Field.IDEEN = 1;
            ep93xx_ide_regs->IDECFG.Field.MODE = hwif->dma_extra - 3;
            ep93xx_ide_regs->IDECFG.Field.UDMA = 1;
            ep93xx_ide_regs->IDEUDMAOP.Field.RWOP = write;
            ep93xx_ide_regs->IDEUDMAOP.Field.UEN = 1;
            break;;
        }

        default:
        {
            break;
        }
    }

}

/*****************************************************************************
 *
 * ep93xx_dmaproc()
 *
 * This function performs the dma read, write and abort operations.
 *
 ****************************************************************************/
int
ep93xx_dmaproc(ide_dma_action_t action, ide_drive_t *drive)
{
    ide_hwif_t *hwif = HWIF(drive);

    switch(action)
    {
		case ide_dma_off:
        {
			printk("%s: DMA disabled\n", drive->name);
        }
		case ide_dma_off_quietly:
        {
			DPRINTK("%s: ide_dma_off_quietly: \n", drive->name);
			
            /*
             *  Clear the using_dma field to indicate that dma is disabled
             *  for this drive.
             */
            drive->using_dma = 0;
            
            /*
             *  Release the dma channel and all memory allocated for dma
             *  purposes.
             */
            ep93xx_ide_release_dma(hwif);

            /*
             *  Success.
             */
			return(0);
        }
		case ide_dma_on:
        {
			DPRINTK("%s: ide_dma_on: \n", drive->name);
			
            if(drive->using_dma)
            {
                DPRINTK("%s: ide_dma_on, already on! \n", drive->name);
                return(0);
            }
            
            /*
             *  Set the using dma field to indicate that dma is enabled.
             */
            drive->using_dma = 1;
            
            /*
             *  Enable the ep93xx dma interface for use with IDE.
             */
			ep93xx_config_ide_dma(hwif);
            
			/*
             *  Success.
             */
            return(0);
        }
		case ide_dma_check:
        {
			DPRINTK("%s: ide_dma_check \n", drive->name);
            
            /*
             *  Check if the drive supports multiword dma or udma modes.
             *  If it does, then call set the device up for that 
             *  type of dma transfer, and call ep93xx_dmaproc with ide_dma_on.
             */
            return(ep93xx_config_ide_device(drive));
        }
		case ide_dma_read:
        {
			unsigned int flags_m2m;
            
            DPRINTK("%s: ide_dma_read: \n", drive->name);
            
            /*
            if(drive->media != ide_disk)
            {
                return(0);
            }
            */
            
            /*
             *  Configure DMA M2M channel flags for a source address hold, h/w
             *  initiated P2M transfer.
             */
            flags_m2m = (SOURCE_HOLD | TRANSFER_MODE_HW_P2M);
            
            /*
             *  This function handles the rest of the setup for a read
             *  operation.
             */
            return(ep93xx_ide_rwdma(drive, action, flags_m2m)); 
        }    
		case ide_dma_write:
        {
			unsigned int flags_m2m;
            
            DPRINTK("%s: ide_dma_write: \n", drive->name);
            
            /*if(drive->media != ide_disk)
            {
                return(0);
            }
            */
            
            /*
             *  Configure DMA M2M channel flags for a destination address
             *  hold, h/w initiated M2P transfer.
             */
            flags_m2m = (DESTINATION_HOLD | TRANSFER_MODE_HW_M2P);
                                     
            /*
             *  This function handles the rest of the setup for a read
             *  operation.
             */
            return(ep93xx_ide_rwdma(drive, action, flags_m2m));
        }		
        case ide_dma_begin:
        {
			DPRINTK("%s: ide_dma_begin: \n", drive->name);
			
            /* Note that this is done *after* the cmd has
			 * been issued to the drive, as per the BM-IDE spec.
			 * The Promise Ultra33 doesn't work correctly when
			 * we do this part before issuing the drive cmd.
			 */

			/*
             *  Init g_done.
             */
            g_done = 0;
            
            /*
             *  Start the dma transfer.
             */
            ep93xx_dma_start(hwif->hw.dma, 1, NULL);
            
            /*
             * TODO: not sure if we need to keep track of the end
             * of the transfer from the dma perspective, but
             * for now, let's do it.
             */
            while(!g_done)
            {
            }
            
            DPRINTK("@@@  DMA thinks it's done!\n");
            			
			return(0);
        }
		/*
         *  Returns 1 if an error occured, and 0 otherwise.
         */
        case ide_dma_end:
        {
			unsigned char dev_stat;
            
            DPRINTK("%s: ide_dma_end: \n", drive->name);
			
            /*
             *  Indicate there's no dma transfer currently in progress.
             */
            drive->waiting_for_dma = 0;
            
            /*
             *  Put the dma interface into pause mode.
             */
            ep93xx_dma_pause(hwif->hw.dma, 1, 0); 
            
            /*
             *  Enable PIO mode on the IDE interface.
             */
            ep93xx_set_pio();
            
            /*
             *  Purge the contents of the dma table of buffer entries.
             */
			ide_destroy_dmatable(drive);
            
            /*
             *  Read the ide device status register.  This clears the interrupt.
             */
			dev_stat = IN_BYTE(IDE_STATUS_REG);
            
            /*
             *  If the error bit is set in the device status register,
             *  return a positive value.
             */
            if((dev_stat & ERR_STAT) || !(dev_stat & READY_STAT))
            {
                return(dev_stat);
            }
            else
            {
                return(0);
            }
        }
		/*
         *  Returns 1 if the IDE interrupt is asserted, and 0 otherwise.
         */
        case ide_dma_test_irq:
        {
            /*
             *  Return the value of the IDE interrupt bit.
             */
			return(ep93xx_ide_regs->IDECR.Field.INTRQ); 
        }
  		case ide_dma_bad_drive:
		case ide_dma_good_drive:
        {
			DPRINTK("%s: ide_dma_bad_good_drive: \n", drive->name);
			
            /*
             *  TODO: need to implement.
             */
            /*return(check_drive_lists(drive, (action == ide_dma_good_drive)));*/
            return(0);
        }
		case ide_dma_verbose:
        {
			DPRINTK("%s: ide_dma_verbose: \n", drive->name);
            
            /*
             *  TODO:
             */
			return(report_drive_dmaing(drive));
        }
		case ide_dma_timeout:
        {
			/*
             *  TODO: need to implement this.
             */
            DPRINTK("%s: ide_dma_timeout: \n", drive->name);
			/* FIXME: Many IDE chipsets do not permit command file register access
			 * FIXME: while the bus-master function is still active.
			 * FIXME: To prevent deadlock with those chipsets, we must be extremely
			 * FIXME: careful here (and in ide_intr() as well) to NOT access any
			 * FIXME: registers from the 0x1Fx/0x17x sets before terminating the
			 * FIXME: bus-master operation via the bus-master control reg.
			 * FIXME: Otherwise, chipset deadlock will occur, and some systems will
			 * FIXME: lock up completely!!
             */

            return(0);
        }
		case ide_dma_retune:
		case ide_dma_lostirq:
        {
			DPRINTK("%s: ide_dma_lostirq\n", drive->name);
			return(1);
        }
		default:
        {
			DPRINTK("%s: unsupported action: %d\n", drive->name, action);
			return(1);
        }
	}
}

#endif /* CONFIG_BLK_DEV_IDEDMA_EP93XX */

#endif //0

/*****************************************************************************
 *
 *  functions to set up the IDE control register and data register to read 
 *  or write a byte of data to/from the specified IDE device register.
 *
 ****************************************************************************/
static void
ep93xx_ide_outb(u8 b, unsigned long addr)
{
    unsigned int uiIDECR;
    
    DPRINTK("ep93xx_ide_outb, addr:%02lx data: %04x\n", addr, b);

    //
    // Write the address out (CS0n, CS1n, DA) fields only.
    //
    uiIDECR = (inl(IDECtrl) & ~IDECR_REG_MASK) | (addr & IDECR_REG_MASK);
    outl(uiIDECR, IDECtrl);
    
    outl((unsigned int)b, IDEDataOut);

    //
    // Toggle Write signal.
    //
    uiIDECR &= ~IDECtrl_DIOWn;
    outl(uiIDECR, IDECtrl);
    uiIDECR |= IDECtrl_DIOWn; 
    outl(uiIDECR, IDECtrl);
}

static void
ep93xx_ide_outbsync(ide_drive_t *drive, u8 b, unsigned long addr)
{
    unsigned int uiIDECR;
    
    DPRINTK("ep93xx_ide_outbsync, addr:%02lx data: %04x\n", addr, b);

    //
    // Write the address out (CS0n, CS1n, DA) fields only.
    //
    uiIDECR = (inl(IDECtrl) & ~IDECR_REG_MASK) | (addr & IDECR_REG_MASK);
    outl(uiIDECR, IDECtrl);

    outl((unsigned int)b, IDEDataOut);

    //
    // Toggle Write signal.
    //
    uiIDECR &= ~IDECtrl_DIOWn;
    outl(uiIDECR, IDECtrl);
    uiIDECR |= IDECtrl_DIOWn; 
    outl(uiIDECR, IDECtrl);
}

static unsigned char
ep93xx_ide_inb(unsigned long addr)
{
    unsigned int uiIDECR;

    DPRINTK("ep93xx_ide_inb addr: %04lx ",addr);

    //
    // Write the address out (CS0n, CS1n, DA) fields only.
    //
    uiIDECR = (inl(IDECtrl) & ~IDECR_REG_MASK) | (addr & IDECR_REG_MASK);
    outl(uiIDECR, IDECtrl);
    
    //
    // Toggle Read signal.
    //
    uiIDECR &= ~IDECtrl_DIORn; 
    outl(uiIDECR, IDECtrl);
    uiIDECR |= IDECtrl_DIORn; 
    outl(uiIDECR, IDECtrl);

    DPRINTK("data: %02x\n", inl(IDEDataIn));
    return((unsigned char)inl(IDEDataIn));
}

/*****************************************************************************
 *
 *  functions to set up the IDE control register and data restister to read 
 *  or write 16 bits of data to/from the specified IDE device register.
 *  These functions should only be used when reading/writing data to/from
 *  the data register.
 *
 ****************************************************************************/
static void
ep93xx_ide_outw(u16 w, unsigned long addr)
{
    unsigned int uiIDECR;

    DPRINTK("ep93xx_ide_outw addr: %04lx data: %04x\n",addr, w);

    //
    // Write the address out (CS0n, CS1n, DA) fields only.
    //
    uiIDECR = (inl(IDECtrl) & ~IDECR_REG_MASK) | (addr & IDECR_REG_MASK);
    outl(uiIDECR, IDECtrl);

    outl((unsigned int)w, IDEDataOut);

    //
    // Toggle Write signal.
    //
    uiIDECR &= ~IDECtrl_DIOWn;
    outl(uiIDECR, IDECtrl);
    uiIDECR |= IDECtrl_DIOWn; 
    outl(uiIDECR, IDECtrl);
}

static u16
ep93xx_ide_inw(unsigned long addr)
{
    unsigned int uiIDECR;

    DPRINTK("ep93xx_ide_inw addr: %04lx ",addr);

    //
    // Write the address out (CS0n, CS1n, DA) fields only.
    //
    uiIDECR = (inl(IDECtrl) & ~IDECR_REG_MASK) | (addr & IDECR_REG_MASK);
    outl(uiIDECR, IDECtrl);

    //
    // Toggle Read signal.
    //
    uiIDECR &= ~IDECtrl_DIORn; 
    outl(uiIDECR, IDECtrl);
    uiIDECR |= IDECtrl_DIORn; 
    outl(uiIDECR, IDECtrl);

    DPRINTK("data: %04x\n", inl(IDEDataIn));
    return((unsigned short)inl(IDEDataIn));
}

/*****************************************************************************
 *
 *  functions to read/write a block of data to/from the ide device using
 *  PIO mode.
 *
 ****************************************************************************/
static void
ep93xx_ide_insw(unsigned long addr, void * buf, u32 count)
{
    unsigned short * data = (unsigned short *)buf;
    unsigned char status;
    DPRINTK("ep93xx_ide_insw\n");
    
    /*
     *  Read in data from the data register 16 bits at a time.
     */
    while(count)
    {
        /*
         *  Read the status register.
         */
        status = ep93xx_ide_inb((STATUSREGISTER << 2) + 2);
        
        /*
         *  Check for the BSY to be clear and DRQ to be set.
         */
        if((status & ATADRQ) && !(status & ATABSY))
        {
            *data = ep93xx_ide_inw(addr);
            data++;
            count--;
        }
    }
}

static void
ep93xx_ide_outsw(unsigned long addr, void * buf, u32 count)
{
    unsigned short * data = (unsigned short *)buf;
    unsigned char status;

    DPRINTK("ep93xx_ide_outsw\n");
    
    /*
     *  Write out data to the data register 16 bits at a time.
     */
    while(count)
    {
        /*
         *  Read the status register.
         */
        status = ep93xx_ide_inb((STATUSREGISTER << 2) + 2);
        
        /*
         *  Check for the BSY to be clear and DRQ to be set.
         */
        if((status & ATADRQ) && !(status & ATABSY))
        {
            ep93xx_ide_outw(*data, addr);
            data++;
            count--;
        }
    }
}

static void
ep93xx_ide_outl(u32 l, unsigned long addr)
{
    printk("ep93xx_ide_outl\n");
}

static u32
ep93xx_ide_inl(unsigned long addr)
{
    printk("ep93xx_ide_inl\n");
    return(0);
}

static void
ep93xx_ide_outsl(unsigned long addr, void * buf, u32 count)
{
    printk("ep93xx_ide_inl\n");
}

static void
ep93xx_ide_insl(unsigned long addr, void * buf, u32 count)
{
    printk("ep93xx_ide_inl\n");
}

/*****************************************************************************
 *
 *  Functions to read/write a block of data to/from the ide device using
 *  PIO mode, using an ATAPI interface.
 *
 ****************************************************************************/
static void
ep93xx_ide_insw_atapi(unsigned int addr, void * buf, int count)
{
    DPRINTK("ep93xx_ide_insw_atapi \n");
    //
    // Convert count from bytes to half words.
    //
    
    if(count % 2)
    {
        count = count / 2;
        count++;
    }
    else
    {
        count = count / 2;    
    }
    
    //
    // Call the function which will read in the data.
    //
    ep93xx_ide_insw(addr, buf, count);
}

static void
ep93xx_ide_outsw_atapi(unsigned int addr, void * buf, int count)
{
    DPRINTK("ep93xx_ide_outsw_atapi \n");
    //
    // Convert count from bytes to half words.
    //
    if(count % 2)
    {
        count = count / 2;
        count++;
    }
    else
    {
        count = count / 2;    
    }
    
    //
    // Call the function which will write out the data.
    //
    ep93xx_ide_outsw(addr, buf, count);
}


void
ep93xx_ata_input_data(ide_drive_t * drive, void * buffer, u32 count)
{
    DPRINTK("ep93xx_ata_input_data \n");
    /*
     *  Read in the specified number of half words from the ide interface.
     */
    ep93xx_ide_insw(IDE_DATA_REG, buffer, count << 1);
}

void
ep93xx_ata_output_data(ide_drive_t * drive, void * buffer, u32 count)
{
    DPRINTK("ep93xx_ata_output_data \n");
    /*
     *  write the specified number of half words from the ide interface
     *  to the ide device.
     */
    ep93xx_ide_outsw(IDE_DATA_REG, buffer, count << 1);
}

void
ep93xx_atapi_input_bytes(ide_drive_t * drive, void * buffer, u32 count)
{
    DPRINTK("ep93xx_atapi_input_bytes \n");
    /*
     *  read in the specified number of bytes from the ide interface.
     */
    ep93xx_ide_insw_atapi(IDE_DATA_REG, buffer, count);
}

void
ep93xx_atapi_output_bytes(ide_drive_t * drive, void * buffer, u32 count)
{
    DPRINTK("ep93xx_atapi_output_bytes \n");
    /*
     *  Write the specified number of bytes from the ide interface
     *  to the ide device.
     */
    ep93xx_ide_outsw_atapi(IDE_DATA_REG, buffer, count);
}

/*****************************************************************************
 *
 * ep93xx_ideproc()
 * 
 * This function is used for most PIO data transfers to/from the IDE
 * interface.
 *
 ****************************************************************************/
///static void
///ep93xx_ideproc(ide_ide_action_t action, ide_drive_t * drive, void * buffer,
///               unsigned int count)
///{
///    switch(action)
///    {
///        case ideproc_ide_input_data:
///        {
///            /*
///             *  Read in the specified number of half words from the ide interface.
///             */
///            ep93xx_ide_insw(IDE_DATA_REG, buffer, count << 1);
///            
///            break;
///        
///        }
///        
///        case ideproc_ide_output_data:
///        {
///            /*
///             *  write the specified number of half words from the ide interface
///             *  to the ide device.
///             */
///         ep93xx_ide_outsw(IDE_DATA_REG, buffer, count << 1);
///            
///            break;
///        
///        }
///        
///        case ideproc_atapi_input_bytes:
///        {
///            /*
///             *  read in the specified number of bytes from the ide interface.
///             */
///            ep93xx_ide_insw_atapi(IDE_DATA_REG, buffer, count);
///            
///            break;
///        }
///        
///        case ideproc_atapi_output_bytes:
///        {
///            /*
///             *  Write the specified number of bytes from the ide interface
///             *  to the ide device.
///             */
///            ep93xx_ide_outsw_atapi(IDE_DATA_REG, buffer, count);
///            
///            break;
///        }
///        
///        default:
///        {
///            break;
///        } 
///    }
///}

/*****************************************************************************
 *
 * ep93xx_ide_init()
 *
 * This function sets up a pointer to the ep93xx specific ideproc funciton.
 *
 ****************************************************************************/
void
ep93xx_ide_init(struct hwif_s * hwif)
{
    //    ide_hwif_t * hwif = (ide_hwif_t *)pointer;
    
    DPRINTK("ep93xx_ide_init \n");
    
    /*
     *  Set up the function pointer within the hwif structure for the ideproc
     *  function.
     */
    hwif->ata_input_data = ep93xx_ata_input_data;
    hwif->ata_output_data = ep93xx_ata_output_data;
    hwif->atapi_input_bytes = ep93xx_atapi_input_bytes;
    hwif->atapi_output_bytes = ep93xx_atapi_output_bytes;

    hwif->OUTB = ep93xx_ide_outb;
    hwif->OUTBSYNC = ep93xx_ide_outbsync;
    hwif->OUTW = ep93xx_ide_outw;
    hwif->OUTL = ep93xx_ide_outl;
    hwif->OUTSW = ep93xx_ide_outsw; 
    hwif->OUTSL = ep93xx_ide_outsl;

    hwif->INB = ep93xx_ide_inb;
    hwif->INW = ep93xx_ide_inw;
    hwif->INL = ep93xx_ide_inl;
    hwif->INSW = ep93xx_ide_insw;
    hwif->INSL = ep93xx_ide_insl;
    
    
    
    
#ifdef CONFIG_BLK_DEV_IDEDMA_EP93XX
    DPRINTK("ep93xx_ide_init- dma enabled \n");
    
    /*
     *  Set up the function pointer within the hwif structure for the dmaproc
     *  function
     */
    hwif->dmaproc = ep93xx_dmaproc;
    
    /*
     *  Set up the function pointer within the hwif structure for the rwproc
     *  function, which is used to set up the IDE controller for a dma read
     *  or write operation.
     */
    hwif->rwproc = ep93xx_rwproc;
    
    /*
     *  Init the dma handle to 0.
     */
    hwif->hw.dma = 0;
    
    /*
     *  Init the dma_extra field to 8, which is an invalid value.
     */
    hwif->dma_extra = 8;
#endif
}


