/* 	Name       : 	ProjectClient.c
	Author     : 	Andrew Shannon
	Description: 	Real Time Embedded Project
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/mman.h>
#include <netinet/in.h>
#include <netdb.h>
#include <arpa/inet.h>
#include <fcntl.h>

#define MSG_SIZE 40

void displayLED(int button, unsigned long *pbdr, unsigned long *pbddr);
int semaphoreWait(int myIP, int sock, struct sockaddr_in from, socklen_t length);
void semaphorePost(int myIP, int sock, struct sockaddr_in server, socklen_t fromlen);

void error(const char *msg)
{
    perror(msg);
    exit(0);
}

int main(void) {

	//Setup FIFO
	int fifo1;
	if((fifo1 = open("/dev/rtf/1", O_RDONLY)) == -1)
		return -1;

	int button;
	int i;
	/*----------set up socket connections----------*/
	int sock, length;
	int boolval = 1;		// for a socket option
	socklen_t fromlen;
	struct sockaddr_in server;
	struct sockaddr_in from;

	sock = socket(AF_INET, SOCK_DGRAM, 0); // Creates socket. Connectionless.
	if (sock < 0)
	   error("Opening socket");

	length = sizeof(server);			// length of structure
	bzero(&server,length);			// sets all values to zero. memset() could be used
	server.sin_family = AF_INET;		// symbol constant for Internet domain
	server.sin_addr.s_addr = INADDR_ANY;		// IP address of the machine on which
											// the server is running
	server.sin_port = htons(2000);	// port number is 2000

	// binds the socket to the address of the host and the port number
	if (bind(sock, (struct sockaddr *)&server, length) < 0)
	   error("binding");

	// set broadcast option
	if (setsockopt(sock, SOL_SOCKET, SO_BROADCAST, &boolval, sizeof(boolval)) < 0)
	{
		printf("error setting socket options\n");
		exit(-1);
	}

	fromlen = sizeof(struct sockaddr_in);	// size of structure
	server.sin_addr.s_addr = inet_addr("10.3.52.255"); //Set to broadcast

	/*----------set up registers for LEDs----------*/
	//Variable declaration
	int fd;
	unsigned long *ptr;
	unsigned long *pbdr;
	unsigned long *pbddr;

	//Open file and map to addresses on board
	fd = open("/dev/mem",O_RDWR);
	ptr = mmap(NULL, getpagesize(), PROT_READ|PROT_WRITE, MAP_SHARED, fd, 0x80840000);

	pbdr = ptr + 0x01;	//Map data register address for port B
	pbddr = ptr + 0x05;	//Map direction register address for port B

	*pbddr |= 0xE0; 	//Set LEDs as outputs
	*pbdr &= ~0xE0;	//turn off LEDs

	/*----------Get own ip----------*/
	//File for pipe & string for reading
	FILE *pipein_fp;
	char readLine[80];

	//Open pipe & run ifconfig, streaming to pipe
	if (( pipein_fp = popen("ifconfig", "r")) == NULL)
	{
			perror("popen");
			exit(1);
	}

	//Read first line and overwrite with second line
	fgets(readLine, 80, pipein_fp);
	fgets(readLine, 80, pipein_fp);

	//Close pipe
	pclose(pipein_fp);

	//Separate readLine into tokens to get ip
	strtok(readLine, ":");

	//Get ip from strtok & convert to int
	int myIP[4];
	for(i=0;i<4;i++) {
		myIP[i] = atoi(strtok(NULL, "."));
	}

	/*----------Wait for startup message----------*/
	//Set message as empty string
	char message[MSG_SIZE];
	bzero(message, MSG_SIZE);

	printf("Waiting for server...\n");

	//Wait for message from server
	if(recvfrom(sock, message, MSG_SIZE, 0, (struct sockaddr *)&from, &length) < 0)
		error("Receive message error");

	printf("Got startup message\n");

	//Set message
	bzero(message, MSG_SIZE);
	sprintf(message, "@ %d.%d.%d.%d", myIP[0], myIP[1], myIP[2], myIP[3]);

	//Reply with IP
	if (sendto(sock, message, 40, 0, (struct sockaddr *)&server, fromlen)  < 0)
		error("Send message error");

	printf("Sent IP: %s\n", message);
	printf("\n");

	sleep(1);	//Wait for other computers to reply

	int pattCount=3;
	int success=1;


	/*--------Game Loop---------*/
	while(success) {

		int newLoop=0;

		printf("Waiting for turn...\n");

		semaphoreWait(myIP[3], sock, from, length); //Wait for turn to play

		/*---------Play game----------*/
		int pattern[30];

		//Receive pattern to display
		bzero(message, MSG_SIZE); //clear message
		if(recvfrom(sock, message, MSG_SIZE, 0, (struct sockaddr *)&from, &length) < 0)
			error("Receive message error");

		if(message[0] == '#') {	//check if pattern
			printf("Received pattern\n");

			if(message[1] == 'I')	//check if new loop
				newLoop = 1;

			for(i=0;i<pattCount;i++) {
				pattern[i] = message[i+3] - 48;
			}
		}
		else if(message[0] == 'W') {	//check if winner
			printf("%s", message);

			for(i=0;i<5;i++) {	//Display winner LEDs
				*pbdr &= ~0xE0;	//turn off all LEDs
				*pbdr |= 0xA0;	//turn on outside LEDs
				printf("WINNER\n");
				usleep(500000);
				*pbdr &= ~0xE0;	//turn off all LEDs
				*pbdr |= 0x40;	//turn on middle LED
				printf("WINNER\n");
				usleep(500000);
			}


			exit(0);
		}

		printf("Begin\n");
		sleep(1);

		/*----------About to display pattern-----------*/
		//Flash LEDs 3 times
		for(i=3;i>0;i--) {
			*pbdr |= 0xE0;	//turn on all LEDs
			printf("%d..\n", i);
			usleep(500000);
			*pbdr &= ~0xE0;	//turn off all LEDs
			printf("\n");
			usleep(500000);
		}

		/*---------Display pattern---------*/
		for(i=0;i<pattCount;i++) {
			displayLED(pattern[i], pbdr, pbddr);
		}

		/*----------Get user input----------*/
		printf("Enter pattern of length: %d\n", pattCount);

		for(i=0;i<pattCount;i++) {

			if(read(fifo1, &button, sizeof(button)) < 0)	//read button from fifo
				return -1;

			printf("button: %d", button);

			displayLED(button, pbdr, pbddr);	//display input
			if(button != pattern[i]) {	//check if correct
				printf("Failed!\n");
				success = 0;
				break;
			}
		}

		//Check if all buttons pressed accurately
		if(i==pattCount) {
			success = 1;

			//Flash LEDs once
			*pbdr |= 0xE0;	//turn on all LEDs
			printf("Success!\n");
			usleep(500000);
			*pbdr &= ~0xE0;	//turn off all LEDs
		}

		/*----------Get new pattern----------*/
		if(success) {

			printf("Enter new pattern of length: %d\n", pattCount+newLoop);

			for(i=0;i<(pattCount+newLoop);i++) {

				if(read(fifo1, &pattern[i], sizeof(pattern[i])) < 0)	//read button from fifo
					return -1;

				printf("button: %d", pattern[i]);

				displayLED(pattern[i], pbdr, pbddr);	//display input
			}

			/*----------Send success message with new pattern-------------*/
			semaphorePost(myIP[3], sock, server, fromlen);

			//Send new pattern
			bzero(message, MSG_SIZE);
			message[0] = '#';

			for(i=1;i<=(pattCount+newLoop);i++) {
				message[i] = pattern[i-1] + 48;	//set pattern to message
			}

			//Send pattern
			if (sendto(sock, message, 40, 0, (struct sockaddr *)&server, fromlen)  < 0)
				error("Send message error");

			printf("%s", message);
			printf("\n");
		}
		else {

			semaphorePost(myIP[3], sock, server, fromlen);

			//Send failed message
			bzero(message, MSG_SIZE); //clear message
			sprintf(message, "Fail");

			if (sendto(sock, message, 40, 0, (struct sockaddr *)&server, fromlen)  < 0)
				error("Send message error");
		}

		//Flash LEDs 2 times
		for(i=2;i>0;i--) {
			*pbdr |= 0xE0;	//turn on all LEDs
			printf("Turn\n");
			usleep(500000);
			*pbdr &= ~0xE0;	//turn off all LEDs
			printf("Over\n");
			usleep(500000);
		}

		pattCount++;	//Increase pattern length
	}

	*pbdr |= 0xE0;	//turn on all LEDs

	return 0;
}

//Displays LED depending on what key is pressed
void displayLED(int button, unsigned long *pbdr, unsigned long *pbddr) {

	//turn on LED
	if(button == 1) {	//check button 1
		*pbdr |= 0x20;
	}
	if(button == 2) {	//check button 2
		*pbdr |= 0x40;
	}
	if(button == 3) {	//check button 3
		*pbdr |= 0x80;
	}

	printf("\n");
	usleep(500000);
	*pbdr &= ~0xE0;	//turn off all LEDs
	printf("\n");
	usleep(500000);
}

int semaphoreWait(int myIP, int sock, struct sockaddr_in from, socklen_t length) {

	char message[MSG_SIZE];

	while(1) {
		bzero(message, MSG_SIZE); //clear message

		//Receive message
		if(recvfrom(sock, message, MSG_SIZE, 0, (struct sockaddr *)&from, &length) < 0)
			error("Receive message error");

		//Check if this computer
		if(message[0] == '@') { //check if IP message
			if(message[1]-48 == myIP) //check if this IP
				return 1;
		}
	}
}

void semaphorePost(int myIP, int sock, struct sockaddr_in server, socklen_t fromlen) {

	char message[MSG_SIZE];
	bzero(message, MSG_SIZE); //clear message

	//Send IP
	message[0] = '!';
	message[1] = myIP;

	//Send message
	if (sendto(sock, message, 40, 0, (struct sockaddr *)&server, fromlen)  < 0)
		error("Send message error");
}

