/* 	Name       : 	ProjectModule.c
	Author     : 	Andrew Shannon
	Description: 	Real Time Embedded Project
*/

#ifndef MODULE
#define MODULE
#endif

#ifndef __KERNEL__
#define __KERNEL__
#endif

#include <linux/module.h>
#include <linux/kernel.h>
#include <asm/io.h>
#include <rtai.h>
#include <rtai_sched.h>
#include <rtai_fifos.h>
#include <linux/time.h>

MODULE_LICENSE("GPL");

//Set interrupt as GPIO combined
#define IRQ_NUM 59

struct timeval timeStart;
struct timeval timeEnd;


int handler(int irq) {

	//Disable interrupt
	rt_disable_irq(irq);

	//Variable declaration
	unsigned long *ptr;
	unsigned long *pbIntStsX;
	unsigned long *pbEOI;

	//Map kernel to addresses on board
	ptr = (unsigned long *) __ioremap(0x80840000, 4096, 0);

	//Map interrupt status address
	pbIntStsX = ptr + 0x2F;

	//GET TIME
	do_gettimeofday(&timeStart);

	//Ensure one button press doesn't send multiple signals
	if((abs((timeStart.tv_usec - timeEnd.tv_usec)) > 400000) || (timeStart.tv_sec - timeEnd.tv_sec) > 1) {

		int button = 0;

		if(*pbIntStsX == 0x01) { //button 1
			button = 1;
		}
		else if(*pbIntStsX == 0x02) { //button 2
			button = 2;
		}
		else if(*pbIntStsX == 0x04) { //button 3
			button = 3;
		}

		//Write button to fifo1
		rtf_put(1, &button, sizeof(button));

		//Get time for next check
		do_gettimeofday(&timeEnd);

	}

	//Clear interrupts
	pbEOI = ptr + 0x2D;
	*pbEOI |= 0x07;

	//Re-enable interrupt
	rt_enable_irq(irq);

	return 0;
}

int init_module(void)
{
	//Create fifo
	int fifo1;
	if((fifo1 = rtf_create(1, sizeof(int))) != 0)
		return -1;

	//Set up ISR for GPIO
	if(rt_request_irq (IRQ_NUM, (irqreturn_t*)handler, NULL, 1) != 0)
		return -1;

	//Enable interrupt for GPIO
	rt_enable_irq(IRQ_NUM);

	//Variable declaration
	unsigned long *ptr;
	unsigned long *pbddr;

	//Map kernel to addresses on board
	ptr = (unsigned long *) __ioremap(0x80840000, 4096, 0);

	//Map direction register address for port B
	pbddr = ptr + 0x05;

	//Set buttons as inputs
	*pbddr &= ~0x07;

	//Map data register to GPIOBIntEn
	unsigned long *pbIntEn;
	pbIntEn= ptr + 0x2E;

	//Set buttons as interrupts
	*pbIntEn |= 0x07;

	//Map data register to GPIOBIntType1
	unsigned long *pbIntType1;
	pbIntType1 = ptr + 0x2B;

	//Set buttons to edge sensitive
	*pbIntType1 |= 0x07;

	//Map data register to GPIOBIntType2
	unsigned long *pbIntType2;
	pbIntType2 = ptr + 0x2C;

	//Set buttons as falling edge
	*pbIntType2 |= 0x07;

	//Set debouncing
	unsigned long *debouncer;
	debouncer = ptr + 0xC8;
	*debouncer |= 0xFF;

	do_gettimeofday(&timeEnd);


	return 0;
}

void cleanup_module(void)
{
	//Disable & delete interrupt
	rt_disable_irq(IRQ_NUM);
	rt_release_irq(IRQ_NUM);

	//Close FIFO
	rtf_destroy(1);
}
