/*
 * Audio Project
 *
 * Name: Russ Butler
 * ID: 12309946
 * Date: 4/13/2012
 */
#include <stdlib.h>
#include <stdio.h>
#include <stdint.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>


//Riff structures

typedef struct RIFF_CHUNK {
	uint32_t identifier;
	uint32_t size;
} RIFF_CHUNK;

typedef struct RIFF_RIFF {
	uint32_t identifier;
	uint32_t size;
	uint32_t format;
} RIFF_RIFF;

//Wave structures

typedef struct WAVE_FORMAT {
	uint16_t AudioFormat;
	uint16_t NumChannels;
	uint32_t SampleRate;
	uint32_t ByteRate;
	uint16_t BlockAlign;
	uint16_t BitsPerSample;
} WAVE_FORMAT;



//If an error occurs print error message along with
//returning an error value
#define WARN_RETURN_VALUE


//Print an error message then terminate
void error(char* message) {
	puts(message);
	exit(0);
}


//round the number up so it is even
uint32_t rount_up_even(uint32_t val) {
	return (val & 1 ? val + 1 : val);
}

//read chunk info, return 0 on success
int read_chunk_info(FILE* file, RIFF_CHUNK* chunk_info) {
	int length = 0;

	//read data
	length = fread(chunk_info,1,sizeof(RIFF_CHUNK),file);
	if (length < 0) {
		#ifdef WARN_RETURN_VALUE
		puts("read_chunk_info error: fread failed");
		#endif
		return length;
	}
	if (length != sizeof(RIFF_CHUNK)) {
		#ifdef WARN_RETURN_VALUE
		printf("read_chunk_info error: only able to read %i bytes\n",length);
		#endif
		return 1;
	}

	return 0;
}

//read chunks until the specified one is found, return 0 on success
int read_chunk_info_skip(FILE* file, RIFF_CHUNK* chunk_info, char* name) {
	RIFF_CHUNK current_chunk;
	int result;

	//chunk name must be 4 bytes
	if (strlen(name) != 4) {
		#ifdef WARN_RETURN_VALUE
		printf("read_chunk_info_skip: Input string not long enough\n");
		#endif
		return -1;
	}

	//read initial chunk
	result = read_chunk_info(file,&current_chunk);
	if (result != 0) {
		#ifdef WARN_RETURN_VALUE
		printf("read_chunk_info_skip: read_chunk_info problem\n");
		#endif
		return -1;
	}

	//search until chunk is found
	while (memcmp(&(current_chunk.identifier),name,4) != 0) {

		result = fseek(file,rount_up_even(current_chunk.size),SEEK_CUR);
		if (result != 0) {
			#ifdef WARN_RETURN_VALUE
			printf("read_chunk_info_skip: Problem Seeking\n");
			#endif
			return -1;
		}

		result = read_chunk_info(file,&current_chunk);
		if (result != 0) {
			#ifdef WARN_RETURN_VALUE
			printf("read_chunk_info_skip: read_chunk_info problem\n");
			#endif
			return -1;
		}
	}
	*chunk_info = current_chunk;

	return 0;
}


int main(int argc, char* argv[]) {
	FILE* file = 0;
	RIFF_RIFF file_header;
	RIFF_CHUNK current_chunk;
	WAVE_FORMAT format;
	char* file_name;

	int length = 0, result = -1;

	if (argc < 2) {
		printf("First parameter must be filename of song.\n");
		return 0;
	}
	file_name = argv[1];

	file = fopen(file_name,"rb");
	if (file == 0) error("Could not open file");


	//read file header
	length = fread(&file_header,1,sizeof(file_header),file);
	if (length < 0) error("Error reading from file");
	if (length != sizeof(file_header)) error("File not large enough");
	if (memcmp(&(file_header.identifier), "RIFF", 4) != 0) error("Incorrect header");
	if (memcmp(&(file_header.format), "WAVE", 4) != 0) error("File is not a wave");


	printf("RIFF data size = %i\n",file_header.size);


	//Read chunks until format chunk is found
	result = read_chunk_info_skip(file,&current_chunk,"fmt ");
	if (result != 0) error("Problem reading \"fmt \" chunk");
	if (current_chunk.size != 16) error("\"fmt \" chunk has wrong size");

	//Read format chunk data
	length = fread(&format,1,sizeof(format),file);
	if (length < 0) error("Error reading from file");
	if (length != sizeof(format)) error("Error reading format");
	if (format.AudioFormat != 1) error("File is not PCM type");

	//Print out file info
	printf("Bits Per Sample = %i\n", (int) format.BitsPerSample);
	printf("Block Align = %i\n", (int) format.BlockAlign);
	printf("Byte Rate = %i\n", (int) format.ByteRate);
	printf("Number of Channels = %i\n", (int) format.NumChannels);
	printf("Sample Rate = %i\n", (int) format.SampleRate);

	//Read the data chunk
	result = read_chunk_info_skip(file,&current_chunk,"data");
	if (result != 0) error("Problem reading \"data\" chunk");
	printf("Data chunk size = %i\n",current_chunk.size);

	//open rtfifo
	int audio_fifo = open("/dev/rtf/0",O_WRONLY);
	if (audio_fifo == -1) error("Could not open output fifo");

	//pump data to rtfifo
	uint8_t audio_data[400];
	while (!feof(file)) {

		//read audio data from file
		result = fread(audio_data,1,400,file);
		if (result < 0) error("Problem reading from file\n");

		//write audio data to driver
		result = write(audio_fifo,audio_data, sizeof(audio_data));
		if (result <= 0) error("Problem writing data to real time buffer\n");
		if (result != 400) printf("Only wrote %i byte\n",result);

	}

	printf("finished reading from audio file\n");

	return 0;
}
