/*
 * Audio Project
 *
 * Name: Russ Butler
 * ID: 12309946
 * Date: 4/13/2012
 */

#define MODULE
#define __KERNEL__


#include <asm/io.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/time.h>
#include <rtai.h>
#include <rtai_sched.h>
#include <rtai_fifos.h>
#include <stdint.h>

MODULE_LICENSE("GPL");


//This header file contains the actual functions and
//not just prototypes.  This is because multiple C
//files create multiple object file and only one
//object file should be installed as a kernel module.
#include "spi_controller.h"


//Define this to print debugging info
//This reports things such as missing audio chunks
//And onboard spi transmission speeds
//Comment this define out to disable debugging
#define DEBUG_MODE


#define GPIO_IRQ_LINE		59


#define GPIO_BASE_ADDR				0x80840000
#define GPIO_PBDR_OFFSET			0x4
#define GPIO_PBDDR_OFFSET			0x14
#define GPIO_GPIOBIntEn_OFFSET		0xB8
#define GPIO_GPIOBIntType1_OFFSET	0xAC
#define GPIO_GPIOBIntType2_OFFSET	0xB0
#define GPIO_GPIOBEOI_OFFSET		0xB4
#define GPIO_IntStsB_OFFSET			0xBC
#define GPIO_PFDR_OFFSET			0x30
#define GPIO_PFDDR_OFFSET			0x34


unsigned long gpio_base_addr = 0;
volatile unsigned long *gpio_p_b_dr = 0;
volatile unsigned long *gpio_p_b_ddr = 0;
volatile unsigned long *gpio_b_int_en = 0;
volatile unsigned long *gpio_b_int_type_1 = 0;
volatile unsigned long *gpio_b_int_type_2 = 0;
volatile unsigned long *gpio_b_eoi = 0;
volatile unsigned long *gpio_b_int_sts = 0;
volatile unsigned long *gpio_p_f_dr = 0;
volatile unsigned long *gpio_p_f_ddr = 0;



//Timer Defines
#define START_DELAY_MS		1000LL		//wait 1 second to run task
#define TASK_PERIOD_MS		1000LL		//run every second
#define TIMER_PERIOD_NS		1000000LL	//1ms period

#ifdef DEBUG_MODE
static void rt_task_handler(int val);
#endif
static void init_interrupt();
static void cleanup_interrupt();


//NOTE:
//The Size of this buffer matters immensely
//With a size of 20000 there are ~10 chunks missing per second (2%)
//By doubling this size to 40000 this loss rate goes to 0%
//Buffer size set to 8k for safety
//The audio delay due to this buffer will be at least 0.9 seconds
#define AUDIO_FIFO			0
#define AUDIO_FIFO_SIZE		80000




#ifdef DEBUG_MODE
uint8_t array[1000];
RT_TASK rt_task;
#endif

unsigned long *portB = 0;
unsigned long *ddrB = 0;
char* baseAddr = 0;
int init_module() {
	int result;

	spi_open();

	result = rtf_create(AUDIO_FIFO,AUDIO_FIFO_SIZE);
	if (result != 0) {
		printk("could not open fifo\n");
		return -1;
	}



	#ifdef DEBUG_MODE

	//initialize timer
	rt_set_periodic_mode();
	start_rt_timer(nano2count(TIMER_PERIOD_NS));


	//calculate values for start and period
	RTIME start_time = nano2count(START_DELAY_MS*1000000LL)+rt_get_time();
	RTIME period = nano2count(TASK_PERIOD_MS*1000000LL);


	//create real time task
	result = rt_task_init(&rt_task,&rt_task_handler,0,3000,0,0,0);
	if (result != 0) {
		printk("Error initializing real time task\n");
		return -1;
	}

	//make the task run periodically
	result = rt_task_make_periodic(&rt_task,start_time,period);
	if (result != 0) {
		printk("Error making task periodic\n");
		rt_task_delete(&rt_task);
		return -1;
	}

	//Collect SPI speed statistics
	RTIME start = rt_get_time_ns();
	spi_write_array(array, sizeof(array));
	RTIME stop = rt_get_time_ns();
	printk("start = %lld, stop = %lld\n",start, stop);
	printk("Spi Frequency = %lld KHz\n",((long long) sizeof(array)*8*1000000/(stop - start)));
	printk("Time to run = %lldus\n",((stop - start)/1000));

	#endif

	init_interrupt();

	return 0;
}

uint8_t audio_buffer[400];

#ifdef DEBUG_MODE
int read_miss_count = 0;
unsigned int could_not_read_400_count = 0;
#endif


//This task is for printing statistics on errors
//this is for debugging only
#ifdef DEBUG_MODE
static void rt_task_handler(int dumby_val) {

	while (1) {

		printk("Read Miss Rate = %i chunks/s\n",read_miss_count);
		if (could_not_read_400_count > 0)
			printk("Incomplete chunks read = %i\n",could_not_read_400_count);
		could_not_read_400_count = 0;
		read_miss_count = 0;

		rt_task_wait_period();
	}
}
#endif

//This is the number of repeated iterations before the request more
//data pin is considered stuck low and interrupts should be disabled
//to prevent freezing. Look at implementation for more details.
#define IRQ_ERROR_ITR			1000
RTIME irq_finish_time = 0;

static void irq_request_more_data() {

	//if less than 100 microseconds have passed since last interrupt
	//then the request more data pin is stuck low.  Turn off interrupts to prevent lockup
	//Normally there is ~4000 - 5000us between interrupts on normal operation
	RTIME irq_start_time = rt_get_time_ns();
	static int irq_error_count = 0;
	if (irq_start_time != 0 && (irq_start_time - irq_finish_time) < 1000000) {
		irq_error_count++;
		if (irq_error_count >= IRQ_ERROR_ITR) {
			rt_disable_irq(GPIO_IRQ_LINE);
			printk("Hardware failure detected - board reset required for audio\n");
			return;
		}
	} else {
		irq_error_count = 0;
	}


	//get the audio data & count errors if any
	int size = rtf_get(AUDIO_FIFO,audio_buffer,400);
	#ifdef DEBUG_MODE
	if (size != 400 && size != 0) could_not_read_400_count++;
	if (size == 0) read_miss_count++;
	#endif

	//zero any extra data
	int i;
	for (i = size; i < sizeof(audio_buffer); i++)
		audio_buffer[i] = 0x100 / 2;


	//write audio data to spi
	spi_write_array(audio_buffer,sizeof(audio_buffer));

	irq_finish_time = rt_get_time_ns();


	rt_ack_irq(GPIO_IRQ_LINE);
}



void cleanup_module() {


	cleanup_interrupt();

	if (gpio_base_addr != 0)
		__iounmap((void*) gpio_base_addr);
	gpio_base_addr = 0;


	#ifdef DEBUG_MODE

	//Delete the task.  If the task it not deleted and is still
	//running when the program terminates major problems can arise.
	int result = rt_task_delete(&rt_task);
	if (result != 0) {
		printk("Error deleting task\n");
	}

	//Stop the timer.  This will halt any tasks using the timer.
	stop_rt_timer();

	rtf_destroy(AUDIO_FIFO);

	#endif

	//close spi resources
	spi_close();



}

static void init_interrupt() {
	int result;

	gpio_base_addr = (unsigned long) __ioremap(GPIO_BASE_ADDR,0x1000,1);
	gpio_p_b_dr = (volatile unsigned long *) (gpio_base_addr + GPIO_PBDR_OFFSET);
	gpio_p_b_ddr = (volatile unsigned long *) (gpio_base_addr + GPIO_PBDDR_OFFSET);
	gpio_b_int_en = (volatile unsigned long *) (gpio_base_addr + GPIO_GPIOBIntEn_OFFSET);
	gpio_b_int_type_1 = (volatile unsigned long *) (gpio_base_addr + GPIO_GPIOBIntType1_OFFSET);
	gpio_b_int_type_2 = (volatile unsigned long *) (gpio_base_addr + GPIO_GPIOBIntType2_OFFSET);
	gpio_b_eoi = (volatile unsigned long *) (gpio_base_addr + GPIO_GPIOBEOI_OFFSET);
	gpio_b_int_sts = (volatile unsigned long *) (gpio_base_addr + GPIO_IntStsB_OFFSET);
	gpio_p_f_dr = (volatile unsigned long *) (gpio_base_addr + GPIO_PFDR_OFFSET);
	gpio_p_f_ddr = (volatile unsigned long *) (gpio_base_addr + GPIO_PFDDR_OFFSET);


	//set interrupt pins to input
	*gpio_p_b_ddr &= ~((1 << 0));

	//register interrupt
	result = rt_request_irq(GPIO_IRQ_LINE,&irq_request_more_data,0,1);
	if (result != 0) printk("Problem registering interrupt\n");

	rt_enable_irq(GPIO_IRQ_LINE);

	//Level triggered
	*gpio_b_int_type_1 &= ~(1 << 0);
	//Interrupt on low
	*gpio_b_int_type_2 &= ~(1 << 0);

	//Enable interrupts
	*gpio_b_int_en |= (1 << 0) | (1 << 1) |(1 << 2) | (1 << 3) | (1 << 4);
}
static void cleanup_interrupt() {
	if (gpio_base_addr != 0)
		__iounmap((void*) gpio_base_addr);
	gpio_base_addr = 0;

	rt_disable_irq(GPIO_IRQ_LINE);

	rt_release_irq(GPIO_IRQ_LINE);
}

