/*
 *	Russ Butler
 *	Audio Controller
 *	6/16/16
 *
 *
 *
 */
#include <avr/io.h>
#include <avr/interrupt.h>


#define setup_error_led()	(DDRC |= (1 << DDC5))
#define error_led_on()		(PORTC |= (1 << PC5))
#define error_led_off()		(PORTC &= ~(1 << PC5))

#define setup_signal_more_data() (DDRB |= (1 << DDB1))
#define signal_more_data_on() (PORTB |= (1 << PB1))
#define signal_more_data_off() (PORTB &= ~(1 << PB1))

volatile uint16_t error_time = 0;
void signal_error() {
	error_time = 60000;
	error_led_on();
}

#define BUFFER_SIZE		400
uint8_t buffer[2][BUFFER_SIZE];

int main(void)
{
	setup_error_led();
	error_led_off();
	setup_signal_more_data();
	signal_more_data_on();
	
	signal_error();
	
	//init buffers
	int i = 0;
	for (i = 0; i < BUFFER_SIZE; i++) {
		buffer[0][i] = 0x100/2;
		buffer[1][i] = 0x100/2;
	}
	
	
	//setup pwm
	DDRD |= (1 <<DDD6) | (1 << DDD5);
	TCCR0A = (1 << COM0A1) | (1 << COM0B1) |	//Non Inverting
			 (1 << WGM00) | (1 << WGM01);		//Fast Pwm Mode
	TCCR0B = (1 << CS00);						//Clock as source
	
	//setup spi - CPOL = 0, CPHA = 0, Mode = Slave
	SPCR = (1 << SPE); 
	
	//setup audio output interrupt
	TCCR1B = (1 << WGM12) |		//CTC mode with A as Top
			 (1 << CS10);		//Freq of main oscillator
	TCNT1H = 0;					//Start clock at 0
	TCNT1L = 0;
	OCR1AH = 0x1;				//Set to a value of 454 - 1 
	OCR1AL = 0xC6 - 1;			//to create an interrupt freq of ~44.1khz
	
	uint8_t* write_pos = 0;
	uint16_t write_size = 0;


	uint8_t* read_pos = 0;
	uint8_t read_remaining = 0;

	uint8_t cur_buf_read = 0;
	
    while(1) {
		
		if (SPSR & (1 << SPIF)) {				//SPI interrupt highest priority
			if (write_size > 0) {
				*write_pos++ = SPDR;
				write_size--;
			} else {
				SPDR;
			}
			if (write_size < BUFFER_SIZE / 2)
				signal_more_data_off();
		} else if (TIFR1 & (1 << OCF1A)) {		//Audio interrupt next highest priority
			
			//clear status flag
			TIFR1 = (1 << OCF1A);
			
			//output data
			OCR0A = *read_pos++;
			OCR0B = *read_pos++;
			read_remaining--;
			
			//load new buffer
			if (read_remaining == 0) {
				
				//sanity check for errors
				if (write_size != 0)
					signal_error();

				//begin filling old buffer with new data
				write_pos = buffer[cur_buf_read];
				write_size = BUFFER_SIZE;
				signal_more_data_on();
		
				//switch read buffer
				if (cur_buf_read == 0) cur_buf_read = 1;
				else cur_buf_read = 0;
				read_pos = buffer[cur_buf_read];
				read_remaining = BUFFER_SIZE / 2;
			} 

		} else if (error_time > 0) {			//Error function lowest priority
			error_time--;
			if (error_time == 0)
				error_led_off();
		}
		
			
			
		
    }
}
