/******************************************************************************
 *
 *  logic_analyzer_client.c
 *
 *  Author     : Dallas Fletchall
 *  Date       : Apr 29, 2014 11:32:45 AM
 *  Description: logic_analyzer_client.c
 *
 *****************************************************************************/
/**************************************
 * Local Includes
 *************************************/
#include <fcntl.h>
#include <netdb.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <unistd.h>
#include "logic_analyzer_client.h"
#include "menu.h"
#include "gnuplot.h"
#include "messages.h"
#include "pin_assignments.h"

//#define CLEAR

/**************************************
 * Functions
 *************************************/
int main( int argc, char *argv[] ){

	MainMenuOption selection = INVALID_MAIN_MENU_OPTION;

	// Setup the client
	int          sockFd;
	int          portNum;
	Sockaddr_in  serverAddress;
	Hostent     *server;

	// Check arguments and store in variables
	if( argc != 3 || 1 != sscanf( argv[2], "%d", &portNum ) ){
		printf( "Usage %s <host ip> <port #>\n", argv[0] );
		return -1;
	}

	// Create socket and connect to server
	sockFd = socket( AF_INET, SOCK_STREAM, IPPROTO_TCP );
	if( sockFd < 0 ){ perror( "socket()" ); return -1; }
	server = gethostbyname( argv[1] );
	if( server == NULL ){ printf( "Unable to connect to %s.", argv[1] ); return -1; }
	memset( &serverAddress, 0, sizeof( serverAddress ) );
	serverAddress.sin_family = AF_INET;
	memcpy( &serverAddress.sin_addr.s_addr, server->h_addr, server->h_length );
	serverAddress.sin_port = htons( portNum );
	if( 0 > connect( sockFd, (struct sockaddr *)&serverAddress, sizeof( serverAddress ) ) ){
		perror( "connect()" );
		return -1;
	}

	// Setup the plot
	gnuplotSetup( "signals.txt" );
	gnuplotPlot();

	while( selection != EXIT_LOGIC_ANALYZER ){

		selection = mainMenu();
		switch( selection ){

			case READ_SIGNALS:
				readSignals( sockFd );
				break;

			case LABEL_SIGNALS:
				changeLabel();
				break;

			case SIGNAL_LEVELS:
				changeLevel( sockFd );
				break;

			case EXIT_LOGIC_ANALYZER:
				break;
			case CLOSE_LOGIC_ANALYZER_SERVER:

			case INVALID_MAIN_MENU_OPTION:
			case MAX_MAIN_MENU_OPTION:
			default:
				printf( "Main Menu Error\n" );
		}

	}

	// Release all resources
	// TODO: clientClose();
	gnuplotClose();

	return 0;

} // end main( int, char ** )

void flushStdin( void ){

	int c = getchar();
	while( c != '\n' ){
		c = getchar();
	}

} // end flushStdin()

void changeLevel( int socket_fd ){

	int   bytes =  0;
	float vLow  = -1;
	float vHigh = -1;
	char  msg[MAX_MSG_SIZE];

	// Format the menu
#ifdef CLEAR
	if( -1 == system( "clear" ) ){ perror( "system( \"clear\" )" ); }
#endif
	printf( "%s\n", TITLE );

	// Get the logic low voltage level
	printf( "\n\tEnter the max voltage for Logic Level Low: " );
	if( 1 != scanf( "%f", &vLow ) ){
		printf( "\t\tError: Voltage must be a number!\n" );
		sleep( 2 );
		return;
	}

	// Get the logic high voltage level
	printf( "\n\tEnter the min voltage for Logic Level High: " );
	if( 1 != scanf( "%f", &vHigh ) ){
		printf( "\t\tError: Voltage must be a number!\n" );
		sleep( 2 );
		return;
	}

	// Write high DAC message to the socket
	memset( msg, 0, MAX_MSG_SIZE );
	sprintf( msg, "%s%c%.2f\n", SET_HIGH_LVL_MSG, DELIMITER, vHigh );
	bytes = send( socket_fd, msg, MAX_MSG_SIZE, 0 );
	if( bytes != MAX_MSG_SIZE ){ perror( "send()" ); return; }

	// If no error wait for return message
	memset( msg, 0, MAX_MSG_SIZE );
	bytes = recv( socket_fd, msg, MAX_MSG_SIZE, MSG_WAITALL );
	if( bytes < 0 ){ perror( "recv()" ); return; }

	// Check return message for error, print
	if( strncmp( SUCCESS_MSG, msg, strlen( SUCCESS_MSG ) ) ){ printf( "ERROR: low DAC, %s\n", msg ); }

	// Write low DAC message to the socket
	memset( msg, 0, MAX_MSG_SIZE );
	sprintf( msg, "%s%c%.2f\n", SET_LOW_LVL_MSG, DELIMITER, vLow );
	bytes = send( socket_fd, msg, MAX_MSG_SIZE, 0 );
	if( bytes != MAX_MSG_SIZE ){ perror( "send()" ); return; }

	// If no error wait for return message
	memset( msg, 0, MAX_MSG_SIZE );
	bytes = recv( socket_fd, msg, MAX_MSG_SIZE, MSG_WAITALL );
	if( bytes < -1 ){ perror( "recv()" ); return; }

	// Check return message for error, print
	if( strncmp( SUCCESS_MSG, msg, strlen( SUCCESS_MSG ) ) ){ printf( "ERROR: high dac, %s\n", msg ); }


	return;

} // end changeLevel( int )

void readSignals( int sock_fd ){

	int       bytes   = 0;
	int       rBytes  = 0;
	int       wFd     = 0;
	float     secs    = 0;
	double    time    = 0;
	char      message[MAX_MSG_SIZE];
	uint32_t  reg     = 0;
	int       rSize = sizeof( reg );

	// Open a file for writing, truncated
	wFd = open( "data.dat", O_TRUNC | O_RDWR | O_CREAT, S_IRWXO );
	if( wFd == -1 ){ perror( "open()" ); return; }

	// Ask the user how many seconds to record
#ifdef CLEAR
	if( system( "clear" ) ){ perror( "system(\"clear\")" ); }
#endif
	printf( "%s\n\n", TITLE );
	printf( "\tEnter the number of seconds to record: " );
	if( 1 != scanf( "%f", &secs ) ){ printf( "Error: Time in seconds must be a number" ); return; }

	// Format the message to be sent
	memset( message, 0, MAX_MSG_SIZE );
	sprintf( message, "%s%c%.3f\n", START_REC_MSG, DELIMITER, secs );


	// Tell the server to start recording
	bytes = send( sock_fd, message, MAX_MSG_SIZE, 0 );
	if( bytes != MAX_MSG_SIZE ){ perror( "send()" ); return; }

	// Read from the socket
	rBytes = recv( sock_fd, &reg, rSize, MSG_WAITALL );
	if( rBytes < 0 ){ perror( "recv()" ); }

	while( reg != 0 ){

		// Write to the file
		bytes = write( wFd, &reg, rSize );
		if( bytes != rSize ){ perror( "write()" ); }

		// Read from the socket
		rBytes = recv( sock_fd, &reg, rSize, MSG_WAITALL );
		if( rBytes < 0 ){ perror( "recv()" ); }


	}

	// Get total time for sampling
	rBytes = recv( sock_fd, &time, sizeof( double ), MSG_WAITALL );
	if( rBytes < 0 ){ perror( "recv()" ); }

	// Get the current position in the file
	printf( "Time: %lf\n", time );

	rBytes = lseek( wFd, 0, SEEK_CUR );
	printf( "Samples: %lf\n", (double)(rBytes) / sizeof( uint32_t ) );
	time   = time / ( rBytes / sizeof( uint32_t ) );

	printf( "Time increment: %lf\n", time );

	// Reset the file before formatting
	if( 0 != lseek( wFd, 0, SEEK_SET ) ){ perror( "lseek()" ); return; }

	formatFileForGnuPlot( wFd, time );
	close( wFd );

	// Update the plot
	gnuplotPlot();

} // end readSignals( int )

void formatFileForGnuPlot( int fd, double time ){

	int       bytes = 1;
	double    t, ch1, ch2, ch3, ch4;
	uint32_t  value;

	FILE *oFile = fopen( "signals.txt", "w" );
	if( oFile == NULL ){ perror( "fopen()" ); }

	// Get the first structure to compare with the time
	t = ch1 = ch2 = ch3 = ch4 = 0;

	while( bytes != 0 ){

		// Read from the .dat file
		bytes = read( fd, (void *)&value, 4 );
		if( bytes < 0 ){ perror( "read()" ); }

		// Extract data, and level shift for channels
		t += time;
		ch1 = extractData( value, P1_HIGH, P1_LOW ) + 3;
		ch2 = extractData( value, P2_HIGH, P2_LOW ) + 2;
		ch3 = extractData( value, P3_HIGH, P3_LOW ) + 1;
		ch4 = extractData( value, P4_HIGH, P4_LOW );

		// Write to signals.txt
		fprintf( oFile, "%.6f %.2f %.2f %.2f %.2f\n", t, ch1, ch2, ch3, ch4 );

	}

	fclose( oFile );

} // end formatFileForGnuPlot( int )

double extractData( uint32_t reg, uint32_t hComparator, uint32_t lComparator ){

	/**************************************************************************
	 * For a valid HIGH, a channel's low and high comparator must be 1. For a
	 * valid LOW, a channel's low and high comparator must be 0. XOR of the
	 * channel's low and high comparators will be false for either of these
	 * cases. ( True indicates floating value ).
	 * There is no need to check both comparators at this point, they must be
	 * equal. If one is high then the signal is high. Else they are both low.
	 *************************************************************************/
	double retVal = 0;

	int a = ( reg & BIT( lComparator ) ) ? 1 : 0;
	int b = ( reg & BIT( hComparator ) ) ? 1 : 0;

	// Get data for channel 1
	if( a ^ b ){
		retVal = .5;  // Floating value
	}else if( a ){
		retVal = .95; // High
	}else{
		retVal = .05; // Low
	}

	return retVal;

} // end extractData( double *, double *, double *, double * )

double timeval2double( Timeval time ){

	return (double)( time.tv_sec + ( ( (double)time.tv_usec ) / 1000000 ) );

} // end timeval2double( Timeval )
