/******************************************************************************
 *
 *  logic_analyzer_server.c
 *
 *  Author     : Dallas Fletchall
 *  Date       : Apr 30, 2014 11:53:37 PM
 *  Description: logic_analyzer_server.c, main implementation file for the
 *  		     project.  Includes program entry point.
 *
 *****************************************************************************/
#include <arpa/inet.h>
#include <signal.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <unistd.h>
#include "rpi.h"
#include "logic_analyzer_server.h"
#include "pin_assignments.h"
#include "mcp4901.h"
#include "messages.h"

#define VOLT_NOT_NUM_ERR "Error: Voltage not a number"
#define VOLT_SET_ERR     "Error: Unable to set voltage"

Mcp4901 lowDac, highDac;

int main( int argc, char *argv[] ){

	// Setup the peripheral
	if( map_peripheral( &gpio ) == -1 ){
		printf( "Failed to map GPIO registers, exiting ...\n\n" );
		return -1;
	}

	// Setup channel pins
	int i;
	for( i = 0; i < 4; i++ ){
		INP_GPIO( i + 8 );
		INP_GPIO( i + 22 );
	}

	// Setup the DAC's with initial voltage 1.5 - 3.5
	mcp4901Setup( &lowDac, SPI_SDI, SPI_SCK, LOW_DAC_CS_N, MCP4901_VCC );
	mcp4901Setup( &highDac, SPI_SDI, SPI_SCK, HIGH_DAC_CS_N, MCP4901_VCC );
	if( -1 == mcp4901SetVout( &lowDac, 1.5 ) || -1 == mcp4901SetVout( &highDac, 3.5 ) ){
		printf( "Unable to setup the DAC's, exiting ...\n\n" ); return -1;
	}

	// Setup port connections
	int portNum, sockFd, connectedSocket;
	Sockadder_in serverAddress, clientAddress;
	if( argc != 2 || 1 != sscanf( argv[1], "%d", &portNum ) ){
		printf( "Invalid arguments, no port provided." ); return -1;
	}
	sockFd = socket( AF_INET, SOCK_STREAM, 0 );
	if( sockFd == -1 ){ perror( "socket()" ); return -1; }
	memset( &serverAddress, 0, sizeof( Sockadder_in ) );
	memset( &clientAddress, 0, sizeof( Sockadder_in ) );

	// Setup server
	serverAddress.sin_family      = AF_INET;
	serverAddress.sin_addr.s_addr = htonl( INADDR_ANY );
	serverAddress.sin_port        = htons( portNum );
	if( 0 > bind( sockFd, (struct sockaddr *)&serverAddress, sizeof( serverAddress ) ) ){
		perror( "bind()" ); return -1;
	}else if( 0 > listen( sockFd, 10 ) ){
		perror( "listen()" ); return -1;
	}

	// Setup the client for acceptance
	socklen_t clientLen = sizeof( clientAddress );

	// Inform user that the server is operational
	printf( "Server Initiated.\n" );

	// Loop listening for connections
	while( 1 ){

		// Try to accept connection, exit if error
		connectedSocket = accept( sockFd, (struct sockaddr *)&clientAddress, &clientLen );
		if( connectedSocket < 0 ){ perror( "accept()" ); exit( -1 ); }

		// New connection established, fork() to listen for another connection
		pid_t pid = fork();
		if( pid < 0 ){ perror( "fork()" ); exit( -1 ); }

		// Handle child
		if( pid == 0 ){

			close( sockFd );

			// Connection established, initiate message transfer
			while( 1 ){
			messageHandler( connectedSocket );
			}
			exit( 0 );

		}

		// Handle parent
		else{
			close( connectedSocket );
			signal( SIGCHLD, SIG_IGN );
		}

	}

	close( connectedSocket );
	close( sockFd );

	return 0;

}

void  messageHandler( int sock_fd ){

	int   n;
	int   bytes;
	char  buffer[MAX_MSG_SIZE];
	char *message;
	char *value;
	char  reply[MAX_MSG_SIZE];

	// Clear message before read
	memset( buffer, 0, MAX_MSG_SIZE );
	n = recv( sock_fd, (void *)buffer, MAX_MSG_SIZE, MSG_WAITALL );
	if( n < 0 ){ perror( "read()" ); }

	printf( "Message: %s\n", buffer );

	// Get message header
	message = strtok( buffer, ":" );
	if( !strcmp( message, SET_LOW_LVL_MSG ) ){

		// Set low DAC
		float  newVoltage = 0;
		int    voltSetErr = 0;

		// Get the specified voltage
		value = strtok( NULL, "\n" );
		if( value == NULL || 1 != sscanf( value, "%f", &newVoltage ) ){

			// Message did not contain a valid number, reply with error messsage
			memset( reply, 0, MAX_MSG_SIZE );
			sprintf( reply, "%s\n", VOLT_NOT_NUM_ERR );
			bytes = send( sock_fd, reply, MAX_MSG_SIZE, 0 );
			if( bytes != MAX_MSG_SIZE ){ perror( "send()" ); return; }

		}else{

			// Try to set the specified voltage
			voltSetErr = mcp4901SetVout( &lowDac, newVoltage );
			if( voltSetErr == -1 ){

				// Unable to set voltage, reply with error message
				memset( reply, 0, MAX_MSG_SIZE );
				sprintf( reply, "%s\n", VOLT_SET_ERR );
				bytes = send( sock_fd, reply, MAX_MSG_SIZE, 0 );
				if( bytes != MAX_MSG_SIZE ){ perror( "send()" ); return; }

			}else{

				// Voltage set successfully, reply with success message
				memset( reply, 0, MAX_MSG_SIZE );
				sprintf( reply, "%s\n", SUCCESS_MSG );
				bytes = send( sock_fd, reply, MAX_MSG_SIZE, 0 );
				if( bytes != MAX_MSG_SIZE ){ perror( "send()" ); return; }

			}
		}

	}else if( !strcmp( message, SET_HIGH_LVL_MSG ) ){

		// Set high DAC
		float  newVoltage   = 0;
		int    voltSetError = 0;

		// Get the specified voltage
		value = strtok( NULL, "\n" );
		if( value == NULL || 1 != sscanf( value, "%f", &newVoltage ) ){

			// Message did not contain a valid number, reply with error messsage
			memset( reply, 0, MAX_MSG_SIZE );
			sprintf( reply, "%s\n", VOLT_NOT_NUM_ERR );
			bytes = send( sock_fd, reply, MAX_MSG_SIZE, 0 );
			if( bytes != MAX_MSG_SIZE ){ perror( "send()" ); return; }

		}else{

			// Try to set the specified voltage
			voltSetError = mcp4901SetVout( &highDac, newVoltage );
			if( voltSetError == -1 ){

				// Unable to set voltage, reply with error message
				memset( reply, 0, MAX_MSG_SIZE );
				sprintf( reply, "%s\n", VOLT_SET_ERR );
				bytes = send( sock_fd, reply, MAX_MSG_SIZE, 0 );
				if( bytes != MAX_MSG_SIZE ){ perror( "send()" ); return; }

			}else{

				// Voltage set successfully, reply with success message
				memset( reply, 0, MAX_MSG_SIZE );
				sprintf( reply, "%s\n", SUCCESS_MSG );
				bytes = send( sock_fd, reply, MAX_MSG_SIZE, 0 );
				if( bytes != MAX_MSG_SIZE ){ perror( "send()" ); return; }

			}
		}
	}else if( !strcmp( message, START_REC_MSG ) ){

		// Determine the amount of time to take samples in msec
		double             start   = 0;
		double             total   = 0;
		float              time_s  = 0;
		int                i       = 0;
		int                error   = 0;
		int                rSize   = sizeof( uint32_t );
		uint32_t           reg;
		Timeval            t;

		value = strtok( NULL, "\n" );
		if( value == NULL || 1 != sscanf( value, "%f", &time_s ) ){

			// Message did not contain a valid number, reply with error messsage
			memset( reply, 0, MAX_MSG_SIZE );
			sprintf( "%s\n", VOLT_NOT_NUM_ERR );
			bytes = send( sock_fd, reply, MAX_MSG_SIZE, 0 );
			if( bytes != MAX_MSG_SIZE ){ perror( "send()" ); return; }
		}

		// At a sample rate of 333.333k/s
		i = 333333 * time_s;

		// Inform user that Recording has started
		printf( "Started Recording\n" );
		printf( "# of samples: %d\n", i );
		error = gettimeofday( &t, NULL );
		if( error < 0 ){ perror( "gettimeofday()" ); }
		start = timeval2double( t );

		while( i-- > 0 ){

			// Get register value
			reg = *(gpio.addr + 13);

			// Write the register to the socket
			bytes = send( sock_fd, &reg, rSize, 0 );
			if( bytes != rSize ){ perror( "send()" ); }

		}

		// Data has been sent, signal end of sending
		i = 0;
		bytes = send( sock_fd, (uint32_t *)&i, rSize, 0 );
		if( bytes != rSize ){ perror( "send()" ); }

		// Let the client know time duration
		error = gettimeofday( &t, NULL );
		if( error < 0 ){ perror( "gettimeofday()" ); }
		total = timeval2double( t ) - start;
		bytes = send( sock_fd, &total, sizeof( double ), 0 );
		if( bytes != sizeof( double ) ){ perror( "send()" ); }

		// Inform user that the recording has stopped
		printf( "Recording Stopped\n" );

	}

	return;

}

double timeval2double( Timeval time ){

	return (double)( time.tv_sec + ( ( (double)time.tv_usec ) / 1000000 ) );

} // end timeval2double( Timeval )
