/*
 * user.c
 *
 *  Created on: Apr 21, 2015
 *      Author: jlavz6
 */

#include <time.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/mman.h>
#include <fcntl.h>
#include <unistd.h>
#include <termios.h>
#include <pthread.h>
#include <semaphore.h>
#include <rtai_fifos.h>
#include <string.h>

//stores temperature readings
typedef struct {
	double temp_read;
	struct timeval temp_time;
}check_Temp;

//variables declared inside of structure to enable shared information between threads
typedef struct{

	int choose_File;//Varaible choosen by the user regarding which output file they would like to log
	int choose_Print;//flag to set printing the screen
	int set_Stop;//flag to stop the collection of temps
	int choose_Periodic;//Variable to choose the period
	int set_Pause;//Flag to pause the collection of temperatures
	int set_Save;//Flag to indicate a temperature was saved
	int fan_set;//Flag to indicate system has reached temperature
	int Reboot;//Flag to reset the system
	struct timeval start_Time;//System start time
	int set_Time;//Flag to indicate that the start time for the system has been set
	int ask_file;//Variable chosen by user on which output file is desired
	double ask_notify;//Variable
	int tempNotifyUnit;
}shared_Info;

void fileWrite(void *ptr);//Will write to an output file
void getTemp();//Collects the temperatures being taken
void UI(void *ptr);//Allows the user to interact with the real time program
int kbhit(void);//Used to allow keyboard usage
void Collect(void *ptr);//Sets up the software side of the program

RTIME Period1;
sem_t sem1;
sem_t sem2;
double time_MS;
unsigned long *ptrSoft = NULL;//Pointer to map to for soft Interrupt
unsigned long *softInt = NULL;//Pointer for soft Interrupt register

void P_SEND(void *ptr)//Thread that will send the temperatures collected through a real time fifo
{
	double Celsius;
	int Fahrenheit;
	*softInt |= 0x80000000;//Enables soft Interrupt
	check_Temp *Temp = (check_Temp *)ptr;
	int k_pipe = 0;

	if((k_pipe = open("/dev/rtf/0", O_RDWR))<0)//Opening real time fifo
	{
		printf("Opening fifo write error.\n");
	}

	//Conversion from AD signal to Celsius temp, and then from Celsius temp to Fahrenheit temp
	Celsius = Temp->temp_read;
	Celsius = Celsius * 5000 / 4096;
	Celsius = (Celsius - 500) / 10;
	Fahrenheit = ((Celsius*1.8) + 32);

	if(write(k_pipe,(void *)&Fahrenheit,sizeof(int))<0)//Write to real time fifo
	{
		printf("pipe write error\n");
		exit(-1);
	}

	pthread_exit(0);
}

int main(void) {

	pthread_t read_Temp;//Thread spawned to read in temperatures
	shared_Info info;//Structure defined type to be passed to other threads in order to share stored information
	volatile unsigned long *enable;//will check if MAX197 installed
	unsigned long  *ptr;

	int fd = open("/dev/mem", O_RDWR|O_SYNC); //file descriptor for mapping
	//Mapping to soft interrupt register
	ptrSoft = (unsigned long *)mmap(NULL,getpagesize(),PROT_READ|PROT_WRITE,MAP_SHARED,fd,0x800C0000);//Mapping to physical memory
	softInt = ptrSoft + 6;//offsetting to soft Interrupt register

	double PeriodTemp = 0;//used for setting the real time period
	int ready;//check to see if user is ready for system start

	system("mkfifo my_pipe >& /dev/null"); //create named pipe
	system("clear");

	sem_init(&sem1, 0, 2);
	//initialize semaphores
	sem_init(&sem2, 0, 1);

	//initialize variables of structure
	info.set_Stop = 0;
	info.choose_Print = 0;
	info.choose_File = 0;
	info.choose_Periodic = 0;
	info.set_Pause = 0;
	info.set_Time = 0;
	info.ask_notify = 0;
	info.fan_set = 0;
	info.set_Save = 0;
	info.Reboot = 0;

	//map address of ptr register
	ptr = mmap(0, getpagesize(), PROT_READ|PROT_WRITE, MAP_SHARED, fd, 0x22400000);

	enable = (unsigned long *)ptr;//check bit 0 to see if MAX197 ADC is installed

	if(*enable & 0x00000001)
	{
		printf("MAX197 Connection\n");
	}

	else
	{
		printf("MAX197 Not Connected\n");
		pthread_exit(0);
	}

	//check if ready to start collecting data
	printf("\nAre you ready to collect temperatures (y/n)? ");
	ready = getchar();

	if(ready == 'y')
	{
		printf("Enter period length in milliseconds? "); //get period value from user
		scanf("%d", &info.choose_Periodic);
		PeriodTemp = info.choose_Periodic;
		PeriodTemp = PeriodTemp * 1000000; //convert to nanoseconds

		Period1 = start_rt_timer(nano2count(PeriodTemp)); //set base period for real time task

		//create thread for data collection
		pthread_create(&read_Temp, NULL, (void*)&Collect, (void *)&info);
		pthread_join(read_Temp, NULL);
	}
	else
	{
		printf("\n");
	}

	usleep(5000);

	return 0;
}

void Collect(void *ptr)
{
	shared_Info *info = (shared_Info *)ptr;
	int iterations;//Variable indicating how long user would like to collect temperatures
	printf("\nEnter how many seconds you would like to collect temperature: ");
	scanf("%d", &iterations);
	//calculation of measurements to be taken
	iterations = ((iterations * 1000) / info->choose_Periodic) + 1;
	//get a temp to indicate system has reached a certain temperature
	printf("\nWould you like to be notified at a specific temp?\n"
			"Specify temperature in degrees Fahrenheit (0 = No): ");
	scanf("%lf", &info->ask_notify);
	info->ask_file = getchar();
	//ask if user would like to save data to files
	printf("\nWould you like to output data collected to an output file?(y/n)\n");
	info->ask_file = getchar();
	//ask user what kind of files they would like and set choice
	if(info->ask_file == 'y')
	{
		printf("Choose type of file:\n");
		printf("1 - (.csv) file\n");
		printf("2 - (.txt) file\n");
		printf("3 - cancel\n");
		scanf("%d", &info->choose_File);
	}

	//create thread for writing to output files and allow user to interact
	pthread_t write_it;
	pthread_create(&write_it, NULL, (void *)&fileWrite, (void *)info);
	system("clear");
	pthread_t user_It;
	pthread_create(&user_It, NULL, (void *)&UI, (void *)info);

	//initialize real time task and make periodic with specified period
	RT_TASK* rtwrite = rt_task_init(nam2num("read_temp"), 0, 512, 256);
	rt_task_make_periodic(rtwrite, rt_get_time() + nano2count(1000000000), Period1);

	gettimeofday(&info->start_Time, NULL);
	//collecting data
	int i=0;
	for(i=0; i < iterations; i++)
	{
		if(info->set_Stop == 1)
		{
			pthread_exit(0); //if stop has been selected, quit data collection
		}
		if(info->set_Pause != 1)
		{
			getTemp(); //get temperature measurement
		}
		else
		{
			if(info->set_Pause == 1)
			{
				printf("\nPaused...\n"); //shows pause if terminal output enabled and paused
			}
		}

		if(info->Reboot == 1)
		{
			i = 0; //if Reboot as has be selected, Reboot data collection
		}
		rt_task_wait_period();
	}
	//System has finished testing
	printf("\nSystem test complete!\n");
	usleep(5000);
	pthread_exit(0);
}

//gets a temperature reading from channel 0 of the MAX 197 ADC on the TS-7250
void getTemp()
{
	volatile unsigned char *op, *control, *busy; //registers for ADC
	check_Temp *Temp; //structure for temperature measurement
	int fd = open("/dev/mem", O_RDWR|O_SYNC); //file descriptor for mapping
	int my_pipe;
	unsigned short *result;
	unsigned char *busy_Val;
	unsigned char *optionValue;

	Temp = (check_Temp*)malloc(sizeof(check_Temp)); //allocate memory

	//map addresses of registers
	op = mmap(0, getpagesize(), PROT_READ|PROT_WRITE, MAP_SHARED, fd, 0x22400000);

	control = mmap(0, getpagesize(), PROT_READ|PROT_WRITE, MAP_SHARED, fd, 0x10F00000);

	busy = mmap(0, getpagesize(), PROT_READ|PROT_WRITE, MAP_SHARED, fd, 0x10800000);

	//open named pipe for writing
	if((my_pipe = open("my_pipe", O_WRONLY)) < 0)
	{
		printf("error creating pipe\n");
		pthread_exit(0);
	}

	//get time of temperature measurement
	gettimeofday(&Temp->temp_time, NULL);

	op = (unsigned char *)op;

	//get sample from ADC Channel 0, Unipolar, 5V Range
	*control = 0x40;


	busy_Val = (unsigned char *)busy;

	//wait until conversion complete
	while(*busy_Val & 0x80)
	{
		busy_Val = (unsigned char *)busy;//bit 7 is 1 when complete
		printf("");
	}
	result = (unsigned short *)control;

	Temp->temp_read = (double)*result;//store result in structure as a double

	pthread_t send_It;//Create pthread to send temps to kernel space
	pthread_create(&send_It, NULL, (void *)&P_SEND, (void *)Temp);

	//write to pipe to get temperature structure to fileWrite thread
	sem_wait(&sem1);
	if(write(my_pipe, Temp, sizeof(*Temp)) != sizeof(*Temp))
	{
		printf("error writing to pipe\n");
		pthread_exit(0);
	}
	sem_post(&sem2);
}


//writes to .csv or .txt (if selected) and displays to terminal (if selected)
void fileWrite(void *ptr)
{
	shared_Info *info = (shared_Info *)ptr;
	check_Temp *Temp;
	Temp = (check_Temp*)malloc(sizeof(check_Temp));
	int my_pipe;
	double Celsius, Fahrenheit;
	FILE *fd1, *fd2;
	double time_MS;

	if(info->choose_File == 1)//if .csv option selected, create .csv file
	{
		fd1 = fopen("dat.csv", "w+");
		fprintf(fd1, "Time, Celsius, Fahrenheit, Save");
	}

	if(info->choose_File == 2)//if .txt option selected, create .txt file
	{
		fd2 = fopen("dat.txt", "w+");
		fprintf(fd2, "Temperature Data\n");
	}

	//open named pipe for reading
	if((my_pipe = open("my_pipe", O_RDONLY)) < 0)
	{
		printf("error creating pipe\n");
		pthread_exit(0);
	}
	while(1)
	{
		sem_wait(&sem2);
		if(info->Reboot == 1)//if Reboot selected, Reboot the system
		{
			if(info->choose_File == 1)
			{
				fclose(fd1);
				fd1 = fopen("dat.csv", "w+");
				fprintf(fd1, "Time, Save");
			}
			if(info->choose_File == 2)
			{
				fclose(fd2);
				fd2 = fopen("dat.txt", "w+");
				fprintf(fd2, "Temperature Data\n");
			}
				info->set_Time = 0; //reset start time
				info->Reboot = 0; //deselect reset
		}

		if((read(my_pipe, Temp, sizeof(*Temp))) < 0)//read from named pipe
		{
			printf("error reading from pipe\n");
			pthread_exit(0);
		}

		if(info->set_Time == 0)//check set_time, (if 0) set start time
		{
			info->start_Time = Temp->temp_time;
			info->set_Time = 1; //set set_Time to show start time has been set
		}

		//convert time to milliseconds
		time_MS = ((Temp->temp_time.tv_sec * 1000.0) - (info->start_Time.tv_sec * 1000.0));
		time_MS += ((Temp->temp_time.tv_usec / 1000.0) - (info->start_Time.tv_usec / 1000.0));

		//convert ADC result to Celsius and then Celsius to Fahrenheit
		Celsius = Temp->temp_read;
		Celsius = Celsius * 5000 / 4096;
		Celsius = (Celsius - 500) / 10;
		Fahrenheit = ((Celsius*1.8) + 32);

		if(info->ask_notify != 0)//If user entered a temperature to be notified of
		{
			if(Fahrenheit >= info->ask_notify)
			{
				info->fan_set = 1;
				printf("Fan will turn on at %d degrees Fahrenheit!\n", (int)info->ask_notify);
			}
			else
			{
				info->fan_set = 0;
			}
		}

		if(info->choose_File == 1)//write Temperatures collected to info.csv
		{
			if(info->set_Save == 1)
			{
				fprintf(fd1, ", Saved...");//if set_Save selected, save measurement
			}
			if(info->fan_set == 1)
			{
				fprintf(fd1, ", Fan turned on...");//If temperature reached for cool down
			}
			fprintf(fd1, "\n%.3f, %.1f, %.lf", time_MS, Celsius, Fahrenheit);
		}

		if(info->choose_File == 2)//write Temperatures collected to info.txt
		{
			if(info->set_Save == 1)
			{
				fprintf(fd2, "Saved...\n"); //if set_Save selected, save measurement
			}
			if(info->fan_set == 1)
			{
				fprintf(fd2, "\nFan turned on...\n");//If temperature reached for cool down
			}

			fprintf(fd2, "\nTime(microseconds): %.4f\n", time_MS);
			fprintf(fd2, "Temp (Celsius): %.1f\n", Celsius);
			fprintf(fd2, "Temp (Fahrenheit): %.lf\n", Fahrenheit);
		}

		if(info->choose_Print == 1)//if choose_Print set, print to the terminal
		{
			printf("\nTime: %.3f\n", time_MS);
			printf("Temp (Celsius): %.1f\n", Celsius);
			printf("Temp (Fahrenheit): %.lf\n", Fahrenheit);
		}

		//deselect flags
		info->set_Save = 0;
		info->fan_set = 0;
		sem_post(&sem1);
	}

	usleep(5000);
	free(Temp);
	pthread_exit(0);
}

void UI(void *ptr)
{
	shared_Info *info = (shared_Info *)ptr;
	int user_Choice;
	printf("System test started...\n"); //system start
	while(1)
	{ //display options to user during data collection
		printf("\nEnter selection by pressing corresponding keyboard number.\n");
		printf("1 - Start/Pause Collection Process\n");
		printf("2 - View Temperatures being collected\n");
		printf("3 - Save Previous Temperature\n");
		printf("4 - Stop Collecting Temperatures\n");
		scanf("%d", &user_Choice);

		//if option 1 selected, start/pause data collection
		if(user_Choice == 1)
		{
			info->set_Pause = info->set_Pause ^ (1 << 0);
			if(info->set_Pause == 0)
			{
				printf("\nCollecting Data...\n");
			}
			else if(info->set_Pause == 1)
			{
				printf("\nPaused...\n");
			}
		}

		//if option 2 selected, flag choose_Print which will enable printing to the terminal
		else if(user_Choice == 2)
		{
			info->choose_Print = 1;
			while(!kbhit())
			{ //wait for keyboard press and then end viewing
				printf("");
			}
			user_Choice = 0;
			info->choose_Print = 0;
		}

		//if option 3 selected, save temperature reading
		else if(user_Choice == 3)
		{
			info->set_Save = 1;
			printf("\nRecorded!\n");
		}

		//if option 4 selected, stop system testing
		else if(user_Choice == 4)
		{
			info->set_Stop = 1;
			printf("\nSystem stop!\n");
			pthread_exit(0);
		}
	}
}

//function used to monitor for a keyboard press
int kbhit(void)
{
	struct timeval tv;
	fd_set rdfs;
	tv.tv_sec = 0;
	tv.tv_usec = 0;
	FD_ZERO(&rdfs);
	FD_SET(STDIN_FILENO, &rdfs);
	select(STDIN_FILENO+1, &rdfs, NULL, NULL, &tv);
	return FD_ISSET(STDIN_FILENO, &rdfs);
}
