
#include <Adafruit_CC3000.h>
#include <Adafruit_CC3000_Server.h>
#include <ccspi.h>
#include <SPI.h>
#include <UDPServer.h>
#include <string.h>
#include <Time.h>
#include "utility/debug.h"
#include "utility/socket.h"

#define UDP_READ_BUFFER_SIZE 10
#define LISTEN_PORT_UDP 2000
UDPServer udpServer(LISTEN_PORT_UDP);

// These are the interrupt and control pins
#define ADAFRUIT_CC3000_IRQ   3  // MUST be an interrupt pin!
// These can be any two pins
#define ADAFRUIT_CC3000_VBAT  5
#define ADAFRUIT_CC3000_CS    10
Adafruit_CC3000_Client udpClient;

// Use hardware SPI for the remaining pins
// On an UNO, SCK = 13, MISO = 12, and MOSI = 11
Adafruit_CC3000 cc3000 = Adafruit_CC3000(ADAFRUIT_CC3000_CS, ADAFRUIT_CC3000_IRQ, ADAFRUIT_CC3000_VBAT,
                                         SPI_CLOCK_DIVIDER); // you can change this clock speed but DI

#define WLAN_SSID       "Tetradecagon"        
#define WLAN_PASS       "lj0t-duun-wune"
#define WLAN_SECURITY   WLAN_SEC_WPA2

const int LedPin = 9;      // Connect the pin to Arduino pin 10
const int MosfetPin = 2;    // Connect the MOSFeT Gate (first pin on the left, when writing on the chip is facing you) to Arduino pin 2
const int MotorPin = 8;
const int WaterSensor = A0;

int sensorValue = 0;         // the sensor value
int sensorMin = 1023;        // minimum sensor value
int sensorMax = 0;           // maximum sensor value

/**************************************************************************/
/*!
    @brief  Sets up the HW and the CC3000 module (called automatically
            on startup)
*/
/**************************************************************************/
void setup(void)
{
  Serial.begin(115200);

  Serial.print("Free RAM: "); Serial.println(getFreeRam(), DEC);
  
  /* Initialise the module */
  Serial.println(F("\nInitialising the CC3000 ..."));
  if (!cc3000.begin())
  {
    Serial.println(F("Unable to initialise the CC3000! Check your wiring?"));
    while(1);
  }
  
  /* Delete any old connection data on the module */
  Serial.println(F("\nDeleting old connection profiles"));
  
  if (!cc3000.deleteProfiles())
  {
    Serial.println(F("Failed!"));
    while(1);
  }

  /* Attempt to connect to an access point */
  char *ssid = WLAN_SSID;             /* Max 32 chars */
  Serial.print(F("\nAttempting to connect to ")); Serial.println(ssid);
  
  if (!cc3000.connectToAP(WLAN_SSID, WLAN_PASS, WLAN_SECURITY)) 
  {
    Serial.println(F("Failed!"));
    while(1);
  }
   
  Serial.println(F("Connected!"));
  
  /* Wait for DHCP to complete */
  Serial.println(F("Request DHCP"));
  while (!cc3000.checkDHCP())
  {
    delay(100); // ToDo: Insert a DHCP timeout!
  }  

  /* Display the IP address DNS, Gateway, etc. */  
  while (! displayConnectionDetails()) 
  {
    delay(1000);
  }
  //Set the pin modes for all pins needed to control circuits
  pinMode(LedPin, OUTPUT);
  pinMode(MosfetPin, OUTPUT);
  pinMode(MotorPin, OUTPUT);
  pinMode(WaterSensor, INPUT);
  
     //Calibrate the sensor
   calibrateWaterSensor();
  
}

void loop() 
{
	//initialize the sockets and buffers
     udpServer.begin();
     size_t messageSize = sizeof(uint8_t) * 48;
     sensorMax += 100;
     int n = 0, i;
     unsigned long piIP;
     const unsigned long connectTimeout  = 15L * 1000L, startTime = millis();
     char *buffer = (char *)malloc(sizeof(char) * 15), *send = (char *)malloc(sizeof(char) * 15);
     memset(buffer, 0, sizeof(buffer));
     memset(send, 0, sizeof(send));
     cc3000.getHostByName("192.168.1.114", &piIP);
     
     do 
     {
       udpClient = cc3000.connectUDP(piIP, LISTEN_PORT_UDP);
     } 
     while((!udpClient.connected()) &&((millis() - startTime) < connectTimeout));
            Serial.println("Connected");       
     
     while(1)
    {
      int startValve, endValve, time;
      n = udpServer.readData(buffer, messageSize);  // n contains # of bytes read into buffer
      
      if(strcmp(buffer, "Water") == 0)
      {      
		//Handle the water command
         Serial.print("From Pi: "); Serial.println(buffer);
         Serial.println("Activating faucet");
		 //Record the start time
         startValve = millis();
		 //Set the led and Mosfet pins high
         digitalWrite(LedPin, HIGH);
         digitalWrite(MosfetPin, HIGH);
         memset(buffer, 0, sizeof(buffer));
                        
          i = 0;
          while( i < 2)
          {
			//Read the sensor and print readings for debugging
            Serial.print("Max:");Serial.println(sensorMax);
            Serial.print("Current Value:");Serial.println(sensorValue);
            sensorValue = analogRead(WaterSensor);
            
            if(sensorValue > sensorMax)
              i++;
            //sleep before next reading
            delay(500);
          }
          //Record end time and calculate duration
          endValve = millis();
          time = endValve - startValve;
          
          Serial.println(sensorMax);
          Serial.println(sensorValue);
		  //Set the pins low and send the completion response to the Pi
          digitalWrite(LedPin, LOW);
          digitalWrite(MosfetPin, LOW);
          uint8_t response[48] = {68, 111, 110, 101, 44, 84, 105, 109, 101, 58};
          
          udpClient.write(response, sizeof(response)); 
             
      }
      else if(strcmp(buffer, "Food") == 0)
      {
		  //Handle the Food Command
         Serial.print("From Pi: "); Serial.println(buffer);
         memset(buffer, 0, sizeof(buffer));
		 //Activate the motor
         digitalWrite(MotorPin, HIGH);
		 //Leave motor on for 5 secs
         delay(5000);
		 //Deactivate the motor
         digitalWrite(MotorPin, LOW);
		 
		 //simulate the Jam sensor using a random number generation
         int jam = rand() % 10;
         
		 //1 in 10 chance of a jam
         if(jam == 1)
         {
			 //If Jammed send message saying so
           uint8_t response[48] = {68, 111, 110, 101, 44, 32, 74, 97, 109, 32, 68, 101, 116, 101, 99, 116, 101, 100, 33};
           udpClient.write(response, sizeof(response));
         }
         else
         {
			 //Otherwise send normal completion message
           uint8_t response[48] = {68, 111, 110, 101, 44, 84, 105, 109, 101, 58};
           udpClient.write(response, sizeof(response));
         }
         
      }
    } 
     
   //CLose the connection
  Serial.println(F("\n\nClosing the connection"));
  cc3000.disconnect();
  delay(5000);
}

void calibrateWaterSensor(void)
{
  Serial.println("Calibrating");
  //Calibrate the sensor taking 10 readings over 5 seconds
  for(int i = 0; i < 10; i ++) 
   {
      sensorValue = analogRead(WaterSensor);

      if (sensorValue > sensorMax) 
      {
        sensorMax = sensorValue;
      }

      if (sensorValue < sensorMin) 
      {
        sensorMin = sensorValue;
      }
      
      delay(500);
  } 
  
  delay(5000);
  //Use maximum and minimum readings to constrain the sensor values
  sensorValue = analogRead(WaterSensor);
  sensorValue = map(sensorValue, sensorMin, sensorMax, 0, 255);
  sensorValue = constrain(sensorValue, 0, 255);
  Serial.println(sensorMax);
  Serial.println(sensorMin);
  
  Serial.println("Calibration Complete!");
}

/**************************************************************************/
/*!
    @brief  Tries to read the IP address and other connection details
*/
/**************************************************************************/
bool displayConnectionDetails(void)
{
  uint32_t ipAddress, netmask, gateway, dhcpserv, dnsserv;
  //Display network information from the CC3000
  if(!cc3000.getIPAddress(&ipAddress, &netmask, &gateway, &dhcpserv, &dnsserv))
  {
    Serial.println(F("Unable to retrieve the IP Address!\r\n"));
    return false;
  }
  else
  {
    Serial.print(F("\nIP Addr: ")); cc3000.printIPdotsRev(ipAddress);
    Serial.print(F("\nNetmask: ")); cc3000.printIPdotsRev(netmask);
    Serial.print(F("\nGateway: ")); cc3000.printIPdotsRev(gateway);
    Serial.print(F("\nDHCPsrv: ")); cc3000.printIPdotsRev(dhcpserv);
    Serial.print(F("\nDNSserv: ")); cc3000.printIPdotsRev(dnsserv);
    Serial.println();
    return true;
  }
}

