#ifndef MODULE
#define MODULE
#endif

#ifndef __KERNEL__
#define __KERNEL__
#endif

#include <linux/module.h>
#include <linux/kernel.h>
#include <unistd.h>
#include <rtai.h>
#include <stdio.h>
#include <asm/io.h>
#include <rtai_sched.h>
#include <rtai_fifos.h>

#define IDLE_TIME_CHECK	10		//time to sleep in an infinite check loop so
								//the process doesn't hog the memory
#define SOUND_PERIOD	2		//the period, in millisecond, to play sound
								//in the sound task

RTIME period;					// 1 ms time.
								// if 1, means threat detected
								// if 0, means just chill
int alert_sig = 1;				// alert signal to report to user space
								// module
int sound = 0;					// toggle to play sound
static RT_TASK sound_task;
unsigned char *map, *mapVIC, *PortB, *PortBDR, *PortF, *PortFDR, *GPIOBIntEn, *GPIOBIntType1, *GPIOBIntType2, *GPIOBEOI, *GPIOBDB, *RawIntStsB;
unsigned int *VIC2IntEnable, *VIC2SoftIntClear;


MODULE_LICENSE("GPL");

//returns either 0 or 1, which is the content of the particular bit
//bitwise "AND" operation to the number shifted to the number of bitPosition
//and the bitvector 0000 0001, thus what will be left is either 1 or 0.
//return -1 if error occurs
int getBitContent(int bitPosition, int content);

//returns the content that had been assigned value at bitPosition
//to set to 1, we use the "OR" operation
//to set to 0, we use the "XOR" operation
// but to check that setting to 0 is valid or not, we first "AND"
// with 0x** to see if the result is 0x** or 0x00. If 0x00, do nothing.
// return -1 in case of error
int assignBitContent(int bitPosition, int content, int value);

//RT TASK, TURN ON RED LIGHT FOR A PERIOD OF TIME
//IF GREEN LIGHT IS DONE OR WALKING BUTTON IS PRESSED
static void rt_process_sound(int t);

//When the software interrupt triggers, send the alert signal
//through the FIFO so that the user space can deal with it
//accordingly.
//FIFO channel is FIFO 0
// alert = 0 --> no alert
// alert = 1 --> threat detected
void alert();

//When the software interrupt triggers, meaning button on auxiliary board
//had been pressed, send the alert signal
//through the FIFO so that the user space can deal with it
//accordingly.
//FIFO channel is FIFO 0
// alert = 0 --> turn off threat detection
//turn off the sound
void chill();

int init_module(void) {

	int temp, j;

	printk("MODULE INSTALLED\n");

	//rt_set_periodic_mode();
	//make the period 1 ms
	period = start_rt_timer(nano2count(1000000));

	//Map to registers and port locations
	//map the base for Port B and the VIC
	map = (unsigned char*)__ioremap(0x80840000, 4096, 0);
	mapVIC = (unsigned char*)__ioremap(0x800C0000, 4096, 0);

	PortB = (unsigned int*) (map+0x04);
	PortBDR = (unsigned int*)(map+0x14);
	PortF = (unsigned int*)(map + 0x30);
	PortFDR = (unsigned int*)(map + 0x34);

	VIC2IntEnable = (unsigned int*)(mapVIC+0x10);
	VIC2SoftIntClear = (unsigned int*)(mapVIC+0x1C);

	//setting the direction of port B, which will output
	//the alert signal
	*PortBDR = 0xE0;
	//set direction to output for speaker
	temp = assignBitContent(1, (int)*PortFDR, 1);
	*PortFDR = temp;
	//reset lights on auxiliary board
	//turn on everything except green
	temp = assignBitContent(5, (int)*PortB, 0);
	*PortB = temp;
	temp = assignBitContent(6, (int)*PortB, 0);
	*PortB = temp;
	temp = assignBitContent(7, (int)*PortB, 1);
	*PortB = temp;

	//map registers to control edge-sensitivity and rising/falling edge signals
	GPIOBEOI = (unsigned int*)(map+0xB4);
	GPIOBDB = (unsigned int*)(map+0xC4);
	GPIOBIntEn = (unsigned int*)(map+0xB8);
	GPIOBIntType1 = (unsigned int*)(map+0xAC);
	GPIOBIntType2 = (unsigned int*)(map+0xB0);

	//map registers to enable specific interrupt lines
	RawIntStsB = (unsigned int*)(map+0xC0);

	//set registers to be edge-sensitive and rising edge signals
	*GPIOBIntType1 |= 0x1F;
	*GPIOBIntType2 &= 0xE0;
	*GPIOBDB |= 0x1F;

	//request irq for hardware
	rt_request_irq(59, chill, 0, 1);
	*GPIOBEOI |= 0x1f;
	*GPIOBIntEn |= 0x1f;
	//Enable irq for hardware
	rt_enable_irq(59);

	//request irq for software
	rt_request_irq(63, alert, 0, 1);
	*VIC2SoftIntClear |= 0x80000000;
	*VIC2IntEnable |= 0x80000000;
	//Enable irq for Software
	rt_enable_irq(63);

	//Create a fifo for extra credit and one for software interrupt
	rtf_create(0, sizeof(int));

	//start rt task
	rt_task_init(&sound_task, rt_process_sound, 0, 256, 0, 0, 0);

	//Set first tone to be C
	rt_task_make_periodic(&sound_task, rt_get_time()+1*period, SOUND_PERIOD*period);

	return 0;

}

void cleanup_module(void){

	printk("MODULE UNINSTALLED\n");

	//disable irq
	rt_disable_irq(63);
	rt_disable_irq(59);

	//release irq
	rt_release_irq(63);
	rt_disable_irq(59);

	//destroy fifos
	rtf_destroy(0);

	//stop rt timer
	stop_rt_timer();
}

//When the software interrupt triggers, meaning button on auxiliary board
//had been pressed, send the alert signal
//through the FIFO so that the user space can deal with it
//accordingly.
//FIFO channel is FIFO 0
// alert = 0 --> turn off threat detection
//turn off the sound
void chill(){
	//Disable software interrupt line
	rt_disable_irq(59);

	alert_sig = 0;
	sound = 0;

	//send alert value to user space module
	rtf_put(0, &alert_sig, sizeof(int));

	//clear interrupt
	*GPIOBEOI |= 0x1f;
	//re-enable the interrrupt line
	rt_enable_irq(59);
}

//When the software interrupt triggers, send the alert signal
//through the FIFO so that the user space can deal with it
//accordingly.
//FIFO channel is FIFO 0
// alert = 1 --> threat detected
//toggle on the sound as well
void alert(){
	//Disable software interrupt line
	rt_disable_irq(63);

	alert_sig = 1;
	sound = 1;

	//send alert value to user space module
	rtf_put(0, &alert_sig, sizeof(int));

	//clear interrupt
	*VIC2SoftIntClear |= 0x80000000;
	//re-enable the interrupt
	rt_enable_irq(63);
}

//RT TASK, TURN ON RED LIGHT FOR A PERIOD OF TIME
//IF GREEN LIGHT IS DONE OR WALKING BUTTON IS PRESSED
static void rt_process_sound(int t){

	int btn_temp = -1;
	int toggle_sound = 0;		//switches between 0 and 1 between periods
	int light_temp = 0;
	int modified_bit_vector;

	while(1){


		while(sound == 1){

			//toggle the value from period to period
			if(toggle_sound == 0){
				toggle_sound = 1;
			} else if(toggle_sound == 1){
				toggle_sound = 0;
			}

			modified_bit_vector = assignBitContent(1, (int)*PortF, toggle_sound);
			*PortF = modified_bit_vector;

			//wait for next period to play
			rt_task_wait_period();
		}

		//when no buttons pressed, wait then check after a while
		rt_sleep(IDLE_TIME_CHECK*period);
	}
}

//returns either 0 or 1, which is the content of the particular bit
//bitwise "AND" operation to the number shifted to the number of bitPosition
//and the bitvector 0000 0001, thus what will be left is either 1 or 0.
//return -1 if error occurs
int getBitContent(int bitPosition, int content){

	int mask = 0x01;
	int shifted = 0;
	int result = -1;

	shifted = content >> bitPosition;
	result = shifted & mask;
	//printk("\nResult at bit %d of %d is %d\n", bitPosition, content, result);
	return result;
}

//returns the content that had been assigned value at bitPosition
//to set to 1, we use the "OR" operation
//to set to 0, we use the "XOR" operation
// but to check that setting to 0 is valid or not, we first "AND"
// with 0x** to see if the result is 0x** or 0x00. If 0x00, do nothing.
// return -1 in case of error
int assignBitContent(int bitPosition, int content, int value){
	int bit_zero = 0x01;
	int bit_one = 0x02;
	int bit_two = 0x04;
	int bit_three = 0x08;
	int bit_four = 0x10;
	int bit_five = 0x20;
	int bit_six = 0x40;
	int bit_seven = 0x80;
	int mask_bit = 0;
	int modified_content = -1;
	int checkContent = -1;

	switch (bitPosition){
		case 0:
			mask_bit = bit_zero;
			break;
		case 1:
			mask_bit = bit_one;
			break;
		case 2:
			mask_bit = bit_two;
			break;
		case 3:
			mask_bit = bit_three;
			break;
		case 4:
			mask_bit = bit_four;
			break;
		case 5:
			mask_bit = bit_five;
			break;
		case 6:
			mask_bit = bit_six;
			break;
		case 7:
			mask_bit = bit_seven;
			break;
		default:
			//printk("\nThe fuck, dude?\n");
			return -1;
	}

	if(value == 1){

		//turn on by using OR
		modified_content = content | mask_bit;
		//printk("\nAssigning to 1\n");
	}else if(value == 0){

		//check if we can turn it off or not
		checkContent = content & mask_bit;

		//if that particular bit is already 0, then nothing to do
		if(checkContent == 0x00){
			modified_content = content;
			//printk("\nAlready 0\n");
		} else if (checkContent == mask_bit) {

			//if that particular bit is 1, then we can turn it to 0 with XOR
			modified_content = content ^ mask_bit;
			//printk("\nAssigning to 0\n");
		}
	} else{
		//printk("\nContent can only be 1 or 0. Twat.\n");
		return -1;
	}

	//printk("\nYou chose to turn bit %d to %d from %d the result is %d\n", bitPosition, value, content, modified_content);
	return modified_content;
}


