#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stddef.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <netinet/in.h>
#include <netdb.h>
#include <arpa/inet.h>
#include <errno.h>
#include <time.h>
#include <math.h>
#include <rtai.h>
#include <rtai_lxrt.h>
#include <rtai_sched.h>
#include <rtai_fifos.h>
#include <pthread>
#include <fcntl.h>
#include <semaphore.h>
#include <assert.h>

#include "peekpoke.h"
#include "eeprog.h"
#include "ep93xx_adc.h"


#define MSG_SIZE 		20// message size
#define HOST_NAME_MAX 	255
#define DATA_PAGE 		0x80840000
#define CALIB_LOC    	2027         //location of calibration values
#define LR_PERIOD		1			//time in ms in which linear regression recalculates
#define WORKING_CHANNEL	1			//the channel we are interested in reading from
#define CALIBRATE_TO_INT 1000		//modifier constant to data to transform
									//from floating point to integer
									//to lessen computation cost
									//determined experimentally with respect to
									//convenience
#define SCALE_UP_TIME	1			//modifier constant to scale up the time stamp
									//in case the number became too small
#define BASELINE_TIME	600			//the ideal starting value for time, given its
									//computational compatibility with the sensor value
									//which ranges from 200 mV to 50 mV (determined
									//experimentally)
									//need this to reset SCALE_DOWN_TIME
									//since slope gets less and less sensitive with
									//greater time. need to also keep in mind not to
									// make it TOO sensitive
#define SCALE_UP_SLOPE	1000000		//a constant modifier that increases the slope value
									//to a manageable number
#define	SAMPLE_SIZE		50				//size of the array to calculate linear regression
#define TRIGGER_SLOPE	5				//the linear regression slope that signifies a threat
									//determined experimentally, subject to change
									//depending upon context

RTIME timer;				//time in milliseconds
int kernel_fifo_id = -1;	//fifo id that will communicate with kernel
int alert_sig = 0;			//signal that determines when to turn on red light
							//and stop all operations
int	SCALE_DOWN_TIME = 0;	//a value to be subtracted from the timestamp
									//to scale down the timestamp value
									// so that any calculations would be measureable
static unsigned long dr_page, adc_page, syscon_page, pld_page; //mapped ports
																//for ADC
long unsigned int *mapVIC, *VIC2SoftInt, *VIC2IntEnable, *VIC2SoftIntClear; //software interrupt register
long unsigned int *map, *PortB, *PortBDR; //port B register
							// assuming direction had already been set by
							// kernel. if not, mask PortBDR to 0xE0
double old_time;			//time when the program receives alert
double new_time;			//time when program finishes dealing with alert
int stored_cal[5][2];  		//stored calibration values for ADC, first dimension are
					   	   	// 5 channel, second dimension are 2 calibration values
int virgin = TRUE;			//calibration detected?

//NETWORKING VARIABLES FOR SENDING DATA
//binding addresses for socket communication
struct sockaddr_in server;
struct sockaddr_in from;
socklen_t fromlen;
int sock;


void error(const char *msg);

//CONVERT TIMEVAL STRUCTURE TO DOUBLE FOR EASIER PROCESSING
//COMPRESS TO FORMAT: (double) <tv_sec>.<usec>
double timeval_to_double(struct timeval in);

/************************************************************************
 *DESCRIPTION: Figure out the calibration value of the board and store them
 *stored_cal global variable. In eeprom 2027->2046 are the calibration values. 2027->2036
*are the zero volt calibration values and 2037->2046 are the 2.5V calibration
*values. Each calibration value is 16 bits written using little endian
 ***********************************************************************/
static void init_adc_channel_calibration();

/******************************************************************************
*DESCRIPTION: Read the data from the desired channel before calibrating it accordingly
*******************************************************************************/
double read_ADC_channel(int ch_num);

//convert string of ip address into integer, replacing '.' wit 0s
int IPToInt(char* ip_addr);

//returns either 0 or 1, which is the content of the particular bit
//bitwise "AND" operation to the number shifted to the number of bitPosition
//and the bitvector 0000 0001, thus what will be left is either 1 or 0.
//return -1 if error occurs
int getBitContent(int bitPosition, int content);

//returns the content that had been assigned value at bitPosition
//to set to 1, we use the "OR" operation
//to set to 0, we use the "XOR" operation
// but to check that setting to 0 is valid or not, we first "AND"
// with 0x** to see if the result is 0x** or 0x00. If 0x00, do nothing.
// return -1 in case of error
int assignBitContent(int bitPosition, int content, int value);

//calculates the linear regression of the incoming data
// ADC only deals with 1 channel as to simplify this project
// thus we read in from only channel 1
// first takes in data for the first 100 values to
//	create a baseline model and set thresholds
//	then each incoming value will be evaluated
//	if it breaches the threshold, then set the software interrupt
void* linear_regression();

//THREAD FUNCTION: reads in data from the fifo
//if software interrupt triggers in kernel space module
// then data will be sent through FIFO ID 0
// if the read data is 1, then alert happens
//  and we turn on the red light on auxiliary board
//	and turn off every other light
void* check_alert();

// when an alert had been triggered, call this function
// this is suppose to be a prototype function where
// anyone can use to implement their own response to a
// threat. For this project, the response function
// will turn off all LEDs in the auxiliary board except
// the red one, then report a time stamp.
int response();

int main(int argc, char *argv[]){

	pthread_t receiver, linear_regression_thr;
	void* ret_val;

    int length, n;
    int boolval = 1; // for a socket option
    int port_no = -1;
    int host_ip_num = -1;
    int temp = 0;

    //structure storing data for acquiring IP address
    struct hostent *he;
    char buffer_msg[MSG_SIZE];
    char host_ip[100];
    char hostname[HOST_NAME_MAX]; // from the linux documentation:
                                    //SUSv2 guarantees that "Host names are limited to 255 bytes".
                                    //POSIX.1-2001 guarantees that "Host names
                                    //(not including the terminating null byte) are
                                    //limited to HOST_NAME_MAX bytes".
                                    //On Linux, HOST_NAME_MAX is defined with the value 64,
                                    //which has been the limit since Linux 1.0
                                    //(earlier kernels imposed a limit of 8 bytes).
    if (argc < 2){
        fprintf(stderr, "ERROR, no port provided\n");
        printf("\nBut it's ok, since you are in this lab, the port is always 2000. Just think about how lucky you are.\n");
        port_no = 2000;
        //exit(0);
    } else {
        port_no = atoi(argv[1]);
    }

    /**************************************************************************
	*	BEGIN NETWORK COMMUNICATION SETUP
	**************************************************************************/

    sock = socket(AF_INET, SOCK_DGRAM, 0); // Creates socket. Connectionless.
    if (sock < 0)
        error("Opening socket");

    length = sizeof(server);             // length of structure
    bzero(&server, length);             // sets all values to zero. memset() could be used
    server.sin_family = AF_INET;         // symbol constant for Internet domain
    server.sin_addr.s_addr = INADDR_ANY; // IP address of the machine on which

    // the server is running
    server.sin_port = htons(port_no); // port number

    // binds the socket to the address of the host and the port number
    if (bind(sock, (struct sockaddr *) &server, length) < 0)
        error("binding");

    // set broadcast option
    if (setsockopt(sock, SOL_SOCKET, SO_BROADCAST, &boolval, sizeof(boolval))< 0){
        printf("error setting socket options\n");
        exit(-1);
    }

    fromlen = sizeof(struct sockaddr_in); // size of structure

    //figuring out the host IP address
    if (gethostname(hostname, HOST_NAME_MAX) != 0) { //get self's host name
                                                    //to figure out IP from
        printf("\nError gethostname: %s\n", strerror(errno));
        exit(-1);
    }

    if ((he = gethostbyname(hostname)) == NULL) { //get the pointer to
                                                //structure hostent
                                                //for the given host name
                                                // get the host info
        herror("gethostbyname");
        exit(-1);
    }

    //h_addr is an element of the hostent structure
    //that stores the host's own address.
    //since the result is returned as a pointer, we need to dereference it
    //as well.
    //inet_ntoa() function converts the Internet host address in,
    //given in network byte order, to a string in IPv4 dotted-decimal notation.
    //The string is returned in a statically allocated buffer, which subsequent
    //calls will overwrite.
    strcpy(host_ip, inet_ntoa(*((struct in_addr*) he->h_addr))); //host_ip now stores
                                                                //the string of IP of
                                                                //the host
    host_ip_num = IPToInt(host_ip);

    printf("\nHost IP string: %s, num %d\n", host_ip, host_ip_num);

    /**************************************************************************
    *	END NETWORK COMMUNICATION SETUP
	***************************************************************************/
    /**************************************************************************
    *	BEGIN MAP PORTS
	***************************************************************************/
    //map board to correct location on TS-7250
	int board = open("/dev/mem", O_RDWR|O_SYNC);

	//VIC registers
	mapVIC = (long unsigned*)mmap(0, getpagesize(), PROT_READ|PROT_WRITE, MAP_SHARED, board, 0x800C0000);
	VIC2SoftInt = (long unsigned)(mapVIC) | 0x18;
	VIC2IntEnable = (long unsigned)(mapVIC) | 0x10;
	VIC2SoftIntClear = (long unsigned)mapVIC | 0x1C;

	//port b registers
	map = (long unsigned*)mmap(0, getpagesize(), PROT_READ|PROT_WRITE, MAP_SHARED, board, 0x80840000);
	PortB = (long unsigned)(map) | 0x04;
	PortBDR = (long unsigned)(map) | 0x14;

	//map ports for ADC
	dr_page = (unsigned long)mmap(0, getpagesize(), PROT_READ|PROT_WRITE, MAP_SHARED, board, DATA_PAGE);
	assert(&dr_page != MAP_FAILED);
	spistart = (unsigned long)mmap(0, getpagesize(), PROT_READ|PROT_WRITE, MAP_SHARED, board, SPI_PAGE);
	assert(&spistart != MAP_FAILED);
	adc_page = (unsigned long)mmap(0, getpagesize(), PROT_READ|PROT_WRITE, MAP_SHARED, board, ADC_PAGE);
	assert(&adc_page != MAP_FAILED);
	syscon_page = (unsigned long)mmap(0, getpagesize(), PROT_READ|PROT_WRITE, MAP_SHARED , board, SYSCON_PAGE);
	assert(&syscon_page != MAP_FAILED);
	pld_page = (unsigned long)mmap(0, getpagesize(), PROT_READ|PROT_WRITE, MAP_SHARED, board, PLD_PAGE);
	assert(&pld_page != MAP_FAILED);

	//initialize ADC
	init_ADC(adc_page, syscon_page);
	init_adc_channel_calibration();
	printf("\nADC initialized\n");

	/**************************************************************************
    *	END MAP PORTS
    **************************************************************************/

	/**************************************************************************
    *	BEGIN REAL TIME TASKS SETUP
	**************************************************************************/

	timer = start_rt_timer(nano2count(1000000));

    //create the thread that awaits data from fifo
    if(pthread_create(&receiver, NULL, (void*)check_alert, NULL) != 0){
        printf("\nError creating alert thread: \n%s\n\n", strerror(errno));
    }

    if(pthread_create(&linear_regression_thr, NULL, (void*)linear_regression, NULL) != 0){
		printf("\nError creating LR thread: \n%s\n\n", strerror(errno));
	}

	/**************************************************************************
	*	END REAL TIME TASKS SETUP
	**************************************************************************/

	/**************************************************************************
    *	RT LOOP
    *	for continuous communication
    **************************************************************************/
    printf("\nSocket setup complete\n");
    while (1){

        // bzero: to "clean up" the buffer. The messages aren't always the same length...
        bzero(buffer_msg, MSG_SIZE); // sets all values to zero. memset() could be used
                                    // receive from a client
        n = recvfrom(sock, buffer_msg, MSG_SIZE, 0, (struct sockaddr *) &from,&fromlen);
        if (n < 0)
            error("recvfrom");

        //printf("Received a datagram. It says:\n%s\n", buffer_msg);

        // To send a broadcast message, we need to change IP address to broadcast address
        // (.255). If we don't change it (with the following line of code), the message
        // would be transmitted to the address from which the message was received.

        from.sin_addr.s_addr = inet_addr("10.3.52.255"); // broadcast address

        //HANDLING DIFFERENT REQUESTS
        if (strcmp(buffer_msg, "GOOD") == 0) { //other machines reporting that
        										//things are ok now
            bzero(buffer_msg, MSG_SIZE);

            printf("\nGOOD received\n");

            if(alert_sig != 0){
				//reset the sig status
				alert_sig = 0;

				//close red light and turn on green light
				temp = assignBitContent(5, (int)*PortB, 0);
				*(PortB) = temp;
				temp = assignBitContent(6, (int)*(PortB), 0);
				*(PortB) = temp;
				temp = assignBitContent(7, (int)*(PortB), 1);
				*(PortB) = temp;
            }

        } else if(strcmp(buffer_msg, "BAD") == 0){	//other machines telling
        											//us that things are
        											//getting bad
        	printf("\nBAD received\n");

        	if(alert_sig != 1){
				alert_sig = 1;

				//close green light and turn on red light
				temp = assignBitContent(5, (int)*(PortB), 1);
				*(PortB) = temp;
				temp = assignBitContent(6, (int)*(PortB), 0);
				*(PortB) = temp;
				temp = assignBitContent(7, (int)*(PortB), 0);
				*(PortB) = temp;

				//triggers software interrupt
				*VIC2SoftInt = 0x80000000;
        	}
        }
    }

    stop_rt_timer();

    return 0;
}

// when an alert had been triggered, call this function
// this is suppose to be a prototype function where
// anyone can use to implement their own response to a
// threat. For this project, the response function
// will turn off all LEDs in the auxiliary board except
// the red one, then report a time stamp.
// IF all goes well, return 1, if not then return 0.
int response(){
	int status = 0;
	int temp;
	double response_time = 0;
	struct timeval time_buffer;
	char message[MSG_SIZE];

    server.sin_addr.s_addr = inet_addr("10.3.52.255"); // broadcast address

	if(alert_sig == 1){

		strcpy(message, "BAD");

		//send out alert messages
		if(sendto(sock, message, MSG_SIZE, 0, (struct sockaddr *) &server, sizeof(struct sockaddr_in)) < 0){
			//printf("Error sending message! %s", strerror(errno));
			error("sendto");
		}

		//close green light and turn on red light
		temp = assignBitContent(5, (int)*(PortB), 1);
		*(PortB) = temp;
		temp = assignBitContent(6, (int)*(PortB), 0);
		*(PortB) = temp;
		temp = assignBitContent(7, (int)*(PortB), 0);
		*(PortB) = temp;

		//report timestamp
		gettimeofday(&time_buffer, NULL);
		new_time = timeval_to_double(time_buffer);

		response_time = new_time - old_time;

		//REPORT RESPONSE TIME

		printf("\nRESPONSE TIME: %lf\n", response_time);
	} else if(alert_sig == 0){

		strcpy(message, "GOOD");

		//send out OK messages
		if(sendto(sock, message, MSG_SIZE, 0, (struct sockaddr *) &server, sizeof(struct sockaddr_in)) < 0){
			//printf("Error sending message! %s", strerror(errno));
			error("sendto");
		}

		//close red light and turn on green light
		temp = assignBitContent(5, (int)*(PortB), 0);
		*(PortB) = temp;
		temp = assignBitContent(6, (int)*(PortB), 0);
		*(PortB) = temp;
		temp = assignBitContent(7, (int)*(PortB), 1);
		*(PortB) = temp;
	}

	return status;
}

//THREAD FUNCTION: reads in data from the fifo
//if software interrupt triggers in kernel space module
// then data will be sent through FIFO ID 0
// if the read data is 1, then alert happens
//  and we turn on the red light on auxiliary board
//	and turn off every other light
// No need to set up RT task, since we depend on
// read as a blocking function, thus it is not
// dependent upon scheduling schemes
void* check_alert(){

	struct timeval time_buffer;
	int temp = -1;

	//OPENING FIFO ACCESS --> ASSUMING THAT ID OF THE FIFO IS 0
    if((kernel_fifo_id = open("/dev/rtf/0", O_RDONLY))<0){
        printf("\nERROR OPENING FIFO from /dev/rtf/0\n");
        exit(-1);
    }

    //continuously read from kernel
    while(1){
    	//reading from the fifo that was sent from kernel space module
    	read(kernel_fifo_id, &temp, sizeof(int));

    	alert_sig = temp;

    	printf("\nAlert signal is: %d\n", alert_sig);

		/* PROTOTYPE BLOCK
			ARBITRARY RESPONSE */
		if(response() == 0){
			printf("\nSomething went wrong responding to threat.\n");
		}
		/* END OF PROTOTYPE BLOCK*/
	}
	pthread_exit(0);
}

//calculates the linear regression of the incoming data
// ADC only deals with 1 channel as to simplify this project
// thus we read in from only channel 1
// first takes in data for the first 100 values to
//	create a baseline model and set thresholds
//	then each incoming value will be evaluated
//	if it breaches the threshold, then set the software interrupt
//
//	Simple Linear Regression: y = b_0 + x*b_1
//	the independent variable x is time
//	the dependent variable y is the sensor value
//	b_0 is the intercept, b_1 is the slope
void* linear_regression(){

	RT_TASK* task1 = rt_task_init(nam2num("lnr"), 0, 512, 256);
	double init_sensor_collection[SAMPLE_SIZE];
	double init_time_collection[SAMPLE_SIZE];
	struct timeval time_buffer;
	double slope, intercept = 0.0;
	double x_sum, y_sum, xy_sum, x_square_sum = 0.0;
	double temp_time, temp_sensor = 0.0;
	int i;

	//set timing for task
	//argumentS: task name, start time, period
	rt_task_make_periodic(task1, rt_get_time()+2*timer, LR_PERIOD*timer);

	while(1){
		//init block
		slope = 0.0;
		intercept = 0.0;
		x_sum = 0.0;
		y_sum = 0.0;
		xy_sum = 0.0;
		x_square_sum = 0.0;

		gettimeofday(&time_buffer, NULL);
		SCALE_DOWN_TIME = timeval_to_double(time_buffer) - BASELINE_TIME;

		if(alert_sig == 0){				//init the linear regression model
										//when there are no threat
			//collect the initial data, 100 samples
			//for both the time and the sensor data
			for(i=0;i<SAMPLE_SIZE;i++){
				init_sensor_collection[i] = CALIBRATE_TO_INT*read_ADC_channel(WORKING_CHANNEL);
				gettimeofday(&time_buffer, NULL);
				init_time_collection[i] = SCALE_UP_TIME*(timeval_to_double(time_buffer) - SCALE_DOWN_TIME);

				//LOOK AT THIS DEBUG LINE TO DETERMINE HOW MUCH OFFSET
				//FOR TIME STAMP
				printf("\nsensor: %lf, time: %lf\n", init_sensor_collection[i], init_time_collection[i]);

				//calculate the sum terms
				x_sum += init_time_collection[i];
				y_sum += init_sensor_collection[i];
				xy_sum += (init_time_collection[i]*init_sensor_collection[i]);
				x_square_sum = (init_time_collection[i]*init_time_collection[i]);
			}

			//calculate slope
			slope = SCALE_UP_SLOPE*(((SAMPLE_SIZE*xy_sum) - (x_sum*y_sum))/((SAMPLE_SIZE*x_square_sum) - (x_sum*x_sum)));

			//calulate intercept
			intercept = (y_sum - slope*x_sum)/SAMPLE_SIZE;

			printf("\nINIT slope = %lf, intercept = %lf\n", slope, intercept);
		}


		while(alert_sig == 0){				//continue detecting threat while
											//there is nothing detected yet
			temp_sensor = CALIBRATE_TO_INT*read_ADC_channel(WORKING_CHANNEL);
			gettimeofday(&time_buffer, NULL);
			temp_time = SCALE_UP_TIME*(timeval_to_double(time_buffer) - SCALE_DOWN_TIME);
			//re-calculate the sum terms
			//removing the oldest term, and calculate first
			x_sum += temp_time - init_time_collection[0];
			y_sum += temp_sensor - init_sensor_collection[0];
			xy_sum += (temp_time*temp_sensor) - (init_time_collection[0]*init_sensor_collection[0]);
			x_square_sum = (temp_time*temp_time) - (init_time_collection[0]*init_time_collection[0]);

			//rearrange the array, pop out oldest element and push in new one
			for(i=0;i<(SAMPLE_SIZE-1);i++){
				init_time_collection[i] = init_time_collection[i+1];
				init_sensor_collection[i] = init_sensor_collection[i+1];
			}
			init_time_collection[SAMPLE_SIZE-1] = temp_time;
			init_sensor_collection[SAMPLE_SIZE-1] = temp_sensor;

			//re-calculate slope
			slope = SCALE_UP_SLOPE*(((SAMPLE_SIZE*xy_sum) - (x_sum*y_sum))/((SAMPLE_SIZE*x_square_sum) - (x_sum*x_sum)));

			//re-calulate intercept
			intercept = (y_sum - slope*x_sum)/SAMPLE_SIZE;
			//printf("\nINIT slope = %lf, intercept = %lf\n", slope, intercept);

			printf("\nslope = %lf, intercept = %lf\n", slope, intercept);

			if(slope >= TRIGGER_SLOPE){		//THREAT DETECTED
				//get time stamp
				gettimeofday(&time_buffer, NULL);
				old_time = timeval_to_double(time_buffer);

				//triggers software interrupt
				*VIC2SoftInt = 0x80000000;
			}

			rt_task_wait_period();
		}

		//let go of the CPU for 10 milliseconds to allow other
		//processes a chance to run
		usleep(10000);
	}


	pthread_exit(0);
}


/************************************************************************
 *DESCRIPTION: Figure out the calibration value of the board and store them
 *stored_cal global variable. In eeprom 2027->2046 are the calibration values. 2027->2036
*are the zero volt calibration values and 2037->2046 are the 2.5V calibration
*values. Each calibration value is 16 bits written using little endian
 ***********************************************************************/
static void init_adc_channel_calibration(){
	int i, j, ch, addr;
	char buffer[20];

	/* intialize the eeprom */
	POKE16(pld_page, (PEEK16(pld_page) & ~CS_MSK)); //disable CS
	POKE32((spistart + SSPCR1), 0x10);  //turn on transmit
	while ((PEEK32(spistart + 0xc) & 0x10) == 0x10); // wait for unbusy
	while ((PEEK32(spistart + 0xc) & 0x5) != 0x1);   // empty FIFO's
	POKE16(pld_page, (PEEK16(pld_page) | CS_MSK));   //enable CS

	POKE32(spistart, 0xc7);                          // set SPI mode 3, 8bit
	POKE32(spistart + 0x10, 0x2);                    // divide clk by 2
	POKE32(spistart + 0x4, 0x0);                     // stop transmit

	while((ee_rdsr(dr_page) & 0x1) == 0x1);//wait for unbusy

	ee_wren(dr_page);
	ee_wrsr(dr_page, 0x0);              // make eeprom R/W

	while((ee_rdsr(dr_page) & 0x1) == 0x1);//wait for unbusy

	/* Read in the calibration values */
	//printf("Calibration Values = \n[ ");

	addr = CALIB_LOC;
	for(i = 0; i < 20; i++)
	{
		ee_read_byte(dr_page, addr, &buffer[i]);

		//check if board has stored calibration values
		if(buffer[i] != 0xFF)
			virgin = FALSE;
		addr++;
	}

	//convert to 16 bit values
	j = 0;
	ch = 0;
	for(i = 0; i < 20; i = i + 2)
	{
		if(i == 10)
		{
			ch = 0;
			j = 1;
		}

		stored_cal[ch][j] = (buffer[i] | (buffer[i+1] << 8));
		//printf("(%d, %d) = 0x%x | ", ch, j, stored_cal[ch][j]);
		ch++;
	}
	//printf(" ]\n");

	//make eeprom RO
	ee_wren(dr_page);
	ee_wrsr(dr_page, 0x1c);
}

/******************************************************************************
*DESCRIPTION: Read the data from the desired channel before calibrating it accordingly
*******************************************************************************/
double read_ADC_channel(int ch_num){
	double val, full_scale;
	int cur_ch;
	int result;

	//printf("\nStored Calibration: %x %x\n", stored_cal[ch_num][0], stored_cal[ch_num][1]);

	/* intialize the eeprom */
	POKE16(pld_page, (PEEK16(pld_page) & ~CS_MSK)); //disable CS
	POKE32((spistart + SSPCR1), 0x10);  //turn on transmit
	while ((PEEK32(spistart + 0xc) & 0x10) == 0x10); // wait for unbusy
	while ((PEEK32(spistart + 0xc) & 0x5) != 0x1);   // empty FIFO's
	POKE16(pld_page, (PEEK16(pld_page) | CS_MSK));   //enable CS

	POKE32(spistart, 0xc7);                          // set SPI mode 3, 8bit
	POKE32(spistart + 0x10, 0x2);                    // divide clk by 2
	POKE32(spistart + 0x4, 0x0);                     // stop transmit

	while((ee_rdsr(dr_page) & 0x1) == 0x1);//wait for unbusy

	ee_wren(dr_page);
	ee_wrsr(dr_page, 0x0);              // make eeprom R/W

	while((ee_rdsr(dr_page) & 0x1) == 0x1);//wait for unbusy

	//select channel address
	switch(ch_num)
	{
		case 0:
			cur_ch = ADC_CH0;
		break;
		case 1:
			cur_ch = ADC_CH1;
		break;
		case 2:
			cur_ch = ADC_CH2;
		break;
		case 3:
			cur_ch = ADC_CH3;
		break;
		case 4:
			cur_ch = ADC_CH4;
		break;
	}

	//Read the adc values for all 5 channels
	//read_7xxx_adc(adc_result);
	//discard first two samples
	read_channel(adc_page, cur_ch);
	read_channel(adc_page, cur_ch);
	usleep(10000);
	result = read_channel(adc_page, cur_ch);

	/*if(virgin == TRUE)
		printf("No calibration values found...\n");
	else
		printf("Board has been calibrated by Technologic Systems...\n");
	*/
	full_scale = (((((double)(stored_cal[ch_num][1] + 0x10000)
		- stored_cal[ch_num][0]) / 2.5 ) * 3.3 ));

	if(result < 0x7000)
		result = result + 0x10000;

	if(virgin == TRUE)  //use approximation
	{
		result = result - 0x9E58;
		val = ((double)result * 3.3) / 0xC350;
	}
	else                //use calibration values
	{
		result = result - stored_cal[ch_num][0];
		val = ((double)result * 3.3) / full_scale;
	}

	//printf("Channel %d: %3.3fV\n", i, val);

	//make eeprom RO
	ee_wren(dr_page);
	ee_wrsr(dr_page, 0x1c);

	return val;
}

//CONVERT TIMEVAL STRUCTURE TO DOUBLE FOR EASIER PROCESSING
//COMPRESS TO FORMAT: (double) <tv_sec>.<usec>
double timeval_to_double(struct timeval in){
	double out = 0.0;
	//double i = ((double)(in.tv_usec))/1000000;
	out = in.tv_sec+(((double)(in.tv_usec))/1000000);
	//printf("\nCONVERTING TO DOUBLE: usec is %lf out time is %lf\n", i, out);
	return out;
}

void error(const char *msg){
    perror(msg);
    exit(0);
}

//convert string of ip address into integer, replacing '.' wit 0s
int IPToInt(char* ip_addr) {

    int i = 0;
    char buffer_ip[100];
    int ip_num = -1;
    for (i = 0; i < strlen(ip_addr); i++) {
        //change all the '.' to 0's
        if (ip_addr[i] == '.') {
            buffer_ip[i] = '0';
        } else {
            buffer_ip[i] = ip_addr[i];
        }
    }
    buffer_ip[i] = '\0';
    ip_num = atoi(buffer_ip);  //this is used to compare the host ip with others
                                //in case of tie vote
    return ip_num;
}

//returns either 0 or 1, which is the content of the particular bit
//bitwise "AND" operation to the number shifted to the number of bitPosition
//and the bitvector 0000 0001, thus what will be left is either 1 or 0.
//return -1 if error occurs
int getBitContent(int bitPosition, int content){

	int mask = 0x01;
	int shifted = 0;
	int result = -1;

	shifted = content >> bitPosition;
	result = shifted & mask;
	//printk("\nResult at bit %d of %d is %d\n", bitPosition, content, result);
	return result;
}

//returns the content that had been assigned value at bitPosition
//to set to 1, we use the "OR" operation
//to set to 0, we use the "XOR" operation
// but to check that setting to 0 is valid or not, we first "AND"
// with 0x** to see if the result is 0x** or 0x00. If 0x00, do nothing.
// return -1 in case of error
int assignBitContent(int bitPosition, int content, int value){
	int bit_zero = 0x01;
	int bit_one = 0x02;
	int bit_two = 0x04;
	int bit_three = 0x08;
	int bit_four = 0x10;
	int bit_five = 0x20;
	int bit_six = 0x40;
	int bit_seven = 0x80;
	int mask_bit = 0;
	int modified_content = -1;
	int checkContent = -1;

	switch (bitPosition){
		case 0:
			mask_bit = bit_zero;
			break;
		case 1:
			mask_bit = bit_one;
			break;
		case 2:
			mask_bit = bit_two;
			break;
		case 3:
			mask_bit = bit_three;
			break;
		case 4:
			mask_bit = bit_four;
			break;
		case 5:
			mask_bit = bit_five;
			break;
		case 6:
			mask_bit = bit_six;
			break;
		case 7:
			mask_bit = bit_seven;
			break;
		default:
			//printk("\nThe fuck, dude?\n");
			return -1;
	}

	if(value == 1){

		//turn on by using OR
		modified_content = content | mask_bit;
		//printk("\nAssigning to 1\n");
	}else if(value == 0){

		//check if we can turn it off or not
		checkContent = content & mask_bit;

		//if that particular bit is already 0, then nothing to do
		if(checkContent == 0x00){
			modified_content = content;
			//printk("\nAlready 0\n");
		} else if (checkContent == mask_bit) {

			//if that particular bit is 1, then we can turn it to 0 with XOR
			modified_content = content ^ mask_bit;
			//printk("\nAssigning to 0\n");
		}
	} else{
		//printk("\nContent can only be 1 or 0. Twat.\n");
		return -1;
	}

	//printk("\nYou chose to turn bit %d to %d from %d the result is %d\n", bitPosition, value, content, modified_content);
	return modified_content;
}

