#include <SPI.h>

/*
    Network / bluetooth communication setup 
*/
const boolean LIGHT_DEBUG = false;
const int bufferLength = 100;
const int MAX_COMMANDS = 100;

char spiBuffer[bufferLength];
char serialBuffer[bufferLength];
volatile byte spiPosition;
volatile boolean process_it = false;

int redLedStatus = 0;
int greenLedStatus = 0;
int blueLedStatus = 0;
int externalA0LedStatus=0;
int externalA1LedStatus=0;

long softwareCommandStart = 0;
int softwareGoalValue = -1;
long softwareDimmingSpeedDivisor = 3;

//Command typedef, currently device is unused
typedef struct commandStruct{
    struct commandStruct* nextCommand = NULL;
    int id;
    int device;
    int value;
};

struct commandStruct* headCommand = NULL;

/*
    Physical Buttons setup
*/
const int button1Pin = 0;    
const int button2Pin = 1;    

int dimmingDirection = 0;

int ledBrightness = 0;
int currentBrightness = 0;
int brightnessChange = 0;

int button1State;
int button2State;
int lastButton1State = LOW;
int lastButton2State = LOW;

long lastPressTime = 0;

long debounceDelay = 50; 
long dimmingDelay = 250;
long dimmingSpeedDivisor = 7;

/*
    Function declarations
*/
void setLedBrightness(int);
void toggleRedLed();
void toggleGreenLed();
void toggleBlueLed();
void toggleA0Led();
void toggleA1Led();
void handleMessage(char[], bool);
void handleCommand(int, int, int, bool);
bool isPastCommand(int);
void saveCommand(struct commandStruct*);
void performAction(int, int, bool);


// Falling edge Slave Select interrupt, start of transaction, no command yet.
void ss_falling (){

} 

void setup (void){
    /*
        Network setup
    */
    Serial.begin (57600);
    
    //Enable bean advertising
    Bean.enableAdvertising(true);

    pinMode(A0, OUTPUT);
    pinMode(A1, OUTPUT);

    // have to send on master in, *slave out*
    pinMode(MISO, OUTPUT);

    // turn on SPI in slave mode
    SPCR |= _BV(SPE);

    // turn on interrupts
    SPCR |= _BV(SPIE);

    // get ready for an interrupt 
    spiPosition = 0;
    process_it = false;

    // interrupt for SS falling edge
    attachInterrupt (0, ss_falling, FALLING);

    /*
        Physical Buttons Setup
    */
    pinMode(button1Pin, INPUT);
    pinMode(button2Pin, INPUT);

    //Initialize led brightness
    Bean.setLedRed(ledBrightness);
}


// SPI interrupt routine
ISR (SPI_STC_vect){
    // grab byte from SPI Data Register
    byte c = SPDR; 

    // add to buffer if room
    if (spiPosition < sizeof spiBuffer)
    {
        spiBuffer[spiPosition++] = c;

        // newline character is termination character
        if (c == '\n')
            process_it = true;

    }
}

// main loop - wait for flag set in interrupt routine
void loop (void){
    //Bluetooth message
    if(Serial.available() > 0){
        /*
        if (LIGHT_DEBUG){
            toggleA1Led();
        }
        */
      
        Serial.readBytesUntil('\n',serialBuffer,bufferLength);
        //Handle bluetooth message
        handleMessage(serialBuffer, false);
        memset(serialBuffer, 0, sizeof(serialBuffer));
    }
  
    //SPI Message
    if (process_it){
        spiBuffer[spiPosition] = 0;  
        //Handle SPI message
        handleMessage(spiBuffer, true);

        memset(spiBuffer, 0, sizeof(spiBuffer));
        spiPosition = 0;
        process_it = false;
    }

    //Read button states
    int button1Reading = digitalRead(button1Pin);
    int button2Reading = digitalRead(button2Pin);

    //Assuming button press is high and not already going the other direction
    if (button1Reading == HIGH && (dimmingDirection == 0 || dimmingDirection == 1)){
        //Record button press if it changes from low to high
        if (button1Reading != lastButton1State) {
            lastPressTime = millis();
        } 

        long delay = millis() - lastPressTime;
        if(delay > debounceDelay){
            dimmingDirection=1;
        }
        if(delay > dimmingDelay){
            //calculate brightness adjustement based on current brightness and direction
            long timeAfter = delay - dimmingDelay;
            brightnessChange = (int) (timeAfter / dimmingSpeedDivisor);
            currentBrightness = min(255,ledBrightness + brightnessChange);
            
            setLedBrightness(currentBrightness);
        }
    }
    else if (button2Reading == HIGH && (dimmingDirection == 0 || dimmingDirection == -1)){
        //Record button press if it changes from low to high
        if (button2Reading != lastButton2State) {
            lastPressTime = millis();
        } 

        long delay = millis() - lastPressTime;
        if(delay > debounceDelay){
            dimmingDirection=-1;
        }
        if(delay > dimmingDelay){
            //calculate brightness adjustement based on current brightness and direction
            long timeAfter = delay - dimmingDelay;
            brightnessChange = (int) -(timeAfter / dimmingSpeedDivisor);
            currentBrightness = max(0,ledBrightness + brightnessChange);
            
            setLedBrightness(currentBrightness);
        }
    }
    else {
        //Finished dimming, brightening, or with button press
        ledBrightness = min(255,ledBrightness+brightnessChange);
        //If a button state changed, we must perform an action
        if (button1Reading != lastButton1State || button2Reading != lastButton2State){
            long delay = millis() - lastPressTime;
            //If shorter than dimming delay, turn on or off light respectively based on dimming direction
            if (delay > debounceDelay && delay < dimmingDelay){
                if (dimmingDirection == 1){
                    ledBrightness = 255;
                }
                else if (dimmingDirection == -1){
                    ledBrightness=0;
                }
                setLedBrightness(ledBrightness);
                brightnessChange = 0;
                dimmingDirection = 0;
            }
            //Else reset values and record currentBrightness as ledBrightness since the dimming is complete
            else if(delay > debounceDelay){
                ledBrightness = currentBrightness;
                brightnessChange = 0;
                dimmingDirection = 0;
            }
        }

    }

    //Save current state of buttons for next loop
    lastButton1State = button1Reading;
    lastButton2State = button2Reading;

    //Software dimming as long as hardware isn't currently dimming
    if (dimmingDirection == 0 && softwareGoalValue != -1){
        //Software goal set and no hardware request present. Dim.
        long currentTime = millis();
        long timeAfter = currentTime - softwareCommandStart;
        int softwareBrightnessChange = (int) (timeAfter / softwareDimmingSpeedDivisor);
        if (ledBrightness == softwareGoalValue){
            softwareGoalValue = -1;
        }
        else if (softwareBrightnessChange >= 1){
            //Move software command start time forward
            softwareCommandStart=currentTime;
            //Increasing brightness
            if (ledBrightness < softwareGoalValue){
                ledBrightness = min(255,ledBrightness+softwareBrightnessChange);
                if (ledBrightness >= softwareGoalValue){
                    softwareGoalValue = -1;
                }
                setLedBrightness(ledBrightness);
            }
            //Decreasing brightness
            else if (ledBrightness > softwareGoalValue){
                ledBrightness = max(0,ledBrightness-softwareBrightnessChange);
                if (ledBrightness <= softwareGoalValue){
                    softwareGoalValue = -1;
                }
                setLedBrightness(ledBrightness);
            }
        }
    }
    //Hardware currently dimming, ignore.
    else{
        //Discard all software goals that have been set once a hardware press has occured.
        softwareGoalValue = -1;
    }
}

void setLedBrightness(int brightness){
    Bean.setLedRed(brightness);
    return;
}

//Toggle red led
void toggleRedLed(){
    if (redLedStatus == 0){
        Bean.setLedRed(255);
        redLedStatus = 1;
    }
    else{
        Bean.setLedRed(0);
        redLedStatus = 0;
    }
    return;
}

//Toggle green led
void toggleGreenLed(){
    if (greenLedStatus == 0){
        Bean.setLedGreen(255);
        greenLedStatus = 1;
    }
    else{
        Bean.setLedGreen(0);
        greenLedStatus = 0;
    }
    return;
}

//Toggle blue led
void toggleBlueLed(){
    if (blueLedStatus == 0){
        Bean.setLedBlue(255);
        blueLedStatus = 1;
    }
    else{
        Bean.setLedBlue(0);
        blueLedStatus = 0;
    }
    return;
}

//Toggle led hooked up to line A0
void toggleA0Led(){
    if (externalA0LedStatus == 0){
        analogWrite(A0,255);
        externalA0LedStatus = 1;
    }
    else{
        analogWrite(A0,0);
        externalA0LedStatus = 0;
    }
    return;
}

//Toggle led hooked up to line A1
void toggleA1Led(){
    if (externalA0LedStatus == 0){
        analogWrite(A1,255);
        externalA0LedStatus = 1;
    }
    else{
        analogWrite(A1,0);
        externalA0LedStatus = 0;
    }
    return;
}

void handleMessage(char buffer[100], bool fromSPI){
    //Split command received based on spaces
    char** splitArray = NULL;
    char* deliminator = strtok(buffer, " ");
    int nSpaces = 0;
    int i=0;

    while (deliminator) {
        splitArray = (char**) realloc (splitArray, sizeof (char*) * ++nSpaces);

        if (splitArray == NULL)
            return;

        splitArray[nSpaces-1] = deliminator;

        deliminator = strtok (NULL, " ");
    }

    splitArray = (char**) realloc (splitArray, sizeof (char*)* (nSpaces+1));
    splitArray[nSpaces] = "\n";

    //If the number of objects in the command was 2, valid command.
    if (nSpaces == 2){
        int id = atoi(splitArray[0]);
        int device = 0;
        int value = atoi(splitArray[1]);
        //Valid, handle command
        handleCommand(id, device, value, fromSPI);
    }

    /*
    if (LIGHT_DEBUG){
        if (nSpaces == 2){
            toggleRedLed();
        }
    }
    */

    //Free array after used
    free(splitArray);
    return;
}

void handleCommand(int id, int device, int value, bool fromSPI){
    //Check to make sure command hasn't already been received
    if (!isPastCommand(id)){
        //Saving command with struct
        struct commandStruct* command = new struct commandStruct;
        command->id = id;
        command->device = device;
        command->value = value;

        //Since new command, save in linked list of past commands.
        saveCommand(command);

        //Perform the action specified by the command
        performAction(device, value, fromSPI);
    }

    return;
}

//Function to loop through linked list and check to see if the past command has been received before based on id
//Returns true if it is a past command, and false if it is a new command
bool isPastCommand(int id){
    struct commandStruct* currentCommand = headCommand;
    if (currentCommand == NULL){
        return false;
    }
    else {
        while (currentCommand != NULL){ 
            //current command not null, bump to next command or return true if matching id
            if (id == currentCommand->id){
                return true;
            }
            else {
                currentCommand = currentCommand->nextCommand;
            }
        }
    }
    //Command not found, return false
    return false;
}

//Function to save command to linked list of commands
void saveCommand(struct commandStruct* command){
    struct commandStruct* currentCommand = headCommand;
    int numberOfCommands = 0;

    //Save as head if no head
    if (currentCommand == NULL){
        headCommand = command;
    }
    else {
        numberOfCommands++;
        //Bump to end of linked list
        while (currentCommand->nextCommand != NULL){ 
            numberOfCommands++;
            currentCommand = currentCommand->nextCommand;
        }
        //Set nextCommand to the new command
        currentCommand->nextCommand = command;

        //Maintain a linked list with a maximum size of commands
        if (numberOfCommands > MAX_COMMANDS){
            if (LIGHT_DEBUG){
                toggleGreenLed();
            }
            struct commandStruct* commandToFree = headCommand;
            headCommand = headCommand->nextCommand;
            //Free old head since our linked list has reached our maximum size.
            free(commandToFree);
        }
    }

    return;
}

//Function for performing an action on a device with a given requested brightness level.
void performAction(int device, int value, bool fromSPI){
    //Toggle correct LED to indicate what interface the command was received with
    if (fromSPI){
        toggleA0Led();
    }
    else {
        toggleA1Led();
    }

    if (LIGHT_DEBUG){
        toggleRedLed();
    }

    //Software command, save current time
    softwareCommandStart = millis();
    //Set softwareGoalValue, keep within 255 and 0
    softwareGoalValue = value;
    softwareGoalValue = min(255, softwareGoalValue);
    softwareGoalValue = max(0, softwareGoalValue);
    return;
}
