package net.quarkworks.bluetoothlightcontrol;

import android.os.Bundle;

import android.support.v4.app.Fragment;
import android.util.Log;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.Button;
import android.widget.SeekBar;

import java.io.UnsupportedEncodingException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;

import nl.littlerobots.bean.Bean;
import nl.littlerobots.bean.BeanListener;
import nl.littlerobots.bean.BeanManager;

/**
 * Created by benjamin on 5/3/15.
 */
public class ParentFragment extends Fragment {
    private static final String TAG = ParentFragment.class.getSimpleName();

    /*
        Constants
     */
    //Port to listen on
    public int COM_PORT = 5000;
    //Threshold for seekbar to send a packet (send every 10 increments in this case)
    public int SEEKBAR_THRESHOLD = 10;

    /*
        References
     */
    private Button toggleButton;
    private Button testButton;
    private SeekBar seekBar;
    private ArrayList<Bean> discoveredBeans = new ArrayList<Bean>();
    private ArrayList<Bean> connectedBeans = new ArrayList<Bean>();
    private Button disconnectButton;
    private Button connectButton;

    /*
        Data
     */

    private int messageId = 0;
    private int lastSeekbarProgress = 0;


    public ParentFragment() {
    }


    @Override
    public void onStart() {
        super.onStart();
    }

    @Override
    public View onCreateView(LayoutInflater inflater, ViewGroup container,
                             Bundle savedInstanceState) {
        View view = inflater.inflate(R.layout.parent_fragment, container, false);


        //Grab views
        toggleButton = (Button) view.findViewById(R.id.toggle_button);
        disconnectButton = (Button) view.findViewById(R.id.disconnect_button);
        connectButton = (Button) view.findViewById(R.id.connect_button);
        seekBar = (SeekBar) view.findViewById(R.id.brightness_seekbar);

        //Set seekbar increments to 1 with a max value of 255
        seekBar.setKeyProgressIncrement(1);
        seekBar.setMax(255);

        seekBar.setOnSeekBarChangeListener(new SeekBar.OnSeekBarChangeListener() {
            @Override
            public void onProgressChanged(SeekBar seekBar, int progress, boolean fromUser) {
                //Assuming user moved seekbar, and greater or less than threshold, send command
                if (progress - lastSeekbarProgress >= SEEKBAR_THRESHOLD || progress - lastSeekbarProgress <= -SEEKBAR_THRESHOLD && fromUser){
                    lastSeekbarProgress = progress;
                    sendBrightnessCommand(progress);
                }
            }

            @Override
            public void onStartTrackingTouch(SeekBar seekBar) {
                //On press track starting position for use with threshold
                lastSeekbarProgress = seekBar.getProgress();
            }

            @Override
            public void onStopTrackingTouch(SeekBar seekBar) {
                //Send brightness command on release
                Log.d(TAG, "Seekbar Progress: " + seekBar.getProgress());
                sendBrightnessCommand(seekBar.getProgress());
            }
        });

        //Toggle light on click listener
        toggleButton.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                //Move progress bar based on current position to the opposite side of current position
                if (seekBar.getProgress() > 127){
                    seekBar.setProgress(0);
                }
                else {
                    seekBar.setProgress(255);
                }

                sendBrightnessCommand(seekBar.getProgress());
            }
        });


        //Disconnect from bluetooth button listener
        disconnectButton.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                for (final Bean bean:connectedBeans){
                    bean.disconnect();
                    connectedBeans.remove(bean);
                }
            }
        });

        //Connect to bluetooth button listener
        connectButton.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                BeanManager.getInstance().startDiscovery(((ParentActivity) getActivity()).listener);
            }
        });



        return view;
    }


    //Send a command to the server with a unique id. Also included in this is the brightness level
    public void sendBrightnessCommand(int brightnessLevel){
        String message = messageId%999+1 + " " + brightnessLevel + "\n";
        messageId++;

        Log.d(TAG, "Sending Message: " + message);
        //Create a new broadcast object
        UDPBroadcast broadcast = new UDPBroadcast(getActivity(), COM_PORT);
        //Send the message via UDP
        broadcast.sendMessage(message);

        //Log.d(TAG, "Sent message: " + message);

        Log.d(TAG, "number of connected beans: " + connectedBeans.size());
        for (final Bean bean:connectedBeans){
            //For all beans, if connected send a bluetooth message to the bean.
            if (bean.isConnected()) {
                Log.d(TAG, "Sending BLE Message: " + message);
                byte[] messageBytes = new byte[0];
                try {
                    messageBytes = message.getBytes("UTF-8");
                } catch (UnsupportedEncodingException e) {
                    e.printStackTrace();
                }
                bean.sendSerialMessage(messageBytes);
            }
        }
    }

    //On bean discovered listener. Log statements describe each function
    public void onBeanDiscovered(final Bean bean){
        Log.d(TAG, "Bean discovered: " + bean.getDevice());
        discoveredBeans.add(bean);
        BeanListener beanListener = new BeanListener() {
            @Override
            public void onConnected() {
                connectedBeans.add(bean);
                Log.d(TAG, "Successfully connected to device: " + bean.getDevice());
                //bean.disconnect();
            }

            @Override
            public void onConnectionFailed() {
                Log.d(TAG, "Connection failed to device: " + bean.getDevice());
            }

            @Override
            public void onDisconnected() {
                Log.d(TAG, "Disconnected from device: " + bean.getDevice());
                //connectedBeans.remove(connectedBeans.indexOf(bean));
            }

            @Override
            public void onSerialMessageReceived(byte[] bytes) {
                String string = "";
                try {
                    string = new String(bytes, "UTF-8");
                } catch (UnsupportedEncodingException e) {
                    e.printStackTrace();
                }

                Log.d(TAG, "Serial message received: " + string);
            }

            @Override
            public void onScratchValueChanged(int i, byte[] bytes) {
                Log.d(TAG, "Scratch value changed, i: " + i + ", bytes: " + bytes.toString());
            }
        };
        bean.connect(getActivity(), beanListener);
    }

    public void onDiscoveryComplete(){

        Log.d(TAG, "onDiscoveryComplete");
    }

    //Unique id, unused currently
    public int getUniqueInt(){
        SimpleDateFormat df = new SimpleDateFormat("DDDhhmmss");
        Date date = new Date();

        String dateString = "1" + df.format(date);
        int uniqueInt = Integer.parseInt(dateString);
        Log.d(TAG, "UniqueInt: " + uniqueInt);

        return uniqueInt;
    }
}
