
 
#include <linux/init.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/gpio.h>                 // Required for the GPIO functions
#include <linux/interrupt.h>            // Required for the IRQ code
#include <linux/fs.h>
#include <asm/segment.h>
#include <asm/uaccess.h>
#include <linux/buffer_head.h>
#include <linux/wait.h>

#define  DEVICE_NAME "looperFIFO"    ///< The device will appear at /dev/looperFIFO using this value
#define  CLASS_NAME  "loops"        ///< The device class -- this is a character device driver

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Sean ODay");
MODULE_DESCRIPTION("A Button driver for the BBB looper");
MODULE_VERSION("0.1");
 
static unsigned int gpioRecord = 49;       
static unsigned int gpioPlay = 117; 
static unsigned int gpioExit = 115;   //GPIOs used for buttons
static unsigned int irqRecord;//irqs to be set  on init
static unsigned int irqPlay;
static unsigned int irqExit;

//this wait queue is for synchronization of the read function and will put the calling thread into a sleep until data is available
wait_queue_head_t my_queue; 
int intflag;//to be used as boolean value for the wait cue

static int    majorNumber;                  ///< Stores the device number -- determined automatically
static char   message[256] = {0};           ///< Memory for the string that is passed from userspace

static short  size_of_message;              ///< Used to remember the size of the string stored
static int    numberOpens = 0;              ///< Counts the number of times the device is opened
static struct class *loopClass  = NULL; ///< The device-driver class struct pointer
static struct device *loopDevice = NULL; ///< The device-driver device struct pointer
 
// The prototype functions for the character driver -- must come before the struct definition
static int     dev_open(struct inode *, struct file *);
static int     dev_release(struct inode *, struct file *);
static ssize_t dev_read(struct file *, char *, size_t, loff_t *);
static ssize_t dev_write(struct file *, const char *, size_t, loff_t *);
          
 
 
/// Function prototype for the custom IRQ handler function -- see below for the implementation
static irq_handler_t  record_handler(unsigned int irq, void *dev_id, struct pt_regs *regs);
static irq_handler_t  play_handler(unsigned int irq, void *dev_id, struct pt_regs *regs);
static irq_handler_t  exit_handler(unsigned int irq, void *dev_id, struct pt_regs *regs);

//maps the functions into the fops struct, this ties the functions created here to the functions in user space
static struct file_operations fops =
{
   .open = dev_open,
   .read = dev_read,
   .write = dev_write,
   .release = dev_release,
};
 

static int __init buttongpio_init(void){
   int result = 0;
   //check that the gpios are valid
   printk(KERN_INFO "GPIO_TEST: Initializing the GPIO_TEST LKM\n");
   // Is the GPIO a valid GPIO number (e.g., the BBB has 4x32 but not all available)
   if (!gpio_is_valid(gpioRecord)){
      printk(KERN_INFO "Invalid GPIO\n");
      return -ENODEV;
   }
   
   if (!gpio_is_valid(gpioPlay)){
      printk(KERN_INFO "Invalid GPIO\n");
      return -ENODEV;
   }
   
   if (!gpio_is_valid(gpioExit)){
      printk(KERN_INFO "Invalid GPIO\n");
      return -ENODEV;
   }
  
   //sets the direction and debounce of the gpios, also exports a low value for them
   gpio_request(gpioRecord, "sysfs");        
   gpio_direction_input(gpioRecord);   
   gpio_set_debounce(gpioRecord, 200);
   gpio_export(gpioRecord, false); 
    
   gpio_request(gpioPlay, "sysfs");       
   gpio_direction_input(gpioPlay);        
   gpio_set_debounce(gpioPlay, 200);     
   gpio_export(gpioPlay, false);          
   
   gpio_request(gpioExit, "sysfs");       
   gpio_direction_input(gpioExit);       
   gpio_set_debounce(gpioPlay, 200);     
   gpio_export(gpioPlay, false);   
   
  
 
   // returns the irq numbers for the gpios
   irqRecord = gpio_to_irq(gpioRecord);
   irqPlay = gpio_to_irq(gpioPlay);
   irqExit = gpio_to_irq(gpioExit);
   printk(KERN_INFO "GPIO_TEST: The button is mapped to IRQ: %d\n", irqRecord);
 
   // This next call requests an interrupt line
   result = request_irq(irqRecord,(irq_handler_t) record_handler,IRQF_TRIGGER_RISING,"recordGPIO_handler",NULL); 
   result = request_irq(irqPlay,(irq_handler_t) play_handler,IRQF_TRIGGER_RISING,"playGPIO_handler",NULL); 
   result = request_irq(irqExit,(irq_handler_t) exit_handler,IRQF_TRIGGER_RISING,"exitGPIO_handler",NULL); 
   
 
     
   
   printk(KERN_INFO "Loop: Initializing the Loop LKM\n");
 
   // allocate major number for device driver
   majorNumber = register_chrdev(0, DEVICE_NAME, &fops);
   if (majorNumber<0){
      printk(KERN_ALERT "Loop failed to register a major number\n");
      return majorNumber;
   }
   printk(KERN_INFO "Loop: registered correctly with major number %d\n", majorNumber);
 
   // Register the device class
   loopClass = class_create(THIS_MODULE, CLASS_NAME);
   if (IS_ERR(loopClass)){                // Check for error and clean up if there is
      unregister_chrdev(majorNumber, DEVICE_NAME);
      printk(KERN_ALERT "Failed to register device class\n");
      return PTR_ERR(loopClass);          // Correct way to return an error on a pointer
   }
   printk(KERN_INFO "loop: device class registered correctly\n");
 
   // Register the device driver
   loopDevice = device_create(loopClass, NULL, MKDEV(majorNumber, 0), NULL, DEVICE_NAME);
   if (IS_ERR(loopDevice)){               // Clean up if there is an error
      class_destroy(loopClass);           // Repeated code but the alternative is goto statements
      unregister_chrdev(majorNumber, DEVICE_NAME);
      printk(KERN_ALERT "Failed to create the device\n");
      return PTR_ERR(loopDevice);
   }
   //initialize the wait queue variable
   init_waitqueue_head(&my_queue);
  
   
   printk(KERN_INFO "loop: device class created correctly\n"); // Made it! device was initialized
   return 0;
   
   return result;
}
 

static void __exit buttongpio_exit(void){
  
   
   free_irq(irqRecord, NULL);              
   free_irq(irqPlay, NULL); 
   free_irq(irqExit, NULL); 
   
   gpio_unexport(gpioRecord);
   gpio_unexport(gpioPlay);
   gpio_unexport(gpioExit);
   
   gpio_free(gpioRecord);                      
   gpio_free(gpioPlay);
   gpio_free(gpioExit);
   
   device_destroy(loopClass, MKDEV(majorNumber, 0));     // remove the device
   class_unregister(loopClass);                          // unregister the device class
   class_destroy(loopClass);                             // remove the device class
   unregister_chrdev(majorNumber, DEVICE_NAME);             // unregister the major number
  
   
   printk(KERN_INFO "Goodbye!\n");
}
 
//interrupt handlers, pretty straightforward, send a character via the message to user space
static irq_handler_t record_handler(unsigned int irq, void *dev_id, struct pt_regs *regs){
  static char buffer = 'R';
   sprintf(message,"%c",buffer);
   //wake up read and set boolean so it will be read by the user space
   wake_up_interruptible(&my_queue);
   intflag=1;
  // printk(KERN_INFO "Record Interrupt! (button state is %d)\n", gpio_get_value(gpioRecord));
   //pipe_write(fd, &buffer,sizeof(buffer));
                      
   return (irq_handler_t) IRQ_HANDLED;      // Announce that the IRQ has been handled correctly
}

static irq_handler_t play_handler(unsigned int irq, void *dev_id, struct pt_regs *regs){
  static char buffer = 'P';
   sprintf(message,"%c",buffer);
   
   wake_up_interruptible(&my_queue);
   intflag=1;
  // printk(KERN_INFO "Play Interrupt! (button state is %d)\n", gpio_get_value(gpioPlay));
  
 
   return (irq_handler_t) IRQ_HANDLED;      // Announce that the IRQ has been handled correctly
}

static irq_handler_t exit_handler(unsigned int irq, void *dev_id, struct pt_regs *regs){
  static char buffer = 'X';
   sprintf(message,"%c",buffer);
  
   wake_up(&my_queue);
   intflag=1;
  // printk(KERN_INFO "Exit Interrupt! (button state is %d)\n", gpio_get_value(gpioExit));
   //pipe_write(fd, &buffer,sizeof(buffer));
   //numberPresses++;                         // Global counter, will be outputted when the module is unloaded
   return (irq_handler_t) IRQ_HANDLED;      // Announce that the IRQ has been handled correctly
}


//function to be called when device is opened, nothing really happens here
static int dev_open(struct inode *inodep, struct file *filep){
   numberOpens++;
   printk(KERN_INFO "loop: Device has been opened %d time(s)\n", numberOpens);
   return 0;
}
 
//read function, the synchronization using the wait queue is key here, 
//calling process will be placed into wait queueuntil data is available

static ssize_t dev_read(struct file *filep, char *buffer, size_t len, loff_t *offset){
   int error_count = 0;
   // copy_to_user has the format ( * to, *from, size) and returns 0 on success
	wait_event_interruptible(my_queue,intflag!=0);
	intflag=0;
	
	error_count = copy_to_user(buffer,&message[0],strlen(message));
	
	
	
	//printk(KERN_INFO "Buffer is %s\n", message);
   if (error_count==0){            // if true then have success
      printk(KERN_INFO "loop: Sent %d characters to the user\n", size_of_message);
      return (size_of_message=0);  // clear the position to the start and return 0
   }
   else {
      printk(KERN_INFO "loop: Failed to send characters to the user\n", error_count);
      return -EFAULT;              // Failed -- return a bad address message (i.e. -14)
   }
}
 
//write function, this will never be used by my program
static ssize_t dev_write(struct file *filep, const char *buffer, size_t len, loff_t *offset){
   //sprintf(message, "%s(%d letters)", buffer, len);   // appending received string with its length
   size_of_message = strlen(message);                 // store the length of the stored message
   printk(KERN_INFO "loop: Received %d characters from the user\n", len);
   return len;
}
 
//release function just states the device is closed
static int dev_release(struct inode *inodep, struct file *filep){
   printk(KERN_INFO "loop: Device successfully closed\n");
   return 0;
}




 
//these calls are mandatory and tie in the init and exit functions
module_init(buttongpio_init);
module_exit(buttongpio_exit);
